#' Rademacher Wild Pairwise Non-Parametric Bootstrap Test (RWPNPBT)
#'
#' Performs a non-parametric bootstrap test using Rademacher wild bootstrap re-sampling.
#' This test compares all pairwise group means using a standardized distance metric,
#' making it robust to violations of normality and heteroscedasticity.
#'
#' The test statistic sums the absolute standardized differences between all pairs of group means.
#' Rademacher weights are applied to centered observations for wild bootstrapping.
#'
#' @param group_list A list where each element is a numeric vector of raw observations for a group.
#' @param B Number of bootstrap re-samples. Default is 100000.
#' @param alpha Significance level for the hypothesis test. Default is 0.05.
#'
#' @return A list of class \code{"RWPNPBT"} containing:
#' \item{test_statistic}{Observed value of the test statistic.}
#' \item{critical_value}{Bootstrap-based critical value at the given alpha level.}
#' \item{decision}{Conclusion of the hypothesis test based on the critical value.}
#'
#' @examples
#' set.seed(123)
#' group1 <- rexp(18, rate = 1/10)     # Exponential distribution
#' group2 <- runif(22, min = 10, max = 18)  # Uniform distribution
#' group3 <- rchisq(20, df = 5) + 7     # Right-skewed Chi-square + shift
#' RWPNPBT(list(group1, group2, group3), B = 1000)
#'
#' @importFrom stats quantile var
#' @importFrom utils combn
#' @export

RWPNPBT = function(group_list, B = 100000, alpha = 0.05) {
  k = length(group_list)
  ns = sapply(group_list, length)
  sample_means = sapply(group_list, mean)
  sample_vars = sapply(group_list, var)
  pair_indices = combn(1:k, 2)

  total_distance = function(means, vars, ns) {
    sum_vals = 0
    for (idx in 1:ncol(pair_indices)) {
      i = pair_indices[1, idx]
      j = pair_indices[2, idx]
      se = sqrt(vars[i]/ns[i] + vars[j]/ns[j])
      if (se == 0 || is.nan(se) || is.na(se)) next
      sum_vals = sum_vals + abs((means[i] - means[j]) / se)
    }
    return(sum_vals)
  }

  lambda_obs = total_distance(sample_means, sample_vars, ns)

  centered_groups = lapply(1:k, function(i) group_list[[i]] - mean(group_list[[i]]))

  get_rademacher_weights = function(n) {
    sample(c(-1, 1), size = n, replace = TRUE, prob = c(0.5, 0.5))
  }

  boot_stats = numeric(B)
  for (b in 1:B) {
    boot_groups = list()
    for (i in 1:k) {
      W = get_rademacher_weights(ns[i])
      boot_groups[[i]] = centered_groups[[i]] * W
    }
    means_b = sapply(boot_groups, mean)
    vars_b = sapply(boot_groups, var)
    boot_stats[b] = total_distance(means_b, vars_b, ns)
  }

  critical_value = quantile(boot_stats, 1 - alpha)

  decision=if (lambda_obs > critical_value) {
    "Reject H0: At least one pair of means differs significantly."
  } else {
    "Do not reject H0: No significant difference among group means."
  }

  result=list(
    test_statistic = lambda_obs,
    critical_value = critical_value,
    decision = decision
  )

  class(result)="RWPNPBT"
  return(result)
}



