% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boots.R
\name{boots}
\alias{boots}
\title{Bootstrap for the I.I.D. Case}
\usage{
boots(x, func, B, smooth = FALSE, moonsize = NULL, mreplace = TRUE, ...)
}
\arguments{
\item{x}{A vector or a matrix representing the data.}

\item{func}{The function to apply to each sample.}

\item{B}{A positive integer; the number of bootstrap replications.}

\item{smooth}{Logical. If \code{TRUE} use a smooth bootstrap.}

\item{moonsize}{A value of m in the 'm out of n' bootstrap.}

\item{mreplace}{Logical. If \code{TRUE}, bootstrap is done with replacement.
If \code{FALSE} the function performs subsampling. In that case, \code{moonsize}
should be specified, else the function stops.}

\item{...}{An optional additional arguments for the \code{func} function.}
}
\value{
An object of class \code{boodd} containing either a vector or a
matrix, depending on whether the \code{func} function returns a single
value or a vector.
If the \code{func} function returns a vector of size \code{n}, then
\code{boots} returns a matrix of size \code{B x n}.
}
\description{
Generic function to bootstrap in the iid case.
}
\details{
The function \code{boots} performs different versions of the bootstrap in the iid
case (see Davison and Hinkley (1997)). With default parameters it performs a
naive bootstrap (see Efron and Tibshirani (1993)).
If \code{moonsize} is specified, the function performs the moon bootstrap (see Bickel et al.(1997))
With \code{smooth=TRUE} the function performs a smooth bootstrap,
based on a kernel density
estimator of the data, with a bandwidth chosen by cross-validation (see Efron
and Tibshirani (1993) and Falk and Reiss (1989)).
Finally, when \code{mreplace=FALSE}, the function performs the subsampling method
of the Politis and Romano (1994) with subsampling size equals to \code{moonsize}.

The \code{func} must be a function whose first argument is a
vector and which returns either a single value or a vector.

The \code{x} argument can be a vector or a matrix. In the case of a matrix,
the \emph{rows} of the matrix are bootstrapped.

The \code{moonsize} and \code{mreplace} arguments concern \emph{m out of n}
bootstrap (aka \emph{moon bootstrap}). The \code{moonsize} argument is an
integer less than the length of \code{x} (or the number of rows if \code{x}
is a matrix). The \code{mreplace} argument is a logical that indicates
whether the bootstrap samples are drawn with or without replacement.
}
\examples{
B <- 999
n <- 200
x <- rnorm(n)
# Naive bootstrap of the mean
boo1 <- boots(x,mean,B)
summary(boo1)
plot(boo1)
confint(boo1)
confint(boo1,method="bperc")

# Naive bootstrap of a multidimentional statistic
mv <- function(data) {c(mean(data),var(data))} # compute mean and variance
boo2 <- boots(x,mv,B)
# Confint can compute percentile and t-percentile confidence intervals 
# when variance is bootstrapped
confint(boo2,method="all")

# Naive Bootstrap of the output parameters of lm (linear regression) function
sigma <- 0.2
y <- x+rnorm(n)
data <- as.matrix(data.frame(x,y))
nlm <- function(dat){lm(dat[,2]~dat[,1])$coefficient}
boo3 <- boots(data,nlm,B)

# Smoothed bootstrap for quantiles 
boo4 <- boots(x,median,B) # without smoothing
plot(boo4)
boo5 <- boots(x,median,B,smooth=TRUE) 
# with smoothing using a cross-validation estimator of the window
plot(boo5)

# Moon bootstrap 
n <- 10000
x <- rnorm(n)
# i.i.d bootstrap is not consistent for the max
boo6 <- boots(x,max,B) 
# Moon bootstrap of the max with a size equals to sqrt(n)
boo7 <- boots(x,max,B,moonsize=sqrt(n)) 
# Subsampling with the moonsize equals to sqrt(n)
boo8 <- boots(x,max,B,moonsize=sqrt(n),mreplace=TRUE) 
}
\references{
Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for
Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.

B. Efron, and Tibshirani, R. (1993). \emph{An Introduction to the
Bootstrap}, Chapman and Hall/CRC Monographs on Statistics and Applied Probability.

Bickel, P. J., Götze, F. and van Zwet, W. R. (1997).
Resampling fewer than n observations: gains, losses, and remedies for losses.
\emph{Statistica Sinica}, \bold{7}, 1-31

A. C. Davison, D. Hinkley (1997). \emph{Bootstrap Methods and Their
Application}, Cambridge Series in Statistical and Probabilistic Mathematics.

Falk, M., & Reiss, R. D. (1989).
Bootstrapping the distance between smooth bootstrap and sample quantile distribution.
\emph{Probability Theory and Related Fields}, \bold{82}, 177–186.

Politis, D. N., & Romano, J. P. (1994). Large Sample Confidence Regions Based on Subsamples
under Minimal Assumptions. \emph{The Annals of Statistics}, \bold{22}. 2031-2050
}
\seealso{
\code{\link{plot.boodd}}, \code{\link{confint.boodd}}.
}
\keyword{"Bootstrap"}
