# ------------------------------------------------------------------------
# 
# "qVar(x,alpha,bandwidth,kernel)" --
# 
# Quantile variance. If qa is the quantile for alpha, the quantile
# variance is given by:
#         alpha(1-alpha)
#     v = --------------
#         fhat_bandwidth(qa)^2
# where
#     fhat_bandwidth(x) = 1/(n bandwidth) \sum_1^n K( (x-X_i)/bandwidth )
# 
# The argument 'alpha' can be a scalar or a vector of probabilities.
# 
# ------------------------------------------------------------------------
##
#' @aliases qVar
#' @title  Estimating Variance of a Quantile
#' @description 
#' This function calculates the quantile variance using kernel density estimation.
#'
#'
#' @param x A numeric vector.
#' @param alpha A numeric value from the interval \eqn{[0,1]} or a vector of probabilities.
#' @param bandwidth A positive numeric value representing the kernel bandwidth smoothing parameter.
#'  If \code{NULL}, bandwidth is estimated using unbiased cross-validation method.
#' @param kernel A character string specifying the smoothing kernel to use:
#'  \code{"gaussian"}, \code{"epanechnikov"}, or \code{"rectangular"}.
#'
#' @return A numeric vector containing the quantile variances.
#'
#' @details 
#' If \code{qa} is the quantile of order \code{alpha}, the quantile
#' variance \eqn{v} is given by:
#' \deqn{v = \frac{\alpha(1 - \alpha)}{\hat{f}_{h}(qa)^2}}
#' where
#' \itemize{
#'   \item \eqn{ \hat{f}_{h}(x) = \frac{1}{n \cdot h} \sum_{i=1}^{n} K\left(\frac{x - X_i}{h}\right)}, 
#'   \item \eqn{h} is the bandwidth, 
#'   \item \eqn{n} is the sample size.
#' }
#' @references 
#'  Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#' Maritz, J. S. and Jarrett, R. G. (1978). A note on estimating the variance of 
#' the sample median. \emph{Journal of the American Statistical Association}, \bold{73}, 194-196. 
#' @examples
#' # Example usage of qVar function
#' data <- rnorm(100)
#' qVar(data, 0.5)
#'
#' @seealso
#' \code{\link{boots}}, \code{\link{confint}}.
#' @keywords Variance Quantile
#' @export



qVar <- function(x,alpha,bandwidth=NULL,kernel=c("gaussian","epanecbandwidthikov","rectangular")) {
  if (!is.vector(x)) {
    stop("x must be a vector")
  }
  n <- length(x)
  if (is.null(bandwidth)) {
    # Get the cross-validation bandwidth estimation
    s <- sd(x)
    bandwidth = bw.ucv(x,lower=s*0.000001,upper=s*2)
    # The estimate provided by the unbiased cross-validation method is of order
    # Cn^(-1/5). We multiply by n^(-2/15) because we need it to be of
    # order n^(-1/3).
    bandwidth <- bandwidth*n^(-2/15)
  }
  kernel <- match.arg(kernel)
  Qs <- quantile(x,alpha)	
  lq <- length(Qs)
  v <- numeric(lq)
  for (i in 1:lq) {
    cr <- (Qs[i]-x)/bandwidth
    K <- switch(kernel,
                gaussian = dnorm(cr,sd=bandwidth),
                epanechnikov = ifelse(abs(x)<1, .75*(1-cr^2), 0),
                rectangular = ifelse(abs(x)<1, .5, 0)
    )
    fhat <- mean(K)/bandwidth
    v[i] <- alpha[i]*(1-alpha[i])/fhat^2	
  }
  return(v)
}