## 
# ------------------------------------------------------------------------
# 
# "naradamar(Sx,Sy,x,bandwidth)" --
# 
# Compute the Naradamar kernel on a grid given by subdivisions Sx and Sy.
# 
# ------------------------------------------------------------------------
##
#' @aliases naradamar
#' @title Nadaraya-Watson Estimator for Transition Densities.
#' @description 
#' Calculates the Nadaraya-Watson estimator for estimating the transition densities 
#' of a Markov chain. 
#' @param x A numeric vector representing a Markov chain.
#' @param bandwidth A positive numeric value representing the kernel bandwidth smoothing parameter.
#' @param Sx A vector of the first coordinate of the grid for which the
#' Nadaraya-Watson kernel estimator will be computed.
#' @param Sy A vector of the second coordinate of the grid for which the
#' Nadaraya-Watson kernel estimator will be computed.
#' @details 
#' The \code{naradamar} function computes the estimated transition densities 
#' \eqn{f(x, y)} of a process \code{x} on a grid given by the \code{Sx} and \code{Sy}
#' values. It utilizes a Nadaraya kernel-type 
#' estimator for the transition density, with a \code{bandwidth} provided by the user. 
#' 
#' @return Returns a matrix of size \code{Sx}\eqn{\cdot}\code{Sy}, containing 
#' the estimated transition densities on the grid.
#' 
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#' @seealso \code{\link{bandw1}}.
#' @keywords "Transition density estimation" "Markov chains"
#' @export
#' @examples
#' set.seed(12345)
#' phi=0.4
#' n=200
#' X <- arima.sim(list(ar=phi),n=n)
#' s=seq(quantile(X,0.01),quantile(X,0.99), length.out=50)
#' h=bandw1(X)
#' res=naradamar(s,s,X,h)
#' persp(s,s,res)


naradamar <- function(Sx,Sy,x,bandwidth) {
  n <- length(x)-1
  Xi <- x[1:n]
  Xip1 <- x[2:(n+1)]
  lx <- length(Sx)
  ly <- length(Sy)
  f <- matrix(0,lx,ly)
  for (k in 1:lx) {
    for (l in 1:ly) {
      f[k,l] <- mean(dnorm((Sx[k]-Xi)/bandwidth)*dnorm((Sy[l]-Xip1)/bandwidth))/mean(dnorm((Sx[k]-Xi)/bandwidth))/bandwidth
    }
  }
  return(f)
}
