##
# ------------------------------------------------------------------------
#
# "freqboot(x,XI,g,B,kernel="normal")" --
#
# Bootstrap in frequency domain (FDB).
#
# ------------------------------------------------------------------------
##
#' @aliases freqboot
#' @title Frequency Domain Bootstrap
#' @description Implements the Frequency Domain Bootstrap (FDB) for time series data.
#' @param x A vector or time series.
#' @param XI A list of functions defined on the interval \eqn{[0, \pi]}.
#' @param g A numeric function accepting \code{length(XI)} arguments, used to compute 
#' the statistic of interest.
#' @param B A positive integer; the number of bootstrap replications.
#' @param kernel A character string specifying the smoothing kernel. The valid values 
#' are:
#' * \code{"normal"} - default,
#' * \code{"epanechnikov"},
#' * \code{"box"} - rectangular kernel.
#' @param bandwidth A real number; the kernel bandwidth smoothing parameter. 
#' If unspecified, an optimal value is computed using formula \eqn{sd(x)*n^(-1/3)}, 
#' which is smaller than the Silverman's rule-of-thumb bandwidth.
#' @details 
#' The input series \code{x} is assumed to be a sample from a real-valued, zero-mean, 
#' stationary time series. The \code{XI} argument consists of functions \eqn{\xi_i} used 
#' to define linear functionals of the spectral density, say 
#' \eqn{A(\xi,f)=\int\xi_i(\omega)f(\omega)d\omega}. The statistic estimates \eqn{T(f)=g(A(\xi,f))}. 
#' The spectral density is estimated by smoothing the periodogram of the series, with 
#' the smoothing kernel specified by \code{kernel} and the smoothing parameter 
#' \code{bandwidth}.
#' The FDB consists in resampling periodogram ordinates standardized by the spectral density 
#' estimates to recompute the bootstrap values of the statistics of interest.
#' @return 
#' Returns an object of class \code{boodd}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Hurvich, C. M. and Zeger, S. L. (1987). Frequency domain bootstrap 
#' methods for time series, Technical Report 87-115, Graduate School of Business
#' Administration, New York Univ.
#' 
#' Bertail, P. and Dudek, A. (2021). Consistency of the Frequency Domain
#' Bootstrap for differentiable functionals, \emph{Electron. J. Statist.}, \bold{15}, 1-36.
#' 
#' Lahiri, S.N. (2003). \emph{Resampling Methods for Dependent Data}. Springer,
#' New York.
#' @seealso \code{\link{aidedboot}}, \code{\link{func_fdb}}, \code{\link{per_boo}}, 
#' \code{\link{tft_boot}}.
#' @keywords "Frequency Domain Bootstrap" "Spectral analysis" "Stationary time series" 
#' @export
#' @examples 
#' set.seed(123)
#' n <- 120
#' x <- arima.sim(list(order=c(1,0,0),ar=0.7),n=n)
#' B <- 999
#' one <- function(x) {1}
#' XI <- list(cos,one)
#' g <- function(x,y) {return(x/y)}
#' # This gives an estimate for the autocorrelation of order 1
#' boo = freqboot(x,XI,g,B,"normal")
#' plot(boo)
##
freqboot <- function(x,XI,g,B,kernel="normal",bandwidth) {
  n <- length(x)
  p <- length(XI)
  # Check the arguments
  for (i in 1:p) {
    if (!is.list(XI) | !is.function(XI[[i]])) {
      stop("XI must be a list of functions")
    }
  }
  if (length(formalArgs(g)) != p) {
    stop("g must be a numeric function with as many arguments as the length of XI (",p,")")
  }
  y <- do.call(g,as.list(rep(1,p)))
  if (!is.vector(y)) {
    stop("Function 'g' must return a vector")
  }
  # 	if (missing(bandwidth)) {h <- bandw1(x)}
  if (missing(bandwidth)) {bandwidth <- sd(x)*n^(-1/3)}
  
  # Periodograms
  x <- ts(x,frequency=1)
  # Compute via FFT
  P <- spec.pgram(x,plot=FALSE,taper=0,fast=FALSE,detrend=FALSE)
  # The returned object P is scaled with factor 1/frequency(x) and without
  # the 1/(2*pi) prefix. It contains the periodograms for the positive
  # Fourier frequencies
  specs <- P$spec/(2*pi)
  freqs <- P$freq*2*pi
  n0 <- length(specs)
  
  # Precompute the \xi_i(\lambda_{jn})
  xiljn <- matrix(nrow=p,ncol=n0)
  for (i in 1:p) {
    xiljn[i,] <- XI[[i]](freqs)
  }
  
  # Precompute the smoothing coefficients
  smc <- smoothingCoefficients(n,bandwidth,kernel)
  
  # Initial statistic
  V <- numeric(p)
  for (i in 1:p) {
    V[i] <- mean(xiljn[i,]*specs)
  }
  y <- do.call(g,as.list(V))
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  cnames <- names(y)
  
  # Extend periodograms and frequencies to ]-pi,pi[
  P0 <- sum(x)^2/(2*pi*n)
  In <- c(rev(specs),P0,specs)
  ljn <- c(-rev(freqs),0,freqs)
  
  # Compute \hat{f_n} for positive Fourier frequencies
  fnhat <- numeric(n0)
  for (i in 1:n0) {
    K <- smc[(n0-i+1):(3*n0-i+1)]
    fnhat[i] <- 2*pi*mean(K*In)/bandwidth
  }
  
  # Compute \hat{\epsilon_jn}
  eps_hat <- specs/fnhat
  # Normalize by the mean
  eps_tilde <- eps_hat/mean(eps_hat)
  # Bootstrap
  for (j in 1:B) {
    # Draw epsilon values randomly with replacement
    eps_star <- sample(eps_tilde,n0,replace=TRUE)
    I_star <-fnhat*eps_star
    for (i in 1:p) {
      V[i] <- mean(xiljn[i,]*I_star)
    }
    res[j,] <- do.call(g,as.list(V))
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "frequency"
  attr(obj,"func") <- g
  return(obj)
}