% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/arc_geo_categories.R
\name{arc_geo_categories}
\alias{arc_geo_categories}
\title{Geocode places on a given area by category}
\usage{
arc_geo_categories(
  category,
  x = NULL,
  y = NULL,
  bbox = NULL,
  name = NULL,
  lat = "lat",
  long = "lon",
  limit = 1,
  full_results = FALSE,
  verbose = FALSE,
  custom_query = list(),
  ...
)
}
\arguments{
\item{category}{A place or address type that can be used to filter results.
Several values can be used as well as a vector (i.e.
\code{c("Cinema", "Museum")}), performing one call for each value. See
\strong{Details}.}

\item{x}{longitude values in numeric format. Must be in the range
\eqn{\left[-180, 180 \right]}.}

\item{y}{latitude values in numeric format. Must be in the range
\eqn{\left[-90, 90 \right]}.}

\item{bbox}{A numeric vector of latitude and longitude
\code{c(minX, minY, maxX, maxY)} that restrict the search area.
See \strong{Details}.}

\item{name}{Optionally, a string indicating the name or address of the
desired results.}

\item{lat}{latitude column name in the output data (default  \code{"lat"}).}

\item{long}{longitude column name in the output data (default  \code{"lon"}).}

\item{limit}{maximum number of results to return per input address. Note
that each query returns a maximum of 50 results.}

\item{full_results}{returns all available data from the API service. This
is a shorthand of \verb{outFields=*}. See \strong{References}. If \code{FALSE} (default)
only the default values of the API are returned. See also
\code{return_addresses} argument.}

\item{verbose}{if \code{TRUE} then detailed logs are output to the console.}

\item{custom_query}{API-specific arguments to be used, passed as a named
list.}

\item{...}{
  Arguments passed on to \code{\link[=arc_geo]{arc_geo}}
  \describe{
    \item{\code{sourcecountry}}{limits the candidates returned to the specified country
or countries. Acceptable values include the three-character country code.
You can specify multiple country codes to limit results to more than one
country.}
    \item{\code{outsr}}{The spatial reference of the \verb{x,y} coordinates returned by a
geocode request. By default is \code{NULL} (i.e. the argument won't be used in
the query). See \strong{Details} and \link{arc_spatial_references}.}
    \item{\code{langcode}}{Sets the language in which reverse-geocoded addresses are
returned.}
  }}
}
\value{
A \link[tibble:tbl_df-class]{tibble} object with the results. See the details of the output in  \href{https://developers.arcgis.com/rest/geocode/api-reference/geocoding-service-output.htm}{ArcGIS REST API Service output}.
}
\description{
This function is useful for extracting places with a given category (or list
of categories) near or within a given location or area. This is a wrapper
of \code{\link[=arc_geo]{arc_geo()}}, but it is vectorized over \code{category}.

See \link{arc_categories} for a detailed explanation and available values.

\strong{Note that} for obtaining results it is needed:
\itemize{
\item Either to provide a pair of coordinates (\verb{x,y} arguments) that would be
used as a reference for geocoding.
\item Or a viewbox (aka bounding box) on the \code{bbox} argument defining a
desired extent of the results.
}

It is possible to combine the two approaches (i.e. providing \verb{x,y,bbox}
values) in order to boost the geocoding process. See \strong{Examples}.
}
\details{
Bounding boxes can be located using different online tools, as
\href{https://boundingbox.klokantech.com/}{Bounding Box Tool}.

For a full list of valid categories see \link{arc_categories}. This function is
vectorized over \code{category}, that means that it performs one independent call
to \code{\link[=arc_geo]{arc_geo()}} for each \code{category} value.

\code{arc_geo_categories()} also understands a single string of categories
separated by commas (\code{"Cinema,Museum"}), that would be internally treated as
\code{c("Cinema", "Museum")}.
}
\section{\code{outsr}}{
The spatial reference can be specified as either a well-known ID (WKID). If
not specified, the spatial reference of the output locations is the same as
that of the service (WGS84, i.e. WKID = 4326)).

See \link{arc_spatial_references} for values and examples.
}

\examples{
\dontshow{if (arcgeocoder_check_access()) withAutoprint(\{ # examplesIf}
\donttest{
# Full workflow: Gas Stations near Carabanchel, Madrid

# Get Carabanchel
carab <- arc_geo("Carabanchel, Madrid, Spain")

# CRS
carab_crs <- unique(carab$latestWkid)


library(ggplot2)

base_map <- ggplot(carab) +
  geom_point(aes(lon, lat), size = 5, color = "red") +
  geom_rect(aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax),
    fill = NA,
    color = "blue"
  ) +
  coord_sf(crs = carab_crs)


# Ex1: Search near Carabanchel (not restricted)
ex1 <- arc_geo_categories("Gas Station",
  # Location
  x = carab$lon, y = carab$lat,
  limit = 50, full_results = TRUE
)


# Reduce number of labels to most common ones
library(dplyr)

labs <- ex1 |>
  count(ShortLabel) |>
  slice_max(n = 7, order_by = n) |>
  pull(ShortLabel)

base_map +
  geom_point(data = ex1, aes(lon, lat, color = ShortLabel)) +
  scale_color_discrete(breaks = labs) +
  labs(
    title = "Example 1",
    subtitle = "Search near (points may be far away)"
  )

# Example 2: Include part of the name, different results
ex2 <- arc_geo_categories("Gas Station",
  # Name
  name = "Repsol",
  # Location
  x = carab$lon, y = carab$lat,
  limit = 50, full_results = TRUE
)

base_map +
  geom_point(data = ex2, aes(lon, lat, color = ShortLabel)) +
  labs(
    title = "Example 2",
    subtitle = "Search near with name"
  )

# Example 3: Near within a extent
ex3 <- arc_geo_categories("Gas Station",
  name = "Repsol",
  bbox = c(carab$xmin, carab$ymin, carab$xmax, carab$ymax),
  limit = 50, full_results = TRUE
)

base_map +
  geom_point(data = ex3, aes(lon, lat, color = ShortLabel)) +
  labs(
    title = "Example 3",
    subtitle = "Search near with name and bbox"
  )
}
\dontshow{\}) # examplesIf}
}
\seealso{
\href{https://developers.arcgis.com/rest/geocode/api-reference/geocoding-category-filtering.htm}{ArcGIS REST Category filtering}.

\link{arc_categories}

Other functions for geocoding:
\code{\link{arc_geo}()},
\code{\link{arc_geo_multi}()},
\code{\link{arc_reverse_geo}()}
}
\concept{geocoding}
