% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit2ts.R
\name{fit2ts}
\alias{fit2ts}
\title{Fit a smooth hazard model with two time scales}
\usage{
fit2ts(
  data2ts = NULL,
  Y = NULL,
  R = NULL,
  Z = NULL,
  bins = NULL,
  Bbases_spec = list(),
  pord = 2,
  optim_method = c("ucminf", "grid_search", "LMMsolver"),
  optim_criterion = c("aic", "bic"),
  lrho = c(0, 0),
  Wprior = NULL,
  ridge = 0,
  control_algorithm = list(),
  par_gridsearch = list()
)
}
\arguments{
\item{data2ts}{(optional) an object of class created by the function
\code{prepare_data()}. Proving this input is the easiest way to use the function
\code{fit2ts}. However, the user can also provide the input data together with
a list of bins, as explained by the following parameters' descriptions.}

\item{Y}{A matrix (or 3d-array) of event counts of dimension nu by ns
(or nu by ns by n).}

\item{R}{A matrix (or 3d-array) of exposure times of dimension nu by ns
(or nu by ns by n).}

\item{Z}{(optional) A regression matrix of covariates values of dimensions
n by p.}

\item{bins}{a list with the specification for the bins. This is created by
the function \code{prepare_data}. If a list prepared externally from such function
if provided, it should contain the following elements:
* \code{bins_u} A vector of bins extremes for the time scale \code{u}.
* \code{midu} A vector with the midpoints of the bins over \code{u}.
* \code{nu} The number of bins over \code{u}.
* \code{bins_s} A vector of bins extremes for the time scale \code{s}.
* \code{mids} A vector with the midpoints of the bins over \code{s}.
* \code{ns} The number of bins over \code{s}.}

\item{Bbases_spec}{A list with the specification for the B-splines basis
with the following elements:
\itemize{
\item \code{bdeg} The degree of the B-splines basis. Default is 3 (for cubic B-splines).
\item \code{nseg_u} The number of segments for the B-splines over \code{u}. Default is 10.
\item \code{min_u} (optional) The lower limit of the domain of \code{Bu}.
Default is \code{min(bins_u)}.
\item \code{max_u} (optional) The upper limit of the domain of \code{Bu}.
Default is \code{max(bins_u)}.
\item \code{nseg_s} The number of segments for the B-splines over \code{s}. Default is 10.
\item \code{min_s} (optional) The lower limit of the domain of \code{Bs}.
Default is \code{min(bins_s)}.
\item \code{max_s} (optional) The upper limit of the domain of \code{Bs}.
Default is \code{max(bins_s)}.
}}

\item{pord}{The order of the penalty. Default is 2.}

\item{optim_method}{The method to be used for optimization:
\code{"ucminf"} (default) for the numerical optimization of the AIC (or BIC),
\code{"grid_search"} for a grid search of the minimum AIC (or BIC)
over a grid of \code{log_10(rho_u)} and \code{log_10(rho_s)} values,
or \code{"LMMsolver"} to solve the model as sparse linear mixed model using the
package LMMsolver.}

\item{optim_criterion}{The criterion to be used for optimization:
\code{"aic"} (default) or \code{"bic"}. BIC penalized model complexity more strongly
than AIC, so that its usage is recommended when a smoother fit is preferable
(see also Camarda, 2012).}

\item{lrho}{A vector of two elements if \code{optim_method == "ucminf"}.
Default is \code{c(0,0)}. A list of two vectors of values for \code{log_10(rho_u)}
and \code{log_10(rho_s)} if \code{optim_method == "grid_search"}. In the latter case,
if a list with two vectors is not provided, a default sequence of
values is used for both \code{log_10(rho_u)} and \code{log_10(rho_s)}.}

\item{Wprior}{An optional matrix of a-priori weights.}

\item{ridge}{A ridge penalty parameter: default is 0. This is useful when, in
some cases the algorithm shows convergence problems. In this case, set to a small
number, for example \code{1e-4}.}

\item{control_algorithm}{A list with optional values for the parameters of
the iterative processes:
\itemize{
\item \code{maxiter} The maximum number of iteration for the IWSL algorithm.
Default is 20.
\item \code{conv_crit} The convergence criteria, expressed as difference between
estimates at iteration i and i+1. Default is \code{1e-5}.
\item \code{verbose} A Boolean. Default is \code{FALSE}. If \code{TRUE} monitors the iteration
process.
\item \code{monitor_ev} A Boolean. Default is \code{FALSE}. If \code{TRUE} monitors the
evaluation of the model over the \code{log_10(rho_s)} values.
}}

\item{par_gridsearch}{A list of parameters for the grid_search:
\itemize{
\item \code{plot_aic} A Boolean. Default is \code{FALSE}. If \code{TRUE}, plot the AIC values
over the grid of \code{log_10(rho_u)} and \code{log_10(rho_s)} values.
\item \code{plot_bic} A Boolean. Default is \code{FALSE}. If \code{TRUE}, plot the BIC values
over the grid of \code{log_10(rho_u)} and \code{log_10(rho_s)} values.
\item \code{return_aic} A Boolean. Default is \code{TRUE}. Return the AIC values.
\item \code{return_bic} A Boolean. Default is \code{TRUE}. Return the BIC values.
\item \code{col} The color palette to be used for the AIC/BIC plot. Default is
\code{grDevices::gray.colors(n=10)}.
\item \code{plot_contour} A Boolean. Default is \code{TRUE}. Adds white contour lines to
the AIC/BIC plot.
\item \code{mark_optimal} A Boolean. Default is \code{TRUE}. If the plot of the AIC or BIC
values is returned, marks the optimal combination of \code{log_10(rho_u)} and
\code{log_10(rho_s)} in the plot.
\item \code{main_aic} The title of the AIC plot. Default is \code{"AIC grid"}.
\item \code{main_bic} The title of the BIC plot. Default is \code{"BIC grid"}.
}}
}
\value{
An object of class \code{haz2ts}, or of class \code{haz2tsLMM}.
For objects of class \code{haz2ts} this is
\itemize{
\item \code{optimal_model} A list with :
\itemize{
\item \code{Alpha} The matrix of estimated P-splines coefficients of dimension
\eqn{c_u} by \eqn{c_s}.
\item \code{Cov_alpha} The variance-covariance matrix of the \code{Alpha} coefficients,
of dimension \eqn{c_uc_s} by \eqn{c_uc_s}.
\item \code{beta} The vector of length p of estimated covariates coefficients
(if model with covariates).
\item \code{Cov_beta} The variance-covariance matrix of the \code{beta} coefficients,
of dimension \eqn{p} by \eqn{p} (if model with covariates).
\item \code{SE_beta} The vector of length \eqn{p} of estimated Standard Errors for the \code{beta}
coefficients (if model with covariates)..
\item \code{Eta} or \code{Eta0} The matrix of values of the (baseline) linear predictor
(log-hazard) of dimension \eqn{n_u} by \eqn{n_s}.
\item \code{H} The hat-matrix.
\item \code{deviance} The deviance.
\item \code{ed} The effective dimension of the model.
\item \code{aic} The value of the AIC.
\item \code{bic} The value of the BIC.
\item \code{Bbases} a list with the B-spline bases \code{Bu} and \code{Bs}
}
\item \code{optimal_logrho} A vector with the optimal values of \code{log10(rho_u)} and
\code{log10(rho_s)}.
\item \code{P_optimal} The optimal penalty matrix P.
\item \code{AIC} (if \code{par_gridsearch$return_aic == TRUE}) The matrix of AIC values.
\item \code{BIC} (if \code{par_gridsearch$return_bic == TRUE}) The matrix of BIC values.
}

Objects of class \code{haz2tsLMM} have a slight different structure. They are
a list with:
\itemize{
\item \code{optimal_model} an object of class \code{LMMsolve}
\item \code{AIC_BIC} a list with, among other things, the AIC and BIC values and the
ED of the model
\item \code{n_events} the number of events
\item \code{nu} the number of bins over the u-axis
\item \code{ns} the number of bins over the s-axis
\item \code{cu} the number of B-splines over the u-axis
\item \code{cs} the number of B-splines over the s-axis
\item \code{covariates} an indicator for PH model
}
}
\description{
\code{fit2ts()} fits a smooth hazard model with two time scales.

Three methods are implemented for the search of the optimal smoothing
parameters (and therefore optimal model): a numerical optimization of the
AIC or BIC of the model, a search for the minimum AIC or BIC of the
model over a grid of \code{log_10} values for the smoothing parameters, and a
solution that uses a sparse mixed model representation of the P-spline model to
estimate the smoothing parameters.
Construction of the B-splines bases and of the penalty matrix is
incorporated within the function. If a matrix of covariates is provided,
the function will estimate a model with covariates.
}
\details{
Some functions from the R-package \code{LMMsolver} are used here.
We refer the interested readers to https://biometris.github.io/LMMsolver/
for more details on \code{LMMsolver} and its usage.
}
\examples{
# Create some fake data - the bare minimum
id <- 1:20
u <- c(5.43, 3.25, 8.15, 5.53, 7.28, 6.61, 5.91, 4.94, 4.25, 3.86, 4.05, 6.86,
       4.94, 4.46, 2.14, 7.56, 5.55, 7.60, 6.46, 4.96)
s <- c(0.44, 4.89, 0.92, 1.81, 2.02, 1.55, 3.16, 6.36, 0.66, 2.02, 1.22, 3.96,
       7.07, 2.91, 3.38, 2.36, 1.74, 0.06, 5.76, 3.00)
ev <- c(1, 0, 0, 1, 0, 1, 0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1)#'

fakedata <- as.data.frame(cbind(id, u, s, ev))
fakedata2ts <- prepare_data(data = fakedata,
                            u = "u",
                            s_out = "s",
                            ev = "ev",
                            ds = .5)
# Fit a fake model - not optimal smoothing
fit2ts(fakedata2ts,
       optim_method = "grid_search",
       lrho = list(seq(1, 1.5, .5), seq(1, 1.5, .5)))
# For more examples please check the vignettes!!! Running more complicated examples
# here would imply longer running times...

}
\references{
Boer, Martin P. 2023. “Tensor Product P-Splines Using a Sparse Mixed Model Formulation.”
Statistical Modelling 23 (5-6): 465–79. https://doi.org/10.1177/1471082X231178591.
Carollo, Angela, Paul H. C. Eilers, Hein Putter, and Jutta Gampe. 2023.
“Smooth Hazards with Multiple Time Scales.” arXiv Preprint:
https://arxiv.org/abs/http://arxiv.org/abs/2305.09342v1
}
