% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeStats}
\alias{computeStats}
\title{Compute Descriptive Statistics for Births, Deaths, Midpoints, and Lifespans in a Persistence Diagram}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeStats()} calculates descriptive statistics of the birth, death, midpoint (the average of birth and death), and lifespan (death minus birth) values. Additionally, it computes the total number of points and entropy of the lifespan values. Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeStats(D, homDim)
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}
\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}
}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes the mean, standard deviation, median, IQR (interquartile range), range, 10th, 25th, 75th and 90th percentiles of the birth, death, midpoint, lifespan (or persistence) values; the total number of bars (or points in the diagram) and the entropy of the lifespan values (-\eqn{\sum_{i=1}^N\frac{l_i}{L}\log_2(\frac{l_i}{L})}, where \eqn{l_i=d_i-b_i} (lifespan) and \eqn{L=\sum_{i=1}^N l_i}). If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.
}


\value{
A (named) 38-dimensional numeric vector containing:
\itemize{
\item \code{mean_births}, \code{stddev_births}, \code{median_births}, \code{iqr_births}, \code{range_births}, \code{p10_births}, \code{p25_births}, \code{p75_births}, \code{p90_births}: Descriptive statistics for birth values.
\item \code{mean_deaths}, \code{stddev_deaths}, \code{median_deaths}, \code{iqr_deaths}, \code{range_deaths}, \code{p10_deaths}, \code{p25_deaths}, \code{p75_deaths}, \code{p90_deaths}: Descriptive statistics for death values.
\item \code{mean_midpoints}, \code{stddev_midpoints}, \code{median_midpoints}, \code{iqr_midpoints}, \code{range_midpoints}, \code{p10_midpoints}, \code{p25_midpoints}, \code{p75_midpoints}, \code{p90_midpoints}: Descriptive statistics for midpoint values (mean of birth and death values).
\item \code{mean_lifespans}, \code{stddev_lifespans}, \code{median_lifespans}, \code{iqr_lifespans}, \code{range_lifespans}, \code{p10_lifespans}, \code{p25_lifespans}, \code{p75_lifespans}, \code{p90_lifespans}: Descriptive statistics for lifespan (or persistence) values (difference between death and birth values).
\item \code{total_bars}: The total number of points in the specified homological dimension.
\item \code{entropy}: The entropy of the lifespan values.
}
}

\author{
Umar Islambekov
}

\references{
1. Ali, D., Asaad, A., Jimenez, M.J., Nanda, V., Paluzo-Hidalgo, E. and Soriano-Trigueros, M., (2023). A survey of vectorization methods in topological data analysis. IEEE Transactions on Pattern Analysis and Machine Intelligence.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

# Compute statistics for homological dimension H_0
computeStats(D, homDim = 0)

# Compute statistics for homological dimension H_1
computeStats(D, homDim = 1)
}
