% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{roc.curve.factor}
\alias{roc.curve.factor}
\title{Reciever Operator Characteristics}
\usage{
\method{roc.curve}{factor}(actual, response, thresholds = NULL, indices = NULL, ...)
}
\arguments{
\item{actual}{A vector \link{length} \eqn{n}, and \eqn{k} levels. Can be of \link{integer} or \link{factor}.}

\item{response}{A \eqn{n \times k} <\link{double}>-matrix of predicted probabilities.
The \eqn{i}-th row should sum to 1 (i.e., a valid probability distribution
over the \eqn{k} classes). The first column corresponds to the first factor
level in \code{actual}, the second column to the second factor level, and so on.}

\item{thresholds}{An optional <\link{double}> vector of \link{length} \eqn{n} (default: \link{NULL}).}

\item{indices}{An optional \eqn{n \times k} matrix of <\link{integer}> values of sorted response probability indices.}

\item{...}{Arguments passed into other methods.}
}
\value{
A \link{data.frame} on the following form,

\item{threshold}{<\link{numeric}> Thresholds used to determine \code{\link[=tpr]{tpr()}} and \code{\link[=fpr]{fpr()}}}
\item{level}{<\link{character}> The level of the actual <\link{factor}>}
\item{label}{<\link{character}> The levels of the actual <\link{factor}>}
\item{fpr}{<\link{numeric}> The false positive rate}
\item{tpr}{<\link{numeric}> The true positve rate}
}
\description{
A generic S3 function to compute the \emph{reciever operator characteristics} score for a classification model. This function dispatches to S3 methods in \code{\link[=roc.curve]{roc.curve()}} and performs no input validation. If you supply \link{NA} values or vectors of unequal \link{length} (e.g. \code{length(x) != length(y)}), the underlying \code{C++} code may trigger undefined behavior and crash your \code{R} session.
\subsection{Defensive measures}{

Because \code{\link[=roc.curve]{roc.curve()}} operates on raw pointers, pointer-level faults (e.g. from \link{NA} or mismatched \link{length}) occur before any \code{R}-level error handling.  Wrapping calls in \code{\link[=try]{try()}} or \code{\link[=tryCatch]{tryCatch()}} will \emph{not} prevent \code{R}-session crashes.

To guard against this, wrap \code{\link[=roc.curve]{roc.curve()}} in a "safe" validator that checks for \link{NA} values and matching \link{length}, for example:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{safe_roc.curve <- function(x, y, ...) \{
  stopifnot(
    !anyNA(x), !anyNA(y),
    length(x) == length(y)
  )
  roc.curve(x, y, ...)
\}
}\if{html}{\out{</div>}}

Apply the same pattern to any custom metric functions to ensure input sanity before calling the underlying \code{C++} code.
}

\subsection{Area under the curve}{

Use \link{auc.roc.curve} for calculating the area under the curve directly.
}

\subsection{Efficient multi-metric evaluation}{

To avoid sorting the same probability matrix multiple times (once per class or curve), you can precompute a single set of sort indices and pass it via the \code{indices} argument. This reduces the overall cost from O(K·N log N) to O(N log N + K·N).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{## presort response
## probabilities
indices <- preorder(response, decreasing = TRUE)

## evaluate reciever operator characteristics
roc.curve(actual, response, indices = indices)
}\if{html}{\out{</div>}}
}
}
\examples{
## Classes and
## seed
set.seed(1903)
classes <- c("Kebab", "Falafel")

## Generate actual classes
## and response probabilities
actual_classes <- factor(
    x = sample(
      x = classes, 
      size = 1e2, 
      replace = TRUE, 
      prob = c(0.7, 0.3)
    )
)

response_probabilities <- ifelse(
    actual_classes == "Kebab", 
    rbeta(sum(actual_classes == "Kebab"), 2, 5), 
    rbeta(sum(actual_classes == "Falafel"), 5, 2)
)

## Construct response
## matrix
probability_matrix <- cbind(
    response_probabilities,
    1 - response_probabilities
)



## Visualize

plot(
    SLmetrics::roc.curve(
     actual   = actual_classes, 
     response = probability_matrix
 )
)


}
\references{
James, Gareth, et al. An introduction to statistical learning. Vol. 112. No. 1. New York: springer, 2013.

Hastie, Trevor. "The elements of statistical learning: data mining, inference, and prediction." (2009).

Pedregosa, Fabian, et al. "Scikit-learn: Machine learning in Python." the Journal of machine Learning research 12 (2011): 2825-2830.
}
\seealso{
Other Classification: 
\code{\link{accuracy}()},
\code{\link{auc.pr.curve}()},
\code{\link{auc.roc.curve}()},
\code{\link{baccuracy}()},
\code{\link{brier.score}()},
\code{\link{ckappa}()},
\code{\link{cmatrix}()},
\code{\link{cross.entropy}()},
\code{\link{dor}()},
\code{\link{fbeta}()},
\code{\link{fdr}()},
\code{\link{fer}()},
\code{\link{fmi}()},
\code{\link{fpr}()},
\code{\link{hammingloss}()},
\code{\link{jaccard}()},
\code{\link{logloss}()},
\code{\link{mcc}()},
\code{\link{nlr}()},
\code{\link{npv}()},
\code{\link{plr}()},
\code{\link{pr.curve}()},
\code{\link{precision}()},
\code{\link{recall}()},
\code{\link{relative.entropy}()},
\code{\link{shannon.entropy}()},
\code{\link{specificity}()},
\code{\link{zerooneloss}()}

Other Supervised Learning: 
\code{\link{accuracy}()},
\code{\link{auc.pr.curve}()},
\code{\link{auc.roc.curve}()},
\code{\link{baccuracy}()},
\code{\link{brier.score}()},
\code{\link{ccc}()},
\code{\link{ckappa}()},
\code{\link{cmatrix}()},
\code{\link{cross.entropy}()},
\code{\link{deviance.gamma}()},
\code{\link{deviance.poisson}()},
\code{\link{deviance.tweedie}()},
\code{\link{dor}()},
\code{\link{fbeta}()},
\code{\link{fdr}()},
\code{\link{fer}()},
\code{\link{fmi}()},
\code{\link{fpr}()},
\code{\link{gmse}()},
\code{\link{hammingloss}()},
\code{\link{huberloss}()},
\code{\link{jaccard}()},
\code{\link{logloss}()},
\code{\link{maape}()},
\code{\link{mae}()},
\code{\link{mape}()},
\code{\link{mcc}()},
\code{\link{mpe}()},
\code{\link{mse}()},
\code{\link{nlr}()},
\code{\link{npv}()},
\code{\link{pinball}()},
\code{\link{plr}()},
\code{\link{pr.curve}()},
\code{\link{precision}()},
\code{\link{rae}()},
\code{\link{recall}()},
\code{\link{relative.entropy}()},
\code{\link{rmse}()},
\code{\link{rmsle}()},
\code{\link{rrmse}()},
\code{\link{rrse}()},
\code{\link{rsq}()},
\code{\link{shannon.entropy}()},
\code{\link{smape}()},
\code{\link{specificity}()},
\code{\link{zerooneloss}()}
}
\concept{Machine learning performance evaluation}
\keyword{classification}
\keyword{evaluation}
