# PoissonGPZ.R
#' Poisson-Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson-Gompertz distribution.
#'
#' The Poisson-Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson-Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1 - \frac{1}{{\left( {1 - {e^{ - \lambda }}} \right)}}
#' \left[ {1 - \exp \left\{ { - \lambda \exp \left( {\frac{\beta }{\alpha }
#' \left( {1 - {e^{\alpha x}}} \right)} \right)} \right\}} \right]  \quad ;\;x > 0.
#' }
#'  
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dpois.gpz()} — Density function
#'   \item \code{ppois.gpz()} — Distribution function
#'   \item \code{qpois.gpz()} — Quantile function
#'   \item \code{rpois.gpz()} — Random generation
#'   \item \code{hpois.gpz()} — Hazard function
#' }
#'
#' @usage
#' dpois.gpz(x, alpha, beta, lambda, log = FALSE)
#' ppois.gpz(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.gpz(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.gpz(n, alpha, beta, lambda)
#' hpois.gpz(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.gpz}: numeric vector of (log-)densities
#'   \item \code{ppois.gpz}: numeric vector of probabilities
#'   \item \code{qpois.gpz}: numeric vector of quantiles
#'   \item \code{rpois.gpz}: numeric vector of random variates
#'   \item \code{hpois.gpz}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary,A.K., Sapkota,L.P., & Kumar, V. (2021). 
#' Poisson Gompertz Distribution with Properties and Applications. 
#' \emph{ International Journal of Applied Engineering Research (IJEAR)},
#' \bold{16(1)},75--84. \doi{10.37622/IJAER/16.1.2021.75-84}     
#'
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dpois.gpz(x, 2.0, 0.5, 0.2)
#' ppois.gpz(x, 2.0, 0.5, 0.2)
#' qpois.gpz(0.5, 2.0, 0.5, 0.2)
#' rpois.gpz(10, 2.0, 0.5, 0.2)
#' hpois.gpz(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=0.2211, beta=0.6540, lambda=6.5456)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.gpz, pfun=ppois.gpz, plot=FALSE)
#' print.gofic(out)
#' 
#' @name PoissonGPZ
#' @aliases dpois.gpz ppois.gpz qpois.gpz rpois.gpz hpois.gpz
#' @export
NULL

#' @rdname PoissonGPZ
#' @usage NULL
#' @export  
dpois.gpz <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	u <- exp(alpha*x[valid])
	z <- exp((beta/alpha)*(1-u))
    gx <- beta * lambda * u * z * exp(-lambda*z)	
    pdf[valid] <- tmp * gx
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonGPZ
#' @usage NULL
#' @export  
ppois.gpz <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- exp((beta/alpha)*(1-exp(alpha*q[valid])))
    cdf[valid] <- 1.0 - tmp * (1-exp(-lambda * z))
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonGPZ
#' @usage NULL
#' @export
qpois.gpz <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
     z <- 1+(1/lambda) * log(1-(1-exp(-lambda))*(1-p[valid]))		
     qtl[valid] <- (1/alpha) * log(1-(alpha/beta)*log(1-z)) 	
	}
    return(qtl)   
 }

#' @rdname PoissonGPZ
#' @usage NULL
#' @export
rpois.gpz <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")

    u <- runif(n)
    q <- qpois.gpz(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonGPZ
#' @usage NULL
#' @export
hpois.gpz <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
    
    nume <- dpois.gpz(x, alpha, beta, lambda)
	surv <- 1.0 - ppois.gpz(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}  

