# LindleyChen.R
#' Lindley-Chen Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley-Chen distribution.
#'
#' The Lindley-Chen distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley-Chen distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta)
#' = 1 - \left[ {1 - \lambda \left( {\frac{\theta }{{1 + \theta }}} \right)
#' \left( {1 - {e^{{x^\alpha }}}} \right)} \right]\;
#' \exp \left\{ {\lambda \theta \left( {1 - {e^{{x^\alpha }}}} \right)} \right\},
#' \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dlindley.chen()} — Density function
#'   \item \code{plindley.chen()} — Distribution function
#'   \item \code{qlindley.chen()} — Quantile function
#'   \item \code{rlindley.chen()} — Random generation
#'   \item \code{hlindley.chen()} — Hazard function
#' }
#'
#' @usage
#' dlindley.chen(x, alpha, lambda, theta, log = FALSE)
#' plindley.chen(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.chen(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.chen(n, alpha, lambda, theta)
#' hlindley.chen(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.chen}: numeric vector of (log-)densities
#'   \item \code{plindley.chen}: numeric vector of probabilities
#'   \item \code{qlindley.chen}: numeric vector of quantiles
#'   \item \code{rlindley.chen}: numeric vector of random variates
#'   \item \code{hlindley.chen}: numeric vector of hazard values
#' }
#'
#' @references
#' Bhati, D., Malik, M. A., & Vaman, H. J. (2015). 
#' Lindley–Exponential distribution: properties and applications. 
#' \emph{Metron}, \bold{73(3)}, 335--357.
#'  
#' Joshi, R. K., & Kumar, V. (2020). 
#' Lindley-Chen Distribution with Applications. \emph{International Journal 
#' of Engineering, Science & Mathematics (IJESM)}, \bold{9(10)}, 12--22.   
#' 
#' @examples
#' x <- seq(1.0, 3.0, 0.25)
#' dlindley.chen(x, 0.5, 2, 0.5)
#' plindley.chen(x, 0.5, 2, 0.5)
#' qlindley.chen(0.5, 0.5, 2, 0.5)
#' rlindley.chen(10, 0.5, 2, 0.5)
#' hlindley.chen(x, 0.5, 2, 0.5)

#' 
#' # Data
#' x <- fibers65
#' # ML estimates
#' params = list(alpha=1.26813, lambda=28.96389, theta=0.00355)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.chen, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.chen, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.chen, pfun=plindley.chen, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyChen
#' @aliases dlindley.chen plindley.chen qlindley.chen rlindley.chen hlindley.chen
#' @export
NULL

#' @rdname LindleyChen
#' @usage NULL
#' @export   
dlindley.chen <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- x[valid] ^ alpha	
	z <- lambda * (1 - exp(u)) 	
    pdf[valid] <- (theta*theta/(1+theta)) * alpha * lambda * (x[valid]^(alpha-1)) * exp(u) * exp(theta*z) * (1 - z)   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyChen
#' @usage NULL
#' @export 
plindley.chen <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	z <- 1 - exp(q[valid] ^ alpha)	
    cdf[valid] <- 1.0 - exp(lambda * theta * z) * (1 - (theta/(1+theta)) * lambda * z)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyChen
#' @usage NULL
#' @export 
qlindley.chen <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Parameters 'alpha', 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	log(1-lambda*(theta/(theta+1))*(1-exp(x^alpha))) +(lambda*theta*(1-exp(x^alpha))) -log(1-wi)   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- 1e-3
      upper <- 1e2
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-6),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LindleyChen
#' @usage NULL
#' @export 
rlindley.chen <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qlindley.chen(u, alpha, lambda, theta)
    return(q)
}

#' @rdname LindleyChen
#' @usage NULL
#' @export 
hlindley.chen <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dlindley.chen(x, alpha, lambda, theta) 
	surv <- 1.0 - plindley.chen(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 


