#' get.R.package
#'
#' Extracts mentioned R packages from text.
#' @param x text string to process.
#' @param update.package.list Logical. If TRUE update of list with available packages is downloaded from CRAN with utils::available.packages().
#' @seealso \code{\link[JATSdecoder]{study.character}} for extracting multiple study characteristics at once.
#' @return Character. Vector with identified R package/s.
#' @importFrom utils available.packages
#' @export
#' @examples
#' get.R.package("We used the R Software packages lme4 (and psych).")

get.R.package<-function(x,update.package.list=FALSE){
# only if text is R use case
if(length(grep("R Core|using R[,\\.]| in R[,\\.]|\\.[Rr] [Pp]roject|oftware 'R'|[A-Za-z]R Foundation| with R[,\\.]|R Development Core|[^A-Z0-9] R \\([Vv1-4]| R [0-9]\\.[0-9]| R[0-9]\\.[0-9]|[Ss]oftware R[^a-zA-z]|[^a-zA-Z]R [Ss]oftware|anguage R|R [Ff]oundation|in R [Vv]ersion|with R [Vv]ersion| in R with " ,unlist(x)))>0 & length(grep("[Pp]ackage|[Ss]oftware|[Ff]unction",unlist(x)))>0){
   # get available R packages
  if(update.package.list==T){
    # get R packages from web page by date
    a<-readLines("https://cran.r-project.org/web/packages/available_packages_by_date.html")
    a<-gsub("/.*","",gsub(".*\\./web/packages/","",grep("/web/packages/",a,value=T)))
    # via cran
    b<-names(utils::available.packages()[,1])
    # combine
    .Rpacks<-unique(c(a,b))
  }
 
   # select lines with package, software or function or algorithm
   x<-grep("[Pp]ackage|[Ss]oftware|[Ff]unction|[Aa]lgorithm",unlist(x),value=T)
   x<-gsub("\\.","",x)
   # get package name before or after term "package", exclude lines with not
   x<-gsub("\\: "," ",x)
   pack1<-gsub(" .*","",gsub(".*[Pp]ackage |.*[Pp]ackages |.*[Ss]oftware |.*[Ff]unction |.*[Ff]unctions |.*[Aa]lgorithm |.*[Aa]lgorithms ","",x))
   pack2<-gsub(".* ","",gsub(" [Pp]ackage.*| [Ff]unction.*| [Ss]oftware.*| [Aa]lgorithm.*","",x))
   pack3<-gsub(" .*","",grep("^not | not |see [A-Z]",unlist(strsplit(x,"and |, |; "))[-1],invert=TRUE,value=TRUE))
   pack4<-gsub(".* ","",unlist(strsplit(x," and |, |; "))[1])
   # compare with available R packages
   p<-unique(gsub("'|[)(,]|^[.]","",c(pack1,pack2,pack3,pack4)))
   return(p[is.element(p,.Rpacks)])
 }else return(character(0))
}


.Rpacks<-c(
"JATSdecoder",
"estimatr"                                            ,
"shadowtext"                                          ,
"AzureAuth"                                           ,
"blockRAR"                                            ,
"checkpoint"                                          ,
"circlize"                                            ,
"cna"                                                 ,
"collidr"                                             ,
"epiGWAS"                                             ,
"FunChisq"                                            ,
"funModeling"                                         ,
"GGIR"                                                ,
"hansard"                                             ,
"joint.Cox"                                           ,
"kernelPSI"                                           ,
"nlsr"                                                ,
"npsf"                                                ,
"OptSig"                                              ,
"pammtools"                                           ,
"pder"                                                ,
"pinp"                                                ,
"plot3logit"                                          ,
"sjstats"                                             ,
"welchADF"                                            ,
"arsenal"                                             ,
"bigreadr"                                            ,
"chebpol"                                             ,
"civis"                                               ,
"Ckmeans.1d.dp"                                       ,
"clustrd"                                             ,
"cvms"                                                ,
"devtools"                                            ,
"distTails"                                           ,
"eph"                                                 ,
"excelR"                                              ,
"ezpickr"                                             ,
"factorial2x2"                                        ,
"fastcmprsk"                                          ,
"fuzzyjoin"                                           ,
"googleAuthR"                                         ,
"gProfileR"                                           ,
"hgutils"                                             ,
"igate"                                               ,
"insight"                                             ,
"laGP"                                                ,
"mapsapi"                                             ,
"matrixStats"                                         ,
"MFAg"                                                ,
"move"                                                ,
"MultinomialCI"                                       ,
"MultNonParam"                                        ,
"nasapower"                                           ,
"NonProbEst"                                          ,
"oai"                                                 ,
"osqp"                                                ,
"plm"                                                 ,
"PracTools"                                           ,
"proj4"                                               ,
"rawr"                                                ,
"RblDataLicense"                                      ,
"RcppEnsmallen"                                       ,
"rdtLite"                                             ,
"riskclustr"                                          ,
"RPEGLMEN"                                            ,
"RProbSup"                                            ,
"selectiveInference"                                  ,
"seqinr"                                              ,
"shadow"                                              ,
"sjPlot"                                              ,
"StanHeaders"                                         ,
"table.express"                                       ,
"textrecipes"                                         ,
"thickmatch"                                          ,
"TraMineRextras"                                      ,
"UCSCXenaTools"                                       ,
"urlshorteneR"                                        ,
"bench"                                               ,
"iDINGO"                                              ,
"LPRelevance"                                         ,
"otsad"                                               ,
"prioritizr"                                          ,
"randomizr"                                           ,
"searchConsoleR"                                      ,
"BayesMallows"                                        ,
"BDgraph"                                             ,
"BVAR"                                                ,
"cate"                                                ,
"CDM"                                                 ,
"clusterlab"                                          ,
"CMplot"                                              ,
"corrcoverage"                                        ,
"dclust"                                              ,
"ggformula"                                           ,
"ggmuller"                                            ,
"glm.predict"                                         ,
"jstor"                                               ,
"link2GI"                                             ,
"LRQMM"                                               ,
"mgm"                                                 ,
 "mixedClust"                                         ,
 "modelStudio"                                        ,
 "mvp"                                                ,
 "nonneg.cg"                                          ,
 "orthoDr"                                            ,
 "PlackettLuce"                                       ,
 "poismf"                                             ,
 "pomp"                                               ,
 "PrevMap"                                            ,
 "projects"                                           ,
 "pyMTurkR"                                           ,
 "QuantileNPCI"                                       ,
 "queryparser"                                        ,
 "Rborist"                                            ,
 "revealedPrefs"                                      ,
 "RMixtComp"                                          ,
 "ssgraph"                                            ,
 "stochQN"                                            ,
 "tidyUSDA"                                           ,
 "WeightIt"                                           ,
 "xspliner"                                           ,
 "bbl"                                                ,
 "BoomSpikeSlab"                                      ,
 "cde"                                                ,
 "EMbC"                                               ,
 "fabricatr"                                          ,
 "freesurferformats"                                  ,
 "FunnelPlotR"                                        ,
 "ggBubbles"                                          ,
 "ggraptR"                                            ,
 "googler"                                            ,
 "GSODR"                                              ,
 "hmmhdd"                                             ,
 "jaod"                                               ,
 "jskm"                                               ,
 "leaflethex"                                         ,
 "log4r"                                              ,
 "mlr3filters"                                        ,
 "mrbayes"                                            ,
 "msigdbr"                                            ,
 "opart"                                              ,
 "pins"                                               ,
 "PLMIX"                                              ,
 "raceland"                                           ,
 "RKelly"                                             ,
 "rsf"                                                ,
 "sclr"                                               ,
 "spc4sts"                                            ,
 "StratifiedMedicine"                                 ,
 "SynthTools"                                         ,
 "tashu"                                              ,
 "wedge"                                              ,
 "bimets"                                             ,
 "BMRSr"                                              ,
 "Boom"                                               ,
 "bracer"                                             ,
 "buildmer"                                           ,
 "ClinReport"                                         ,
 "cloudml"                                            ,
 "crch"                                               ,
 "distreg.vis"                                        ,
 "distributions3"                                     ,
 "do"                                                 ,
 "effects"                                            ,
 "evmix"                                              ,
 "fdaMixed"                                           ,
 "feather"                                            ,
 "fixest"                                             ,
 "gdtools"                                            ,
 "geometry"                                           ,
 "geosed"                                             ,
 "getTBinR"                                           ,
 "ggeffects"                                          ,
 "ggpubr"                                             ,
 "ggquickeda"                                         ,
 "grf"                                                ,
 "HSPOR"                                              ,
 "ledger"                                             ,
 "lg"                                                 ,
 "liquidSVM"                                          ,
 "metamicrobiomeR"                                    ,
 "mockery"                                            ,
 "MSG"                                                ,
 "multiApply"                                         ,
 "openblender"                                        ,
 "optweight"                                          ,
 "pkgdown"                                            ,
 "PPQplan"                                            ,
 "predict3d"                                          ,
 "QBAsyDist"                                          ,
 "RODBC"                                              ,
 "Rraven"                                             ,
 "rstatix"                                            ,
 "scModels"                                           ,
 "snahelper"                                          ,
 "spNNGP"                                             ,
 "survminer"                                          ,
 "textfeatures"                                       ,
 "tidypredict"                                        ,
 "tsBSS"                                              ,
 "tvthemes"                                           ,
 "vivo"                                               ,
 "warbleR"                                            ,
 "wtest"                                              ,
 "chngpt"                                             ,
 "creditmodel"                                        ,
 "dispRity"                                           ,
 "emuR"                                               ,
 "episensr"                                           ,
 "fastDummies"                                        ,
 "FateID"                                             ,
 "feasts"                                             ,
 "forecastSNSTS"                                      ,
 "fullfact"                                           ,
 "genesysr"                                           ,
 "ggalluvial"                                         ,
 "ggraph"                                             ,
 "ghibli"                                             ,
 "gld"                                                ,
 "glmmboot"                                           ,
 "jstable"                                            ,
 "mfbvar"                                             ,
 "nilde"                                              ,
 "nomogramFormula"                                    ,
 "outliertree"                                        ,
 "Rcmdr"                                              ,
 "RcppArmadillo"                                      ,
 "rerddapXtracto"                                     ,
 "RMixtCompIO"                                        ,
 "SDMtune"                                            ,
 "sirus"                                              ,
 "SparkR"                                             ,
 "spray"                                              ,
 "ssr"                                                ,
 "tidylog"                                            ,
 "distances"                                          ,
 "dtw"                                                ,
 "ExtractTrainData"                                   ,
 "hBayesDM"                                           ,
 "jmvconnect"                                         ,
 "msma"                                               ,
 "scorecard"                                          ,
 "vegan"                                              ,
 "zip"                                                ,
 "archivist"                                          ,
 "auditor"                                            ,
 "AzureKusto"                                         ,
 "fdapace"                                            ,
 "googleCloudStorageR"                                ,
 "HARModel"                                           ,
 "HMP"                                                ,
 "JointAI"                                            ,
 "MetabolAnalyze"                                     ,
 "nFactors"                                           ,
 "pbapply"                                            ,
 "riskParityPortfolio"                                ,
 "sdcTable"                                           ,
 "TruncatedNormal"                                    ,
 "addinslist"                                         ,
 "avar"                                               ,
 "elrm"                                               ,
 "embryogrowth"                                       ,
 "EQL"                                                ,
 "ExomeDepth"                                         ,
 "formulaic"                                          ,
 "ggstatsplot"                                        ,
 "gitgadget"                                          ,
 "gWQSRS"                                             ,
 "HaDeX"                                              ,
 "L0Learn"                                            ,
 "LPCM"                                               ,
 "MicSim"                                             ,
 "multicastR"                                         ,
 "multiDimBio"                                        ,
 "nppbib"                                             ,
 "NTS"                                                ,
 "OwenQ"                                              ,
 "PowerTOST"                                          ,
 "prioGene"                                           ,
 "QuantumOps"                                         ,
 "resampledata"                                       ,
 "rlas"                                               ,
 "rMVP"                                               ,
 "ruv"                                                ,
 "semTools"                                           ,
 "spsurvey"                                           ,
 "srm"                                                ,
 "staRdom"                                            ,
 "swissdd"                                            ,
 "wv"                                                 ,
 "anytime"                                            ,
 "arules"                                             ,
 "BayesNSGP"                                          ,
 "brms"                                               ,
 "CovTools"                                           ,
 "DTSg"                                               ,
 "edgar"                                              ,
 "enrichR"                                            ,
 "EpiModel"                                           ,
 "ExtremalDep"                                        ,
 "fhidata"                                            ,
 "getCRUCLdata"                                       ,
 "GPoM.FDLyapu"                                       ,
 "gtrendsR"                                           ,
 "HBV.IANIGLA"                                        ,
 "HelpersMG"                                          ,
 "HH"                                                 ,
 "highfrequency"                                      ,
 "iAdapt"                                             ,
 "IrishDirectorates"                                  ,
 "LBLGXE"                                             ,
 "LEGIT"                                              ,
 "LocalControlStrategy"                               ,
 "Mercator"                                           ,
 "mudata2"                                            ,
 "nlmixr"                                             ,
 "nord"                                               ,
 "oclust"                                             ,
 "OncoBayes2"                                         ,
 "pct"                                                ,
 "periscope"                                          ,
 "psychReport"                                        ,
 "restatapi"                                          ,
 "Rnightlights"                                       ,
 "runjags"                                            ,
 "RxODE"                                              ,
 "Ryacas"                                             ,
 "s2"                                                 ,
 "SECFISH"                                            ,
 "SingleCaseES"                                       ,
 "survPen"                                            ,
 "symmetry"                                           ,
 "tsdb"                                               ,
 "assignPOP"                                          ,
 "bcf"                                                ,
 "bivquant"                                           ,
 "cholera"                                            ,
 "cmenet"                                             ,
 "coin"                                               ,
 "cusum"                                              ,
 "datarobot"                                          ,
 "dbparser"                                           ,
 "ddsPLS"                                             ,
 "fastNaiveBayes"                                     ,
 "geomedb"                                            ,
 "getspres"                                           ,
 "ggpointdensity"                                     ,
 "gmvarkit"                                           ,
 "groupedstats"                                       ,
 "GSED"                                               ,
 "hdbm"                                               ,
 "lavaan"                                             ,
 "lodi"                                               ,
 "lumberjack"                                         ,
 "magickGUI"                                          ,
 "MANOVA.RM"                                          ,
 "MCPModGeneral"                                      ,
 "mdsOpt"                                             ,
 "memochange"                                         ,
 "MVar"                                               ,
 "ncdfgeom"                                           ,
 "ncmeta"                                             ,
 "oceanis"                                            ,
 "otpr"                                               ,
 "pairwiseComparisons"                                ,
 "phylopath"                                          ,
 "provParseR"                                         ,
 "quadrupen"                                          ,
 "ragg"                                               ,
 "rAmCharts"                                          ,
 "RBesT"                                              ,
 "rje"                                                ,
 "rmumps"                                             ,
 "segmentr"                                           ,
 "SHAPforxgboost"                                     ,
 "spinebil"                                           ,
 "sse"                                                ,
 "statsExpressions"                                   ,
 "swephR"                                             ,
 "TestDesign"                                         ,
 "textdata"                                           ,
 "TITAN2"                                             ,
 "TOHM"                                               ,
 "trackter"                                           ,
 "tradestatistics"                                    ,
 "tripsAndDipR"                                       ,
 "validate"                                           ,
 "vardpoor"                                           ,
 "visNetwork"                                         ,
 "wgaim"                                              ,
 "whisker"                                            ,
 "wyz.code.testthat"                                  ,
 "afex"                                               ,
 "AzureVM"                                            ,
 "CC"                                                 ,
 "cofad"                                              ,
 "ContaminatedMixt"                                   ,
 "DiPs"                                               ,
 "distr6"                                             ,
 "dynsurv"                                            ,
 "exceedProb"                                         ,
 "explore"                                            ,
 "flexsurvcure"                                       ,
 "fmcmc"                                              ,
 "fpop"                                               ,
 "ggExtra"                                            ,
 "gWQS"                                               ,
 "libcoin"                                            ,
 "MatchIt.mice"                                       ,
 "mixmeta"                                            ,
 "mma"                                                ,
 "MPTmultiverse"                                      ,
 "multiviewtest"                                      ,
 "MVar.pt"                                            ,
 "nhdplusTools"                                       ,
 "pcalg"                                              ,
 "puniform"                                           ,
 "Rankcluster"                                        ,
 "Rata"                                               ,
 "rcompanion"                                         ,
 "recommenderlab"                                     ,
 "RGISTools"                                          ,
 "RMixtCompUtilities"                                 ,
 "rrpack"                                             ,
 "SACOBRA"                                            ,
 "seriation"                                          ,
 "tidycells"                                          ,
 "uGMAR"                                              ,
 "windfarmGA"                                         ,
 "wyz.code.offensiveProgramming"                      ,
 "zen4R"                                              ,
 "BCEA"                                               ,
 "cansim"                                             ,
 "chron"                                              ,
 "cmsaf"                                              ,
 "dextergui"                                          ,
 "epimdr"                                             ,
 "epoc"                                               ,
 "fusedest"                                           ,
 "geohashTools"                                       ,
 "GMMAT"                                              ,
 "gprofiler2"                                         ,
 "hagis"                                              ,
 "iBreakDown"                                         ,
 "ingredients"                                        ,
 "inspectdf"                                          ,
 "KATforDCEMRI"                                       ,
 "LBSPR"                                              ,
 "lingtypology"                                       ,
 "magick"                                             ,
 "markovchain"                                        ,
 "mfe"                                                ,
 "mlr3db"                                             ,
 "mlr3learners"                                       ,
 "oceanmap"                                           ,
 "parameters"                                         ,
 "pkgbuild"                                           ,
 "pmdplyr"                                            ,
 "PMwR"                                               ,
 "provViz"                                            ,
 "pstest"                                             ,
 "rotasym"                                            ,
 "SimComp"                                            ,
 "sleepwalk"                                          ,
 "truelies"                                           ,
 "vimp"                                               ,
 "visit"                                              ,
 "yardstick"                                          ,
 "Coinprofile"                                        ,
 "comat"                                              ,
 "DALEX"                                              ,
 "expss"                                              ,
 "fipe"                                               ,
 "flashlight"                                         ,
 "gfoRmula"                                           ,
 "hpa"                                                ,
 "jack"                                               ,
 "kntnr"                                              ,
 "lessR"                                              ,
 "mlr3"                                               ,
 "npreg"                                              ,
 "phyclust"                                           ,
 "regtools"                                           ,
 "replicateBE"                                        ,
 "spGARCH"                                            ,
 "tidytransit"                                        ,
 "arcos"                                              ,
 "AzureVMmetadata"                                    ,
 "BANOVA"                                             ,
 "BayesX"                                             ,
 "bikm1"                                              ,
 "Cascade"                                            ,
 "citecorp"                                           ,
 "cosa"                                               ,
 "crone"                                              ,
 "dbflobr"                                            ,
 "dLagM"                                              ,
 "gripp"                                              ,
 "HCD"                                                ,
 "idiogramFISH"                                       ,
 "ijtiff"                                             ,
 "irtplay"                                            ,
 "lax"                                                ,
 "matrixsampling"                                     ,
 "MEDseq"                                             ,
 "oompaBase"                                          ,
 "Patterns"                                           ,
 "phonfieldwork"                                      ,
 "PostcodesioR"                                       ,
 "randomizeBE"                                        ,
 "RcppExamples"                                       ,
 "ShiftShareSE"                                       ,
 "SimBIID"                                            ,
 "solitude"                                           ,
 "sommer"                                             ,
 "sprintr"                                            ,
 "summarytools"                                       ,
 "bcrm"                                               ,
 "completejourney"                                    ,
 "covr"                                               ,
 "ctsem"                                              ,
 "dfadjust"                                           ,
 "dsr"                                                ,
 "economiccomplexity"                                 ,
 "EIAdata"                                            ,
 "epiR"                                               ,
 "fdm2id"                                             ,
 "gamair"                                             ,
 "GLDEX"                                              ,
 "gllvm"                                              ,
 "gmailr"                                             ,
 "hchinamap"                                          ,
 "hms"                                                ,
 "ICtest"                                             ,
 "iSTATS"                                             ,
 "komaletter"                                         ,
 "miceadds"                                           ,
 "mod"                                                ,
 "nph"                                                ,
 "PetfindeR"                                          ,
 "PML"                                                ,
 "portfolioBacktest"                                  ,
 "pricesensitivitymeter"                              ,
 "RcmdrPlugin.NMBU"                                   ,
 "RecordLinkage"                                      ,
 "Rwordseg"                                           ,
 "sicegar"                                            ,
 "sigora"                                             ,
 "TAM"                                                ,
 "tinytest"                                           ,
 "UCSCXenaShiny"                                      ,
 "wyz.code.rdoc"                                      ,
 "apexcharter"                                        ,
 "behaviorchange"                                     ,
 "bigutilsr"                                          ,
 "cursory"                                            ,
 "dexter"                                             ,
 "DoE.multi.response"                                 ,
 "Eagle"                                              ,
 "enviGCMS"                                           ,
 "esquisse"                                           ,
 "forecast"                                           ,
 "FPDclustering"                                      ,
 "genepop"                                            ,
 "GenEst"                                             ,
 "GillespieSSA2"                                      ,
 "gretel"                                             ,
 "MetaboList"                                         ,
 "MixGHD"                                             ,
 "mlr3misc"                                           ,
 "mRpostman"                                          ,
 "nandb"                                              ,
 "nat"                                                ,
 "paws"                                               ,
 "pmd"                                                ,
 "pulsar"                                             ,
 "radiant.data"                                       ,
 "raster"                                             ,
 "Rirt"                                               ,
 "rpart.plot"                                         ,
 "rscielo"                                            ,
 "simGWAS"                                            ,
 "STMotif"                                            ,
 "tdthap"                                             ,
 "ufs"                                                ,
 "units"                                              ,
 "zoib"                                               ,
 "apcluster"                                          ,
 "av"                                                 ,
 "billboarder"                                        ,
 "Biopeak"                                            ,
 "blogdown"                                           ,
 "bnviewer"                                           ,
 "bookdown"                                           ,
 "burnr"                                              ,
 "CBPS"                                               ,
 "Chaos01"                                            ,
 "circumplex"                                         ,
 "clustree"                                           ,
 "Countr"                                             ,
 "d3r"                                                ,
 "DataSpaceR"                                         ,
 "DClusterm"                                          ,
 "denvax"                                             ,
 "deTestSet"                                          ,
 "DRAYL"                                              ,
 "DTRreg"                                             ,
 "durmod"                                             ,
 "factory"                                            ,
 "GD"                                                 ,
 "gdalcubes"                                          ,
 "GENEAclassify"                                      ,
 "genodds"                                            ,
 "ggimage"                                            ,
 "heuristica"                                         ,
 "ICD10gm"                                            ,
 "ipfr"                                               ,
 "KernSmoothIRT"                                      ,
 "longitudinalcascade"                                ,
 "ludic"                                              ,
 "NetworkToolbox"                                     ,
 "NPP"                                                ,
 "optionstrat"                                        ,
 "pagedown"                                           ,
 "partitionBEFsp"                                     ,
 "paws.analytics"                                     ,
 "paws.application.integration"                       ,
 "paws.compute"                                       ,
 "paws.cost.management"                               ,
 "paws.customer.engagement"                           ,
 "paws.database"                                      ,
 "paws.machine.learning"                              ,
 "paws.management"                                    ,
 "paws.networking"                                    ,
 "paws.security.identity"                             ,
 "paws.storage"                                       ,
 "pccc"                                               ,
 "qs"                                                 ,
 "rdlocrand"                                          ,
 "rmarkdown"                                          ,
 "rticles"                                            ,
 "samc"                                               ,
 "scorepeak"                                          ,
 "shinylogs"                                          ,
 "spANOVA"                                            ,
 "splot"                                              ,
 "SWMPrExtension"                                     ,
 "sys"                                                ,
 "umap"                                               ,
 "vici"                                               ,
 "viromeBrowser"                                      ,
 "webutils"                                           ,
 "xfun"                                               ,
 "xgxr"                                               ,
 "BallMapper"                                         ,
 "bestNormalize"                                      ,
 "bioacoustics"                                       ,
 "birdring"                                           ,
 "blackbox"                                           ,
 "bvartools"                                          ,
 "cbsodataR"                                          ,
 "clean"                                              ,
 "comorbidity"                                        ,
 "crseEventStudy"                                     ,
 "DataVisualizations"                                 ,
 "dexterMST"                                          ,
 "difNLR"                                             ,
 "disk.frame"                                         ,
 "dlsem"                                              ,
 "dst"                                                ,
 "dtpcrm"                                             ,
 "electivity"                                         ,
 "GENEAread"                                          ,
 "ggforce"                                            ,
 "graphlayouts"                                       ,
 "gtsummary"                                          ,
 "HiCblock"                                           ,
 "hutilscpp"                                          ,
 "icarus"                                             ,
 "ihpdr"                                              ,
 "implicitMeasures"                                   ,
 "kernscr"                                            ,
 "lactcurves"                                         ,
 "lgr"                                                ,
 "MazamaCoreUtils"                                    ,
 "mDAG"                                               ,
 "metadynminer"                                       ,
 "monotonicity"                                       ,
 "NetSimR"                                            ,
 "noisyCE2"                                           ,
 "overlapping"                                        ,
 "pass.lme"                                           ,
 "paws.common"                                        ,
 "Peptides"                                           ,
 "photobiologyInOut"                                  ,
 "pksensi"                                            ,
 "qrandom"                                            ,
 "RCzechia"                                           ,
 "readabs"                                            ,
 "RgoogleMaps"                                        ,
 "rSPDE"                                              ,
 "scoringRules"                                       ,
 "semPlot"                                            ,
 "Seurat"                                             ,
 "spread"                                             ,
 "ssizeRNA"                                           ,
 "suddengains"                                        ,
 "sugarbag"                                           ,
 "testequavar"                                        ,
 "vcr"                                                ,
 "visualR"                                            ,
 "xlink"                                              ,
 "APfun"                                              ,
 "APIS"                                               ,
 "ari"                                                ,
 "asnipe"                                             ,
 "benchmarkme"                                        ,
 "benchmarkmeData"                                    ,
 "biclique"                                           ,
 "BiDAG"                                              ,
 "BNPTSclust"                                         ,
 "BRISC"                                              ,
 "broomExtra"                                         ,
 "CAISEr"                                             ,
 "CalibratR"                                          ,
 "citr"                                               ,
 "ClickClustCont"                                     ,
 "cotram"                                             ,
 "cytometree"                                         ,
 "dlookr"                                             ,
 "drake"                                              ,
 "edeaR"                                              ,
 "EmbedSOM"                                           ,
 "envDocument"                                        ,
 "eSDM"                                               ,
 "fields"                                             ,
 "FrF2"                                               ,
 "GDINA"                                              ,
 "ggstance"                                           ,
 "googledrive"                                        ,
 "handlr"                                             ,
 "harmonicmeanp"                                      ,
 "HellCor"                                            ,
 "HiCglmi"                                            ,
 "hommel"                                             ,
 "hpackedbubble"                                      ,
 "Infusion"                                           ,
 "ispd"                                               ,
 "JADE"                                               ,
 "jsmodule"                                           ,
 "khroma"                                             ,
 "kiwisR"                                             ,
 "lpSolve"                                            ,
 "MAINT.Data"                                         ,
 "MEtest"                                             ,
 "MetricsWeighted"                                    ,
 "MGDrivE"                                            ,
 "miceFast"                                           ,
 "mlflow"                                             ,
 "MSPRT"                                              ,
 "onewaytests"                                        ,
 "opalr"                                              ,
 "pangaear"                                           ,
 "PLmixed"                                            ,
 "RcppQuantuccia"                                     ,
 "rgl"                                                ,
 "rjags"                                              ,
 "rmapzen"                                            ,
 "rmytarget"                                          ,
 "rnassqs"                                            ,
 "Rpoet"                                              ,
 "rqdatatable"                                        ,
 "RTransferEntropy"                                   ,
 "RWsearch"                                           ,
 "ryandexdirect"                                      ,
 "scam"                                               ,
 "securitytxt"                                        ,
 "sharpData"                                          ,
 "shinybusy"                                          ,
 "smooth"                                             ,
 "sparsevar"                                          ,
 "Spectrum"                                           ,
 "spiderbar"                                          ,
 "spiritR"                                            ,
 "svd"                                                ,
 "tidyLPA"                                            ,
 "TLMoments"                                          ,
 "transport"                                          ,
 "Tushare"                                            ,
 "txtq"                                               ,
 "UniprotR"                                           ,
 "uroot"                                              ,
 "weibullness"                                        ,
 "march"                                              ,
 "RLumModel"                                          ,
 "ElemStatLearn"                                      ,
 "GlmSimulatoR"                                       ,
 "MHCtools"                                           ,
 "ds4psy"                                             ,
 "ggplot2"                                            ,
 "oppr"                                               ,
 "overture"                                           ,
 "yarr"                                               ,
 "BESTree"                                            ,
 "butcher"                                            ,
 "censusxy"                                           ,
 "cyanoFilter"                                        ,
 "grainscape"                                         ,
 "greta"                                              ,
 "interimApp"                                         ,
 "PPMR"                                               ,
 "protag"                                             ,
 "scrypt"                                             ,
 "shinyML"                                            ,
 "simstudy"                                           ,
 "sos4R"                                              ,
 "spatstat.Knet"                                      ,
 "SuperLearner"                                       ,
 "tabulog"                                            ,
 "thurstonianIRT"                                     ,
 "tidytree"                                           ,
 "valetr"                                             ,
 "VOSONDash"                                          ,
 "webmockr"                                           ,
 "xml2"                                               ,
 "accept"                                             ,
 "arrow"                                              ,
 "aRxiv"                                              ,
 "aws.signature"                                      ,
 "blavaan"                                            ,
 "bootnet"                                            ,
 "classyfireR"                                        ,
 "CorBin"                                             ,
 "diffman"                                            ,
 "EventDetectR"                                       ,
 "fabletools"                                         ,
 "GetLattesData"                                      ,
 "ggsignif"                                           ,
 "joinet"                                             ,
 "knitr"                                              ,
 "lidR"                                               ,
 "lpSolveAPI"                                         ,
 "lspartition"                                        ,
 "meteor"                                             ,
 "modelr"                                             ,
 "mrfse"                                              ,
 "mssm"                                               ,
 "NNS"                                                ,
 "sabarsi"                                            ,
 "sass"                                               ,
 "shar"                                               ,
 "shinyMatrix"                                        ,
 "swCRTdesign"                                        ,
 "symbolicDA"                                         ,
 "tidygenomics"                                       ,
 "tmcn"                                               ,
 "websocket"                                          ,
 "ADAPTS"                                             ,
 "bioOED"                                             ,
 "cder"                                               ,
 "ClimMobTools"                                       ,
 "compstatr"                                          ,
 "DistributionTest"                                   ,
 "DT"                                                 ,
 "dyndimred"                                          ,
 "EBPRS"                                              ,
 "Epi"                                                ,
 "exactextractr"                                      ,
 "featuretoolsR"                                      ,
 "geojsonR"                                           ,
 "hmm.discnp"                                         ,
 "hwordcloud"                                         ,
 "landscapemetrics"                                   ,
 "markdown"                                           ,
 "mlrCPO"                                             ,
 "MWright"                                            ,
 "netmeta"                                            ,
 "nngeo"                                              ,
 "onelogin"                                           ,
 "opencpu"                                            ,
 "OpenML"                                             ,
 "planets"                                            ,
 "qrng"                                               ,
 "quantreg"                                           ,
 "rapbase"                                            ,
 "RQuantLib"                                          ,
 "SCORPIUS"                                           ,
 "servr"                                              ,
 "ShortForm"                                          ,
 "shrinkTVP"                                          ,
 "SimJoint"                                           ,
 "sindyr"                                             ,
 "tikzDevice"                                         ,
 "tinytex"                                            ,
 "TrendSLR"                                           ,
 "webTRISr"                                           ,
 "bayesmeta"                                          ,
 "bayestestR"                                         ,
 "cat.dt"                                             ,
 "classInt"                                           ,
 "correlationfunnel"                                  ,
 "cvcqv"                                              ,
 "distantia"                                          ,
 "ethnobotanyR"                                       ,
 "exdex"                                              ,
 "flamingos"                                          ,
 "flightplanning"                                     ,
 "ggplotify"                                          ,
 "hdfqlr"                                             ,
 "HextractoR"                                         ,
 "huxtable"                                           ,
 "inlmisc"                                            ,
 "IRISSeismic"                                        ,
 "meta"                                               ,
 "metasens"                                           ,
 "mlr"                                                ,
 "nnTensor"                                           ,
 "parsnip"                                            ,
 "PupilPre"                                           ,
 "RaProR"                                             ,
 "rgrass7"                                            ,
 "rlfsm"                                              ,
 "RobStatTM"                                          ,
 "SemNetCleaner"                                      ,
 "shinyTime"                                          ,
 "squid"                                              ,
 "tsbox"                                              ,
 "visvow"                                             ,
 "BANEScarparkinglite"                                ,
 "BatchGetSymbols"                                    ,
 "BayesPostEst"                                       ,
 "bnlearn"                                            ,
 "climatol"                                           ,
 "dbscan"                                             ,
 "DRAFT"                                              ,
 "dssd"                                               ,
 "experiment"                                         ,
 "FCSlib"                                             ,
 "fslr"                                               ,
 "geobr"                                              ,
 "gMOIP"                                              ,
 "golem"                                              ,
 "groupdata2"                                         ,
 "httr"                                               ,
 "imagefx"                                            ,
 "importinegi"                                        ,
 "IsoplotR"                                           ,
 "JFE"                                                ,
 "jmuOutlier"                                         ,
 "kofdata"                                            ,
 "kStatistics"                                        ,
 "meme"                                               ,
 "MetabolicSurv"                                      ,
 "mets"                                               ,
 "mlmm.gwas"                                          ,
 "mvLSW"                                              ,
 "negenes"                                            ,
 "NewmanOmics"                                        ,
 "oncrawlR"                                           ,
 "performance"                                        ,
 "pivottabler"                                        ,
 "polycor"                                            ,
 "projmgr"                                            ,
 "REndo"                                              ,
 "Rfast2"                                             ,
 "rflights"                                           ,
 "rgeos"                                              ,
 "RISCA"                                              ,
 "rSPARCS"                                            ,
 "rTorch"                                             ,
 "scan"                                               ,
 "sccr"                                               ,
 "scoper"                                             ,
 "SemNeT"                                             ,
 "SemNetDictionaries"                                 ,
 "sensemakr"                                          ,
 "seqgendiff"                                         ,
 "slga"                                               ,
 "stuart"                                             ,
 "tensorsparse"                                       ,
 "valueEQ5D"                                          ,
 "cdparcoord"                                         ,
 "exampletestr"                                       ,
 "genderizeR"                                         ,
 "labdsv"                                             ,
 "makedummies"                                        ,
 "MetaLandSim"                                        ,
 "rsparse"                                            ,
 "Spbsampling"                                        ,
 "TPmsm"                                              ,
 "trackdf"                                            ,
 "vegperiod"                                          ,
 "AzureStor"                                          ,
 "bnspatial"                                          ,
 "CAWaR"                                              ,
 "EcoIndR"                                            ,
 "foreSIGHT"                                          ,
 "hmlasso"                                            ,
 "intsurv"                                            ,
 "ktaucenters"                                        ,
"MPV"                                                 ,
"NHSRdatasets"                                        ,
"pCODE"                                               ,
"RcppCCTZ"                                            ,
"RMaCzek"                                             ,
"rmdplugr"                                            ,
"admisc"                                              ,
"BayesCTDesign"                                       ,
"BayesianFROC"                                        ,
"bayest"                                              ,
"BHSBVAR"                                             ,
"bigKRLS"                                             ,
"BMhyb"                                               ,
"bpgmm"                                               ,
"c14bazAAR"                                           ,
"cairoDevice"                                         ,
"CHNOSZ"                                              ,
"cpcens"                                              ,
"crul"                                                ,
"datrProfile"                                         ,
"describedata"                                        ,
"doParallel"                                          ,
"DTDA.cif"                                            ,
"easyreg"                                             ,
"elo"                                                 ,
"fillr"                                               ,
"flare"                                               ,
"foreign"                                             ,
"GeneralisedCovarianceMeasure"                        ,
"Ghat"                                                ,
"GWASinspector"                                       ,
"iCellR"                                              ,
"installr"                                            ,
"Kernelheaping"                                       ,
"leafR"                                               ,
"lognorm"                                             ,
"malariaAtlas"                                        ,
"MCI2"                                                ,
"mlmi"                                                ,
"neonUtilities"                                       ,
"phenModel"                                           ,
"pmsampsize"                                          ,
"pyinit"                                              ,
"RcppHungarian"                                       ,
"ROpenCVLite"                                         ,
"rQCC"                                                ,
"rtika"                                               ,
"shades"                                              ,
"shapper"                                             ,
"shinyMolBio"                                         ,
"simdistr"                                            ,
"simExam"                                             ,
"sociome"                                             ,
"SpatialExtremes"                                     ,
"tokenizers.bpe"                                      ,
"topoDistance"                                        ,
"TSplotly"                                            ,
"whoa"                                                ,
"aaSEA"                                               ,
"algorithmia"                                         ,
"baggr"                                               ,
"bioinactivation"                                     ,
"biwavelet"                                           ,
"bnpa"                                                ,
"CA3variants"                                         ,
"ClustImpute"                                         ,
"coxed"                                               ,
"DGLMExtPois"                                         ,
"emmeans"                                             ,
"GeneBook"                                            ,
"h2o"                                                 ,
"idefix"                                              ,
"idmodelr"                                            ,
"lava"                                                ,
"lazytrade"                                           ,
"LDATS"                                               ,
"LDlinkR"                                             ,
"lifecycle"                                           ,
"MachineShop"                                         ,
"mpath"                                               ,
"mvSLOUCH"                                            ,
"nlme"                                                ,
"nltm"                                                ,
"nowcasting"                                          ,
"obfuscatoR"                                          ,
"plothelper"                                          ,
"Polychrome"                                          ,
"rbart"                                               ,
"Replication"                                         ,
"Rmosek"                                              ,
"RNifti"                                              ,
"see"                                                 ,
"sirt"                                                ,
"sitree"                                              ,
"sitreeE"                                             ,
"tbl2xts"                                             ,
"tdigest"                                             ,
"tensorflow"                                          ,
"tfCox"                                               ,
"tidypmc"                                             ,
"wikifacts"                                           ,
"xgboost"                                             ,
"zenplots"                                            ,
"zFactor"                                             ,
"activityCounts"                                      ,
"AER"                                                 ,
"BinaryEPPM"                                          ,
"biplotbootGUI"                                       ,
"coxrt"                                               ,
"deldir"                                              ,
"DiceDesign"                                          ,
"fdasrvf"                                             ,
"fivethirtyeight"                                     ,
"GOFShiny"                                            ,
"greybox"                                             ,
"lfda"                                                ,
"Luminescence"                                        ,
"miWQS"                                               ,
"mobForest"                                           ,
"MoEClust"                                            ,
"mortAAR"                                             ,
"mwshiny"                                             ,
"namedCapture"                                        ,
"nlr"                                                 ,
"ParBayesianOptimization"                             ,
"PeakSegDisk"                                         ,
"rcosmo"                                              ,
"RJafroc"                                             ,
"rless"                                               ,
"segRDA"                                              ,
"simex"                                               ,
"sizeMat"                                             ,
"SpatialPack"                                         ,
"updog"                                               ,
"velociraptr"                                         ,
"AGHmatrix"                                           ,
"bamlss"                                              ,
"canvasXpress"                                        ,
"elasticsearchr"                                      ,
"FEprovideR"                                          ,
"forwards"                                            ,
"frailtyEM"                                           ,
"jagsUI"                                              ,
"LMest"                                               ,
"mfGARCH"                                             ,
"MVTests"                                             ,
"pdfetch"                                             ,
"quanteda"                                            ,
"radiant.basics"                                      ,
"Recon"                                               ,
"ShrinkCovMat"                                        ,
"simmer"                                              ,
"stplanr"                                             ,
"timereg"                                             ,
"trustedtimestamping"                                 ,
"unifed"                                              ,
"aplpack"                                             ,
"bayesm"                                              ,
"bhm"                                                 ,
"bsamGP"                                              ,
"castor"                                              ,
"clr"                                                 ,
"corr2D"                                              ,
"fieldRS"                                             ,
"gitignore"                                           ,
"gradeR"                                              ,
"JointNets"                                           ,
"jomo"                                                ,
"LAGOSNE"                                             ,
"MSEtool"                                             ,
"newsmap"                                             ,
"PKNCA"                                               ,
"pvaluefunctions"                                     ,
"reportROC"                                           ,
"RNewsflow"                                           ,
"RootsExtremaInflections"                             ,
"rquery"                                              ,
"rsvd"                                                ,
"seewave"                                             ,
"spatialrisk"                                         ,
"tidytext"                                            ,
"tricolore"                                           ,
"tsibble"                                             ,
"AHM"                                                 ,
"cancensus"                                           ,
"costsensitive"                                       ,
"fastmap"                                             ,
"FluMoDL"                                             ,
"IMAGE"                                               ,
"ktsolve"                                             ,
"longclust"                                           ,
"MatSkew"                                             ,
"metadynminer3d"                                      ,
"MixMatrix"                                           ,
"RAhrefs"                                             ,
"rnbp"                                                ,
"rsleep"                                              ,
"rvkstat"                                             ,
"samurais"                                            ,
"stats19"                                             ,
"bannerCommenter"                                     ,
"brranching"                                          ,
"dataMaid"                                            ,
"doMC"                                                ,
"doSNOW"                                              ,
"easyanova"                                           ,
"flexmixNL"                                           ,
"foreach"                                             ,
"MapGAM"                                              ,
"offlineChange"                                       ,
"rankdist"                                            ,
"slfm"                                                ,
"SmoothWin"                                           ,
"vtreat"                                              ,
"baggedcv"                                            ,
"bdpar"                                               ,
"cNORM"                                               ,
"ergMargins"                                          ,
"gargle"                                              ,
"geoparser"                                           ,
"GillespieSSA"                                        ,
"GMDHreg"                                             ,
"imgpalr"                                             ,
"iterators"                                           ,
"litteR"                                              ,
"mmetrics"                                            ,
"NetMix"                                              ,
"PACVr"                                               ,
"plotdap"                                             ,
"radiant.model"                                       ,
"Rbeast"                                              ,
"stmgp"                                               ,
"tcpl"                                                ,
"tsviz"                                               ,
"WeibullFit"                                          ,
"actuar"                                              ,
"bigstep"                                             ,
"bs4Dash"                                             ,
"ChemoSpec"                                           ,
"ChemoSpecUtils"                                      ,
"diceR"                                               ,
"eia"                                                 ,
"flobr"                                               ,
"HACSim"                                              ,
"MBNMAdose"                                           ,
"mistr"                                               ,
"ordinalClust"                                        ,
"pcFactorStan"                                        ,
"PRIMME"                                              ,
"Rcpp"                                                ,
"Rlabkey"                                             ,
"rosetta"                                             ,
"samplingVarEst"                                      ,
"sars"                                                ,
"SCMA"                                                ,
"SCRT"                                                ,
"SCVA"                                                ,
"SimCorMultRes"                                       ,
"SpatialAcc"                                          ,
"svars"                                               ,
"taxotools"                                           ,
"tesseract"                                           ,
"testthat"                                            ,
"treeplyr"                                            ,
"UncertainInterval"                                   ,
"vein"                                                ,
"vwline"                                              ,
"cdata"                                               ,
"DCEM"                                                ,
"detpack"                                             ,
"EGAnet"                                              ,
"EMP"                                                 ,
"FSA"                                                 ,
"hmstimer"                                            ,
"IATscores"                                           ,
"ICAMS"                                               ,
"lacunaritycovariance"                                ,
"lslx"                                                ,
"mediacloudr"                                         ,
"OmicsPLS"                                            ,
"phylocomr"                                           ,
"RcmdrPlugin.SCDA"                                    ,
"RcppDynProg"                                         ,
"replyr"                                              ,
"reticulate"                                          ,
"rpact"                                               ,
"RSQLite"                                             ,
"rWind"                                               ,
"seplyr"                                              ,
"sf"                                                  ,
"sigr"                                                ,
"socialmixr"                                          ,
"spatialEco"                                          ,
"spmoran"                                             ,
"stopwords"                                           ,
"wrapr"                                               ,
"WVPlots"                                             ,
"acfMPeriod"                                          ,
"BetaPASS"                                            ,
"BNPmix"                                              ,
"ckanr"                                               ,
"clipr"                                               ,
"cmna"                                                ,
"CNVScope"                                            ,
"CorrectOverloadedPeaks"                              ,
"cvGEE"                                               ,
"fastGraph"                                           ,
"gitlink"                                             ,
"graph4lg"                                            ,
"leiden"                                              ,
"MedSurvey"                                           ,
"MetaIntegrator"                                      ,
"parade"                                              ,
"ppclust"                                             ,
"PredPsych"                                           ,
"prevederer"                                          ,
"ProFit"                                              ,
"pubmed.mineR"                                        ,
"readwritesqlite"                                     ,
"Rnets"                                               ,
"robcbi"                                              ,
"rray"                                                ,
"Rsagacmd"                                            ,
"RStoolbox"                                           ,
"slm"                                                 ,
"tidymargins"                                         ,
"cold"                                                ,
"cranly"                                              ,
"curl"                                                ,
"evaluator"                                           ,
"gert"                                                ,
"GetDFPData"                                          ,
"GMDH2"                                               ,
"its.analysis"                                        ,
"mlogit"                                              ,
"PELVIS"                                              ,
"purrrogress"                                         ,
"Renvlp"                                              ,
"ResourceSelection"                                   ,
"rosm"                                                ,
"rsconnect"                                           ,
"sae2"                                                ,
"shopifyr"                                            ,
"synthACS"                                            ,
"AdvDif4"                                             ,
"augmentedRCBD"                                       ,
"bridgesampling"                                      ,
"implyr"                                              ,
"permutes"                                            ,
"pROC"                                                ,
"proxyC"                                              ,
"RPushbullet"                                         ,
"simts"                                               ,
"ceramic"                                             ,
"CeRNASeek"                                           ,
"dendroTools"                                         ,
"doFuture"                                            ,
"ezplot"                                              ,
"modeldb"                                             ,
"munsellinterpol"                                     ,
"qif"                                                 ,
"rerddap"                                             ,
"RZigZag"                                             ,
"alfr"                                                ,
"available"                                           ,
"borrowr"                                             ,
"dataPreparation"                                     ,
"ddpca"                                               ,
"deaR"                                                ,
"finalfit"                                            ,
"forestRK"                                            ,
"GLMpack"                                             ,
"jaggR"                                               ,
"jubilee"                                             ,
"KCSKNNShiny"                                         ,
"LilRhino"                                            ,
"lutz"                                                ,
"meanr"                                               ,
"mnreadR"                                             ,
"MomTrunc"                                            ,
"mortyr"                                              ,
"MplusTrees"                                          ,
"plsmselect"                                          ,
"rSHAPE"                                              ,
"sdcSpatial"                                          ,
"sglg"                                                ,
"shazam"                                              ,
"ssmrob"                                              ,
"SWIM"                                                ,
"TestDimorph"                                         ,
"tigger"                                              ,
"TSEind"                                              ,
"acebayes"                                            ,
"activPAL"                                            ,
"alakazam"                                            ,
"animint2"                                            ,
"ANOVAreplication"                                    ,
"antaresEditObject"                                   ,
"antaresRead"                                         ,
"AnthropMMD"                                          ,
"AzureKeyVault"                                       ,
"AzureRMR"                                            ,
"basetheme"                                           ,
"basicMCMCplots"                                      ,
"bltm"                                                ,
"bootImpute"                                          ,
"callr"                                               ,
"CautiousLearning"                                    ,
"cld3"                                                ,
"ClimInd"                                             ,
"ClusterR"                                            ,
"contribution"                                        ,
"DatabaseConnector"                                   ,
"dplR"                                                ,
"echarts4r"                                           ,
"effsize"                                             ,
"fastJT"                                              ,
"fingertipscharts"                                    ,
"fingertipsR"                                         ,
"ForestFit"                                           ,
"freesurfer"                                          ,
"gdalUtilities"                                       ,
"GEEmediate"                                          ,
"ggfittext"                                           ,
"ggparty"                                             ,
"greenclust"                                          ,
"horseshoe"                                           ,
"iai"                                                 ,
"kosel"                                               ,
"loon.data"                                           ,
"matrixStrucTest"                                     ,
"MatTransMix"                                         ,
"metacoder"                                           ,
"mined"                                               ,
"moderndive"                                          ,
"MSbox"                                               ,
"multiColl"                                           ,
"openssl"                                             ,
"opticskxi"                                           ,
"partykit"                                            ,
"perfectphyloR"                                       ,
"prettydoc"                                           ,
"processx"                                            ,
"PWFSLSmoke"                                          ,
"qad"                                                 ,
"R2BEAT"                                              ,
"RATest"                                              ,
"RcppXsimd"                                           ,
"rfinterval"                                          ,
"RKEEL"                                               ,
"robeth"                                              ,
"RobustAFT"                                           ,
"RVenn"                                               ,
"sabre"                                               ,
"SCINA"                                               ,
"sdcHierarchies"                                      ,
"simPop"                                              ,
"sismonr"                                             ,
"splinetree"                                          ,
"SRTtools"                                            ,
"strapgod"                                            ,
"teamr"                                               ,
"tmap"                                                ,
"tmaptools"                                           ,
"TSdist"                                              ,
"vosonSML"                                            ,
"VSURF"                                               ,
"VTShiny"                                             ,
"aws.ec2metadata"                                     ,
"AzureContainers"                                     ,
"coalitions"                                          ,
"damr"                                                ,
"DescToolsAddIns"                                     ,
"deSolve"                                             ,
"DVHmetrics"                                          ,
"DynamicGP"                                           ,
"exuber"                                              ,
"GetTDData"                                           ,
"grec"                                                ,
"growthPheno"                                         ,
"GWmodel"                                             ,
"icemelt"                                             ,
"imagine"                                             ,
"leaflet.providers"                                   ,
"mergeTrees"                                          ,
"modelsummary"                                        ,
"NonNorMvtDist"                                       ,
"pcIRT"                                               ,
"pyramid"                                             ,
"Repliscope"                                          ,
"repurrrsive"                                         ,
"revtools"                                            ,
"RRI"                                                 ,
"rsimsum"                                             ,
"simecol"                                             ,
"smerc"                                               ,
"sparsesvd"                                           ,
"support"                                             ,
"surveysd"                                            ,
"teamcolors"                                          ,
"TrackReconstruction"                                 ,
"TSsmoothing"                                         ,
"VineCopula"                                          ,
"dslabs"                                              ,
"HURDAT"                                              ,
"inegiR"                                              ,
"LDheatmap"                                           ,
"logging"                                             ,
"rolldown"                                            ,
"shotGroups"                                          ,
"DiscreteFDR"                                         ,
"DoE.MIParray"                                        ,
"EcoNetGen"                                           ,
"egg"                                                 ,
"INDperform"                                          ,
"mrgsolve"                                            ,
"multinet"                                            ,
"qgcomp"                                              ,
"Qtools"                                              ,
"repeated"                                            ,
"spotifyr"                                            ,
"wordspace"                                           ,
"ClimProjDiags"                                       ,
"corrr"                                               ,
"cricketr"                                            ,
"diffeqr"                                             ,
"embed"                                               ,
"GET"                                                 ,
"hindex"                                              ,
"imgrec"                                              ,
"intervcomp"                                          ,
"IRkernel"                                            ,
"lda.svi"                                             ,
"LogConcDEAD"                                         ,
"lpdensity"                                           ,
"mipred"                                              ,
"odds.n.ends"                                         ,
"optmatch"                                            ,
"osrm"                                                ,
"panelr"                                              ,
"paradox"                                             ,
"qMRI"                                                ,
"quint"                                               ,
"rddensity"                                           ,
"Rfast"                                               ,
"rosr"                                                ,
"rsample"                                             ,
"sgd"                                                 ,
"sinx"                                                ,
"SSDforR"                                             ,
"vitae"                                               ,
"xaringan"                                            ,
"XGR"                                                 ,
"altair"                                              ,
"AsyK"                                                ,
"beyondWhittle"                                       ,
"BNPdensity"                                          ,
"chandwich"                                           ,
"commonsMath"                                         ,
"cowplot"                                             ,
"DEEVD"                                               ,
"DELTD"                                               ,
"genlasso"                                            ,
"heatwaveR"                                           ,
"hisse"                                               ,
"httk"                                                ,
"iCARH"                                               ,
"jackalope"                                           ,
"kdensity"                                            ,
"microseq"                                            ,
"mize"                                                ,
"NeuralSens"                                          ,
"packageRank"                                         ,
"parsetools"                                          ,
"ramchoice"                                           ,
"rayshader"                                           ,
"redlistr"                                            ,
"rehh"                                                ,
"rematch2"                                            ,
"utile.visuals"                                       ,
"asymmetry"                                           ,
"balance"                                             ,
"BSL"                                                 ,
"cdcsis"                                              ,
"concurve"                                            ,
"depmixS4"                                            ,
"detzrcr"                                             ,
"DramaAnalysis"                                       ,
"ds"                                                  ,
"dti"                                                 ,
"FLSSS"                                               ,
"FractalParameterEstimation"                          ,
"gets"                                                ,
"graphTweets"                                         ,
"idealstan"                                           ,
"IndexNumR"                                           ,
"jaatha"                                              ,
"merDeriv"                                            ,
"metaBMA"                                             ,
"mice"                                                ,
"MinBAR"                                              ,
"PAutilities"                                         ,
"pbmcapply"                                           ,
"portalr"                                             ,
"propr"                                               ,
"rbi"                                                 ,
"rbi.helpers"                                         ,
"retistruct"                                          ,
"RItools"                                             ,
"shinyglide"                                          ,
"twitterwidget"                                       ,
"VCA"                                                 ,
"AzureGraph"                                          ,
"babelwhale"                                          ,
"bamdit"                                              ,
"blob"                                                ,
"bnstruct"                                            ,
"catchr"                                              ,
"coala"                                               ,
"electionsBR"                                         ,
"GpGp"                                                ,
"LexisNexisTools"                                     ,
"matrixNormal"                                        ,
"matrixTests"                                         ,
"mboxr"                                               ,
"micemd"                                              ,
"MKmisc"                                              ,
"reproducible"                                        ,
"rjmcmc"                                              ,
"rscala"                                              ,
"rstan"                                               ,
"serieslcb"                                           ,
"shinyAce"                                            ,
"srvyr"                                               ,
"stoRy"                                               ,
"svgViewR"                                            ,
"tsiR"                                                ,
"ulid"                                                ,
"VWPre"                                               ,
"WMWssp"                                              ,
"AnalyzeFMRI"                                         ,
"autothresholdr"                                      ,
"AWR.Athena"                                          ,
"bzinb"                                               ,
"ClustBlock"                                          ,
"contextual"                                          ,
"detrendr"                                            ,
"DSAIRM"                                              ,
"econetwork"                                          ,
"EmpiricalCalibration"                                ,
"EpiEstim"                                            ,
"eplusr"                                              ,
"fastICA"                                             ,
"genBaRcode"                                          ,
"manymodelr"                                          ,
"mclust"                                              ,
"metR"                                                ,
"ppgmmga"                                             ,
"randomForestSRC"                                     ,
"wand"                                                ,
"AMPLE"                                               ,
"crso"                                                ,
"dtwclust"                                            ,
"E4tools"                                             ,
"Gmisc"                                               ,
"intkrige"                                            ,
"netjack"                                             ,
"pdR"                                                 ,
"probably"                                            ,
"rcheology"                                           ,
"reactR"                                              ,
"rSymbiota"                                           ,
"sptemExp"                                            ,
"SQRL"                                                ,
"endtoend"                                            ,
"energy"                                              ,
"exams"                                               ,
"hopbyhop"                                            ,
"lcopula"                                             ,
"miniCRAN"                                            ,
"nopaco"                                              ,
"psychonetrics"                                       ,
"RKHSMetaMod"                                         ,
"rLandsat"                                            ,
"scbursts"                                            ,
"TestCor"                                             ,
"vioplot"                                             ,
"weightr"                                             ,
"BioMedR"                                             ,
"boot"                                                ,
"coda"                                                ,
"DataViz"                                             ,
"esmprep"                                             ,
"ggnewscale"                                          ,
"GUTS"                                                ,
"imputeTestbench"                                     ,
"interactions"                                        ,
"magicaxis"                                           ,
"ptmixed"                                             ,
"rrd"                                                 ,
"rvinecopulib"                                        ,
"searcher"                                            ,
"soundgen"                                            ,
"statar"                                              ,
"StratifiedBalancing"                                 ,
"ternvis"                                             ,
"tmt"                                                 ,
"tsensembler"                                         ,
"udpipe"                                              ,
"vctrs"                                               ,
"Xplortext"                                           ,
"abn"                                                 ,
"bomrang"                                             ,
"checkmate"                                           ,
"compound.Cox"                                        ,
"coxme"                                               ,
"ctrlGene"                                            ,
"dabestr"                                             ,
"digest"                                              ,
"dplyr"                                               ,
"DSAIDE"                                              ,
"EloChoice"                                           ,
"emax.glm"                                            ,
"ggpval"                                              ,
"haven"                                               ,
"iCiteR"                                              ,
"ips"                                                 ,
"mds"                                                 ,
"mkin"                                                ,
"not"                                                 ,
"spacyr"                                              ,
"sparklyr"                                            ,
"TSMSN"                                               ,
"usethis"                                             ,
"blockForest"                                         ,
"cassandRa"                                           ,
"CrossVA"                                             ,
"dad"                                                 ,
"expoRkit"                                            ,
"FactoInvestigate"                                    ,
"FactoMineR"                                          ,
"factorMerger"                                        ,
"foieGras"                                            ,
"fxtract"                                             ,
"gam"                                                 ,
"ggmcmc"                                              ,
"glinternet"                                          ,
"heemod"                                              ,
"hesim"                                               ,
"kernelTDA"                                           ,
"KScorrect"                                           ,
"LeafArea"                                            ,
"MarketMatching"                                      ,
"marmap"                                              ,
"modeLLtest"                                          ,
"Omisc"                                               ,
"plot.matrix"                                         ,
"PRSim"                                               ,
"psychNET"                                            ,
"radir"                                               ,
"rbacon"                                              ,
"RcppAlgos"                                           ,
"retractcheck"                                        ,
"RPEIF"                                               ,
"RSGHB"                                               ,
"SAFD"                                                ,
"sdols"                                               ,
"semnar"                                              ,
"shallot"                                             ,
"simmr"                                               ,
"SmartEDA"                                            ,
"socceR"                                              ,
"vip"                                                 ,
"WebGestaltR"                                         ,
"zCompositions"                                       ,
"adoptr"                                              ,
"bayesAB"                                             ,
"bigrquery"                                           ,
"ClussCluster"                                        ,
"cocktailApp"                                         ,
"coda.base"                                           ,
"conf"                                                ,
"copBasic"                                            ,
"cquad"                                               ,
"crassmat"                                            ,
"dbplot"                                              ,
"DSL"                                                 ,
"eGST"                                                ,
"elitism"                                             ,
"ellipsis"                                            ,
"epsiwal"                                             ,
"fitODBOD"                                            ,
"future"                                              ,
"hierarchicalDS"                                      ,
"isni"                                                ,
"jSDM"                                                ,
"lobstr"                                              ,
"MBCbook"                                             ,
"mcmcderive"                                          ,
"NNLM"                                                ,
"npsp"                                                ,
"odin"                                                ,
"OptimaRegion"                                        ,
"phd"                                                 ,
"politeness"                                          ,
"primePCA"                                            ,
"protolite"                                           ,
"rai"                                                 ,
"rasterVis"                                           ,
"rclipboard"                                          ,
"recipes"                                             ,
"rexpokit"                                            ,
"robust"                                              ,
"RRF"                                                 ,
"SAM"                                                 ,
"SemiMarkov"                                          ,
"SparseBiplots"                                       ,
"TPD"                                                 ,
"traipse"                                             ,
"TSEwgt"                                              ,
"UNPaC"                                               ,
"V8"                                                  ,
"benchr"                                              ,
"easyPSID"                                            ,
"errorlocate"                                         ,
"imputeTS"                                            ,
"jetpack"                                             ,
"KCSNBShiny"                                          ,
"label.switching"                                     ,
"mcmcabn"                                             ,
"medExtractR"                                         ,
"prodigenr"                                           ,
"r2pmml"                                              ,
"Rcan"                                                ,
"rngtools"                                            ,
"SimilaR"                                             ,
"systemfonts"                                         ,
"tidytidbits"                                         ,
"utile.tables"                                        ,
"utile.tools"                                         ,
"viafr"                                               ,
"densratio"                                           ,
"missRanger"                                          ,
"openCR"                                              ,
"RProtoBuf"                                           ,
"sensitivity"                                         ,
"tvReg"                                               ,
"aricode"                                             ,
"BiodiversityR"                                       ,
"envirem"                                             ,
"EZtune"                                              ,
"git2r"                                               ,
"HardyWeinberg"                                       ,
"newsanchor"                                          ,
"phylotate"                                           ,
"pillar"                                              ,
"SAGMM"                                               ,
"traits"                                              ,
"walkr"                                               ,
"wdpar"                                               ,
"ActFrag"                                             ,
"asciiSetupReader"                                    ,
"bayes4psy"                                           ,
"bwimage"                                             ,
"ClustVarLV"                                          ,
"CoImp"                                               ,
"condvis2"                                            ,
"Directional"                                         ,
"dttr2"                                               ,
"FlowCAr"                                             ,
"fpc"                                                 ,
"gainML"                                              ,
"gglogo"                                              ,
"GiniDistance"                                        ,
"googleCloudVisionR"                                  ,
"grainchanger"                                        ,
"gstar"                                               ,
"HETOP"                                               ,
"HMMmlselect"                                         ,
"inflection"                                          ,
"ioanalysis"                                          ,
"mirtCAT"                                             ,
"mlxR"                                                ,
"movecost"                                            ,
"NetworkComparisonTest"                               ,
"palaeoSig"                                           ,
"PersianStemmer"                                      ,
"phylosim"                                            ,
"piecepackr"                                          ,
"plotGMM"                                             ,
"qtlbook"                                             ,
"rmargint"                                            ,
"saqgetr"                                             ,
"SimDesign"                                           ,
"slippymath"                                          ,
"taxize"                                              ,
"threeBrain"                                          ,
"vcvComp"                                             ,
"vroom"                                               ,
"worrms"                                              ,
"yatah"                                               ,
"AlphaPart"                                           ,
"analogsea"                                           ,
"bibliometrix"                                        ,
"bold"                                                ,
"coinmarketcapr"                                      ,
"dialr"                                               ,
"eggCounts"                                           ,
"eq5d"                                                ,
"factorstochvol"                                      ,
"felp"                                                ,
"filesstrings"                                        ,
"fulltext"                                            ,
"gestalt"                                             ,
"IncDTW"                                              ,
"mcmcr"                                               ,
"plotmo"                                              ,
"rblt"                                                ,
"rfishbase"                                           ,
"visreg"                                              ,
"WeMix"                                               ,
"cgdsr"                                               ,
"dialrjars"                                           ,
"DTAT"                                                ,
"eemR"                                                ,
"GADMTools"                                           ,
"lcmm"                                                ,
"mschart"                                             ,
"photobiologyFilters"                                 ,
"plotluck"                                            ,
"R62S3"                                               ,
"readtext"                                            ,
"Rilostat"                                            ,
"RJDemetra"                                           ,
"RNGforGPD"                                           ,
"ShinyItemAnalysis"                                   ,
"SignifReg"                                           ,
"stochvol"                                            ,
"trade"                                               ,
"tuber"                                               ,
"varclust"                                            ,
"abbyyR"                                              ,
"aws.alexa"                                           ,
"DAMisc"                                              ,
"flextable"                                           ,
"Gifi"                                                ,
"GLMMadaptive"                                        ,
"gsrsb"                                               ,
"islasso"                                             ,
"jsonvalidate"                                        ,
"nparcomp"                                            ,
"PakPMICS2018"                                        ,
"pavo"                                                ,
"pgraph"                                              ,
"RcppTOML"                                            ,
"RDML"                                                ,
"rlang"                                               ,
"RSocrata"                                            ,
"rvg"                                                 ,
"secsse"                                              ,
"trialr"                                              ,
"wpp2019"                                             ,
"bamboo"                                              ,
"binb"                                                ,
"blrm"                                                ,
"ClusVis"                                             ,
"colorSpec"                                           ,
"devEMF"                                              ,
"enc"                                                 ,
"forestplot"                                          ,
"GenTag"                                              ,
"gggenes"                                             ,
"IceCast"                                             ,
"IDE"                                                 ,
"inlabru"                                             ,
"interim"                                             ,
"lime"                                                ,
"mcgfa"                                               ,
"mcglm"                                               ,
"metapost"                                            ,
"mev"                                                 ,
"pbdSLAP"                                             ,
"pivmet"                                              ,
"predictoR"                                           ,
"psda"                                                ,
"QDComparison"                                        ,
"remotes"                                             ,
"sensibo.sky"                                         ,
"SeqNet"                                              ,
"spatialfusion"                                       ,
"SqlRender"                                           ,
"text2speech"                                         ,
"vdiffr"                                              ,
"vegawidget"                                          ,
"AMR"                                                 ,
"aroma.affymetrix"                                    ,
"auk"                                                 ,
"guardianapi"                                         ,
"hdnom"                                               ,
"hmi"                                                 ,
"hR"                                                  ,
"icr"                                                 ,
"ipc"                                                 ,
"missingHE"                                           ,
"mlt.docreg"                                          ,
"objectremover"                                       ,
"officer"                                             ,
"PerformanceAnalytics"                                ,
"samplesize4surveys"                                  ,
"spatstat"                                            ,
"strex"                                               ,
"tabr"                                                ,
"tram"                                                ,
"biscale"                                             ,
"xtensor"                                             ,
"airGR"                                               ,
"alphahull"                                           ,
"antitrust"                                           ,
"basefun"                                             ,
"BEDMatrix"                                           ,
"BIGL"                                                ,
"CARBayes"                                            ,
"conflicted"                                          ,
"cplexAPI"                                            ,
"cppRouting"                                          ,
"DDD"                                                 ,
"did"                                                 ,
"frostr"                                              ,
"ICRanks"                                             ,
"imgw"                                                ,
"lvnet"                                               ,
"mefa4"                                               ,
"mlt"                                                 ,
"mmmgee"                                              ,
"MODIStsp"                                            ,
"morse"                                               ,
"nlMS"                                                ,
"ormPlot"                                             ,
"parSim"                                              ,
"pesel"                                               ,
"phonics"                                             ,
"plotrix"                                             ,
"praznik"                                             ,
"psSubpathway"                                        ,
"revdbayes"                                           ,
"rmdcev"                                              ,
"spaMM"                                               ,
"subgroup.discovery"                                  ,
"SuperExactTest"                                      ,
"TraMineR"                                            ,
"usdarnass"                                           ,
"ace2fastq"                                           ,
"ARPobservation"                                      ,
"autoimage"                                           ,
"CausalKinetiX"                                       ,
"DynNom"                                              ,
"echo.find"                                           ,
"eoffice"                                             ,
"gemlog"                                              ,
"interflex"                                           ,
"mlVAR"                                               ,
"owdbr"                                               ,
"poolfstat"                                           ,
"PReMiuM"                                             ,
"sequoia"                                             ,
"skimr"                                               ,
"TSE"                                                 ,
"variables"                                           ,
"BisqueRNA"                                           ,
"cluster"                                             ,
"CoRpower"                                            ,
"CSTools"                                             ,
"fgeo"                                                ,
"ggdemetra"                                           ,
"klustR"                                              ,
"krige"                                               ,
"lgrdata"                                             ,
"liftr"                                               ,
"metaheuristicOpt"                                    ,
"mixSPE"                                              ,
"mvtnorm"                                             ,
"MXM"                                                 ,
"phangorn"                                            ,
"plinkQC"                                             ,
"psycho"                                              ,
"qgraph"                                              ,
"QUALYPSO"                                            ,
"rfacebookstat"                                       ,
"rspa"                                                ,
"scriptName"                                          ,
"shinymanager"                                        ,
"spc"                                                 ,
"StratigrapheR"                                       ,
"tidyqwi"                                             ,
"ui"                                                  ,
"wwntests"                                            ,
"adept"                                               ,
"APFr"                                                ,
"bigchess"                                            ,
"CholWishart"                                         ,
"CompareCausalNetworks"                               ,
"comparer"                                            ,
"Compositional"                                       ,
"dataesgobr"                                          ,
"drugCombo"                                           ,
"fgeo.analyze"                                        ,
"fgeo.plot"                                           ,
"future.apply"                                        ,
"gausscov"                                            ,
"HiResTEC"                                            ,
"LogicReg"                                            ,
"mexhaz"                                              ,
"MortalityLaws"                                       ,
"multivariance"                                       ,
"neatRanges"                                          ,
"ows4R"                                               ,
"phuse"                                               ,
"phytools"                                            ,
"PolynomF"                                            ,
"poppr"                                               ,
"prevtoinc"                                           ,
"provSummarizeR"                                      ,
"RadioGx"                                             ,
"rskey"                                               ,
"splithalf"                                           ,
"sudokuAlt"                                           ,
"aroma.core"                                          ,
"BART"                                                ,
"Bergm"                                               ,
"bootstrap"                                           ,
"changedetection"                                     ,
"comprehenr"                                          ,
"dbplyr"                                              ,
"DesignLibrary"                                       ,
"earthtide"                                           ,
"fgeo.tool"                                           ,
"git2rdata"                                           ,
"npcure"                                              ,
"oce"                                                 ,
"pandocfilters"                                       ,
"politicaldata"                                       ,
"prediction"                                          ,
"quantmod"                                            ,
"quietR"                                              ,
"R.temis"                                             ,
"rjdqa"                                               ,
"rotor"                                               ,
"roxygen2md"                                          ,
"rPackedBar"                                          ,
"segmented"                                           ,
"seqHMM"                                              ,
"sjmisc"                                              ,
"statnetWeb"                                          ,
"tab"                                                 ,
"tidymv"                                              ,
"CGE"                                                 ,
"assist"                                              ,
"climwin"                                             ,
"DDPNA"                                               ,
"modelDown"                                           ,
"NST"                                                 ,
"OutlierDetection"                                    ,
"rjwsacruncher"                                       ,
"tsibbledata"                                         ,
"unikn"                                               ,
"uplifteval"                                          ,
"abcrf"                                               ,
"bayesLife"                                           ,
"catfun"                                              ,
"ceRtainty"                                           ,
"CGGP"                                                ,
"CGPfunctions"                                        ,
"clpAPI"                                              ,
"CoreGx"                                              ,
"DET"                                                 ,
"dglars"                                              ,
"DRHotNet"                                            ,
"EFA.MRFA"                                            ,
"EloRating"                                           ,
"expstudies"                                          ,
"FisPro"                                              ,
"FLightR"                                             ,
"gear"                                                ,
"gexp"                                                ,
"kmed"                                                ,
"metapro"                                             ,
"modifiedmk"                                          ,
"photobiologyLamps"                                   ,
"photobiologyPlants"                                  ,
"QFASA"                                               ,
"RBF"                                                 ,
"rflexscan"                                           ,
"Rlgt"                                                ,
"rock"                                                ,
"rotl"                                                ,
"safetyGraphics"                                      ,
"ScorePlus"                                           ,
"SHELF"                                               ,
"SimVitD"                                             ,
"SMITIDstruct"                                        ,
"SMITIDvisu"                                          ,
"SSDM"                                                ,
"statnet"                                             ,
"STOPES"                                              ,
"subgxe"                                              ,
"Ternary"                                             ,
"tfdeploy"                                            ,
"tidyrules"                                           ,
"tinyProject"                                         ,
"apcf"                                                ,
"archetypal"                                          ,
"crmPack"                                             ,
"crochet"                                             ,
"dfConn"                                              ,
"ggpage"                                              ,
"ggspectra"                                           ,
"hIRT"                                                ,
"JSmediation"                                         ,
"logspline"                                           ,
"lvm4net"                                             ,
"nomnoml"                                             ,
"nor1mix"                                             ,
"oddsratio"                                           ,
"outbreaker2"                                         ,
"polspline"                                           ,
"qrnn"                                                ,
"R.utils"                                             ,
"recorder"                                            ,
"variantspark"                                        ,
"augSIMEX"                                            ,
"BIFIEsurvey"                                         ,
"BVSNLP"                                              ,
"cblasr"                                              ,
"divDyn"                                              ,
"folderfun"                                           ,
"MBESS"                                               ,
"Numero"                                              ,
"processanimateR"                                     ,
"pwr2ppl"                                             ,
"pwt9"                                                ,
"Quandl"                                              ,
"shipunov"                                            ,
"stabm"                                               ,
"tergm"                                               ,
"tidycensus"                                          ,
"tigris"                                              ,
"timeR"                                               ,
"AdhereR"                                             ,
"aweek"                                               ,
"bayesTFR"                                            ,
"cglasso"                                             ,
"DAIME"                                               ,
"EHRtemporalVariability"                              ,
"EstimationTools"                                     ,
"FHDI"                                                ,
"ForecastFramework"                                   ,
"formatR"                                             ,
"HybridFS"                                            ,
"lineqGPR"                                            ,
"MetaSubtract"                                        ,
"mime"                                                ,
"monomvn"                                             ,
"MortCast"                                            ,
"OUwie"                                               ,
"padr"                                                ,
"PakPMICS2018mn"                                      ,
"pmmlTransformations"                                 ,
"POD"                                                 ,
"RcppNumerical"                                       ,
"regressoR"                                           ,
"robsurvey"                                           ,
"RPANDA"                                              ,
"RSpectra"                                            ,
"showtext"                                            ,
"SpatialVx"                                           ,
"cimir"                                               ,
"ergm"                                                ,
"extraDistr"                                          ,
"KFAS"                                                ,
"qte"                                                 ,
"RSuite"                                              ,
"rubias"                                              ,
"salesforcer"                                         ,
"tractor.base"                                        ,
"trajr"                                               ,
"ubci"                                                ,
"wnl"                                                 ,
"BMAmevt"                                             ,
"ChemoSpec2D"                                         ,
"clusterSEs"                                          ,
"datasus"                                             ,
"econullnetr"                                         ,
"gss"                                                 ,
"hablar"                                              ,
"kstMatrix"                                           ,
"littler"                                             ,
"maditr"                                              ,
"motmot"                                              ,
"penaltyLearning"                                     ,
"Ricetl"                                              ,
"rLakeAnalyzer"                                       ,
"rrecsys"                                             ,
"RSwissMaps"                                          ,
"sim1000G"                                            ,
"sundialr"                                            ,
"WgtEff"                                              ,
"AFM"                                                 ,
"bigleaf"                                             ,
"chinese.misc"                                        ,
"missSBM"                                             ,
"PakPMICS2018mm"                                      ,
"pense"                                               ,
"photobiology"                                        ,
"regnet"                                              ,
"Rinstapkg"                                           ,
"stream"                                              ,
"workflowr"                                           ,
"binnednp"                                            ,
"bsts"                                                ,
"cpd"                                                 ,
"crunch"                                              ,
"es.dif"                                              ,
"exiftoolr"                                           ,
"fgeo.x"                                              ,
"GNAR"                                                ,
"intrinsicDimension"                                  ,
"isocat"                                              ,
"mgcViz"                                              ,
"mombf"                                               ,
"PakPMICS2018fs"                                      ,
"PakPMICS2018hh"                                      ,
"phenofit"                                            ,
"qgam"                                                ,
"RandomCoefficients"                                  ,
"REAT"                                                ,
"rorcid"                                              ,
"RtextSummary"                                        ,
"Rxnat"                                               ,
"semTable"                                            ,
"splithalfr"                                          ,
"tacmagic"                                            ,
"TUWmodel"                                            ,
"vanddraabe"                                          ,
"YPInterimTesting"                                    ,
"alterryx"                                            ,
"CircSpaceTime"                                       ,
"crunchy"                                             ,
"derivmkts"                                           ,
"DriftBurstHypothesis"                                ,
"eRm"                                                 ,
"fuzzyreg"                                            ,
"ggPMX"                                               ,
"interactionTest"                                     ,
"JuliaCall"                                           ,
"numDeriv"                                            ,
"provenance"                                          ,
"sjlabelled"                                          ,
"smacof"                                              ,
"spinBayes"                                           ,
"stabreg"                                             ,
"stringdist"                                          ,
"tibble"                                              ,
"TopicScore"                                          ,
"WRS2"                                                ,
"XML"                                                 ,
"BibPlots"                                            ,
"bWGR"                                                ,
"clustermq"                                           ,
"coarseDataTools"                                     ,
"dataRetrieval"                                       ,
"dodgr"                                               ,
"e1071"                                               ,
"gap"                                                 ,
"HistDAWass"                                          ,
"ICODS"                                               ,
"Iso"                                                 ,
"prioritylasso"                                       ,
"promotionImpact"                                     ,
"QZ"                                                  ,
"rdfp"                                                ,
"RobustGaSP"                                          ,
"rxylib"                                              ,
"SoilHyP"                                             ,
"traitdataform"                                       ,
"tseries"                                             ,
"tsmp"                                                ,
"wiqid"                                               ,
"adespatial"                                          ,
"cascsim"                                             ,
"clrdag"                                              ,
"cluscov"                                             ,
"geiger"                                              ,
"hydroscoper"                                         ,
"ipumsr"                                              ,
"jenkins"                                             ,
"manhplot"                                            ,
"memapp"                                              ,
"mFilter"                                             ,
"paleotree"                                           ,
"PINSPlus"                                            ,
"polyRAD"                                             ,
"prabclus"                                            ,
"QTOCen"                                              ,
"RJSONIO"                                             ,
"SCCI"                                                ,
"SCCS"                                                ,
"serial"                                              ,
"vtable"                                              ,
"AeRobiology"                                         ,
"bayesreg"                                            ,
"climextRemes"                                        ,
"cpsurvsim"                                           ,
"gap.datasets"                                        ,
"gen2stage"                                           ,
"generalhoslem"                                       ,
"grangers"                                            ,
"GuessCompx"                                          ,
"jacpop"                                              ,
"json64"                                              ,
"justifier"                                           ,
"LeMaRns"                                             ,
"lmboot"                                              ,
"MPSEM"                                               ,
"mSimCC"                                              ,
"MTLR"                                                ,
"multicross"                                          ,
"MultiRobust"                                         ,
"naivebayes"                                          ,
"PakPMICS2018bh"                                      ,
"pinochet"                                            ,
"pmml"                                                ,
"popsom"                                              ,
"PowerfulMaxEigenpair"                                ,
"PSTR"                                                ,
"RCarb"                                               ,
"recmap"                                              ,
"robcp"                                               ,
"robis"                                               ,
"secr"                                                ,
"sensmediation"                                       ,
"spectral"                                            ,
"spectralGraphTopology"                               ,
"spectrino"                                           ,
"statVisual"                                          ,
"stima"                                               ,
"TreeSearch"                                          ,
"uCAREChemSuiteCLI"                                   ,
"volesti"                                             ,
"compareDF"                                           ,
"CVXR"                                                ,
"fssemR"                                              ,
"ggwordcloud"                                         ,
"kzfs"                                                ,
"nimble"                                              ,
"reproj"                                              ,
"sparseIndexTracking"                                 ,
"spex"                                                ,
"statnet.common"                                      ,
"tidybayes"                                           ,
"agriTutorial"                                        ,
"beastier"                                            ,
"gtWAS"                                               ,
"PowerUpR"                                            ,
"Ball"                                                ,
"BayesianPower"                                       ,
"binsmooth"                                           ,
"BTM"                                                 ,
"colocr"                                              ,
"crfsuite"                                            ,
"datos"                                               ,
"drifter"                                             ,
"EIX"                                                 ,
"ergm.ego"                                            ,
"Evapotranspiration"                                  ,
"fcros"                                               ,
"fishtree"                                            ,
"float"                                               ,
"ggResidpanel"                                        ,
"hutils"                                              ,
"HWxtest"                                             ,
"insurancerating"                                     ,
"kequate"                                             ,
"kinship2"                                            ,
"loose.rock"                                          ,
"MedDietCalc"                                         ,
"mppR"                                                ,
"MultiwayRegression"                                  ,
"NPMLENCC"                                            ,
"OpenStreetMap"                                       ,
"pkgcache"                                            ,
"projpred"                                            ,
"RDS"                                                 ,
"rdwd"                                                ,
"rstanemax"                                           ,
"ruimtehol"                                           ,
"semiArtificial"                                      ,
"simglm"                                              ,
"StepReg"                                             ,
"studentlife"                                         ,
"touchard"                                            ,
"tsdf"                                                ,
"tsfknn"                                              ,
"TSrepr"                                              ,
"yonder"                                              ,
"atable"                                              ,
"brunnermunzel"                                       ,
"DendroSync"                                          ,
"geigen"                                              ,
"MIAmaxent"                                           ,
"parlitools"                                          ,
"rpms"                                                ,
"smotefamily"                                         ,
"spp"                                                 ,
"SWMPr"                                               ,
"tframe"                                              ,
"viewshed3d"                                          ,
"zoltr"                                               ,
"AICcmodavg"                                          ,
"DChaos"                                              ,
"EditImputeCont"                                      ,
"g3viz"                                               ,
"gasfluxes"                                           ,
"gpuR"                                                ,
"icenReg"                                             ,
"IOHanalyzer"                                         ,
"madrat"                                              ,
"magclass"                                            ,
"matlib"                                              ,
"MCDA"                                                ,
"medfate"                                             ,
"metaSEM"                                             ,
"meteoland"                                           ,
"MtreeRing"                                           ,
"princurve"                                           ,
"psychTools"                                          ,
"rCBA"                                                ,
"rgdal"                                               ,
"rintrojs"                                            ,
"rIP"                                                 ,
"sparsebnUtils"                                       ,
"statmod"                                             ,
"SuperpixelImageSegmentation"                         ,
"tsna"                                                ,
"BIOdry"                                              ,
"blocksdesign"                                        ,
"caRamel"                                             ,
"eigenmodel"                                          ,
"EncDNA"                                              ,
"etm"                                                 ,
"evaluate"                                            ,
"expperm"                                             ,
"genio"                                               ,
"ggenealogy"                                          ,
"jmvcore"                                             ,
"kaos"                                                ,
"measurements"                                        ,
"mem"                                                 ,
"mnlogit"                                             ,
"openxlsx"                                            ,
"RViennaCL"                                           ,
"spant"                                               ,
"swirlify"                                            ,
"Tinflex"                                             ,
"trinROC"                                             ,
"zoo"                                                 ,
"car"                                                 ,
"ChainLadder"                                         ,
"cmprsk"                                              ,
"compareGroups"                                       ,
"devRate"                                             ,
"funbarRF"                                            ,
"ggnormalviolin"                                      ,
"kmi"                                                 ,
"lagged"                                              ,
"minerva"                                             ,
"packagefinder"                                       ,
"pedquant"                                            ,
"piratings"                                           ,
"radsafer"                                            ,
"respirometry"                                        ,
"rnn"                                                 ,
"rtf"                                                 ,
"SelectBoost"                                         ,
"shinyTree"                                           ,
"Sim.DiffProc"                                        ,
"startup"                                             ,
"tempoR"                                              ,
"testthis"                                            ,
"WCM"                                                 ,
"WGScan"                                              ,
"chillR"                                              ,
"evtree"                                              ,
"FAMoS"                                               ,
"geometa"                                             ,
"geonapi"                                             ,
"geosphere"                                           ,
"ggfortify"                                           ,
"labelled"                                            ,
"muRty"                                               ,
"OpenMx"                                              ,
"R2MLwiN"                                             ,
"SimInf"                                              ,
"tsDyn"                                               ,
"CFC"                                                 ,
"cRegulome"                                           ,
"hergm"                                               ,
"hsdar"                                               ,
"IDmining"                                            ,
"nanotime"                                            ,
"pks"                                                 ,
"rdataretriever"                                      ,
"ReinforcementLearning"                               ,
"snakecase"                                           ,
"WriteXLS"                                            ,
"alpaca"                                              ,
"armspp"                                              ,
"bayesvl"                                             ,
"bife"                                                ,
"bivariate"                                           ,
"BNSP"                                                ,
"climateStability"                                    ,
"clusterSim"                                          ,
"EpiStats"                                            ,
"georob"                                              ,
"icesTAF"                                             ,
"icpsrdata"                                           ,
"integr"                                              ,
"mdatools"                                            ,
"MDSPCAShiny"                                         ,
"parcoords"                                           ,
"rcites"                                              ,
"restriktor"                                          ,
"RobinHood"                                           ,
"rwebstat"                                            ,
"simml"                                               ,
"spaero"                                              ,
"SuperPCA"                                            ,
"SupMZ"                                               ,
"tranSurv"                                            ,
"vfprogression"                                       ,
"wavScalogram"                                        ,
"xtune"                                               ,
"barsurf"                                             ,
"bayesplot"                                           ,
"evian"                                               ,
"gcerisk"                                             ,
"gimme"                                               ,
"jpndistrict"                                         ,
"pedgene"                                             ,
"Rd"                                                  ,
"Routliers"                                           ,
"s20x"                                                ,
"speaq"                                               ,
"tidync"                                              ,
"unfoldr"                                             ,
"WGCNA"                                               ,
"APSIM"                                               ,
"bayesdfa"                                            ,
"bib2df"                                              ,
"cgmanalysis"                                         ,
"dixonTest"                                           ,
"eddi"                                                ,
"gridBezier"                                          ,
"IsoSpecR"                                            ,
"jcolors"                                             ,
"ndtv"                                                ,
"osmdata"                                             ,
"personalized"                                        ,
"predictrace"                                         ,
"RcppParallel"                                        ,
"refGenome"                                           ,
"regsem"                                              ,
"seqDesign"                                           ,
"TCA"                                                 ,
"tidyRSS"                                             ,
"TSP"                                                 ,
"UpSetR"                                              ,
"AdaSampling"                                         ,
"areal"                                               ,
"biomartr"                                            ,
"bipartite"                                           ,
"codebook"                                            ,
"collector"                                           ,
"datarium"                                            ,
"drgee"                                               ,
"IPPP"                                                ,
"MCMCtreeR"                                           ,
"planor"                                              ,
"QCA"                                                 ,
"R6DS"                                                ,
"redshiftTools"                                       ,
"rscorecard"                                          ,
"scBio"                                               ,
"SEMID"                                               ,
"STraTUS"                                             ,
"xoi"                                                 ,
"YRmisc"                                              ,
"AF"                                                  ,
"algaeClassify"                                       ,
"arulesViz"                                           ,
"CalSim"                                              ,
"changepoint.np"                                      ,
"cleancall"                                           ,
"CLONETv2"                                            ,
"cmvnorm"                                             ,
"ContourFunctions"                                    ,
"datapackage.r"                                       ,
"EFDR"                                                ,
"errors"                                              ,
"geomorph"                                            ,
"glmnet"                                              ,
"grattan"                                             ,
"gridGraphics"                                        ,
"hdme"                                                ,
"hexSticker"                                          ,
"kmer"                                                ,
"ks"                                                  ,
"kSamples"                                            ,
"learnPopGen"                                         ,
"macleish"                                            ,
"mastif"                                              ,
"mc.heterogeneity"                                    ,
"mdsr"                                                ,
"networktools"                                        ,
"paco"                                                ,
"peRspective"                                         ,
"phenocamapi"                                         ,
"ragtop"                                              ,
"simputation"                                         ,
"tidycode"                                            ,
"trendyy"                                             ,
"volleystat"                                          ,
"chorrrds"                                            ,
"ggcorrplot"                                          ,
"InjurySeverityScore"                                 ,
"inventorize"                                         ,
"lavaSearch2"                                         ,
"maxLik"                                              ,
"metamedian"                                          ,
"moveHMM"                                             ,
"RRPP"                                                ,
"rtweet"                                              ,
"spaceNet"                                            ,
"CommEcol"                                            ,
"dmai"                                                ,
"enpls"                                               ,
"FSAdata"                                             ,
"funrar"                                              ,
"gamlss"                                              ,
"glmmfields"                                          ,
"mdsstat"                                             ,
"nLTT"                                                ,
"OHPL"                                                ,
"protr"                                               ,
"radiant"                                             ,
"Thermimage"                                          ,
"tmbstan"                                             ,
"AFheritability"                                      ,
"bmass"                                               ,
"bundesbank"                                          ,
"CodataGS"                                            ,
"coloc"                                               ,
"CorrToolBox"                                         ,
"Delaporte"                                           ,
"dqrng"                                               ,
"egoTERGM"                                            ,
"EMMAgeo"                                             ,
"fclust"                                              ,
"grex"                                                ,
"JMcmprsk"                                            ,
"matahari"                                            ,
"memoria"                                             ,
"msaenet"                                             ,
"parallelMap"                                         ,
"partition"                                           ,
"PLNmodels"                                           ,
"prettyB"                                             ,
"pulseTD"                                             ,
"radarBoxplot"                                        ,
"RECA"                                                ,
"rlmDataDriven"                                       ,
"SMARTp"                                              ,
"SNPknock"                                            ,
"sparseMatEst"                                        ,
"SPOT"                                                ,
"statsguRu"                                           ,
"svglite"                                             ,
"tokenbrowser"                                        ,
"AID"                                                 ,
"anim.plots"                                          ,
"bamp"                                                ,
"betareg"                                             ,
"BHMSMAfMRI"                                          ,
"BinNonNor"                                           ,
"BivUnifBin"                                          ,
"briqr"                                               ,
"bysykkel"                                            ,
"CB2"                                                 ,
"dde"                                                 ,
"DevTreatRules"                                       ,
"expose"                                              ,
"FENmlm"                                              ,
"frailtypack"                                         ,
"fy"                                                  ,
"glancedata"                                          ,
"googleVis"                                           ,
"gstat"                                               ,
"hyper2"                                              ,
"intoo"                                               ,
"ivmte"                                               ,
"josaplay"                                            ,
"LagSequential"                                       ,
"lmvar"                                               ,
"MChtest"                                             ,
"MobileTrigger"                                       ,
"Morpho"                                              ,
"multinomineq"                                        ,
"N2H4"                                                ,
"orders"                                              ,
"packMBPLSDA"                                         ,
"PoisBinNonNor"                                       ,
"PoisBinOrdNonNor"                                    ,
"progress"                                            ,
"psfmi"                                               ,
"radiant.multivariate"                                ,
"ramlegacy"                                           ,
"rdoc"                                                ,
"reprex"                                              ,
"rromeo"                                              ,
"ruler"                                               ,
"sdcMicro"                                            ,
"vaultr"                                              ,
"virtualPollen"                                       ,
"blaise"                                              ,
"bnpsd"                                               ,
"cffdrs"                                              ,
"ctmm"                                                ,
"dae"                                                 ,
"daff"                                                ,
"EleChemr"                                            ,
"ergm.count"                                          ,
"ergm.rank"                                           ,
"ergm.userterms"                                      ,
"esc"                                                 ,
"FitUltD"                                             ,
"gamlss.data"                                         ,
"gamlss.dist"                                         ,
"imageData"                                           ,
"ipcwswitch"                                          ,
"libsoc"                                              ,
"lmviz"                                               ,
"Map2NCBI"                                            ,
"mlergm"                                              ,
"mrfDepth"                                            ,
"nlstimedist"                                         ,
"PHEindicatormethods"                                 ,
"PhenotypeSimulator"                                  ,
"PoisBinOrd"                                          ,
"PoisBinOrdNor"                                       ,
"PoisNonNor"                                          ,
"ratematrix"                                          ,
"rcube"                                               ,
"rtimicropem"                                         ,
"rvest"                                               ,
"SoftRandomForest"                                    ,
"streamMOA"                                           ,
"trackeR"                                             ,
"ukbtools"                                            ,
"vtree"                                               ,
"waterYearType"                                       ,
"ASSOCShiny"                                          ,
"basket"                                              ,
"BatchJobs"                                           ,
"BinNor"                                              ,
"BLPestimatoR"                                        ,
"btergm"                                              ,
"c212"                                                ,
"clubSandwich"                                        ,
"DetSel"                                              ,
"ecoval"                                              ,
"fungible"                                            ,
"ggsoccer"                                            ,
"gower"                                               ,
"grImport"                                            ,
"heuristicsmineR"                                     ,
"hexbin"                                              ,
"hipread"                                             ,
"lotri"                                               ,
"metafor"                                             ,
"metawho"                                             ,
"mplot"                                               ,
"mthapower"                                           ,
"MultiOrd"                                            ,
"MultiRNG"                                            ,
"nsrr"                                                ,
"OrdNor"                                              ,
"PoisNor"                                             ,
"pomdp"                                               ,
"profvis"                                             ,
"prophet"                                             ,
"repr"                                                ,
"Rpipedrive"                                          ,
"servosphereR"                                        ,
"sn"                                                  ,
"spatsoc"                                             ,
"ukgasapi"                                            ,
"wbs"                                                 ,
"amap"                                                ,
"asremlPlus"                                          ,
"aster"                                               ,
"diffobj"                                             ,
"doex"                                                ,
"ggthemes"                                            ,
"koRpus.lang.en"                                      ,
"mapsRinteractive"                                    ,
"mapview"                                             ,
"merTools"                                            ,
"popkin"                                              ,
"poplite"                                             ,
"SGB"                                                 ,
"somspace"                                            ,
"spdplyr"                                             ,
"SurfaceTortoise"                                     ,
"tables"                                              ,
"UKgrid"                                              ,
"wevid"                                               ,
"hSDM"                                                ,
"kutils"                                              ,
"ldatuning"                                           ,
"MALDIquant"                                          ,
"MetaStan"                                            ,
"neatmaps"                                            ,
"ppitables"                                           ,
"RcppAnnoy"                                           ,
"rmcfs"                                               ,
"robustbase"                                          ,
"sarima"                                              ,
"superml"                                             ,
"tufte"                                               ,
"AnaCoDa"                                             ,
"analogue"                                            ,
"BradleyTerry2"                                       ,
"CLA"                                                 ,
"cocorresp"                                           ,
"dendextend"                                          ,
"DoE.wrapper"                                         ,
"fmri"                                                ,
"heatmaply"                                           ,
"ISOpureR"                                            ,
"PopGenome"                                           ,
"riingo"                                              ,
"seg"                                                 ,
"sfsmisc"                                             ,
"smurf"                                               ,
"arse"                                                ,
"cartograflow"                                        ,
"dplyr.teradata"                                      ,
"ECOSolveR"                                           ,
"essHist"                                             ,
"ggroups"                                             ,
"glmaag"                                              ,
"graphon"                                             ,
"intsvy"                                              ,
"ipflasso"                                            ,
"mosum"                                               ,
"mscstts"                                             ,
"radlibs"                                             ,
"rankFD"                                              ,
"robustreg"                                           ,
"roccv"                                               ,
"spanish"                                             ,
"SpatialPosition"                                     ,
"abtest"                                              ,
"bikedata"                                            ,
"BLRShiny"                                            ,
"caesar"                                              ,
"capitalR"                                            ,
"CoSMoS"                                              ,
"GPareto"                                             ,
"IDmeasurer"                                          ,
"intccr"                                              ,
"jpmesh"                                              ,
"KNNShiny"                                            ,
"mapcan"                                              ,
"metacart"                                            ,
"mongolite"                                           ,
"MRPC"                                                ,
"nanostringr"                                         ,
"nmadb"                                               ,
"OrthoPanels"                                         ,
"pewdata"                                             ,
"photosynthesis"                                      ,
"qvcalc"                                              ,
"R.SamBada"                                           ,
"rENA"                                                ,
"rhierbaps"                                           ,
"rr2"                                                 ,
"rversions"                                           ,
"sequenza"                                            ,
"SpatialML"                                           ,
"adimpro"                                             ,
"aphid"                                               ,
"apollo"                                              ,
"astsa"                                               ,
"aws"                                                 ,
"awsMethods"                                          ,
"cepreader"                                           ,
"congressbr"                                          ,
"dimRed"                                              ,
"dragon"                                              ,
"empichar"                                            ,
"emstreeR"                                            ,
"ensemblepp"                                          ,
"flan"                                                ,
"gravity"                                             ,
"MAVE"                                                ,
"Mediana"                                             ,
"msgl"                                                ,
"multilevelMatching"                                  ,
"NetworkChange"                                       ,
"onlineVAR"                                           ,
"parsedate"                                           ,
"PRISMAstatement"                                     ,
"rabhit"                                              ,
"rgbif"                                               ,
"todor"                                               ,
"assertable"                                          ,
"ConR"                                                ,
"ggrepel"                                             ,
"GRAPE"                                               ,
"icd"                                                 ,
"Matching"                                            ,
"optimParallel"                                       ,
"psychomix"                                           ,
"rchie"                                               ,
"rcmdcheck"                                           ,
"rdatacite"                                           ,
"SEERaBomb"                                           ,
"seminr"                                              ,
"SFS"                                                 ,
"sglOptim"                                            ,
"sigminer"                                            ,
"survsup"                                             ,
"x13binary"                                           ,
"batchscr"                                            ,
"BimodalIndex"                                        ,
"ClassComparison"                                     ,
"ClassDiscovery"                                      ,
"clhs"                                                ,
"CloneSeeker"                                         ,
"cmocean"                                             ,
"CrossValidate"                                       ,
"dapr"                                                ,
"febr"                                                ,
"fs"                                                  ,
"grImport2"                                           ,
"integIRTy"                                           ,
"jrt"                                                 ,
"jwutil"                                              ,
"kcpRS"                                               ,
"LAM"                                                 ,
"localIV"                                             ,
"lwgeom"                                              ,
"minimaxdesign"                                       ,
"Modeler"                                             ,
"multiwave"                                           ,
"NBShiny"                                             ,
"OasisR"                                              ,
"OTclust"                                             ,
"PCDimension"                                         ,
"PreProcess"                                          ,
"prevR"                                               ,
"quadprog"                                            ,
"shinytest"                                           ,
"Sojourn"                                             ,
"SSOSVM"                                              ,
"styler"                                              ,
"tabplot"                                             ,
"Thresher"                                            ,
"Umpire"                                              ,
"vapour"                                              ,
"virtualspecies"                                      ,
"attachment"                                          ,
"Crossover"                                           ,
"equSA"                                               ,
"future.batchtools"                                   ,
"genius"                                              ,
"hydrostats"                                          ,
"incadata"                                            ,
"PSCBS"                                               ,
"Riex"                                                ,
"rmdfiltr"                                            ,
"SIMMS"                                               ,
"stUPscales"                                          ,
"texPreview"                                          ,
"whereami"                                            ,
"XRJulia"                                             ,
"ActivePathways"                                      ,
"bivrp"                                               ,
"blmeco"                                              ,
"extremefit"                                          ,
"googleComputeEngineR"                                ,
"interp"                                              ,
"ivx"                                                 ,
"jointMeanCov"                                        ,
"LOGAN"                                               ,
"RcmdrPlugin.EZR"                                     ,
"rcrossref"                                           ,
"spls"                                                ,
"tealeaves"                                           ,
"amerika"                                             ,
"ASSISTant"                                           ,
"BIOMASS"                                             ,
"CTShiny"                                             ,
"dani"                                                ,
"devoid"                                              ,
"dirichletprocess"                                    ,
"distill"                                             ,
"DoE.base"                                            ,
"EmiStatR"                                            ,
"farff"                                               ,
"FMradio"                                             ,
"geoviz"                                              ,
"gridGeometry"                                        ,
"gunit"                                               ,
"iarm"                                                ,
"istacr"                                              ,
"Monte.Carlo.se"                                      ,
"ncodeR"                                              ,
"PlotContour"                                         ,
"postGIStools"                                        ,
"rbison"                                              ,
"RBtest"                                              ,
"rcartocolor"                                         ,
"rplos"                                               ,
"RPresto"                                             ,
"rTRNG"                                               ,
"sampleSelection"                                     ,
"scgwr"                                               ,
"Sojourn.Data"                                        ,
"SPCAvRP"                                             ,
"surveyplanning"                                      ,
"swirl"                                               ,
"trendsegmentR"                                       ,
"varImp"                                              ,
"WaMaSim"                                             ,
"webex"                                               ,
"airGRteaching"                                       ,
"beginr"                                              ,
"Bessel"                                              ,
"bookdownplus"                                        ,
"distrMod"                                            ,
"dynutils"                                            ,
"geosapi"                                             ,
"HEMDAG"                                              ,
"ipwErrorY"                                           ,
"Lahman"                                              ,
"mindr"                                               ,
"nestfs"                                              ,
"ouch"                                                ,
"pinyin"                                              ,
"RelimpPCR"                                           ,
"rentrez"                                             ,
"RFishBC"                                             ,
"SAR"                                                 ,
"schoRsch"                                            ,
"youtubecaption"                                      ,
"ANOVAShiny"                                          ,
"berryFunctions"                                      ,
"cgam"                                                ,
"colourlovers"                                        ,
"Conigrave"                                           ,
"ctrdata"                                             ,
"Distance"                                            ,
"divest"                                              ,
"gpg"                                                 ,
"gStream"                                             ,
"MIIVsem"                                             ,
"nhdR"                                                ,
"origami"                                             ,
"PoissonPCA"                                          ,
"rangeModelMetadata"                                  ,
"rqPen"                                               ,
"sapfluxnetr"                                         ,
"sgmcmc"                                              ,
"TDMR"                                                ,
"textutils"                                           ,
"VIRF"                                                ,
"aLFQ"                                                ,
"DeclareDesign"                                       ,
"DJL"                                                 ,
"giphyr"                                              ,
"glmmML"                                              ,
"haploR"                                              ,
"HLSM"                                                ,
"leafem"                                              ,
"lmtest"                                              ,
"ltmix"                                               ,
"michelRodange"                                       ,
"moveVis"                                             ,
"reproducer"                                          ,
"Rsmlx"                                               ,
"sdnet"                                               ,
"sensobol"                                            ,
"shinyFiles"                                          ,
"TcGSA"                                               ,
"waveformlidar"                                       ,
"Anthropometry"                                       ,
"BlakerCI"                                            ,
"cranlogs"                                            ,
"DLMtool"                                             ,
"ggdmc"                                               ,
"kvh"                                                 ,
"lcc"                                                 ,
"NACHO"                                               ,
"NBAloveR"                                            ,
"PACLasso"                                            ,
"parsec"                                              ,
"qqid"                                                ,
"sdprisk"                                             ,
"serrsBayes"                                          ,
"sobolnp"                                             ,
"spsann"                                              ,
"VLMC"                                                ,
"CLUSTShiny"                                          ,
"exactRankTests"                                      ,
"ipred"                                               ,
"legocolors"                                          ,
"liayson"                                             ,
"markmyassignment"                                    ,
"mbgraphic"                                           ,
"OptM"                                                ,
"PUlasso"                                             ,
"RNOmni"                                              ,
"telegram.bot"                                        ,
"xrf"                                                 ,
"caret"                                               ,
"CRWRM"                                               ,
"DarkDiv"                                             ,
"DeRezende.Ferreira"                                  ,
"paramhetero"                                         ,
"survey"                                              ,
"baRcodeR"                                            ,
"bisque"                                              ,
"DDIwR"                                               ,
"gestate"                                             ,
"GGMselect"                                           ,
"glcm"                                                ,
"histogram"                                           ,
"irace"                                               ,
"LINselect"                                           ,
"MASS"                                                ,
"microPop"                                            ,
"mitools"                                             ,
"nutriNetwork"                                        ,
"nzilbb.labbcat"                                      ,
"phateR"                                              ,
"powdR"                                               ,
"pre"                                                 ,
"rglobi"                                              ,
"rollRegres"                                          ,
"sazedR"                                              ,
"simode"                                              ,
"sivipm"                                              ,
"SplitReg"                                            ,
"tree"                                                ,
"ypr"                                                 ,
"BALLI"                                               ,
"checkr"                                              ,
"childesr"                                            ,
"csv"                                                 ,
"EMSNM"                                               ,
"encode"                                              ,
"encryptr"                                            ,
"err"                                                 ,
"estudy2"                                             ,
"excerptr"                                            ,
"finch"                                               ,
"GeneCycle"                                           ,
"geojsonio"                                           ,
"inum"                                                ,
"mcStats"                                             ,
"metaplot"                                            ,
"MLDAShiny"                                           ,
"mpt"                                                 ,
"nonmemica"                                           ,
"openEBGM"                                            ,
"ordinal"                                             ,
"Phase123"                                            ,
"PREPShiny"                                           ,
"RMKL"                                                ,
"RobAStBase"                                          ,
"ROptEst"                                             ,
"rstatscn"                                            ,
"spec"                                                ,
"stcos"                                               ,
"streamDepletr"                                       ,
"suppdata"                                            ,
"umx"                                                 ,
"wrangle"                                             ,
"conditionz"                                          ,
"CoordinateCleaner"                                   ,
"countHMM"                                            ,
"dbx"                                                 ,
"DisImpact"                                           ,
"FARDEEP"                                             ,
"flyio"                                               ,
"HoRM"                                                ,
"ibd"                                                 ,
"IMak"                                                ,
"jsonify"                                             ,
"kmc"                                                 ,
"MCMCglmm"                                            ,
"memisc"                                              ,
"missDeaths"                                          ,
"modelplotr"                                          ,
"MonoPoly"                                            ,
"platetools"                                          ,
"reverseR"                                            ,
"RTest"                                               ,
"SortedEffects"                                       ,
"srp"                                                 ,
"bmixture"                                            ,
"Brundle"                                             ,
"dMod"                                                ,
"ecmwfr"                                              ,
"ed50"                                                ,
"EML"                                                 ,
"GetBCBData"                                          ,
"ISOcodes"                                            ,
"kdtools"                                             ,
"lctools"                                             ,
"nptest"                                              ,
"polyreg"                                             ,
"PROBShiny"                                           ,
"PythonInR"                                           ,
"RealVAMS"                                            ,
"rIntervalTree"                                       ,
"rioja"                                               ,
"robustSingleCell"                                    ,
"SimRVSequences"                                      ,
"socviz"                                              ,
"stars"                                               ,
"Unicode"                                             ,
"archetypes"                                          ,
"codemetar"                                           ,
"coop"                                                ,
"copula"                                              ,
"demography"                                          ,
"DepthProc"                                           ,
"DiagrammeR"                                          ,
"FedData"                                             ,
"FSelectorRcpp"                                       ,
"genetics"                                            ,
"igraph"                                              ,
"infer"                                               ,
"jmv"                                                 ,
"LCA"                                                 ,
"MLRShiny"                                            ,
"mpmi"                                                ,
"multimark"                                           ,
"oak"                                                 ,
"pamr"                                                ,
"quotedargs"                                          ,
"rcdd"                                                ,
"rfPermute"                                           ,
"rms"                                                 ,
"robustloggamma"                                      ,
"Rramas"                                              ,
"SamplingStrata"                                      ,
"shiny"                                               ,
"smapr"                                               ,
"Smisc"                                               ,
"subtee"                                              ,
"taxonomizr"                                          ,
"tidyquant"                                           ,
"TSSS"                                                ,
"Zelig"                                               ,
"boxoffice"                                           ,
"ftsa"                                                ,
"janitor"                                             ,
"ocp"                                                 ,
"randtoolbox"                                         ,
"splitstackshape"                                     ,
"xtable"                                              ,
"hdf5r"                                               ,
"music"                                               ,
"NNMIS"                                               ,
"Power2Stage"                                         ,
"akmedoids"                                           ,
"capm"                                                ,
"clickR"                                              ,
"FADA"                                                ,
"forestChange"                                        ,
"frailtySurv"                                         ,
"funHDDC"                                             ,
"gibble"                                              ,
"ICAOD"                                               ,
"iMRMC"                                               ,
"madness"                                             ,
"memery"                                              ,
"palasso"                                             ,
"splusTimeDate"                                       ,
"tint"                                                ,
"blockseg"                                            ,
"cba"                                                 ,
"cOde"                                                ,
"deisotoper"                                          ,
"GerminaR"                                            ,
"MetaCycle"                                           ,
"MiDA"                                                ,
"Orcs"                                                ,
"PDFEstimator"                                        ,
"POT"                                                 ,
"R.filesets"                                          ,
"SDMTools"                                            ,
"textmineR"                                           ,
"cAIC4"                                               ,
"eNetXplorer"                                         ,
"EpiILMCT"                                            ,
"explor"                                              ,
"filehash"                                            ,
"gamCopula"                                           ,
"geodist"                                             ,
"gjam"                                                ,
"gRc"                                                 ,
"Haplin"                                              ,
"hypergate"                                           ,
"swdft"                                               ,
"wfe"                                                 ,
"worldmet"                                            ,
"arulesSequences"                                     ,
"BoolNet"                                             ,
"catnet"                                              ,
"cmmr"                                                ,
"discSurv"                                            ,
"dstat"                                               ,
"fpeek"                                               ,
"frequentdirections"                                  ,
"geostatsp"                                           ,
"HDMT"                                                ,
"holodeck"                                            ,
"mdmb"                                                ,
"PTXQC"                                               ,
"scatterD3"                                           ,
"SPUTNIK"                                             ,
"survtmle"                                            ,
"TCIApathfinder"                                      ,
"tsfeatures"                                          ,
"tuneRanger"                                          ,
"allelematch"                                         ,
"BayesSenMC"                                          ,
"Brq"                                                 ,
"DiSSMod"                                             ,
"FastRWeb"                                            ,
"funtimes"                                            ,
"Irescale"                                            ,
"island"                                              ,
"JamendoR"                                            ,
"JavaGD"                                              ,
"lsbclust"                                            ,
"MFDFA"                                               ,
"pts2polys"                                           ,
"PwrGSD"                                              ,
"rnrfa"                                               ,
"robCompositions"                                     ,
"Rserve"                                              ,
"trip"                                                ,
"upwaver"                                             ,
"word.alignment"                                      ,
"glmulti"                                             ,
"KernelKnn"                                           ,
"localModel"                                          ,
"Rdpack"                                              ,
"RobRex"                                              ,
"ROptRegTS"                                           ,
"scorecardModelUtils"                                 ,
"textTinyR"                                           ,
"urltools"                                            ,
"ANN2"                                                ,
"BlockCov"                                            ,
"censusapi"                                           ,
"complmrob"                                           ,
"CTAShiny"                                            ,
"equivUMP"                                            ,
"fic"                                                 ,
"MetaLonDA"                                           ,
"multicolor"                                          ,
"nodbi"                                               ,
"qkerntool"                                           ,
"seqCBS"                                              ,
"spatstat.local"                                      ,
"TSstudio"                                            ,
"unine"                                               ,
"wildlifeDI"                                          ,
"earth"                                               ,
"FastGaSP"                                            ,
"frost"                                               ,
"GSE"                                                 ,
"HS"                                                  ,
"konfound"                                            ,
"NAM"                                                 ,
"netgwas"                                             ,
"precrec"                                             ,
"psychmeta"                                           ,
"qualtRics"                                           ,
"rpart"                                               ,
"rpf"                                                 ,
"RSCAT"                                               ,
"SARP.moodle"                                         ,
"scriptexec"                                          ,
"sctransform"                                         ,
"SeqFeatR"                                            ,
"subscore"                                            ,
"trackeRapp"                                          ,
"bdclean"                                             ,
"broman"                                              ,
"CaPO4Sim"                                            ,
"cogmapr"                                             ,
"diagmeta"                                            ,
"eiPartialID"                                         ,
"elastic"                                             ,
"entropart"                                           ,
"ePCR"                                                ,
"epidata"                                             ,
"eurostat"                                            ,
"futility"                                            ,
"HDclust"                                             ,
"kde1d"                                               ,
"ksNN"                                                ,
"kstIO"                                               ,
"lpirfs"                                              ,
"PAFit"                                               ,
"permutations"                                        ,
"PxWebApiData"                                        ,
"raptr"                                               ,
"RobLox"                                              ,
"RobLoxBioC"                                          ,
"ROptEstOld"                                          ,
"SIRE"                                                ,
"wdm"                                                 ,
"backports"                                           ,
"ChannelAttribution"                                  ,
"chromoMap"                                           ,
"cinterpolate"                                        ,
"deductive"                                           ,
"HDPenReg"                                            ,
"MetaUtility"                                         ,
"micar"                                               ,
"microsynth"                                          ,
"MOTE"                                                ,
"MRFcov"                                              ,
"plotly"                                              ,
"rdnb"                                                ,
"stemmatology"                                        ,
"SVN"                                                 ,
"tpr"                                                 ,
"TR8"                                                 ,
"bssm"                                                ,
"CEAmarkov"                                           ,
"codyn"                                               ,
"crank"                                               ,
"DCG"                                                 ,
"dynamac"                                             ,
"fst"                                                 ,
"ggiraph"                                             ,
"gluvarpro"                                           ,
"gremlin"                                             ,
"lmreg"                                               ,
"MendelianRandomization"                              ,
"MuMIn"                                               ,
"mvgraphnorm"                                         ,
"noaastormevents"                                     ,
"pracma"                                              ,
"prettyR"                                             ,
"rainfarmr"                                           ,
"reinforcelearn"                                      ,
"RHMS"                                                ,
"RIA"                                                 ,
"sigmajs"                                             ,
"spinifex"                                            ,
"ssh"                                                 ,
"STRAH"                                               ,
"SurrogateOutcome"                                    ,
"tbm"                                                 ,
"tfdatasets"                                          ,
"tm1r"                                                ,
"TSHRC"                                               ,
"TULIP"                                               ,
"attempt"                                             ,
"cgraph"                                              ,
"comf"                                                ,
"emdi"                                                ,
"evolqg"                                              ,
"foretell"                                            ,
"GetHFData"                                           ,
"huge"                                                ,
"jtools"                                              ,
"KenSyn"                                              ,
"lagsarlmtree"                                        ,
"LeArEst"                                             ,
"openair"                                             ,
"ProjectManagement"                                   ,
"rhub"                                                ,
"RobExtremes"                                         ,
"sboost"                                              ,
"shinydashboardPlus"                                  ,
"simLife"                                             ,
"SMUT"                                                ,
"TreeSim"                                             ,
"unival"                                              ,
"adehabitatHR"                                        ,
"adehabitatHS"                                        ,
"adehabitatLT"                                        ,
"adehabitatMA"                                        ,
"Binarize"                                            ,
"BiTrinA"                                             ,
"broom"                                               ,
"countToFPKM"                                         ,
"data.table"                                          ,
"DatabaseConnectorJars"                               ,
"ddalpha"                                             ,
"enviPat"                                             ,
"fourierin"                                           ,
"GFORCE"                                              ,
"KSPM"                                                ,
"kst"                                                 ,
"leafpop"                                             ,
"pbdMPI"                                              ,
"pendensity"                                          ,
"postinfectious"                                      ,
"PP"                                                  ,
"Rblpapi"                                             ,
"treespace"                                           ,
"uwot"                                                ,
"ExPanDaR"                                            ,
"FinTS"                                               ,
"IBDsim"                                              ,
"isoband"                                             ,
"MLML2R"                                              ,
"ProPublicaR"                                         ,
"QPot"                                                ,
"quadmesh"                                            ,
"sandwich"                                            ,
"sugrrants"                                           ,
"syntaxr"                                             ,
"tRophicPosition"                                     ,
"virtuoso"                                            ,
"argo"                                                ,
"cplots"                                              ,
"DHS.rates"                                           ,
"dynamichazard"                                       ,
"FlowScreen"                                          ,
"geonetwork"                                          ,
"hopit"                                               ,
"httpuv"                                              ,
"keras"                                               ,
"lconnect"                                            ,
"mnlfa"                                               ,
"networkDynamic"                                      ,
"pkgnet"                                              ,
"RadData"                                             ,
"rfVarImpOOB"                                         ,
"spdep"                                               ,
"adnuts"                                              ,
"agricolae"                                           ,
"armada"                                              ,
"calcWOI"                                             ,
"chartql"                                             ,
"collections"                                         ,
"CoxPhLb"                                             ,
"CRFCSD"                                              ,
"distrEllipse"                                        ,
"dsa"                                                 ,
"dynparam"                                            ,
"ebirdst"                                             ,
"flagr"                                               ,
"geneHummus"                                          ,
"genpwr"                                              ,
"glmdisc"                                             ,
"hglm"                                                ,
"hglm.data"                                           ,
"hydra"                                               ,
"llbayesireg"                                         ,
"SADISA"                                              ,
"saotd"                                               ,
"TreeBUGS"                                            ,
"TSEtools"                                            ,
"unifDAG"                                             ,
"wikisourcer"                                         ,
"aniDom"                                              ,
"bmrm"                                                ,
"desplot"                                             ,
"elect"                                               ,
"mmabig"                                              ,
"pcmabc"                                              ,
"profileModel"                                        ,
"RandVar"                                             ,
"RefManageR"                                          ,
"RiverLoad"                                           ,
"rpsftm"                                              ,
"suncalc"                                             ,
"ALA4R"                                               ,
"brglm"                                               ,
"conting"                                             ,
"crblocks"                                            ,
"distillery"                                          ,
"eba"                                                 ,
"EmissV"                                              ,
"gaston"                                              ,
"gganimate"                                           ,
"ggpmisc"                                             ,
"GPCMlasso"                                           ,
"gscounts"                                            ,
"MetaSKAT"                                            ,
"mlbgameday"                                          ,
"network"                                             ,
"nmfem"                                               ,
"optparse"                                            ,
"org"                                                 ,
"ptsuite"                                             ,
"SCBiclust"                                           ,
"steps"                                               ,
"SUMMER"                                              ,
"ukpolice"                                            ,
"adoption"                                            ,
"alfred"                                              ,
"bedr"                                                ,
"cheese"                                              ,
"desctable"                                           ,
"easyalluvial"                                        ,
"EValue"                                              ,
"extRemes"                                            ,
"grplassocat"                                         ,
"gwer"                                                ,
"hurricaneexposure"                                   ,
"lawstat"                                             ,
"lisa"                                                ,
"MAP"                                                 ,
"NMOF"                                                ,
"opencv"                                              ,
"OptimalCutpoints"                                    ,
"PublicationBias"                                     ,
"rhoR"                                                ,
"rPraat"                                              ,
"simrel"                                              ,
"siplab"                                              ,
"smbinning"                                           ,
"spatialreg"                                          ,
"STAT"                                                ,
"survival"                                            ,
"Umoments"                                            ,
"WRSS"                                                ,
"AmigaFFH"                                            ,
"BDEsize"                                             ,
"imfr"                                                ,
"jrc"                                                 ,
"llogistic"                                           ,
"LNIRT"                                               ,
"nhds"                                                ,
"priceR"                                              ,
"adeptdata"                                           ,
"aire.zmvm"                                           ,
"clustringr"                                          ,
"DataPackageR"                                        ,
"NDP"                                                 ,
"REddyProc"                                           ,
"smcfcs"                                              ,
"unpivotr"                                            ,
"autoFRK"                                             ,
"clusternor"                                          ,
"distrEx"                                             ,
"dotdot"                                              ,
"DZEXPM"                                              ,
"easyPubMed"                                          ,
"EdSurvey"                                            ,
"EWOC2"                                               ,
"fakemake"                                            ,
"funcy"                                               ,
"gdmp"                                                ,
"hdi"                                                 ,
"IMWatson"                                            ,
"IndexConstruction"                                   ,
"intensity.analysis"                                  ,
"lori"                                                ,
"rapidxmlr"                                           ,
"rJava"                                               ,
"snipEM"                                              ,
"tapkee"                                              ,
"tatoo"                                               ,
"bipartiteD3"                                         ,
"Cairo"                                               ,
"drat"                                                ,
"dyads"                                               ,
"ffmanova"                                            ,
"homologene"                                          ,
"languageserver"                                      ,
"MLCM"                                                ,
"nlrx"                                                ,
"orloca"                                              ,
"photobiologyWavebands"                               ,
"saeSim"                                              ,
"SurrogateTest"                                       ,
"tgcd"                                                ,
"cutpointr"                                           ,
"lfe"                                                 ,
"netgsa"                                              ,
"npcp"                                                ,
"photobiologySun"                                     ,
"POUMM"                                               ,
"Rnmr1D"                                              ,
"tidyhydat"                                           ,
"unitizer"                                            ,
"x3ptools"                                            ,
"blockcluster"                                        ,
"childsds"                                            ,
"ecotox"                                              ,
"estmeansd"                                           ,
"iMediate"                                            ,
"namer"                                               ,
"netdiffuseR"                                         ,
"rangeMapper"                                         ,
"Rpolyhedra"                                          ,
"RVAideMemoire"                                       ,
"SentimentAnalysis"                                   ,
"spatialsegregation"                                  ,
"theiaR"                                              ,
"BTLLasso"                                            ,
"clustcurv"                                           ,
"Copula.Markov"                                       ,
"CopulaREMADA"                                        ,
"cvcrand"                                             ,
"dbmss"                                               ,
"doremi"                                              ,
"dydea"                                               ,
"ems"                                                 ,
"gginnards"                                           ,
"gsl"                                                 ,
"gtable"                                              ,
"JMbayes"                                             ,
"latte"                                               ,
"QTLRel"                                              ,
"quiddich"                                            ,
"SGCS"                                                ,
"sybilccFBA"                                          ,
"tcR"                                                 ,
"Tmisc"                                               ,
"tryCatchLog"                                         ,
"fbroc"                                               ,
"ideamdb"                                             ,
"listarrays"                                          ,
"MaxMC"                                               ,
"Replicate"                                           ,
"rocTree"                                             ,
"spsh"                                                ,
"texteffect"                                          ,
"vistime"                                             ,
"WufooR"                                              ,
"anthro"                                              ,
"cyphr"                                               ,
"english"                                             ,
"exprso"                                              ,
"ggasym"                                              ,
"IBCF.MTME"                                           ,
"loon"                                                ,
"MNLR"                                                ,
"OutliersO3"                                          ,
"pls"                                                 ,
"spsur"                                               ,
"SurvBoost"                                           ,
"ActiveDriverWGS"                                     ,
"apparent"                                            ,
"cleanr"                                              ,
"dclone"                                              ,
"EMCluster"                                           ,
"getopt"                                              ,
"gtfsrouter"                                          ,
"heatmap3"                                            ,
"ideq"                                                ,
"indelmiss"                                           ,
"jcp"                                                 ,
"kayadata"                                            ,
"kmcudaR"                                             ,
"markophylo"                                          ,
"Matrix"                                              ,
"MM2S"                                                ,
"mwaved"                                              ,
"ncf"                                                 ,
"osc"                                                 ,
"plainview"                                           ,
"pmhtutorial"                                         ,
"polynom"                                             ,
"reactlog"                                            ,
"rMEA"                                                ,
"RZooRoH"                                             ,
"seacarb"                                             ,
"sentimentr"                                          ,
"shiftR"                                              ,
"shinyhttr"                                           ,
"simTool"                                             ,
"strvalidator"                                        ,
"support.BWS2"                                        ,
"toxEval"                                             ,
"treeclim"                                            ,
"xxIRT"                                               ,
"assertthat"                                          ,
"basictabler"                                         ,
"cornet"                                              ,
"crimedata"                                           ,
"datetimeutils"                                       ,
"DynTxRegime"                                         ,
"edfReader"                                           ,
"expm"                                                ,
"FamEvent"                                            ,
"geotopbricks"                                        ,
"GPM"                                                 ,
"lexicon"                                             ,
"lorentz"                                             ,
"maGUI"                                               ,
"mgcv"                                                ,
"MultiVarSel"                                         ,
"picante"                                             ,
"Rodam"                                               ,
"rtiff"                                               ,
"slouch"                                              ,
"tensorBSS"                                           ,
"WeightedCluster"                                     ,
"xpose"                                               ,
"CJAMP"                                               ,
"clifro"                                              ,
"CopulaCenR"                                          ,
"countTransformers"                                   ,
"gamRR"                                               ,
"highr"                                               ,
"iprior"                                              ,
"MFPCA"                                               ,
"microsamplingDesign"                                 ,
"MlBayesOpt"                                          ,
"moveWindSpeed"                                       ,
"MPN"                                                 ,
"MUS"                                                 ,
"psd"                                                 ,
"psymonitor"                                          ,
"RadOnc"                                              ,
"RNiftyReg"                                           ,
"siland"                                              ,
"swgee"                                               ,
"synthpop"                                            ,
"tabula"                                              ,
"tidystringdist"                                      ,
"adaptivetau"                                         ,
"amt"                                                 ,
"audio"                                               ,
"binsreg"                                             ,
"BioCircos"                                           ,
"cctools"                                             ,
"cli"                                                 ,
"CoopGame"                                            ,
"dfcomb"                                              ,
"dockerfiler"                                         ,
"eeptools"                                            ,
"emulator"                                            ,
"GGMM"                                                ,
"KONPsurv"                                            ,
"PBSadmb"                                             ,
"Perc"                                                ,
"rags2ridges"                                         ,
"rdbnomics"                                           ,
"rdhs"                                                ,
"rfordummies"                                         ,
"RHRV"                                                ,
"rstudioapi"                                          ,
"s2dverification"                                     ,
"SpaDES.core"                                         ,
"SpaDES.tools"                                        ,
"sparseBC"                                            ,
"subcopem2D"                                          ,
"TauStar"                                             ,
"TempCont"                                            ,
"trackdem"                                            ,
"whoami"                                              ,
"xesreadR"                                            ,
"amelie"                                              ,
"BalanceCheck"                                        ,
"catlearn"                                            ,
"clusteredinterference"                               ,
"colorspace"                                          ,
"DistributionOptimization"                            ,
"ecp"                                                 ,
"eechidna"                                            ,
"flexsurv"                                            ,
"GMSE"                                                ,
"msm"                                                 ,
"MVN"                                                 ,
"ohtadstats"                                          ,
"parglm"                                              ,
"PoA"                                                 ,
"popEpi"                                              ,
"rmatio"                                              ,
"RoundAndRound"                                       ,
"ruta"                                                ,
"RxCEcolInf"                                          ,
"SARP.compo"                                          ,
"shinyWidgets"                                        ,
"swmmr"                                               ,
"synoptReg"                                           ,
"thriftr"                                             ,
"tosca"                                               ,
"wsyn"                                                ,
"ape"                                                 ,
"benthos"                                             ,
"cprr"                                                ,
"Cyclops"                                             ,
"DataExplorer"                                        ,
"DescTools"                                           ,
"lexRankr"                                            ,
"mmand"                                               ,
"MultiFit"                                            ,
"pbdDMAT"                                             ,
"pointdexter"                                         ,
"QAIG"                                                ,
"rcss"                                                ,
"ROptSpace"                                           ,
"SPSL"                                                ,
"sqlscore"                                            ,
"WaveletANN"                                          ,
"automl"                                              ,
"BAMBI"                                               ,
"bigdist"                                             ,
"bulletcp"                                            ,
"clustMixType"                                        ,
"haplotypes"                                          ,
"iheatmapr"                                           ,
"kableExtra"                                          ,
"kamila"                                              ,
"mapedit"                                             ,
"mediation"                                           ,
"MM2Sdata"                                            ,
"qicharts2"                                           ,
"RAppArmor"                                           ,
"resautonet"                                          ,
"SPreg"                                               ,
"tableHTML"                                           ,
"utiml"                                               ,
"AlphaSimR"                                           ,
"bazar"                                               ,
"clustDRM"                                            ,
"cubeview"                                            ,
"cvar"                                                ,
"EGRETci"                                             ,
"getmstatistic"                                       ,
"gitlabr"                                             ,
"jrvFinance"                                          ,
"lazyeval"                                            ,
"Lmoments"                                            ,
"MachineLearning"                                     ,
"mcBFtest"                                            ,
"msir"                                                ,
"neighbr"                                             ,
"odpc"                                                ,
"ordinalgmifs"                                        ,
"PBSmapping"                                          ,
"PCMBase"                                             ,
"pocrm"                                               ,
"purrr"                                               ,
"purrrlyr"                                            ,
"pushbar"                                             ,
"qwraps2"                                             ,
"Relexon"                                             ,
"rerf"                                                ,
"ridge"                                               ,
"Rmalschains"                                         ,
"rosetteApi"                                          ,
"StatMatch"                                           ,
"stdReg"                                              ,
"TIMP"                                                ,
"vetr"                                                ,
"AMIAS"                                               ,
"baytrends"                                           ,
"bnclassify"                                          ,
"diversitree"                                         ,
"elliptic"                                            ,
"EventStudy"                                          ,
"fsdaR"                                               ,
"ggpol"                                               ,
"incidence"                                           ,
"PBSmodelling"                                        ,
"polyclip"                                            ,
"Ropj"                                                ,
"Runuran"                                             ,
"rwavelet"                                            ,
"thor"                                                ,
"AGread"                                              ,
"bayesCT"                                             ,
"camtrapR"                                            ,
"cloudSimplifieR"                                     ,
"cxhull"                                              ,
"DepLogo"                                             ,
"discfrail"                                           ,
"eiPack"                                              ,
"flsa"                                                ,
"gaselect"                                            ,
"geofacet"                                            ,
"hexView"                                             ,
"leafpm"                                              ,
"librarian"                                           ,
"linearQ"                                             ,
"lmfor"                                               ,
"loo"                                                 ,
"mgsub"                                               ,
"modeest"                                             ,
"multiway"                                            ,
"obliqueRSF"                                          ,
"panelView"                                           ,
"pgbart"                                              ,
"phylin"                                              ,
"plantecophys"                                        ,
"pubtatordb"                                          ,
"raustats"                                            ,
"readxl"                                              ,
"rrscale"                                             ,
"runstats"                                            ,
"sparkxgb"                                            ,
"TreeLS"                                              ,
"waiter"                                              ,
"xgb2sql"                                             ,
"yum"                                                 ,
"blorr"                                               ,
"Compind"                                             ,
"credentials"                                         ,
"diffdf"                                              ,
"disclapmix"                                          ,
"expSBM"                                              ,
"FixedPoint"                                          ,
"gamesGA"                                             ,
"gamlss.inf"                                          ,
"gfcanalysis"                                         ,
"ggperiodic"                                          ,
"glue"                                                ,
"GMCM"                                                ,
"gratia"                                              ,
"HHG"                                                 ,
"icesDatras"                                          ,
"icesSAG"                                             ,
"icesVocab"                                           ,
"ivtools"                                             ,
"jiebaR"                                              ,
"lmom"                                                ,
"lmQCM"                                               ,
"LPWC"                                                ,
"optimStrat"                                          ,
"paleoTS"                                             ,
"PBSddesolve"                                         ,
"permute"                                             ,
"RCircos"                                             ,
"rivr"                                                ,
"rprev"                                               ,
"SimplifyStats"                                       ,
"sitar"                                               ,
"SPCDAnalyze"                                         ,
"stringi"                                             ,
"SuperGauss"                                          ,
"sylcount"                                            ,
"alphavantager"                                       ,
"bdpv"                                                ,
"corTest"                                             ,
"distr"                                               ,
"distrDoc"                                            ,
"distrRmetrics"                                       ,
"distrSim"                                            ,
"distrTeach"                                          ,
"distrTEst"                                           ,
"dlnm"                                                ,
"Equalden.HD"                                         ,
"gmp"                                                 ,
"jarbes"                                              ,
"MFT"                                                 ,
"mixdir"                                              ,
"ncdf4"                                               ,
"neariso"                                             ,
"pacman"                                              ,
"pacotest"                                            ,
"pairwiseCI"                                          ,
"pimeta"                                              ,
"pmxTools"                                            ,
"rbtc"                                                ,
"readobj"                                             ,
"Rmixmod"                                             ,
"RobAStRDA"                                           ,
"SHT"                                                 ,
"spelling"                                            ,
"spThin"                                              ,
"startupmsg"                                          ,
"ConfIntVariance"                                     ,
"easyCODA"                                            ,
"funData"                                             ,
"JacobiEigen"                                         ,
"lmomRFA"                                             ,
"mcmc"                                                ,
"mimsy"                                               ,
"myTAI"                                               ,
"pdftools"                                            ,
"pm4py"                                               ,
"potts"                                               ,
"prais"                                               ,
"simcdm"                                              ,
"simmer.plot"                                         ,
"FusionLearn"                                         ,
"gSeg"                                                ,
"ifaTools"                                            ,
"networktree"                                         ,
"agop"                                                ,
"argparse"                                            ,
"arrApply"                                            ,
"biomod2"                                             ,
"conquestr"                                           ,
"deepNN"                                              ,
"discharge"                                           ,
"ELISAtools"                                          ,
"EntropyMCMC"                                         ,
"fastglm"                                             ,
"findpython"                                          ,
"glmlep"                                              ,
"liteq"                                               ,
"loadr"                                               ,
"MODIS"                                               ,
"mulset"                                              ,
"needmining"                                          ,
"petrinetR"                                           ,
"phmm"                                                ,
"retrodesign"                                         ,
"RGtk2"                                               ,
"rivernet"                                            ,
"robustrank"                                          ,
"rockchalk"                                           ,
"RSSL"                                                ,
"runner"                                              ,
"seastests"                                           ,
"spam"                                                ,
"spam64"                                              ,
"tablerDash"                                          ,
"tanaka"                                              ,
"tenispolaR"                                          ,
"TestDataImputation"                                  ,
"validatetools"                                       ,
"calibrator"                                          ,
"CARRoT"                                              ,
"cdfquantreg"                                         ,
"ChillModels"                                         ,
"cptcity"                                             ,
"DTAXG"                                               ,
"FishResp"                                            ,
"furniture"                                           ,
"GaussianHMM1d"                                       ,
"ggfan"                                               ,
"GJRM"                                                ,
"invctr"                                              ,
"lmSubsets"                                           ,
"longpower"                                           ,
"mimi"                                                ,
"ML.MSBD"                                             ,
"moc.gapbk"                                           ,
"mri"                                                 ,
"nvctr"                                               ,
"partitionComparison"                                 ,
"processR"                                            ,
"qpdf"                                                ,
"randomNames"                                         ,
"ranger"                                              ,
"SCRABBLE"                                            ,
"SDraw"                                               ,
"SOMbrero"                                            ,
"spBayes"                                             ,
"stapler"                                             ,
"STV"                                                 ,
"sybil"                                               ,
"TSDFGS"                                              ,
"varbvs"                                              ,
"vistributions"                                       ,
"adamethods"                                          ,
"binostics"                                           ,
"caTools"                                             ,
"CoDiNA"                                              ,
"CorrBin"                                             ,
"DHARMa"                                              ,
"ECGofTestDx"                                         ,
"emld"                                                ,
"gastempt"                                            ,
"GCalcium"                                            ,
"gcite"                                               ,
"ggupset"                                             ,
"glmnetUtils"                                         ,
"hettx"                                               ,
"KnowGRRF"                                            ,
"lenses"                                              ,
"lineup"                                              ,
"mvord"                                               ,
"networkABC"                                          ,
"noteMD"                                              ,
"party"                                               ,
"polysat"                                             ,
"PROMETHEE"                                           ,
"Quartet"                                             ,
"radiant.design"                                      ,
"RandomFields"                                        ,
"rasterize"                                           ,
"RCAL"                                                ,
"regrrr"                                              ,
"RInside"                                             ,
"ROI.plugin.scs"                                      ,
"rptR"                                                ,
"scs"                                                 ,
"slideview"                                           ,
"sparseHessianFD"                                     ,
"stacomiR"                                            ,
"TextForecast"                                        ,
"ttTensor"                                            ,
"tvm"                                                 ,
"vqtl"                                                ,
"automagic"                                           ,
"CEGO"                                                ,
"cplm"                                                ,
"dvmisc"                                              ,
"firebehavioR"                                        ,
"GPSCDF"                                              ,
"HGNChelper"                                          ,
"leafsync"                                            ,
"lifecontingencies"                                   ,
"LLSR"                                                ,
"lme4"                                                ,
"midasr"                                              ,
"multcomp"                                            ,
"OBsMD"                                               ,
"POINT"                                               ,
"prcbench"                                            ,
"probhat"                                             ,
"proxy"                                               ,
"Rd2roxygen"                                          ,
"registry"                                            ,
"scifigure"                                           ,
"seqmon"                                              ,
"SongEvo"                                             ,
"sonicLength"                                         ,
"sparktf"                                             ,
"spMC"                                                ,
"TooManyCellsR"                                       ,
"apa"                                                 ,
"ccrs"                                                ,
"Corbi"                                               ,
"CRF"                                                 ,
"crosswalkr"                                          ,
"disttools"                                           ,
"equivalenceTest"                                     ,
"fftw"                                                ,
"hash"                                                ,
"ifultools"                                           ,
"MarginalMediation"                                   ,
"mpcmp"                                               ,
"packHV"                                              ,
"RandomFieldsUtils"                                   ,
"RCurl"                                               ,
"REBayes"                                             ,
"RKUM"                                                ,
"rmutil"                                              ,
"RMySQL"                                              ,
"rNOMADS"                                             ,
"SmCCNet"                                             ,
"snem"                                                ,
"SparseMSE"                                           ,
"sunburstR"                                           ,
"tau"                                                 ,
"tis"                                                 ,
"walker"                                              ,
"webp"                                                ,
"bigstatsr"                                           ,
"CSESA"                                               ,
"discoveR"                                            ,
"gambin"                                              ,
"rasciidoc"                                           ,
"shinymaterial"                                       ,
"untb"                                                ,
"WDI"                                                 ,
"zipR"                                                ,
"DAAG"                                                ,
"documenter"                                          ,
"ganalytics"                                          ,
"prnsamplr"                                           ,
"spongebob"                                           ,
"BalancedSampling"                                    ,
"beautier"                                            ,
"bigMap"                                              ,
"BinOrdNonNor"                                        ,
"EWGoF"                                               ,
"feisr"                                               ,
"linguisticsdown"                                     ,
"lulcc"                                               ,
"MODISTools"                                          ,
"PubMedWordcloud"                                     ,
"rzmq"                                                ,
"tfio"                                                ,
"threshr"                                             ,
"tidyr"                                               ,
"tracerer"                                            ,
"BALCONY"                                             ,
"dief"                                                ,
"dosearch"                                            ,
"geoBayes"                                            ,
"ISCO08ConveRsions"                                   ,
"live"                                                ,
"Mega2R"                                              ,
"moc"                                                 ,
"NetworkInference"                                    ,
"selac"                                               ,
"SmallCountRounding"                                  ,
"xplorerr"                                            ,
"AutoPipe"                                            ,
"bst"                                                 ,
"gWidgets2"                                           ,
"LSAfun"                                              ,
"NLMR"                                                ,
"pcadapt"                                             ,
"pez"                                                 ,
"r.blip"                                              ,
"RegSDC"                                              ,
"RMTL"                                                ,
"SDLfilter"                                           ,
"segclust2d"                                          ,
"spdynmod"                                            ,
"tsvr"                                                ,
"anipaths"                                            ,
"canprot"                                             ,
"ECoL"                                                ,
"forge"                                               ,
"gama"                                                ,
"grpreg"                                              ,
"hmeasure"                                            ,
"list"                                                ,
"mmapcharr"                                           ,
"modelObj"                                            ,
"motoRneuron"                                         ,
"ncvreg"                                              ,
"NetWeaver"                                           ,
"paleobioDB"                                          ,
"phenocamr"                                           ,
"plsr"                                                ,
"ProjectTemplate"                                     ,
"RWeka"                                               ,
"sdm"                                                 ,
"sievePH"                                             ,
"simpleCache"                                         ,
"slam"                                                ,
"splashr"                                             ,
"ThresholdROC"                                        ,
"AGSDest"                                             ,
"cellWise"                                            ,
"CINNA"                                               ,
"clue"                                                ,
"coveffectsplot"                                      ,
"dtplyr"                                              ,
"easyDes"                                             ,
"Ecfun"                                               ,
"franc"                                               ,
"GWEX"                                                ,
"LadR"                                                ,
"landscapetools"                                      ,
"RcppStreams"                                         ,
"robustX"                                             ,
"RRphylo"                                             ,
"bootstrapFP"                                         ,
"cp4p"                                                ,
"dr4pl"                                               ,
"fChange"                                             ,
"jcext"                                               ,
"kernplus"                                            ,
"LocalControl"                                        ,
"MCMCvis"                                             ,
"migest"                                              ,
"PriorCD"                                             ,
"processmapR"                                         ,
"SIBER"                                               ,
"snowboot"                                            ,
"toOrdinal"                                           ,
"tsoutliers"                                          ,
"UnivRNG"                                             ,
"unix"                                                ,
"wpp2017"                                             ,
"ashr"                                                ,
"deepgmm"                                             ,
"deTS"                                                ,
"ecosim"                                              ,
"ipeadatar"                                           ,
"JQL"                                                 ,
"lin.eval"                                            ,
"MoonFinder"                                          ,
"NScluster"                                           ,
"pathlibr"                                            ,
"PointFore"                                           ,
"RichR"                                               ,
"rmio"                                                ,
"RVowpalWabbit"                                       ,
"shinyEventLogger"                                    ,
"spm"                                                 ,
"stlcsb"                                              ,
"surveillance"                                        ,
"UtilityFrailtyPH12"                                  ,
"BeSS"                                                ,
"bmlm"                                                ,
"broom.mixed"                                         ,
"dang"                                                ,
"FossilSim"                                           ,
"GB2group"                                            ,
"nonlinearTseries"                                    ,
"ordinalNet"                                          ,
"picasso"                                             ,
"RcppCWB"                                             ,
"rSEA"                                                ,
"urbin"                                               ,
"wru"                                                 ,
"easySdcTable"                                        ,
"glmertree"                                           ,
"PlayerRatings"                                       ,
"pliable"                                             ,
"polidata"                                            ,
"rstiefel"                                            ,
"Scalelink"                                           ,
"SGP"                                                 ,
"simpleboot"                                          ,
"tmle"                                                ,
"acs"                                                 ,
"bupaR"                                               ,
"configural"                                          ,
"curveDepth"                                          ,
"fugue"                                               ,
"geonames"                                            ,
"ggvoronoi"                                           ,
"KOGMWU"                                              ,
"pak"                                                 ,
"perturbR"                                            ,
"pgdraw"                                              ,
"relMix"                                              ,
"rmdformats"                                          ,
"rollmatch"                                           ,
"rsppfp"                                              ,
"stationery"                                          ,
"bdchecks"                                            ,
"cchs"                                                ,
"dkanr"                                               ,
"fiberLD"                                             ,
"flexmix"                                             ,
"ggsn"                                                ,
"imageviewer"                                         ,
"LongMemoryTS"                                        ,
"maptools"                                            ,
"mixsqp"                                              ,
"nonmem2R"                                            ,
"nsga3"                                               ,
"openVA"                                              ,
"optimos.prime"                                       ,
"pcgen"                                               ,
"PUPAIM"                                              ,
"segmenTier"                                          ,
"SPAS"                                                ,
"telefit"                                             ,
"tidygraph"                                           ,
"VGAM"                                                ,
"BAwiR"                                               ,
"foghorn"                                             ,
"forcats"                                             ,
"quadprogXT"                                          ,
"sloop"                                               ,
"survELtest"                                          ,
"tableone"                                            ,
"ThreeArmedTrials"                                    ,
"areaplot"                                            ,
"hoa"                                                 ,
"matsbyname"                                          ,
"ROI.plugin.neos"                                     ,
"rPref"                                               ,
"ahpsurvey"                                           ,
"BacArena"                                            ,
"bcmaps"                                              ,
"dbhydroR"                                            ,
"FNN"                                                 ,
"irg"                                                 ,
"MATA"                                                ,
"naniar"                                              ,
"neo4r"                                               ,
"rwty"                                                ,
"scopr"                                               ,
"visdat"                                              ,
"BoltzMM"                                             ,
"brglm2"                                              ,
"mRMRe"                                               ,
"R6"                                                  ,
"triangle"                                            ,
"auth0"                                               ,
"carbonate"                                           ,
"DSsim"                                               ,
"HierDpart"                                           ,
"isqg"                                                ,
"logmult"                                             ,
"ordinalCont"                                         ,
"paletteer"                                           ,
"philentropy"                                         ,
"qsub"                                                ,
"SoyNAM"                                              ,
"staplr"                                              ,
"survivalAnalysis"                                    ,
"xROI"                                                ,
"autocogs"                                            ,
"BMTME"                                               ,
"cgwtools"                                            ,
"Claddis"                                             ,
"CornerstoneR"                                        ,
"emdbook"                                             ,
"forecastHybrid"                                      ,
"gridSVG"                                             ,
"hoardeR"                                             ,
"IDPmisc"                                             ,
"languagelayeR"                                       ,
"r6extended"                                          ,
"randnet"                                             ,
"rcitoid"                                             ,
"seedwater"                                           ,
"tidystopwords"                                       ,
"tourr"                                               ,
"BEACH"                                               ,
"calendar"                                            ,
"HeritSeq"                                            ,
"HRW"                                                 ,
"IrregLong"                                           ,
"kaphom"                                              ,
"later"                                               ,
"lmerTest"                                            ,
"matsindf"                                            ,
"mccmeiv"                                             ,
"metaMix"                                             ,
"Rfssa"                                               ,
"rPACI"                                               ,
"rrum"                                                ,
"sdpt3r"                                              ,
"smicd"                                               ,
"sparsenet"                                           ,
"tfse"                                                ,
"tibbletime"                                          ,
"TmCalculator"                                        ,
"uclust"                                              ,
"VIM"                                                 ,
"bpbounds"                                            ,
"cleanerR"                                            ,
"CombinePortfolio"                                    ,
"GWLelast"                                            ,
"kangar00"                                            ,
"modules"                                             ,
"OpenImageR"                                          ,
"rust"                                                ,
"stringr"                                             ,
"valaddin"                                            ,
"daarem"                                              ,
"docxtools"                                           ,
"FuzzyStatProb"                                       ,
"jipApprox"                                           ,
"Rglpk"                                               ,
"SpatioTemporal"                                      ,
"timeSeq"                                             ,
"bain"                                                ,
"deconvolveR"                                         ,
"EGRET"                                               ,
"geojsonlint"                                         ,
"GPfit"                                               ,
"GVARX"                                               ,
"iC10"                                                ,
"ICcforest"                                           ,
"MEMSS"                                               ,
"mlmRev"                                              ,
"noaaoceans"                                          ,
"RationalExp"                                         ,
"rehydratoR"                                          ,
"robFitConGraph"                                      ,
"ropendata"                                           ,
"Sequential"                                          ,
"StempCens"                                           ,
"abjutils"                                            ,
"cartography"                                         ,
"CascadeData"                                         ,
"CLME"                                                ,
"CongreveLamsdell2016"                                ,
"dartR"                                               ,
"daymetr"                                             ,
"metamisc"                                            ,
"NestedCategBayesImpute"                              ,
"neuralnet"                                           ,
"piggyback"                                           ,
"polyCub"                                             ,
"PortfolioOptim"                                      ,
"rijkspalette"                                        ,
"TDA"                                                 ,
"wmwpow"                                              ,
"bayfoxr"                                             ,
"EFAutilities"                                        ,
"GenomicTools"                                        ,
"GenomicTools.fileHandler"                            ,
"geosample"                                           ,
"GWSDAT"                                              ,
"lucid"                                               ,
"metap"                                               ,
"mfx"                                                 ,
"mstknnclust"                                         ,
"ordinalForest"                                       ,
"overlapptest"                                        ,
"pcLasso"                                             ,
"promote"                                             ,
"ProTrackR"                                           ,
"PTAk"                                                ,
"roll"                                                ,
"rstap"                                               ,
"Surrogate"                                           ,
"testDriveR"                                          ,
"totalcensus"                                         ,
"vortexR"                                             ,
"anapuce"                                             ,
"bayesdistreg"                                        ,
"carpenter"                                           ,
"CMatching"                                           ,
"dqshiny"                                             ,
"enveomics.R"                                         ,
"FuzzyNumbers"                                        ,
"ggmap"                                               ,
"ggquiver"                                            ,
"iml"                                                 ,
"irlba"                                               ,
"jsonld"                                              ,
"mcca"                                                ,
"missCompare"                                         ,
"MScombine"                                           ,
"NSO1212"                                             ,
"PMA"                                                 ,
"proustr"                                             ,
"qtlcharts"                                           ,
"R.rsp"                                               ,
"RiemBase"                                            ,
"unmarked"                                            ,
"bender"                                              ,
"bfw"                                                 ,
"eulerr"                                              ,
"FactorsR"                                            ,
"fedregs"                                             ,
"glmmsr"                                              ,
"growth"                                              ,
"IMIFA"                                               ,
"matchingMarkets"                                     ,
"mephas"                                              ,
"multistateutils"                                     ,
"PopGenReport"                                        ,
"RMark"                                               ,
"rsinaica"                                            ,
"secrdesign"                                          ,
"stable"                                              ,
"trread"                                              ,
"trtf"                                                ,
"vortexRdata"                                         ,
"zTree"                                               ,
"ARTool"                                              ,
"cbsem"                                               ,
"countfitteR"                                         ,
"dpcR"                                                ,
"echor"                                               ,
"expp"                                                ,
"GeneralizedUmatrix"                                  ,
"gnlm"                                                ,
"helminthR"                                           ,
"impimp"                                              ,
"lhs"                                                 ,
"MMeM"                                                ,
"MMLR"                                                ,
"phase1PRMD"                                          ,
"plsRcox"                                             ,
"prototest"                                           ,
"robustlmm"                                           ,
"rts"                                                 ,
"shutterstock"                                        ,
"simone"                                              ,
"SoftClustering"                                      ,
"SpaDES"                                              ,
"SpaDES.addins"                                       ,
"vagalumeR"                                           ,
"clogitL1"                                            ,
"maximin"                                             ,
"metScanR"                                            ,
"mousetrap"                                           ,
"npmr"                                                ,
"phylobase"                                           ,
"plsRglm"                                             ,
"readbulk"                                            ,
"rSymPy"                                              ,
"tbrf"                                                ,
"bootLR"                                              ,
"colorfindr"                                          ,
"dina"                                                ,
"egor"                                                ,
"fda.usc"                                             ,
"lawn"                                                ,
"localICE"                                            ,
"ltsk"                                                ,
"MALDIquantForeign"                                   ,
"NMAoutlier"                                          ,
"Nonpareil"                                           ,
"plsRbeta"                                            ,
"polymapR"                                            ,
"QCEWAS"                                              ,
"QuantNorm"                                           ,
"readsdmx"                                            ,
"rmd"                                                 ,
"SCAT"                                                ,
"SensusR"                                             ,
"simsurv"                                             ,
"UPMASK"                                              ,
"AdaptGauss"                                          ,
"DistatisR"                                           ,
"fGarch"                                              ,
"gendist"                                             ,
"geojson"                                             ,
"GSA"                                                 ,
"hyper.fit"                                           ,
"InPosition"                                          ,
"IRISMustangMetrics"                                  ,
"KnowBR"                                              ,
"MonteCarlo"                                          ,
"mpoly"                                               ,
"OpenCL"                                              ,
"pairwise"                                            ,
"plsdof"                                              ,
"protoclust"                                          ,
"PTE"                                                 ,
"ropenaq"                                             ,
"Rwinsteps"                                           ,
"TExPosition"                                         ,
"TInPosition"                                         ,
"agriwater"                                           ,
"bitsqueezr"                                          ,
"busdater"                                            ,
"dominanceanalysis"                                   ,
"FlickrAPI"                                           ,
"fromo"                                               ,
"geoknife"                                            ,
"languageR"                                           ,
"MazamaSpatialUtils"                                  ,
"nlreg"                                               ,
"ordinalLBM"                                          ,
"phreeqc"                                             ,
"ROCit"                                               ,
"rsparkling"                                          ,
"spacesRGB"                                           ,
"SPEDInstabR"                                         ,
"splitfngr"                                           ,
"treemapify"                                          ,
"uptasticsearch"                                      ,
"xSub"                                                ,
"blscrapeR"                                           ,
"ceterisParibus"                                      ,
"credsubs"                                            ,
"customizedTraining"                                  ,
"dataone"                                             ,
"mirt"                                                ,
"rainbow"                                             ,
"ResistorArray"                                       ,
"riskRegression"                                      ,
"simr"                                                ,
"supernova"                                           ,
"tools4uplift"                                        ,
"TotalCopheneticIndex"                                ,
"weibulltools"                                        ,
"bhrcr"                                               ,
"caschrono"                                           ,
"dtwSat"                                              ,
"iccbeta"                                             ,
"mldr"                                                ,
"patentsview"                                         ,
"pseval"                                              ,
"RAdwords"                                            ,
"SemiCompRisks"                                       ,
"DatabionicSwarm"                                     ,
"gplots"                                              ,
"gWidgets"                                            ,
"mizer"                                               ,
"mme"                                                 ,
"msgps"                                               ,
"sampler"                                             ,
"stepR"                                               ,
"wavelets"                                            ,
"aod"                                                 ,
"BioStatR"                                            ,
"dfcrm"                                               ,
"Hmisc"                                               ,
"irr"                                                 ,
"mblm"                                                ,
"muhaz"                                               ,
"RcmdrPlugin.IPSUR"                                   ,
"strider"                                             ,
"SVMMaj"                                              ,
"waveslim"                                            ,
"BENMMI"                                              ,
"BGData"                                              ,
"catseyes"                                            ,
"DEVis"                                               ,
"ETAS"                                                ,
"GAS"                                                 ,
"genoPlotR"                                           ,
"gensemble"                                           ,
"ICDS"                                                ,
"imagerExtra"                                         ,
"IMaGES"                                              ,
"MAMS"                                                ,
"NetLogoR"                                            ,
"npsm"                                                ,
"peperr"                                              ,
"permuco"                                             ,
"qrjoint"                                             ,
"ROI.models.miplib"                                   ,
"ROI.models.netlib"                                   ,
"ROI.plugin.optimx"                                   ,
"scalreg"                                             ,
"Semblance"                                           ,
"Sleuth3"                                             ,
"ssev"                                                ,
"stratEst"                                            ,
"SwarmSVM"                                            ,
"vennLasso"                                           ,
"zyp"                                                 ,
"BNSL"                                                ,
"cdcfluview"                                          ,
"cIRT"                                                ,
"Data2LD"                                             ,
"iNEXT"                                               ,
"rabi"                                                ,
"rgenoud"                                             ,
"ri2"                                                 ,
"RNeXML"                                              ,
"ROI.plugin.deoptim"                                  ,
"ROI.plugin.nloptr"                                   ,
"Sleuth2"                                             ,
"ZVCV"                                                ,
"asbio"                                               ,
"carfima"                                             ,
"colorednoise"                                        ,
"episheet"                                            ,
"face"                                                ,
"fitdistrplus"                                        ,
"homomorpheR"                                         ,
"imager"                                              ,
"missMDA"                                             ,
"natserv"                                             ,
"phylosignal"                                         ,
"prc"                                                 ,
"ROI"                                                 ,
"rtext"                                               ,
"rym"                                                 ,
"surveydata"                                          ,
"woeR"                                                ,
"assertr"                                             ,
"bfp"                                                 ,
"BigVAR"                                              ,
"confreq"                                             ,
"CUFF"                                                ,
"descriptr"                                           ,
"diffeR"                                              ,
"fourPNO"                                             ,
"mapdeck"                                             ,
"mbir"                                                ,
"nlaR"                                                ,
"NlinTS"                                              ,
"panelvar"                                            ,
"pssmooth"                                            ,
"qtl"                                                 ,
"RaceID"                                              ,
"Rmpfr"                                               ,
"simtimer"                                            ,
"tor"                                                 ,
"vegtable"                                            ,
"AcceptanceSampling"                                  ,
"BayesianTools"                                       ,
"ensr"                                                ,
"hrbrthemes"                                          ,
"ical"                                                ,
"Immigrate"                                           ,
"MCAvariants"                                         ,
"MHTcop"                                              ,
"outbreaks"                                           ,
"ParallelLogger"                                      ,
"pubchunks"                                           ,
"RcppHNSW"                                            ,
"RcppXPtrUtils"                                       ,
"RSmartlyIO"                                          ,
"SSBtools"                                            ,
"stylo"                                               ,
"taxlist"                                             ,
"tensorr"                                             ,
"TH.data"                                             ,
"abstractr"                                           ,
"Bayesrel"                                            ,
"CARS"                                                ,
"Cluster.OBeu"                                        ,
"crossword.r"                                         ,
"DescriptiveStats.OBeu"                               ,
"geniusr"                                             ,
"harrypotter"                                         ,
"HiClimR"                                             ,
"jose"                                                ,
"RGCxGC"                                              ,
"TimeSeries.OBeu"                                     ,
"trendchange"                                         ,
"BeyondBenford"                                       ,
"dsm"                                                 ,
"effectFusion"                                        ,
"germinationmetrics"                                  ,
"handyplots"                                          ,
"hcandersenr"                                         ,
"iva"                                                 ,
"NormalBetaPrime"                                     ,
"shinyrecap"                                          ,
"smartdata"                                           ,
"arpr"                                                ,
"ClinicalTrialSummary"                                ,
"cumSeg"                                              ,
"DynaRankR"                                           ,
"hdm"                                                 ,
"iotables"                                            ,
"NlcOptim"                                            ,
"norris"                                              ,
"palettesForR"                                        ,
"ParamHelpers"                                        ,
"spatialwidget"                                       ,
"surrosurv"                                           ,
"testextra"                                           ,
"casino"                                              ,
"colourvalues"                                        ,
"foto"                                                ,
"lcpm"                                                ,
"mldr.datasets"                                       ,
"pogit"                                               ,
"quRan"                                               ,
"randomsearch"                                        ,
"RCALI"                                               ,
"rstpm2"                                              ,
"segMGarch"                                           ,
"textrank"                                            ,
"TropFishR"                                           ,
"activity"                                            ,
"BuyseTest"                                           ,
"climdex.pcic"                                        ,
"DRomics"                                             ,
"insol"                                               ,
"meditations"                                         ,
"memor"                                               ,
"modQR"                                               ,
"netchain"                                            ,
"nnfor"                                               ,
"perturb"                                             ,
"rugarch"                                             ,
"SMPracticals"                                        ,
"ssMousetrack"                                        ,
"steemr"                                              ,
"timevis"                                             ,
"treedater"                                           ,
"TukeyC"                                              ,
"BUCSS"                                               ,
"ciuupi"                                              ,
"epitrix"                                             ,
"GEint"                                               ,
"highcharter"                                         ,
"misaem"                                              ,
"MM4LMM"                                              ,
"msde"                                                ,
"nonet"                                               ,
"rdflib"                                              ,
"RGF"                                                 ,
"RIdeogram"                                           ,
"rmgarch"                                             ,
"SnowballC"                                           ,
"soilDB"                                              ,
"CAvariants"                                          ,
"clam"                                                ,
"cts"                                                 ,
"double.truncation"                                   ,
"gbm"                                                 ,
"ibmsunburst"                                         ,
"JumpTest"                                            ,
"landsepi"                                            ,
"MESS"                                                ,
"metaviz"                                             ,
"multipanelfigure"                                    ,
"osrmr"                                               ,
"particles"                                           ,
"RapidPolygonLookup"                                  ,
"recombinator"                                        ,
"rfm"                                                 ,
"rnoaa"                                               ,
"segregation"                                         ,
"ssa"                                                 ,
"TSVC"                                                ,
"askpass"                                             ,
"auRoc"                                               ,
"bspmma"                                              ,
"crypto"                                              ,
"earthtones"                                          ,
"exifr"                                               ,
"KDViz"                                               ,
"petro.One"                                           ,
"SACCR"                                               ,
"shinyanimate"                                        ,
"systemicrisk"                                        ,
"xptr"                                                ,
"babynames"                                           ,
"colordistance"                                       ,
"comperes"                                            ,
"countcolors"                                         ,
"CPBayes"                                             ,
"lolog"                                               ,
"mosaic"                                              ,
"orsk"                                                ,
"psych"                                               ,
"Trading"                                             ,
"whitening"                                           ,
"cPCG"                                                ,
"DirichletReg"                                        ,
"enrichwith"                                          ,
"epubr"                                               ,
"FMAdist"                                             ,
"forestSAS"                                           ,
"ggdark"                                              ,
"glmmTMB"                                             ,
"GMSimpute"                                           ,
"HydeNet"                                             ,
"jointseg"                                            ,
"LVGP"                                                ,
"MeTo"                                                ,
"pixiedust"                                           ,
"prithulib"                                           ,
"RDFTensor"                                           ,
"RLumShiny"                                           ,
"rsoi"                                                ,
"rtrek"                                               ,
"states"                                              ,
"suropt"                                              ,
"TPES"                                                ,
"tsutils"                                             ,
"voronoiTreemap"                                      ,
"BiProbitPartial"                                     ,
"DtD"                                                 ,
"fExpressCertificates"                                ,
"GA"                                                  ,
"geojsonsf"                                           ,
"hetGP"                                               ,
"KeyboardSimulator"                                   ,
"laeken"                                              ,
"NITPicker"                                           ,
"nprobust"                                            ,
"poweRlaw"                                            ,
"statGraph"                                           ,
"TableToLongForm"                                     ,
"ARpLMEC"                                             ,
"curvHDR"                                             ,
"demoGraphic"                                         ,
"docxtractr"                                          ,
"dynsbm"                                              ,
"esreg"                                               ,
"GLIDE"                                               ,
"mergedblocks"                                        ,
"nomisr"                                              ,
"ODEsensitivity"                                      ,
"pdfsearch"                                           ,
"plotKML"                                             ,
"prereg"                                              ,
"scriptuRs"                                           ,
"SDALGCP"                                             ,
"SILM"                                                ,
"simmer.bricks"                                       ,
"spm12r"                                              ,
"tbd"                                                 ,
"treestartr"                                          ,
"trelliscopejs"                                       ,
"vagam"                                               ,
"yaImpute"                                            ,
"analysisPipelines"                                   ,
"astrochron"                                          ,
"badger"                                              ,
"blockmodeling"                                       ,
"CARBayesST"                                          ,
"ciTools"                                             ,
"decoder"                                             ,
"elec"                                                ,
"fabMix"                                              ,
"frapplot"                                            ,
"fRLR"                                                ,
"germanpolls"                                         ,
"gethr"                                               ,
"hitandrun"                                           ,
"ISEtools"                                            ,
"lemon"                                               ,
"ontologyIndex"                                       ,
"paleofire"                                           ,
"plotlyGeoAssets"                                     ,
"polmineR"                                            ,
"PPRL"                                                ,
"pzfx"                                                ,
"QHOT"                                                ,
"rangemodelR"                                         ,
"RANN"                                                ,
"revengc"                                             ,
"rv"                                                  ,
"scrobbler"                                           ,
"seqminer"                                            ,
"snotelr"                                             ,
"spbabel"                                             ,
"vegclust"                                            ,
"badgecreatr"                                         ,
"BCDating"                                            ,
"bgsmtr"                                              ,
"BH"                                                  ,
"ChemometricsWithR"                                   ,
"cvequality"                                          ,
"errorist"                                            ,
"ExPosition"                                          ,
"FIT"                                                 ,
"ForestGapR"                                          ,
"future.callr"                                        ,
"gclus"                                               ,
"ggfocus"                                             ,
"Greg"                                                ,
"htmlTable"                                           ,
"ipptoolbox"                                          ,
"majesticR"                                           ,
"matconv"                                             ,
"mitml"                                               ,
"MRFA"                                                ,
"mtconnectR"                                          ,
"mvabund"                                             ,
"NatureSounds"                                        ,
"nmathresh"                                           ,
"optigrab"                                            ,
"orderedLasso"                                        ,
"pool"                                                ,
"pxweb"                                               ,
"rgdax"                                               ,
"rzeit2"                                              ,
"simcausal"                                           ,
"simexaft"                                            ,
"simstandard"                                         ,
"sitmo"                                               ,
"spData"                                              ,
"sport"                                               ,
"subscreen"                                           ,
"survidm"                                             ,
"tseriesChaos"                                        ,
"wpp2015"                                             ,
"dynlm"                                               ,
"mvcwt"                                               ,
"cramer"                                              ,
"gvlma"                                               ,
"HMMEsolver"                                          ,
"neotoma"                                             ,
"PivotalR"                                            ,
"RedditExtractoR"                                     ,
"SGPdata"                                             ,
"bayesImageS"                                         ,
"bwd"                                                 ,
"clinDR"                                              ,
"CombMSC"                                             ,
"COST"                                                ,
"dHSIC"                                               ,
"DirectEffects"                                       ,
"ecm"                                                 ,
"EMMIXgene"                                           ,
"envnames"                                            ,
"FitARMA"                                             ,
"future.BatchJobs"                                    ,
"ggsolvencyii"                                        ,
"GSIF"                                                ,
"lpc"                                                 ,
"memgene"                                             ,
"modesto"                                             ,
"pheatmap"                                            ,
"phenology"                                           ,
"PoiClaClu"                                           ,
"protViz"                                             ,
"raws.profile"                                        ,
"rbin"                                                ,
"RQEntangle"                                          ,
"Stat2Data"                                           ,
"utc"                                                 ,
"aqp"                                                 ,
"behavr"                                              ,
"bikeshare14"                                         ,
"BivGeo"                                              ,
"conStruct"                                           ,
"ddpcr"                                               ,
"dint"                                                ,
"distrr"                                              ,
"diveMove"                                            ,
"DTRlearn2"                                           ,
"EffectLiteR"                                         ,
"fastqcr"                                             ,
"gammSlice"                                           ,
"liger"                                               ,
"PAC"                                                 ,
"papeR"                                               ,
"Rdistance"                                           ,
"riskyr"                                              ,
"RSelenium"                                           ,
"Rsomoclu"                                            ,
"striprtf"                                            ,
"tidystats"                                           ,
"valr"                                                ,
"zoomgrid"                                            ,
"ArchaeoPhases"                                       ,
"BaMORC"                                              ,
"bigmatch"                                            ,
"CatDyn"                                              ,
"CIM"                                                 ,
"ctmcd"                                               ,
"DIFtree"                                             ,
"FiRE"                                                ,
"forestmangr"                                         ,
"IPWboxplot"                                          ,
"Langevin"                                            ,
"logger"                                              ,
"mbbefd"                                              ,
"metamer"                                             ,
"OSDR"                                                ,
"plethem"                                             ,
"ptstem"                                              ,
"qdap"                                                ,
"restorepoint"                                        ,
"RTD"                                                 ,
"stevedore"                                           ,
"taxa"                                                ,
"tidyxl"                                              ,
"class"                                               ,
"ActivityIndex"                                       ,
"mmsample"                                            ,
"coroICA"                                             ,
"dpcid"                                               ,
"gameofthrones"                                       ,
"lvmcomp"                                             ,
"RcmdrPlugin.UCA"                                     ,
"weightQuant"                                         ,
"jlctree"                                             ,
"CPAT"                                                ,
"Fstability"                                          ,
"infix"                                               ,
"codetools"                                           ,
"DALEX2"                                              ,
"svydiags"                                            ,
"Rsconctdply"                                         ,
"tsxtreme"                                            ,
"benford.analysis"                                    ,
"bigalgebra"                                          ,
"Buddle"                                              ,
"fam.recrisk"                                         ,
"FastLZeroSpikeInference"                             ,
"gcKrig"                                              ,
"GenoScan"                                            ,
"googleAnalyticsR"                                    ,
"hctrial"                                             ,
"hdrcde"                                              ,
"HyRiM"                                               ,
"kernhaz"                                             ,
"LUCIDus"                                             ,
"mirtjml"                                             ,
"ps"                                                  ,
"RcmdrPlugin.TeachStat"                               ,
"RDieHarder"                                          ,
"Rdimtools"                                           ,
"readr"                                               ,
"SGL"                                                 ,
"tm"                                                  ,
"topicmodels"                                         ,
"webddx"                                              ,
"psychotree"                                          ,
"R2GUESS"                                             ,
"ragt2ridges"                                         ,
"ROI.plugin.lpsolve"                                  ,
"ROI.plugin.qpoases"                                  ,
"seasonal"                                            ,
"spatialwarnings"                                     ,
"tailr"                                               ,
"variosig"                                            ,
"varrank"                                             ,
"datastructures"                                      ,
"ggtern"                                              ,
"ocedata"                                             ,
"OmnibusFisher"                                       ,
"osmplotr"                                            ,
"primerTree"                                          ,
"psychotools"                                         ,
"rcrtan"                                              ,
"RKEELjars"                                           ,
"Rlibeemd"                                            ,
"visualFields"                                        ,
"adegraphics"                                         ,
"breathtestcore"                                      ,
"ChaosGame"                                           ,
"cubature"                                            ,
"drtmle"                                              ,
"dynRB"                                               ,
"fluspect"                                            ,
"fuzzySim"                                            ,
"kuiper.2samp"                                        ,
"LabRS"                                               ,
"metaRMST"                                            ,
"NetworkSim"                                          ,
"preputils"                                           ,
"prettyGraphs"                                        ,
"r2d3"                                                ,
"readABF"                                             ,
"ritis"                                               ,
"sensitivityCalibration"                              ,
"sentometrics"                                        ,
"SIGN"                                                ,
"VDJgermlines"                                        ,
"waterquality"                                        ,
"FLAME"                                               ,
"HCmodelSets"                                         ,
"jvcoords"                                            ,
"kdevine"                                             ,
"kohonen"                                             ,
"qcv"                                                 ,
"rfUtilities"                                         ,
"arulesCBA"                                           ,
"bfsl"                                                ,
"cliapp"                                              ,
"convexjlr"                                           ,
"netCoin"                                             ,
"tinter"                                              ,
"foolbox"                                             ,
"hunspell"                                            ,
"linl"                                                ,
"MHTdiscrete"                                         ,
"redist"                                              ,
"WordPools"                                           ,
"bezier"                                              ,
"bioRad"                                              ,
"ConvergenceClubs"                                    ,
"ggspatial"                                           ,
"norm2"                                               ,
"pgmm"                                                ,
"RefFreeEWAS"                                         ,
"RSA"                                                 ,
"TPMplt"                                              ,
"tweenr"                                              ,
"datetime"                                            ,
"msBP"                                                ,
"Partiallyoverlapping"                                ,
"Rlda"                                                ,
"rwalkr"                                              ,
"scclust"                                             ,
"solrium"                                             ,
"STARTS"                                              ,
"glmc"                                                ,
"healthcareai"                                        ,
"jsr223"                                              ,
"metaboGSE"                                           ,
"mvbutils"                                            ,
"NetworkDistance"                                     ,
"parallelDist"                                        ,
"skynet"                                              ,
"SMFilter"                                            ,
"animation"                                           ,
"CsChange"                                            ,
"geogrid"                                             ,
"glmm"                                                ,
"RNAsmc"                                              ,
"rsMove"                                              ,
"smoothr"                                             ,
"antaresProcessing"                                   ,
"erhcv"                                               ,
"immer"                                               ,
"metansue"                                            ,
"mudfold"                                             ,
"prism"                                               ,
"RANN.L1"                                             ,
"rEDM"                                                ,
"COMPoissonReg"                                       ,
"dials"                                               ,
"iptools"                                             ,
"LinkedGASP"                                          ,
"LSDsensitivity"                                      ,
"multiRDPG"                                           ,
"pedigreeTools"                                       ,
"radix"                                               ,
"streamR"                                             ,
"transformr"                                          ,
"depmix"                                              ,
"palm"                                                ,
"pbdBASE"                                             ,
"Rdrools"                                             ,
"rtk"                                                 ,
"ammistability"                                       ,
"beadplexr"                                           ,
"gimms"                                               ,
"httptest"                                            ,
"icesAdvice"                                          ,
"jsonlite"                                            ,
"lomb"                                                ,
"parmsurvfit"                                         ,
"tcgsaseq"                                            ,
"VARSEDIG"                                            ,
"clordr"                                              ,
"mdendro"                                             ,
"metatest"                                            ,
"pleio"                                               ,
"rvcheck"                                             ,
"SAMM"                                                ,
"spacesXYZ"                                           ,
"trajectories"                                        ,
"vlad"                                                ,
"acopula"                                             ,
"AHR"                                                 ,
"csvread"                                             ,
"EMD"                                                 ,
"fun"                                                 ,
"humanleague"                                         ,
"pkgsearch"                                           ,
"profr"                                               ,
"quantities"                                          ,
"rpql"                                                ,
"spCP"                                                ,
"testit"                                              ,
"biospear"                                            ,
"HiCfeat"                                             ,
"pdfCluster"                                          ,
"rTensor"                                             ,
"seismicRoll"                                         ,
"womblR"                                              ,
"aGE"                                                 ,
"argonDash"                                           ,
"cglm"                                                ,
"CopCTS"                                              ,
"Copula.surv"                                         ,
"customsteps"                                         ,
"cvmgof"                                              ,
"DIMORA"                                              ,
"ev.trawl"                                            ,
"eyetrackingR"                                        ,
"fbar"                                                ,
"featurefinder"                                       ,
"GENMETA"                                             ,
"gmat"                                                ,
"jackstraw"                                           ,
"meanShiftR"                                          ,
"mixRaschTools"                                       ,
"npExact"                                             ,
"PKconverter"                                         ,
"pkgcond"                                             ,
"ppcc"                                                ,
"qccrs"                                               ,
"refuge"                                              ,
"remedy"                                              ,
"semds"                                               ,
"solartime"                                           ,
"twosamples"                                          ,
"vmsbase"                                             ,
"ar.matrix"                                           ,
"BayesianNetwork"                                     ,
"compas"                                              ,
"covatest"                                            ,
"DBKGrad"                                             ,
"eAnalytics"                                          ,
"energyr"                                             ,
"hoardr"                                              ,
"TreeDep"                                             ,
"writexl"                                             ,
"cartogram"                                           ,
"causalweight"                                        ,
"cleandata"                                           ,
"commonmark"                                          ,
"container"                                           ,
"ggQC"                                                ,
"graphql"                                             ,
"HMMcopula"                                           ,
"keyholder"                                           ,
"lbfgsb3c"                                            ,
"scrime"                                              ,
"SeedCalc"                                            ,
"snfa"                                                ,
"ARTP2"                                               ,
"HMB"                                                 ,
"ImputeRobust"                                        ,
"JGL"                                                 ,
"MetaboLouise"                                        ,
"nCopula"                                             ,
"rCRM"                                                ,
"relsurv"                                             ,
"smartR"                                              ,
"albopictus"                                          ,
"causaleffect"                                        ,
"celestial"                                           ,
"changepoint.mv"                                      ,
"ectotemp"                                            ,
"fitplc"                                              ,
"generics"                                            ,
"IRdisplay"                                           ,
"leaflet.opacity"                                     ,
"preference"                                          ,
"psica"                                               ,
"RStorm"                                              ,
"selectapref"                                         ,
"SkewHyperbolic"                                      ,
"tmod"                                                ,
"WebPower"                                            ,
"wiseR"                                               ,
"elevatr"                                             ,
"GCD"                                                 ,
"netSEM"                                              ,
"physiology"                                          ,
"ProFound"                                            ,
"riskPredictClustData"                                ,
"CUB"                                                 ,
"DistributionUtils"                                   ,
"FRESA.CAD"                                           ,
"gllm"                                                ,
"lasso2"                                              ,
"pseudorank"                                          ,
"SpatialGraph"                                        ,
"StatDA"                                              ,
"TAShiny"                                             ,
"TeachNet"                                            ,
"tidymodels"                                          ,
"trafo"                                               ,
"cranlike"                                            ,
"FlexParamCurve"                                      ,
"FSMUMI"                                              ,
"hsphase"                                             ,
"ipdw"                                                ,
"kappaSize"                                           ,
"mixKernel"                                           ,
"NormalLaplace"                                       ,
"npordtests"                                          ,
"PDShiny"                                             ,
"postlogic"                                           ,
"questionr"                                           ,
"read.gb"                                             ,
"readODS"                                             ,
"rio"                                                 ,
"SetMethods"                                          ,
"TRES"                                                ,
"VarianceGamma"                                       ,
"Z10"                                                 ,
"zscorer"                                             ,
"document"                                            ,
"insect"                                              ,
"MplusAutomation"                                     ,
"RonFHIR"                                             ,
"SPREDA"                                              ,
"ssdtools"                                            ,
"stackoverflow"                                       ,
"yuimaGUI"                                            ,
"isoph"                                               ,
"RcppEigen"                                           ,
"Rmagic"                                              ,
"stminsights"                                         ,
"STPGA"                                               ,
"CytobankAPI"                                         ,
"esDesign"                                            ,
"googlePolylines"                                     ,
"patternize"                                          ,
"pid"                                                 ,
"RcppThread"                                          ,
"tadaatoolbox"                                        ,
"goeveg"                                              ,
"iClick"                                              ,
"momentuHMM"                                          ,
"ngstk"                                               ,
"olsrr"                                               ,
"SIfEK"                                               ,
"assertive.data"                                      ,
"cladoRcpp"                                           ,
"glassdoor"                                           ,
"PMmisc"                                              ,
"assertive.matrices"                                  ,
"BioInstaller"                                        ,
"dobson"                                              ,
"farver"                                              ,
"fishmethods"                                         ,
"meteoForecast"                                       ,
"modi"                                                ,
"neurobase"                                           ,
"OjaNP"                                               ,
"optional"                                            ,
"pathfindR"                                           ,
"rhandsontable"                                       ,
"RSE"                                                 ,
"scico"                                               ,
"whiboclustering"                                     ,
"CAST"                                                ,
"clustvarsel"                                         ,
"collateral"                                          ,
"deckgl"                                              ,
"epibasix"                                            ,
"fdq"                                                 ,
"ffmetadata"                                          ,
"FlexScan"                                            ,
"gginference"                                         ,
"GLSE"                                                ,
"hillR"                                               ,
"ITGM"                                                ,
"LindleyPowerSeries"                                  ,
"NBDesign"                                            ,
"ODS"                                                 ,
"ordering"                                            ,
"pollimetry"                                          ,
"ref.ICAR"                                            ,
"rphylopic"                                           ,
"rscopus"                                             ,
"scrm"                                                ,
"spartan"                                             ,
"spreadr"                                             ,
"StMoSim"                                             ,
"argonR"                                              ,
"cleanNLP"                                            ,
"Fgmutils"                                            ,
"NPBayesImputeCat"                                    ,
"pems.utils"                                          ,
"pinbasic"                                            ,
"RainfallErosivityFactor"                             ,
"RcppMsgPack"                                         ,
"rfviz"                                               ,
"scPDSI"                                              ,
"shinyEffects"                                        ,
"SII"                                                 ,
"tiler"                                               ,
"TSMCP"                                               ,
"dslice"                                              ,
"ncpen"                                               ,
"rpdo"                                                ,
"wTO"                                                 ,
"BivRec"                                              ,
"evtclass"                                            ,
"MIXFIM"                                              ,
"mltest"                                              ,
"nproc"                                               ,
"pterrace"                                            ,
"randcorr"                                            ,
"RcppGetconf"                                         ,
"TGS"                                                 ,
"EpiSignalDetection"                                  ,
"FRegSigCom"                                          ,
"ggdistribute"                                        ,
"HMMpa"                                               ,
"ITRLearn"                                            ,
"MultiJoin"                                           ,
"OsteoBioR"                                           ,
"pkggraph"                                            ,
"pvar"                                                ,
"rrcov"                                               ,
"signalHsmm"                                          ,
"spocc"                                               ,
"sumFREGAT"                                           ,
"switchr"                                             ,
"tidyposterior"                                       ,
"UncDecomp"                                           ,
"AutoStepwiseGLM"                                     ,
"BondValuation"                                       ,
"bReeze"                                              ,
"dlbayes"                                             ,
"EpiReport"                                           ,
"forestHES"                                           ,
"gamlss.countKinf"                                    ,
"histry"                                              ,
"metsyn"                                              ,
"mixggm"                                              ,
"Modalclust"                                          ,
"packrat"                                             ,
"pkgverse"                                            ,
"pluscode"                                            ,
"RCSF"                                                ,
"RGenData"                                            ,
"rJST"                                                ,
"RPyGeo"                                              ,
"semdrw"                                              ,
"smoothROCtime"                                       ,
"SolveRationalMatrixEquation"                         ,
"sparklyr.nested"                                     ,
"tableschema.r"                                       ,
"BiocManager"                                         ,
"bioset"                                              ,
"configr"                                             ,
"deming"                                              ,
"helixvis"                                            ,
"multimode"                                           ,
"Rmpi"                                                ,
"skpr"                                                ,
"fastLink"                                            ,
"MetabolomicsBasics"                                  ,
"model4you"                                           ,
"NCmisc"                                              ,
"PMCMRplus"                                           ,
"PResiduals"                                          ,
"QGglmm"                                              ,
"qrmtools"                                            ,
"reqres"                                              ,
"rmweather"                                           ,
"roptim"                                              ,
"rpostgis"                                            ,
"RSAGA"                                               ,
"trekfont"                                            ,
"AMCTestmakeR"                                        ,
"BGLR"                                                ,
"Rcrawler"                                            ,
"reportReg"                                           ,
"SpATS"                                               ,
"textcat"                                             ,
"interep"                                             ,
"jmcm"                                                ,
"rstanarm"                                            ,
"Rtsne"                                               ,
"sparkavro"                                           ,
"SpatialVS"                                           ,
"spTimer"                                             ,
"VICmodel"                                            ,
"antiword"                                            ,
"dfpk"                                                ,
"dtangle"                                             ,
"ibs"                                                 ,
"idem"                                                ,
"MALDIrppa"                                           ,
"quickPlot"                                           ,
"survHE"                                              ,
"TMB"                                                 ,
"unrtf"                                               ,
"ZeBook"                                              ,
"arrangements"                                        ,
"econet"                                              ,
"lintr"                                               ,
"MADPop"                                              ,
"MDFS"                                                ,
"breathteststan"                                      ,
"Factoshiny"                                          ,
"gcookbook"                                           ,
"Nmisc"                                               ,
"nodiv"                                               ,
"parfm"                                               ,
"roxygen2"                                            ,
"RXshrink"                                            ,
"switchrGist"                                         ,
"tfestimators"                                        ,
"trackr"                                              ,
"BoSSA"                                               ,
"CENFA"                                               ,
"eegkit"                                              ,
"endorse"                                             ,
"ffbase"                                              ,
"gmnl"                                                ,
"GRANCore"                                            ,
"pander"                                              ,
"pbv"                                                 ,
"Rchoice"                                             ,
"roprov"                                              ,
"shinythemes"                                         ,
"anyLib"                                              ,
"beanz"                                               ,
"binequality"                                         ,
"garray"                                              ,
"PerseusR"                                            ,
"RcmdrPlugin.orloca"                                  ,
"robmed"                                              ,
"semtree"                                             ,
"sessioninfo"                                         ,
"solrad"                                              ,
"TDAstats"                                            ,
"xts"                                                 ,
"convevol"                                            ,
"KarsTS"                                              ,
"lattice"                                             ,
"modelgrid"                                           ,
"Tejapi"                                              ,
"TwoSampleTest.HD"                                    ,
"visTree"                                             ,
"BPM"                                                 ,
"kmeRs"                                               ,
"memnet"                                              ,
"ModelMetrics"                                        ,
"numform"                                             ,
"OWEA"                                                ,
"owmr"                                                ,
"SEMrushR"                                            ,
"biostat3"                                            ,
"crov"                                                ,
"halfcircle"                                          ,
"kofnGA"                                              ,
"lsmeans"                                             ,
"MARSS"                                               ,
"NPHazardRate"                                        ,
"polylabelr"                                          ,
"pubh"                                                ,
"SAutomata"                                           ,
"VARtests"                                            ,
"wosr"                                                ,
"yll"                                                 ,
"arfima"                                              ,
"ballr"                                               ,
"hazer"                                               ,
"mleap"                                               ,
"polyglot"                                            ,
"arkdb"                                               ,
"automultinomial"                                     ,
"customLayout"                                        ,
"fds"                                                 ,
"logisticRR"                                          ,
"reinforcedPred"                                      ,
"RImpact"                                             ,
"spatstat.utils"                                      ,
"ViSiElse"                                            ,
"AMModels"                                            ,
"BAS"                                                 ,
"BCRA"                                                ,
"biva"                                                ,
"cpt"                                                 ,
"export"                                              ,
"freetypeharfbuzz"                                    ,
"GHS"                                                 ,
"graphframes"                                         ,
"hero"                                                ,
"lass0"                                               ,
"LearningRlab"                                        ,
"mapi"                                                ,
"mudens"                                              ,
"oem"                                                 ,
"PROSPER"                                             ,
"REPLesentR"                                          ,
"sdStaf"                                              ,
"siebanxicor"                                         ,
"sleepr"                                              ,
"temperatureresponse"                                 ,
"TSDT"                                                ,
"ttestshiny"                                          ,
"zbank"                                               ,
"bucky"                                               ,
"condformat"                                          ,
"InSilicoVA"                                          ,
"pkgload"                                             ,
"qualpalr"                                            ,
"RColetum"                                            ,
"ROI.plugin.ecos"                                     ,
"strip"                                               ,
"Tariff"                                              ,
"atmopt"                                              ,
"atus"                                                ,
"blsAPI"                                              ,
"compboost"                                           ,
"CountsEPPM"                                          ,
"ebimetagenomics"                                     ,
"extremis"                                            ,
"frequency"                                           ,
"koRpus"                                              ,
"kza"                                                 ,
"NitrogenUptake2016"                                  ,
"orloca.es"                                           ,
"PCRedux"                                             ,
"pgsc"                                                ,
"plfm"                                                ,
"poissonMT"                                           ,
"RcppDist"                                            ,
"rhmc"                                                ,
"shinyhelper"                                         ,
"spectrolab"                                          ,
"uavRmp"                                              ,
"ASMap"                                               ,
"countrycode"                                         ,
"dub"                                                 ,
"RcppRedis"                                           ,
"Rwhois"                                              ,
"BayesS5"                                             ,
"bivgeom"                                             ,
"docopulae"                                           ,
"FinePop"                                             ,
"HCR"                                                 ,
"latexpdf"                                            ,
"loa"                                                 ,
"lsbs"                                                ,
"pingers"                                             ,
"qsort"                                               ,
"reportr"                                             ,
"rgsp"                                                ,
"rtsdata"                                             ,
"SAMCpack"                                            ,
"sbl"                                                 ,
"AgreementInterval"                                   ,
"dhglm"                                               ,
"electoral"                                           ,
"mdhglm"                                              ,
"microdemic"                                          ,
"np"                                                  ,
"PPCI"                                                ,
"rem"                                                 ,
"solvebio"                                            ,
"vegan3d"                                             ,
"DBfit"                                               ,
"DiPhiSeq"                                            ,
"FourScores"                                          ,
"GroupComparisons"                                    ,
"inpdfr"                                              ,
"mathgraph"                                           ,
"mmpf"                                                ,
"nipals"                                              ,
"sparcl"                                              ,
"taskscheduleR"                                       ,
"vegdata"                                             ,
"DEploid"                                             ,
"geodetector"                                         ,
"isdparser"                                           ,
"kyotil"                                              ,
"reshape"                                             ,
"RMixpanel"                                           ,
"SuperRanker"                                         ,
"websearchr"                                          ,
"BALD"                                                ,
"BreedingSchemeLanguage"                              ,
"fiery"                                               ,
"gamlss.tr"                                           ,
"jqr"                                                 ,
"pkmon"                                               ,
"reReg"                                               ,
"assertive.code"                                      ,
"assertive.data.uk"                                   ,
"assertive.data.us"                                   ,
"assertive.models"                                    ,
"boral"                                               ,
"changer"                                             ,
"RApiDatetime"                                        ,
"deal"                                                ,
"arena2r"                                             ,
"cjoint"                                              ,
"EMMIXmfa"                                            ,
"gcForest"                                            ,
"HSDiC"                                               ,
"JMI"                                                 ,
"linkprediction"                                      ,
"MGLM"                                                ,
"MPS"                                                 ,
"poliscidata"                                         ,
"rTephra"                                             ,
"SpecDetec"                                           ,
"wikitaxa"                                            ,
"ABPS"                                                ,
"betapart"                                            ,
"bsplinePsd"                                          ,
"BWStest"                                             ,
"charlatan"                                           ,
"eva"                                                 ,
"hNMF"                                                ,
"KODAMA"                                              ,
"microbenchmark"                                      ,
"NLP"                                                 ,
"NLPutils"                                            ,
"psidR"                                               ,
"PWEALL"                                              ,
"smam"                                                ,
"storr"                                               ,
"BBI"                                                 ,
"DeLorean"                                            ,
"fergm"                                               ,
"GenAlgo"                                             ,
"gofCopula"                                           ,
"HRM"                                                 ,
"nhanesA"                                             ,
"QTL.gCIMapping"                                      ,
"shinydashboard"                                      ,
"bcROCsurface"                                        ,
"carrier"                                             ,
"logNormReg"                                          ,
"Minirand"                                            ,
"queuecomputer"                                       ,
"rmapshaper"                                          ,
"samr"                                                ,
"SitesInterest"                                       ,
"weights"                                             ,
"clusterRepro"                                        ,
"cobiclust"                                           ,
"cosinor2"                                            ,
"crminer"                                             ,
"decisionSupport"                                     ,
"define"                                              ,
"readJDX"                                             ,
"stormwindmodel"                                      ,
"ungroup"                                             ,
"analytics"                                           ,
"galgo"                                               ,
"imputeR"                                             ,
"jjb"                                                 ,
"MLGL"                                                ,
"nprotreg"                                            ,
"nvmix"                                               ,
"paran"                                               ,
"rocsvm.path"                                         ,
"SAVER"                                               ,
"touch"                                               ,
"MetFns"                                              ,
"bayesPop"                                            ,
"BOIN"                                                ,
"crqa"                                                ,
"descstatsr"                                          ,
"mase"                                                ,
"mMPA"                                                ,
"nlts"                                                ,
"pmpp"                                                ,
"RcppNLoptExample"                                    ,
"rococo"                                              ,
"TBFmultinomial"                                      ,
"varbin"                                              ,
"VFS"                                                 ,
"antaresViz"                                          ,
"docopt"                                              ,
"drfit"                                               ,
"EcoVirtual"                                          ,
"FisherEM"                                            ,
"globals"                                             ,
"GROAN"                                               ,
"ITNr"                                                ,
"itunesr"                                             ,
"partialCI"                                           ,
"sysfonts"                                            ,
"tidyselect"                                          ,
"ttScreening"                                         ,
"VBLPCM"                                              ,
"ca"                                                  ,
"CutpointsOEHR"                                       ,
"MTS"                                                 ,
"RSDA"                                                ,
"saemix"                                              ,
"spotGUI"                                             ,
"uwIntroStats"                                        ,
"wooldridge"                                          ,
"BayesGOF"                                            ,
"bayou"                                               ,
"Bolstad"                                             ,
"nucim"                                               ,
"RGoogleAnalytics"                                    ,
"RndTexExams"                                         ,
"StereoMorph"                                         ,
"COMBIA"                                              ,
"CopyDetect"                                          ,
"crossrun"                                            ,
"DiceKriging"                                         ,
"epos"                                                ,
"FourgameteP"                                         ,
"IntrinioStockAPI"                                    ,
"InvariantCausalPrediction"                           ,
"MCSim"                                               ,
"nlgeocoder"                                          ,
"processcheckR"                                       ,
"TSEntropies"                                         ,
"useful"                                              ,
"vamc"                                                ,
"weathercan"                                          ,
"directotree"                                         ,
"DRIP"                                                ,
"listviewer"                                          ,
"pollen"                                              ,
"ratelimitr"                                          ,
"SharpeR"                                             ,
"simukde"                                             ,
"Temporal"                                            ,
"UPSvarApprox"                                        ,
"wallace"                                             ,
"airportr"                                            ,
"eixport"                                             ,
"GUIDE"                                               ,
"lamme"                                               ,
"mcbiopi"                                             ,
"Opt5PL"                                              ,
"outcomerate"                                         ,
"pdSpecEst"                                           ,
"PredictionR"                                         ,
"sdPrior"                                             ,
"stacomirtools"                                       ,
"tls"                                                 ,
"comtradr"                                            ,
"dma"                                                 ,
"ERP"                                                 ,
"fansi"                                               ,
"filelock"                                            ,
"fwsim"                                               ,
"gbutils"                                             ,
"ggetho"                                              ,
"mixedsde"                                            ,
"texmex"                                              ,
"uiucthemes"                                          ,
"Delta"                                               ,
"GGUM"                                                ,
"PRIMsrc"                                             ,
"rise"                                                ,
"spatstat.data"                                       ,
"textreg"                                             ,
"TileManager"                                         ,
"ArArRedux"                                           ,
"jmdl"                                                ,
"nloptr"                                              ,
"odr"                                                 ,
"PeakSegJoint"                                        ,
"PerFit"                                              ,
"RcmdrPlugin.RiskDemo"                                ,
"SpNMF"                                               ,
"BASiNET"                                             ,
"CalibrateSSB"                                        ,
"equateIRT"                                           ,
"stability"                                           ,
"SwissAir"                                            ,
"tensorBF"                                            ,
"whitechapelR"                                        ,
"coRanking"                                           ,
"FactoClass"                                          ,
"hrIPW"                                               ,
"IMmailgun"                                           ,
"rcarbon"                                             ,
"rtkore"                                              ,
"SimRVPedigree"                                       ,
"webuse"                                              ,
"WeightedROC"                                         ,
"BRETIGEA"                                            ,
"cap"                                                 ,
"carData"                                             ,
"codified"                                            ,
"effectR"                                             ,
"fastGHQuad"                                          ,
"ggparliament"                                        ,
"htmlwidgets"                                         ,
"optimx"                                              ,
"portsort"                                            ,
"RBitmoji"                                            ,
"Rfolding"                                            ,
"sym.arma"                                            ,
"tdsc"                                                ,
"ClueR"                                               ,
"CORElearn"                                           ,
"fDMA"                                                ,
"GFD"                                                 ,
"LearnGeom"                                           ,
"MazamaWebUtils"                                      ,
"mdir.logrank"                                        ,
"RcppAPT"                                             ,
"ADMM"                                                ,
"BETS"                                                ,
"ddiv"                                                ,
"deflateBR"                                           ,
"ggvis"                                               ,
"gifski"                                              ,
"HDcpDetect"                                          ,
"MaxPro"                                              ,
"roperators"                                          ,
"Rvcg"                                                ,
"scalpel"                                             ,
"SCGLR"                                               ,
"textreadr"                                           ,
"titeIR"                                              ,
"webshot"                                             ,
"fluxweb"                                             ,
"ggridges"                                            ,
"hyfo"                                                ,
"iptmnetr"                                            ,
"mclogit"                                             ,
"orcutt"                                              ,
"R.matlab"                                            ,
"rebird"                                              ,
"rebmix"                                              ,
"rsq"                                                 ,
"sm"                                                  ,
"taipan"                                              ,
"utility"                                             ,
"adjclust"                                            ,
"bdvis"                                               ,
"CRM"                                                 ,
"LncPath"                                             ,
"MEPDF"                                               ,
"movMF"                                               ,
"rcdk"                                                ,
"rdrobust"                                            ,
"addhaz"                                              ,
"boxcoxmix"                                           ,
"CPE"                                                 ,
"freegroup"                                           ,
"gWidgets2tcltk"                                      ,
"multigraph"                                          ,
"nonnest2"                                            ,
"bigdatadist"                                         ,
"checkLuhn"                                           ,
"data.tree"                                           ,
"distdichoR"                                          ,
"EcoHydRology"                                        ,
"flexclust"                                           ,
"frequencyConnectedness"                              ,
"ITRSelect"                                           ,
"PBRF"                                                ,
"userfriendlyscience"                                 ,
"DescribeDisplay"                                     ,
"geex"                                                ,
"NHMSAR"                                              ,
"nullabor"                                            ,
"SI"                                                  ,
"ether"                                               ,
"rapidjsonr"                                          ,
"RMOA"                                                ,
"RMOAjars"                                            ,
"sparsepp"                                            ,
"anomaly"                                             ,
"APtools"                                             ,
"blin"                                                ,
"BootstrapQTL"                                        ,
"gbfs"                                                ,
"inctools"                                            ,
"multiplex"                                           ,
"RInno"                                               ,
"rsdmx"                                               ,
"trigpoints"                                          ,
"zeroEQpart"                                          ,
"adpss"                                               ,
"datasauRus"                                          ,
"DensParcorr"                                         ,
"DIconvex"                                            ,
"lmomco"                                              ,
"LPKsample"                                           ,
"quantregGrowth"                                      ,
"rcoreoa"                                             ,
"rsnps"                                               ,
"soiltexture"                                         ,
"TTR"                                                 ,
"BayLum"                                              ,
"CIAAWconsensus"                                      ,
"dataCompareR"                                        ,
"droptest"                                            ,
"EffectTreat"                                         ,
"SplitSoftening"                                      ,
"wilson"                                              ,
"ccfa"                                                ,
"cowsay"                                              ,
"cymruservices"                                       ,
"eivtools"                                            ,
"mhurdle"                                             ,
"molaR"                                               ,
"ndjson"                                              ,
"netrankr"                                            ,
"packcircles"                                         ,
"ReIns"                                               ,
"rmetalog"                                            ,
"shiny.router"                                        ,
"CorDiff"                                             ,
"EloOptimized"                                        ,
"emhawkes"                                            ,
"gdns"                                                ,
"googleway"                                           ,
"HMMextra0s"                                          ,
"kergp"                                               ,
"magic"                                               ,
"n1qn1"                                               ,
"salty"                                               ,
"spikeSlabGAM"                                        ,
"treeman"                                             ,
"xmlparsedata"                                        ,
"xopen"                                               ,
"aiRthermo"                                           ,
"caMST"                                               ,
"duawranglr"                                          ,
"filling"                                             ,
"knockoff"                                            ,
"rgeopat2"                                            ,
"roundhouse"                                          ,
"sparr"                                               ,
"stylest"                                             ,
"tvR"                                                 ,
"usmap"                                               ,
"VBTree"                                              ,
"voteogram"                                           ,
"dcemriS4"                                            ,
"FFTrees"                                             ,
"JWileymisc"                                          ,
"PropClust"                                           ,
"RcmdrPlugin.BiclustGUI"                              ,
"rtsplot"                                             ,
"asht"                                                ,
"BMA"                                                 ,
"crawl"                                               ,
"cruts"                                               ,
"demogR"                                              ,
"episcan"                                             ,
"fts"                                                 ,
"glpkAPI"                                             ,
"kitagawa"                                            ,
"LabourMarketAreas"                                   ,
"longmemo"                                            ,
"MCMCpack"                                            ,
"MmgraphR"                                            ,
"mstrio"                                              ,
"reglogit"                                            ,
"RtutoR"                                              ,
"snow"                                                ,
"Blaunet"                                             ,
"CKLRT"                                               ,
"condvis"                                             ,
"covafillr"                                           ,
"EnvCpt"                                              ,
"EstMix"                                              ,
"HDtest"                                              ,
"knor"                                                ,
"nse"                                                 ,
"PHeval"                                              ,
"shiny.i18n"                                          ,
"SimPhe"                                              ,
"VarfromPDB"                                          ,
"CTT"                                                 ,
"flipscores"                                          ,
"ggmosaic"                                            ,
"qualmap"                                             ,
"TempleMetrics"                                       ,
"fastnet"                                             ,
"Hotelling"                                           ,
"KrigInv"                                             ,
"ModelMap"                                            ,
"PhylogeneticEM"                                      ,
"prettycode"                                          ,
"sensiPhy"                                            ,
"tstools"                                             ,
"yacca"                                               ,
"zoocat"                                              ,
"AsioHeaders"                                         ,
"BHTSpack"                                            ,
"datasets.load"                                       ,
"itsmr"                                               ,
"kidney.epi"                                          ,
"MEGENA"                                              ,
"metRology"                                           ,
"MVR"                                                 ,
"ndl"                                                 ,
"nesRdata"                                            ,
"PopED"                                               ,
"RcmdrMisc"                                           ,
"rly"                                                 ,
"RWekajars"                                           ,
"SamplerCompare"                                      ,
"season"                                              ,
"sklarsomega"                                         ,
"spagmix"                                             ,
"downscale"                                           ,
"MetSizeR"                                            ,
"rWishart"                                            ,
"vanquish"                                            ,
"cmrutils"                                            ,
"CTTinShiny"                                          ,
"datadigest"                                          ,
"ELMSO"                                               ,
"ggTimeSeries"                                        ,
"itemanalysis"                                        ,
"medflex"                                             ,
"pysd2r"                                              ,
"redR"                                                ,
"rgoogleslides"                                       ,
"ROP"                                                 ,
"SamplingBigData"                                     ,
"tiobeindexr"                                         ,
"rsm"                                                 ,
"fitur"                                               ,
"ade4"                                                ,
"bcaboot"                                             ,
"bdDwC"                                               ,
"bdribs"                                              ,
"CARBayesdata"                                        ,
"corpora"                                             ,
"DOS"                                                 ,
"ECLRMC"                                              ,
"elasticnet"                                          ,
"excursions"                                          ,
"fastR2"                                              ,
"gustave"                                             ,
"HGSL"                                                ,
"ldbounds"                                            ,
"logbin"                                              ,
"MAc"                                                 ,
"MAd"                                                 ,
"nearfar"                                             ,
"normalp"                                             ,
"npmlreg"                                             ,
"pencopula"                                           ,
"plotscale"                                           ,
"radjust"                                             ,
"Rambo"                                               ,
"repec"                                               ,
"rifle"                                               ,
"sinew"                                               ,
"SparseLPM"                                           ,
"stocks"                                              ,
"takos"                                               ,
"TauP.R"                                              ,
"wrassp"                                              ,
"ambient"                                             ,
"aods3"                                               ,
"AssetCorr"                                           ,
"chicane"                                             ,
"detect"                                              ,
"doBy"                                                ,
"earlygating"                                         ,
"evoper"                                              ,
"idm"                                                 ,
"lero.lero"                                           ,
"lsm"                                                 ,
"OOBCurve"                                            ,
"ore"                                                 ,
"PeerPerformance"                                     ,
"profExtrema"                                         ,
"qoma.smuggler"                                       ,
"RiskPortfolios"                                      ,
"rmake"                                               ,
"smfsb"                                               ,
"surrosurvROC"                                        ,
"apaTables"                                           ,
"approximator"                                        ,
"BPEC"                                                ,
"fdANOVA"                                             ,
"IsoriX"                                              ,
"mipfp"                                               ,
"rtdists"                                             ,
"SpNetPrep"                                           ,
"vaersvax"                                            ,
"VarSelLCM"                                           ,
"cwhmisc"                                             ,
"fingerPro"                                           ,
"geoelectrics"                                        ,
"gmapsdistance"                                       ,
"mBvs"                                                ,
"PoweR"                                               ,
"rtypeform"                                           ,
"SSLASSO"                                             ,
"ZIM"                                                 ,
"calpassapi"                                          ,
"eDMA"                                                ,
"leaflet"                                             ,
"miceMNAR"                                            ,
"pdp"                                                 ,
"prob"                                                ,
"projections"                                         ,
"RRreg"                                               ,
"vpc"                                                 ,
"funique"                                             ,
"GPRMortality"                                        ,
"mclustcomp"                                          ,
"sspse"                                               ,
"supc"                                                ,
"coopProductGame"                                     ,
"flip"                                                ,
"latentnet"                                           ,
"mrMLM"                                               ,
"mrMLM.GUI"                                           ,
"symDMatrix"                                          ,
"TeXCheckR"                                           ,
"tfruns"                                              ,
"accelerometry"                                       ,
"adiv"                                                ,
"bacistool"                                           ,
"binGroup"                                            ,
"BMRBr"                                               ,
"gsynth"                                              ,
"iC10TrainingData"                                    ,
"mvglmmRank"                                          ,
"ncappc"                                              ,
"optiSel"                                             ,
"perccalc"                                            ,
"plsgenomics"                                         ,
"portfolio.optimization"                              ,
"quickmatch"                                          ,
"rtrim"                                               ,
"stepPenal"                                           ,
"survxai"                                             ,
"twl"                                                 ,
"twoway"                                              ,
"wordcloud"                                           ,
"xergm"                                               ,
"xRing"                                               ,
"ars"                                                 ,
"BLR"                                                 ,
"brnn"                                                ,
"essurvey"                                            ,
"fixerapi"                                            ,
"IndepTest"                                           ,
"kerasformula"                                        ,
"manet"                                               ,
"PNADcIBGE"                                           ,
"predictmeans"                                        ,
"roahd"                                               ,
"rtide"                                               ,
"sparseinv"                                           ,
"xmlrpc2"                                             ,
"BayesNetBP"                                          ,
"collapsibleTree"                                     ,
"eikosograms"                                         ,
"hedgehog"                                            ,
"highSCREEN"                                          ,
"lmridge"                                             ,
"mboost"                                              ,
"pwrFDR"                                              ,
"RJSDMX"                                              ,
"RSEIS"                                               ,
"rstantools"                                          ,
"TE"                                                  ,
"decido"                                              ,
"NCA"                                                 ,
"quickblock"                                          ,
"rdpower"                                             ,
"rsolr"                                               ,
"SIDES"                                               ,
"synchronicity"                                       ,
"AdaptiveSparsity"                                    ,
"cartools"                                            ,
"con2aqi"                                             ,
"getProxy"                                            ,
"ISAT"                                                ,
"ltmle"                                               ,
"MRHawkes"                                            ,
"nCal"                                                ,
"pGPx"                                                ,
"rfoaas"                                              ,
"RLT"                                                 ,
"shinyFeedback"                                       ,
"shp2graph"                                           ,
"datadr"                                              ,
"dwapi"                                               ,
"ELYP"                                                ,
"IPEC"                                                ,
"latticeDensity"                                      ,
"lazyrmd"                                             ,
"microplot"                                           ,
"SteinerNet"                                          ,
"sweep"                                               ,
"timetk"                                              ,
"emplik2"                                             ,
"krm"                                                 ,
"mar1s"                                               ,
"modopt.matlab"                                       ,
"narray"                                              ,
"shapes"                                              ,
"truncreg"                                            ,
"udapi"                                               ,
"AbSim"                                               ,
"airr"                                                ,
"BAT"                                                 ,
"covTestR"                                            ,
"DMMF"                                                ,
"emplik"                                              ,
"GPrank"                                              ,
"hhi"                                                 ,
"imptree"                                             ,
"jsonstat"                                            ,
"LatticeDesign"                                       ,
"matrixProfile"                                       ,
"metacom"                                             ,
"permubiome"                                          ,
"rattle"                                              ,
"rhli"                                                ,
"VarBundle"                                           ,
"AST"                                                 ,
"batchtools"                                          ,
"dyncomp"                                             ,
"EM.Fuzzy"                                            ,
"leanpubr"                                            ,
"lest"                                                ,
"merlin"                                              ,
"mixchar"                                             ,
"MVNBayesian"                                         ,
"pkgconfig"                                           ,
"RWDataPlyr"                                          ,
"stpm"                                                ,
"tm.plugin.mail"                                      ,
"trawl"                                               ,
"BMisc"                                               ,
"choroplethr"                                         ,
"ENMeval"                                             ,
"mumm"                                                ,
"stilt"                                               ,
"tensr"                                               ,
"BootValidation"                                      ,
"DemoDecomp"                                          ,
"epiflows"                                            ,
"fc"                                                  ,
"GmAMisc"                                             ,
"LMfilteR"                                            ,
"powerlmm"                                            ,
"ProliferativeIndex"                                  ,
"ZipRadius"                                           ,
"bcp"                                                 ,
"BLOQ"                                                ,
"boostmtree"                                          ,
"brandwatchR"                                         ,
"Brobdingnag"                                         ,
"CSFA"                                                ,
"EL2Surv"                                             ,
"frailtyHL"                                           ,
"grplasso"                                            ,
"junctions"                                           ,
"kirby21.base"                                        ,
"kirby21.fmri"                                        ,
"kirby21.t1"                                          ,
"ltable"                                              ,
"matlabr"                                             ,
"mccf1"                                               ,
"painter"                                             ,
"patrick"                                             ,
"quadmatrix"                                          ,
"RQGIS"                                               ,
"Rspotify"                                            ,
"SCPME"                                               ,
"subprocess"                                          ,
"togglr"                                              ,
"RagGrid"                                             ,
"clusrank"                                            ,
"fasterElasticNet"                                    ,
"glm2"                                                ,
"nsapi"                                               ,
"atlas"                                               ,
"BGGE"                                                ,
"BrailleR"                                            ,
"dnet"                                                ,
"imp4p"                                               ,
"kernlab"                                             ,
"libstableR"                                          ,
"MCMCprecision"                                       ,
"RSNNS"                                               ,
"smoothSurv"                                          ,
"SnakeCharmR"                                         ,
"GNE"                                                 ,
"hddtools"                                            ,
"HMDHFDplus"                                          ,
"logKDE"                                              ,
"MDSMap"                                              ,
"PairViz"                                             ,
"reclin"                                              ,
"scales"                                              ,
"VariableScreening"                                   ,
"vinereg"                                             ,
"CarletonStats"                                       ,
"datafsm"                                             ,
"hpiR"                                                ,
"iterpc"                                              ,
"populationPDXdesign"                                 ,
"WCE"                                                 ,
"archivist.github"                                    ,
"ForecastComb"                                        ,
"gRbase"                                              ,
"gridsample"                                          ,
"gumbel"                                              ,
"Mqrcm"                                               ,
"MSMwRA"                                              ,
"nsROC"                                               ,
"rhosp"                                               ,
"roadoi"                                              ,
"rplotengine"                                         ,
"tsallisqexp"                                         ,
"LinkedMatrix"                                        ,
"MatrixLDA"                                           ,
"nseval"                                              ,
"PhySortR"                                            ,
"PQLseq"                                              ,
"timeseriesdb"                                        ,
"vars"                                                ,
"nbc4va"                                              ,
"FDboost"                                             ,
"kendallRandomWalks"                                  ,
"mvMORPH"                                             ,
"ctl"                                                 ,
"hwwntest"                                            ,
"NPMLEcmprsk"                                         ,
"rare"                                                ,
"Tides"                                               ,
"TOSTER"                                              ,
"ADMMsigma"                                           ,
"cromwellDashboard"                                   ,
"disto"                                               ,
"GFE"                                                 ,
"HCT"                                                 ,
"LncFinder"                                           ,
"omu"                                                 ,
"readmnist"                                           ,
"rmi"                                                 ,
"rolocISCCNBS"                                        ,
"sparkbq"                                             ,
"SparseMDC"                                           ,
"tribe"                                               ,
"ARIbrain"                                            ,
"aws.kms"                                             ,
"CANSIM2R"                                            ,
"colourvision"                                        ,
"csvy"                                                ,
"ggsom"                                               ,
"ICcalib"                                             ,
"lfl"                                                 ,
"mvctm"                                               ,
"netregR"                                             ,
"redcapAPI"                                           ,
"spef"                                                ,
"adaptMT"                                             ,
"BDP2"                                                ,
"beezdemand"                                          ,
"biglmm"                                              ,
"dhh"                                                 ,
"EMSHS"                                               ,
"flifo"                                               ,
"geomerge"                                            ,
"makeParallel"                                        ,
"phylotaR"                                            ,
"robmixglm"                                           ,
"statip"                                              ,
"uniformly"                                           ,
"venn"                                                ,
"VFP"                                                 ,
"cregg"                                               ,
"CrossClustering"                                     ,
"cytofan"                                             ,
"dcmodify"                                            ,
"debugr"                                              ,
"DecisionAnalysis"                                    ,
"dotCall64"                                           ,
"draw"                                                ,
"glassoFast"                                          ,
"gren"                                                ,
"growthcurver"                                        ,
"IntClust"                                            ,
"lilikoi"                                             ,
"lintools"                                            ,
"logitnorm"                                           ,
"MultipleBubbles"                                     ,
"poolVIM"                                             ,
"QuClu"                                               ,
"roloc"                                               ,
"Rspc"                                                ,
"ruin"                                                ,
"stinepack"                                           ,
"syt"                                                 ,
"trend"                                               ,
"copulaedas"                                          ,
"geoR"                                                ,
"metaDigitise"                                        ,
"RcppCNPy"                                            ,
"reghelper"                                           ,
"sylly"                                               ,
"taRifx"                                              ,
"tensorA"                                             ,
"catR"                                                ,
"demu"                                                ,
"fredr"                                               ,
"Hapi"                                                ,
"InterVA5"                                            ,
"labelVector"                                         ,
"RanglaPunjab"                                        ,
"SurvTrunc"                                           ,
"woeBinning"                                          ,
"BinQuasi"                                            ,
"CBCgrps"                                             ,
"CSeqpat"                                             ,
"epanet2toolkit"                                      ,
"exact2x2"                                            ,
"funLBM"                                              ,
"mable"                                               ,
"MMDvariance"                                         ,
"rncl"                                                ,
"RPS"                                                 ,
"rrtable"                                             ,
"RSentiment"                                          ,
"rviewgraph"                                          ,
"UnitCircle"                                          ,
"WeibullR"                                            ,
"basicTrendline"                                      ,
"CEC"                                                 ,
"clusterCrit"                                         ,
"conjoint"                                            ,
"dnr"                                                 ,
"GPoM"                                                ,
"INLABMA"                                             ,
"lspls"                                               ,
"NeuralNetTools"                                      ,
"numOSL"                                              ,
"pec"                                                 ,
"penalizedSVM"                                        ,
"prozor"                                              ,
"reconstructr"                                        ,
"regplot"                                             ,
"stepp"                                               ,
"colorscience"                                        ,
"forward"                                             ,
"measures"                                            ,
"shinyjqui"                                           ,
"yaml"                                                ,
"aftgee"                                              ,
"anMC"                                                ,
"CaliCo"                                              ,
"ERSA"                                                ,
"mvst"                                                ,
"pcr"                                                 ,
"piecewiseSEM"                                        ,
"Rhpc"                                                ,
"RhpcBLASctl"                                         ,
"safer"                                               ,
"sigQC"                                               ,
"meta4diag"                                           ,
"moonBook"                                            ,
"nmaINLA"                                             ,
"TeachingSampling"                                    ,
"textclean"                                           ,
"WordR"                                               ,
"xmrr"                                                ,
"CAinterprTools"                                      ,
"ggiraphExtra"                                        ,
"linkspotter"                                         ,
"rbit"                                                ,
"survutils"                                           ,
"ztable"                                              ,
"diffusionMap"                                        ,
"elmNNRcpp"                                           ,
"nmslibR"                                             ,
"R.devices"                                           ,
"rda"                                                 ,
"wdman"                                               ,
"calibrateBinary"                                     ,
"FKF"                                                 ,
"forestControl"                                       ,
"lazy"                                                ,
"MortalityGaps"                                       ,
"POCRE"                                               ,
"ppls"                                                ,
"redland"                                             ,
"spind"                                               ,
"trimcluster"                                         ,
"ApacheLogProcessor"                                  ,
"BE"                                                  ,
"blastula"                                            ,
"bor"                                                 ,
"GWASinlps"                                           ,
"happytime"                                           ,
"logistf"                                             ,
"ncar"                                                ,
"NonCompart"                                          ,
"portes"                                              ,
"rredlist"                                            ,
"sdmpredictors"                                       ,
"table1"                                              ,
"tkrplot"                                             ,
"tmvmixnorm"                                          ,
"binhf"                                               ,
"binman"                                              ,
"CASMAP"                                              ,
"CliftLRD"                                            ,
"CNLTreg"                                             ,
"CNLTtsa"                                             ,
"IntNMF"                                              ,
"liftLRD"                                             ,
"LS2Wstat"                                            ,
"qqplotr"                                             ,
"r4lineups"                                           ,
"r511"                                                ,
"rmpw"                                                ,
"robotstxt"                                           ,
"TangPoemR"                                           ,
"tatest"                                              ,
"TaxicabCA"                                           ,
"abctools"                                            ,
"BioPET"                                              ,
"Boruta"                                              ,
"CATkit"                                              ,
"chemCal"                                             ,
"CodeDepends"                                         ,
"dgo"                                                 ,
"rcorpora"                                            ,
"rmda"                                                ,
"RTransProb"                                          ,
"scatterpie"                                          ,
"spacetime"                                           ,
"Arothron"                                            ,
"dmm"                                                 ,
"GCalignR"                                            ,
"InterSIM"                                            ,
"keyring"                                             ,
"modeltools"                                          ,
"MSwM"                                                ,
"NHPoisson"                                           ,
"rdmulti"                                             ,
"rFerns"                                              ,
"vdmR"                                                ,
"WRTDStidal"                                          ,
"EnvStats"                                            ,
"GreedyExperimentalDesign"                            ,
"GreedyExperimentalDesignJARs"                        ,
"hypervolume"                                         ,
"RcppClassic"                                         ,
"sValues"                                             ,
"Rlinsolve"                                           ,
"splm"                                                ,
"DTWUMI"                                              ,
"eha"                                                 ,
"FRK"                                                 ,
"glasso"                                              ,
"googlenlp"                                           ,
"htree"                                               ,
"ofGEM"                                               ,
"pompom"                                              ,
"sgmodel"                                             ,
"descomponer"                                         ,
"diseasemapping"                                      ,
"gDefrag"                                             ,
"generalCorr"                                         ,
"JGR"                                                 ,
"MAGNAMWAR"                                           ,
"multifluo"                                           ,
"penalized"                                           ,
"qGaussian"                                           ,
"washeR"                                              ,
"xdcclarge"                                           ,
"xwf"                                                 ,
"dpmixsim"                                            ,
"DTWBI"                                               ,
"dygraphs"                                            ,
"erah"                                                ,
"highriskzone"                                        ,
"llama"                                               ,
"nabor"                                               ,
"polyPK"                                              ,
"tauturri"                                            ,
"tweet2r"                                             ,
"xkcd"                                                ,
"adlift"                                              ,
"banter"                                              ,
"bayesDP"                                             ,
"cetcolor"                                            ,
"countgmifs"                                          ,
"FPCA3D"                                              ,
"imsig"                                               ,
"netdep"                                              ,
"nlt"                                                 ,
"poisFErobust"                                        ,
"RGeckoboard"                                         ,
"rt.test"                                             ,
"stablelearner"                                       ,
"SubgrPlots"                                          ,
"syllabifyr"                                          ,
"CommonJavaJars"                                      ,
"corrgram"                                            ,
"gMCP"                                                ,
"JuniperKernel"                                       ,
"Metrics"                                             ,
"pegas"                                               ,
"sidier"                                              ,
"wtss"                                                ,
"agRee"                                               ,
"flatxml"                                             ,
"sft"                                                 ,
"tuneR"                                               ,
"betaboost"                                           ,
"jeek"                                                ,
"LCAextend"                                           ,
"multinets"                                           ,
"rggobi"                                              ,
"SQB"                                                 ,
"AcousticNDLCodeR"                                    ,
"agridat"                                             ,
"meaRtools"                                           ,
"alignfigR"                                           ,
"ExtMallows"                                          ,
"fda"                                                 ,
"gppm"                                                ,
"IntegratedMRF"                                       ,
"mason"                                               ,
"mineCitrus"                                          ,
"MMAC"                                                ,
"nmixgof"                                             ,
"peRiodiCS"                                           ,
"permDep"                                             ,
"satscanMapper"                                       ,
"scRNAtools"                                          ,
"survMisc"                                            ,
"epitab"                                              ,
"gghighlight"                                         ,
"kernelboot"                                          ,
"pointdensityP"                                       ,
"RcppMeCab"                                           ,
"Rrdrand"                                             ,
"swamp"                                               ,
"timeordered"                                         ,
"varhandle"                                           ,
"Blendstat"                                           ,
"diffee"                                              ,
"ggedit"                                              ,
"horizon"                                             ,
"optiRum"                                             ,
"rsyslog"                                             ,
"scholar"                                             ,
"scraEP"                                              ,
"spup"                                                ,
"ACA"                                                 ,
"adaptalint"                                          ,
"disco"                                               ,
"EDA"                                                 ,
"rde"                                                 ,
"simule"                                              ,
"STRMPS"                                              ,
"TROM"                                                ,
"CircStats"                                           ,
"eChem"                                               ,
"editrules"                                           ,
"GenomicMating"                                       ,
"ggrasp"                                              ,
"knnp"                                                ,
"NBLDA"                                               ,
"plde"                                                ,
"PRISM.forecast"                                      ,
"SimCorrMix"                                          ,
"spemd"                                               ,
"tippy"                                               ,
"tropicalSparse"                                      ,
"biolink"                                             ,
"expands"                                             ,
"glmpathcr"                                           ,
"GuardianR"                                           ,
"interplot"                                           ,
"LaplacesDemon"                                       ,
"lmm"                                                 ,
"OECD"                                                ,
"svMisc"                                              ,
"TSA"                                                 ,
"bayesboot"                                           ,
"expint"                                              ,
"flexdashboard"                                       ,
"googlesheets"                                        ,
"groupedSurv"                                         ,
"learnr"                                              ,
"multicmp"                                            ,
"nlcv"                                                ,
"pan"                                                 ,
"Rttf2pt1"                                            ,
"svrpath"                                             ,
"zooimage"                                            ,
"dynOmics"                                            ,
"gistr"                                               ,
"hyperSpec"                                           ,
"ropercenter"                                         ,
"SimMultiCorrData"                                    ,
"svIDE"                                               ,
"svTools"                                             ,
"svWidgets"                                           ,
"BTdecayLasso"                                        ,
"cbinom"                                              ,
"clogitLasso"                                         ,
"CytobankAPIstats"                                    ,
"dineq"                                               ,
"dng"                                                 ,
"dotwhisker"                                          ,
"ecospat"                                             ,
"GBJ"                                                 ,
"mmap"                                                ,
"mrds"                                                ,
"nsRFA"                                               ,
"NutrienTrackeR"                                      ,
"phyloseqGraphTest"                                   ,
"preseqR"                                             ,
"readbitmap"                                          ,
"VGAMextra"                                           ,
"backpipe"                                            ,
"ClusterBootstrap"                                    ,
"epanetReader"                                        ,
"fancycut"                                            ,
"fetchR"                                              ,
"gtools"                                              ,
"multiROC"                                            ,
"nycflights13"                                        ,
"otvPlots"                                            ,
"RcmdrPlugin.lfstat"                                  ,
"sbpiper"                                             ,
"schoenberg"                                          ,
"tableMatrix"                                         ,
"treeDA"                                              ,
"voxel"                                               ,
"autoshiny"                                           ,
"bizdays"                                             ,
"catcont"                                             ,
"gmodels"                                             ,
"mlf"                                                 ,
"na.tools"                                            ,
"phylogram"                                           ,
"readroper"                                           ,
"rrepast"                                             ,
"cofeatureR"                                          ,
"coreCT"                                              ,
"DDPGPSurv"                                           ,
"EcoGenetics"                                         ,
"GSAfisherCombined"                                   ,
"ICEinfer"                                            ,
"mosaicCore"                                          ,
"MuChPoint"                                           ,
"nat.templatebrains"                                  ,
"patternplot"                                         ,
"RGreenplum"                                          ,
"subrank"                                             ,
"vsgoftest"                                           ,
"mosaicData"                                          ,
"MPTinR"                                              ,
"plotROC"                                             ,
"schumaker"                                           ,
"zeitgebr"                                            ,
"alr3"                                                ,
"doRNG"                                               ,
"mcmcplots"                                           ,
"RcmdrPlugin.temis"                                   ,
"gnm"                                                 ,
"googleLanguageR"                                     ,
"iilasso"                                             ,
"mlrMBO"                                              ,
"sigmoid"                                             ,
"thinkr"                                              ,
"TSPred"                                              ,
"emojifont"                                           ,
"vertexenum"                                          ,
"adwordsR"                                            ,
"careless"                                            ,
"centralplot"                                         ,
"convey"                                              ,
"cstab"                                               ,
"DPtree"                                              ,
"groupICA"                                            ,
"M2SMF"                                               ,
"Mcomp"                                               ,
"otuSummary"                                          ,
"prodest"                                             ,
"PRROC"                                               ,
"rdiversity"                                          ,
"RepoGenerator"                                       ,
"ahnr"                                                ,
"bayeslm"                                             ,
"causalsens"                                          ,
"choiceDes"                                           ,
"CircularDDM"                                         ,
"designmatch"                                         ,
"dragonking"                                          ,
"gamclass"                                            ,
"geeM"                                                ,
"geoops"                                              ,
"growthrates"                                         ,
"IAPWS95"                                             ,
"MetaComp"                                            ,
"rdryad"                                              ,
"belg"                                                ,
"dragulaR"                                            ,
"fuser"                                               ,
"iplots"                                              ,
"refinr"                                              ,
"statprograms"                                        ,
"walrus"                                              ,
"graphicalVAR"                                        ,
"sads"                                                ,
"alphastable"                                         ,
"Bchron"                                              ,
"distfree.cr"                                         ,
"gamboostLSS"                                         ,
"hddplot"                                             ,
"hybridModels"                                        ,
"IntervalSurgeon"                                     ,
"loder"                                               ,
"mcen"                                                ,
"mmtfa"                                               ,
"MultisiteMediation"                                  ,
"QTL.gCIMapping.GUI"                                  ,
"randomizeR"                                          ,
"RcppEigenAD"                                         ,
"simIReff"                                            ,
"SLIDE"                                               ,
"sssc"                                                ,
"breakDown"                                           ,
"compositions"                                        ,
"eseis"                                               ,
"IPCAPS"                                              ,
"jaccard"                                             ,
"lue"                                                 ,
"mixor"                                               ,
"morpheus"                                            ,
"nparMD"                                              ,
"qpcR"                                                ,
"splines2"                                            ,
"stockR"                                              ,
"tree.bins"                                           ,
"TrendInTrend"                                        ,
"tuckerR.mmgg"                                        ,
"varjmcm"                                             ,
"bshazard"                                            ,
"dlm"                                                 ,
"driftR"                                              ,
"IATanalytics"                                        ,
"iTOP"                                                ,
"PAS"                                                 ,
"ROpenDota"                                           ,
"sae"                                                 ,
"sensitivitymv"                                       ,
"CaseBasedReasoning"                                  ,
"CGP"                                                 ,
"fpCompare"                                           ,
"ggseas"                                              ,
"hett"                                                ,
"hydroPSO"                                            ,
"IDSpatialStats"                                      ,
"iRepro"                                              ,
"lestat"                                              ,
"linpk"                                               ,
"lpridge"                                             ,
"MLZ"                                                 ,
"munsell"                                             ,
"NetRep"                                              ,
"quickpsy"                                            ,
"spectralAnalysis"                                    ,
"tightClust"                                          ,
"tuts"                                                ,
"bpca"                                                ,
"CALIBERrfimpute"                                     ,
"Deriv"                                               ,
"gamlss.cens"                                         ,
"leaflet.minicharts"                                  ,
"manipulateWidget"                                    ,
"ompr"                                                ,
"ompr.roi"                                            ,
"PlasmaMutationDetector"                              ,
"PPforest"                                            ,
"xlsx"                                                ,
"AROC"                                                ,
"cheddar"                                             ,
"CMLS"                                                ,
"FILEST"                                              ,
"freeknotsplines"                                     ,
"GetoptLong"                                          ,
"humidity"                                            ,
"inplace"                                             ,
"mvMISE"                                              ,
"RcppGSL"                                             ,
"RcppZiggurat"                                        ,
"teachingApps"                                        ,
"biclust"                                             ,
"GlobalOptions"                                       ,
"HDInterval"                                          ,
"Libra"                                               ,
"odbc"                                                ,
"railtrails"                                          ,
"RcppDE"                                              ,
"bigQueryR"                                           ,
"boxplotdbl"                                          ,
"CompRandFld"                                         ,
"diaplt"                                              ,
"elliplot"                                            ,
"fitDRC"                                              ,
"LSAmitR"                                             ,
"MOCCA"                                               ,
"rjson"                                               ,
"TunePareto"                                          ,
"understandBPMN"                                      ,
"wavemulcor"                                          ,
"briskaR"                                             ,
"fastcluster"                                         ,
"gnFit"                                               ,
"LPmerge"                                             ,
"mixR"                                                ,
"pcse"                                                ,
"RegularizedSCA"                                      ,
"ScottKnott"                                          ,
"sievetest"                                           ,
"tabulizer"                                           ,
"textshape"                                           ,
"tigreBrowserWriter"                                  ,
"wikilake"                                            ,
"BICORN"                                              ,
"bjscrapeR"                                           ,
"CREAM"                                               ,
"pds3"                                                ,
"rERR"                                                ,
"rppo"                                                ,
"sambia"                                              ,
"SPAtest"                                             ,
"cdlTools"                                            ,
"Dodge"                                               ,
"gdm"                                                 ,
"LendingClub"                                         ,
"nsprcomp"                                            ,
"PLordprob"                                           ,
"plumber"                                             ,
"powerCompRisk"                                       ,
"RcppRoll"                                            ,
"seroincidence"                                       ,
"sp"                                                  ,
"SpatialNP"                                           ,
"Tcomp"                                               ,
"tm.plugin.lexisnexis"                                ,
"autovarCore"                                         ,
"beepr"                                               ,
"bootRes"                                             ,
"CePa"                                                ,
"CVglasso"                                            ,
"glmnetcr"                                            ,
"KRIS"                                                ,
"mixdist"                                             ,
"pkr"                                                 ,
"qrcmNP"                                              ,
"rcane"                                               ,
"RmarineHeatWaves"                                    ,
"rres"                                                ,
"rwc"                                                 ,
"tangram"                                             ,
"wISAM"                                               ,
"cardidates"                                          ,
"ibmdbR"                                              ,
"marelac"                                             ,
"mokken"                                              ,
"quickmapr"                                           ,
"simsem"                                              ,
"PairedData"                                          ,
"seas"                                                ,
"sparseSVM"                                           ,
"SpatialEpi"                                          ,
"ccdrAlgorithm"                                       ,
"FeaLect"                                             ,
"fso"                                                 ,
"nitrcbot"                                            ,
"odds.converter"                                      ,
"RNentropy"                                           ,
"WaveletArima"                                        ,
"CBT"                                                 ,
"metaforest"                                          ,
"OpenRepGrid"                                         ,
"phylolm"                                             ,
"redux"                                               ,
"comclim"                                             ,
"comperank"                                           ,
"DDoutlier"                                           ,
"GENLIB"                                              ,
"graphkernels"                                        ,
"hotspots"                                            ,
"lacm"                                                ,
"mixsmsn"                                             ,
"nricens"                                             ,
"occ"                                                 ,
"plan"                                                ,
"PVR"                                                 ,
"SimEUCartelLaw"                                      ,
"stargazer"                                           ,
"swissMrP"                                            ,
"AGD"                                                 ,
"bit"                                                 ,
"brainGraph"                                          ,
"c3"                                                  ,
"FarmTest"                                            ,
"glmmEP"                                              ,
"goric"                                               ,
"GrimR"                                               ,
"hdbinseg"                                            ,
"infoDecompuTE"                                       ,
"iqLearn"                                             ,
"KoulMde"                                             ,
"metaBLUE"                                            ,
"metaLik"                                             ,
"NFWdist"                                             ,
"OpVaR"                                               ,
"ramcmc"                                              ,
"RatingScaleReduction"                                ,
"RchivalTag"                                          ,
"sbgcop"                                              ,
"SiZer"                                               ,
"splusTimeSeries"                                     ,
"StructFDR"                                           ,
"tempR"                                               ,
"waved"                                               ,
"yasp"                                                ,
"deeplr"                                              ,
"HSAUR3"                                              ,
"joineRML"                                            ,
"manifestoR"                                          ,
"plotMElm"                                            ,
"PriorGen"                                            ,
"survBootOutliers"                                    ,
"bestglm"                                             ,
"FloodMapper"                                         ,
"irregulAR1"                                          ,
"messaging"                                           ,
"mritc"                                               ,
"rusk"                                                ,
"aspect"                                              ,
"CVST"                                                ,
"GoFKernel"                                           ,
"hamlet"                                              ,
"homals"                                              ,
"MarkowitzR"                                          ,
"MPsychoR"                                            ,
"OTUtable"                                            ,
"PRISMA"                                              ,
"readstata13"                                         ,
"sglasso"                                             ,
"StratSel"                                            ,
"aws.s3"                                              ,
"bigsplines"                                          ,
"clampSeg"                                            ,
"directlabels"                                        ,
"DisHet"                                              ,
"edgarWebR"                                           ,
"FuncMap"                                             ,
"mbest"                                               ,
"PeakSegOptimal"                                      ,
"phyloclim"                                           ,
"pkgmaker"                                            ,
"ptw"                                                 ,
"ALEPlot"                                             ,
"cfma"                                                ,
"cmce"                                                ,
"dfmta"                                               ,
"DPP"                                                 ,
"epiphy"                                              ,
"event"                                               ,
"fastM"                                               ,
"gamlss.spatial"                                      ,
"ica"                                                 ,
"knitrBootstrap"                                      ,
"ldat"                                                ,
"locpol"                                              ,
"lvec"                                                ,
"neurohcp"                                            ,
"PCPS"                                                ,
"riv"                                                 ,
"SEA"                                                 ,
"tclust"                                              ,
"utf8"                                                ,
"AdaptFitOS"                                          ,
"btb"                                                 ,
"excel.link"                                          ,
"gsEasy"                                              ,
"histmdl"                                             ,
"icd.data"                                            ,
"latdiag"                                             ,
"LS2W"                                                ,
"mratios"                                             ,
"rvalues"                                             ,
"synlik"                                              ,
"unsystation"                                         ,
"AppliedPredictiveModeling"                           ,
"aprof"                                               ,
"BIS"                                                 ,
"C50"                                                 ,
"Cprob"                                               ,
"FastHCS"                                             ,
"FastPCS"                                             ,
"genotypeR"                                           ,
"interlineaR"                                         ,
"intRegGOF"                                           ,
"lgcp"                                                ,
"mapplots"                                            ,
"margins"                                             ,
"MaskJointDensity"                                    ,
"miniGUI"                                             ,
"ODEnetwork"                                          ,
"prioritizrdata"                                      ,
"qmethod"                                             ,
"qut"                                                 ,
"smdata"                                              ,
"spatsurv"                                            ,
"spMaps"                                              ,
"templates"                                           ,
"ziphsmm"                                             ,
"AIG"                                                 ,
"BINCOR"                                              ,
"Cubist"                                              ,
"easycsv"                                             ,
"FuzzyAHP"                                            ,
"hnp"                                                 ,
"logiBin"                                             ,
"mixAK"                                               ,
"neutralitytestr"                                     ,
"nima"                                                ,
"SIRItoGTFS"                                          ,
"smaa"                                                ,
"spuRs"                                               ,
"tabularaster"                                        ,
"yuima"                                               ,
"ini"                                                 ,
"NUCOMBog"                                            ,
"TAQMNGR"                                             ,
"BayesFactor"                                         ,
"DetMCD"                                              ,
"DetR"                                                ,
"mvnmle"                                              ,
"PMCMR"                                               ,
"popReconstruct"                                      ,
"primefactr"                                          ,
"StepSignalMargiLike"                                 ,
"synchrony"                                           ,
"wakefield"                                           ,
"Biocomb"                                             ,
"BusinessDuration"                                    ,
"DstarM"                                              ,
"GAparsimony"                                         ,
"ggjoy"                                               ,
"inline"                                              ,
"joineR"                                              ,
"Kpart"                                               ,
"miniUI"                                              ,
"MixSAL"                                              ,
"multDM"                                              ,
"NormPsy"                                             ,
"npsr"                                                ,
"PhysicalActivity"                                    ,
"randgeo"                                             ,
"rbgm"                                                ,
"rdist"                                               ,
"rsurface"                                            ,
"RUnit"                                               ,
"SIBERG"                                              ,
"smoothAPC"                                           ,
"spt"                                                 ,
"stR"                                                 ,
"adagio"                                              ,
"adwave"                                              ,
"ASGS.foyer"                                          ,
"bootsPLS"                                            ,
"colocalization"                                      ,
"contfrac"                                            ,
"CrypticIBDcheck"                                     ,
"ctv"                                                 ,
"datadogr"                                            ,
"diffusr"                                             ,
"emg"                                                 ,
"ExpDes"                                              ,
"ExpDes.pt"                                           ,
"fat2Lpoly"                                           ,
"fracprolif"                                          ,
"GGally"                                              ,
"hermite"                                             ,
"lambda.r"                                            ,
"LeLogicielR"                                         ,
"matchMulti"                                          ,
"MMS"                                                 ,
"modTurPoint"                                         ,
"mvinfluence"                                         ,
"nleqslv"                                             ,
"numbers"                                             ,
"pmc"                                                 ,
"popprxl"                                             ,
"PortfolioAnalytics"                                  ,
"rcbsubset"                                           ,
"rfbCNPJ"                                             ,
"Rgbp"                                                ,
"robfilter"                                           ,
"sdef"                                                ,
"sFFLHD"                                              ,
"simsalapar"                                          ,
"survsim"                                             ,
"TailRank"                                            ,
"tukeytrend"                                          ,
"VNM"                                                 ,
"YuGene"                                              ,
"zetadiv"                                             ,
"BiDimRegression"                                     ,
"biogas"                                              ,
"ChoR"                                                ,
"codep"                                               ,
"edci"                                                ,
"eive"                                                ,
"FSelector"                                           ,
"furrr"                                               ,
"GLSME"                                               ,
"icsw"                                                ,
"iemisc"                                              ,
"kdist"                                               ,
"MM"                                                  ,
"mmeln"                                               ,
"MSGARCH"                                             ,
"NipponMap"                                           ,
"NSM3"                                                ,
"poibin"                                              ,
"RcmdrPlugin.MPAStats"                                ,
"renpow"                                              ,
"repo"                                                ,
"sharpr2"                                             ,
"SK"                                                  ,
"spgs"                                                ,
"SurvGSD"                                             ,
"TFMPvalue"                                           ,
"timedelay"                                           ,
"tkrgl"                                               ,
"evidence"                                            ,
"ff"                                                  ,
"GeneralizedHyperbolic"                               ,
"GERGM"                                               ,
"gge"                                                 ,
"HTSSIP"                                              ,
"PBNPA"                                               ,
"quantable"                                           ,
"RAM"                                                 ,
"RankAggreg"                                          ,
"refund"                                              ,
"RnavGraphImageData"                                  ,
"rTableICC"                                           ,
"SpatialTools"                                        ,
"ASSA"                                                ,
"base64url"                                           ,
"catch"                                               ,
"datamaps"                                            ,
"difR"                                                ,
"dtp"                                                 ,
"facerec"                                             ,
"ffstream"                                            ,
"ggsci"                                               ,
"KSgeneral"                                           ,
"MTA"                                                 ,
"ph2hetero"                                           ,
"RKEELdata"                                           ,
"RobustCalibration"                                   ,
"smacpod"                                             ,
"spearmanCI"                                          ,
"tabulizerjars"                                       ,
"brotli"                                              ,
"ComICS"                                              ,
"FastRCS"                                             ,
"gamlr"                                               ,
"LPGraph"                                             ,
"mcga"                                                ,
"rJPSGCS"                                             ,
"RSiena"                                              ,
"sodavis"                                             ,
"vitality"                                            ,
"mltools"                                             ,
"PSM"                                                 ,
"SSN"                                                 ,
"treeClust"                                           ,
"ASIP"                                                ,
"cld2"                                                ,
"CleanBSequences"                                     ,
"CPP"                                                 ,
"HDclassif"                                           ,
"Knoema"                                              ,
"LPower"                                              ,
"mgwrsar"                                             ,
"nVennR"                                              ,
"Observation"                                         ,
"openSTARS"                                           ,
"randquotes"                                          ,
"RcppProgress"                                        ,
"REddyProcNCDF"                                       ,
"Select"                                              ,
"shiny.semantic"                                      ,
"spFSR"                                               ,
"asympTest"                                           ,
"Benchmarking"                                        ,
"blandr"                                              ,
"childhoodmortality"                                  ,
"epiDisplay"                                          ,
"groc"                                                ,
"ismev"                                               ,
"MVisAGe"                                             ,
"nadiv"                                               ,
"QuasiSeq"                                            ,
"quhomology"                                          ,
"RcmdrPlugin.HH"                                      ,
"ROI.plugin.alabama"                                  ,
"rPowerSampleSize"                                    ,
"rsvg"                                                ,
"TrajDataMining"                                      ,
"TRSbook"                                             ,
"BIEN"                                                ,
"EMSaov"                                              ,
"GRANBase"                                            ,
"qtbase"                                              ,
"xyloplot"                                            ,
"Amelia"                                              ,
"coga"                                                ,
"fscaret"                                             ,
"fuzzr"                                               ,
"gdpc"                                                ,
"leerSIECyL"                                          ,
"phiDelta"                                            ,
"red"                                                 ,
"ScottKnottESD"                                       ,
"statsr"                                              ,
"survRM2adapt"                                        ,
"BlockFeST"                                           ,
"CondIndTests"                                        ,
"CTRE"                                                ,
"dlib"                                                ,
"gdistance"                                           ,
"JSM"                                                 ,
"MPkn"                                                ,
"nardl"                                               ,
"OscillatorGenerator"                                 ,
"oXim"                                                ,
"RcmdrPlugin.OptimClassifier"                         ,
"rdi"                                                 ,
"rpanel"                                              ,
"sensors4plumes"                                      ,
"webr"                                                ,
"covsep"                                              ,
"JoSAE"                                               ,
"lmeVarComp"                                          ,
"MOLHD"                                               ,
"propagate"                                           ,
"RMariaDB"                                            ,
"RPostgres"                                           ,
"uncertainty"                                         ,
"AssocAFC"                                            ,
"crsra"                                               ,
"oglmx"                                               ,
"pbdRPC"                                              ,
"pbdZMQ"                                              ,
"ameco"                                               ,
"bartMachine"                                         ,
"Barycenter"                                          ,
"catenary"                                            ,
"DAC"                                                 ,
"filenamer"                                           ,
"matR"                                                ,
"MSCMT"                                               ,
"ProjectionBasedClustering"                           ,
"SanzCircos"                                          ,
"spduration"                                          ,
"statcheck"                                           ,
"SurvDisc"                                            ,
"beam"                                                ,
"cyclestreets"                                        ,
"elementR"                                            ,
"Formula"                                             ,
"InterpretMSSpectrum"                                 ,
"overlap"                                             ,
"popbio"                                              ,
"psyosphere"                                          ,
"SBSDiff"                                             ,
"sstModel"                                            ,
"turboEM"                                             ,
"weco"                                                ,
"AR"                                                  ,
"Calculator.LR.FNs"                                   ,
"cattonum"                                            ,
"DBI"                                                 ,
"goodpractice"                                        ,
"intamap"                                             ,
"intcensROC"                                          ,
"mrbsizeR"                                            ,
"phenoCDM"                                            ,
"pointblank"                                          ,
"RAC"                                                 ,
"xplain"                                              ,
"brt"                                                 ,
"CRPClustering"                                       ,
"crs"                                                 ,
"desc"                                                ,
"ECctmc"                                              ,
"estprod"                                             ,
"Jdmbs"                                               ,
"Rnumerai"                                            ,
"rwunderground"                                       ,
"sbart"                                               ,
"scape"                                               ,
"shinystan"                                           ,
"Sstack"                                              ,
"stosim"                                              ,
"UsingR"                                              ,
"BBMV"                                                ,
"bigIntegerAlgos"                                     ,
"gcmr"                                                ,
"getlandsat"                                          ,
"LatticeKrig"                                         ,
"originr"                                             ,
"vMask"                                               ,
"CompGLM"                                             ,
"hyperSMURF"                                          ,
"predtoolsTS"                                         ,
"anesrake"                                            ,
"BigQuic"                                             ,
"sensR"                                               ,
"brainKCCA"                                           ,
"envalysis"                                           ,
"jmdem"                                               ,
"jmetrik"                                             ,
"mathpix"                                             ,
"MultiplierDEA"                                       ,
"smartsizer"                                          ,
"BLSM"                                                ,
"efts"                                                ,
"MixfMRI"                                             ,
"pooling"                                             ,
"raw"                                                 ,
"rddapp"                                              ,
"SpatialFloor"                                        ,
"svDialogs"                                           ,
"TableMonster"                                        ,
"timechange"                                          ,
"base2grob"                                           ,
"equalCovs"                                           ,
"evd"                                                 ,
"forestmodel"                                         ,
"gensvm"                                              ,
"lavaanPlot"                                          ,
"ManlyMix"                                            ,
"MiRKAT"                                              ,
"MittagLeffleR"                                       ,
"onnx"                                                ,
"shinyLP"                                             ,
"SOR"                                                 ,
"sourcetools"                                         ,
"survivALL"                                           ,
"svmplus"                                             ,
"worms"                                               ,
"BayesXsrc"                                           ,
"cultevo"                                             ,
"EMVS"                                                ,
"EpiCurve"                                            ,
"frambgrowth"                                         ,
"imdbapi"                                             ,
"MARX"                                                ,
"PieceExpIntensity"                                   ,
"pmultinom"                                           ,
"SNFtool"                                             ,
"SourceSet"                                           ,
"VarReg"                                              ,
"water"                                               ,
"ACTCD"                                               ,
"APML0"                                               ,
"cubing"                                              ,
"dcurver"                                             ,
"fbati"                                               ,
"gdalUtils"                                           ,
"komadown"                                            ,
"leaflet.esri"                                        ,
"relations"                                           ,
"semantic.dashboard"                                  ,
"sigmaNet"                                            ,
"spatial.tools"                                       ,
"spBayesSurv"                                         ,
"svGUI"                                               ,
"tactile"                                             ,
"walkalytics"                                         ,
"psgp"                                                ,
"R.oo"                                                ,
"RSiteCatalyst"                                       ,
"VetResearchLMM"                                      ,
"autoplotly"                                          ,
"fgui"                                                ,
"kerndwd"                                             ,
"leaflet.extras"                                      ,
"smovie"                                              ,
"alr4"                                                ,
"BayesSAE"                                            ,
"binMto"                                              ,
"BSagri"                                              ,
"C443"                                                ,
"corpustools"                                         ,
"ctmcmove"                                            ,
"dissever"                                            ,
"europepmc"                                           ,
"futile.matrix"                                       ,
"futile.options"                                      ,
"NetOrigin"                                           ,
"pbatR"                                               ,
"PET"                                                 ,
"polywog"                                             ,
"qlcMatrix"                                           ,
"RM.weights"                                          ,
"semsfa"                                              ,
"tawny"                                               ,
"tawny.types"                                         ,
"utilsIPEA"                                           ,
"votesys"                                             ,
"wesanderson"                                         ,
"BEST"                                                ,
"coneproj"                                            ,
"denseFLMM"                                           ,
"FarmSelect"                                          ,
"Matrix.utils"                                        ,
"profileR"                                            ,
"arc"                                                 ,
"deBInfer"                                            ,
"DrInsight"                                           ,
"GPvam"                                               ,
"pedantics"                                           ,
"prodlim"                                             ,
"rbraries"                                            ,
"stampr"                                              ,
"thgenetics"                                          ,
"TP.idm"                                              ,
"UniDOE"                                              ,
"WhiteStripe"                                         ,
"anomalize"                                           ,
"caper"                                               ,
"coxphw"                                              ,
"genderBR"                                            ,
"ltm"                                                 ,
"Publish"                                             ,
"rvertnet"                                            ,
"Rvmmin"                                              ,
"uncmbb"                                              ,
"vcfR"                                                ,
"apaText"                                             ,
"batch"                                               ,
"bisect"                                              ,
"CBDA"                                                ,
"classiFunc"                                          ,
"cond"                                                ,
"corehunter"                                          ,
"csampling"                                           ,
"effectsizescr"                                       ,
"JM"                                                  ,
"loggle"                                              ,
"marg"                                                ,
"newTestSurvRec"                                      ,
"NightDay"                                            ,
"PCICt"                                               ,
"RJDBC"                                               ,
"rnr"                                                 ,
"SemiPar"                                             ,
"sGMRFmix"                                            ,
"SimRepeat"                                           ,
"idbr"                                                ,
"meltt"                                               ,
"SixSigma"                                            ,
"CensMixReg"                                          ,
"influence.SEM"                                       ,
"mlegp"                                               ,
"Rprofet"                                             ,
"yesno"                                               ,
"arm"                                                 ,
"clinfun"                                             ,
"DBHC"                                                ,
"fpp2"                                                ,
"gldrm"                                               ,
"GraphPCA"                                            ,
"hydrolinks"                                          ,
"lqmm"                                                ,
"mgc"                                                 ,
"MixSIAR"                                             ,
"monkeylearn"                                         ,
"nlirms"                                              ,
"nna"                                                 ,
"promises"                                            ,
"sdat"                                                ,
"StMoMo"                                              ,
"waveband"                                            ,
"aMNLFA"                                              ,
"aws.comprehend"                                      ,
"aws.translate"                                       ,
"bayesammi"                                           ,
"CCM"                                                 ,
"cem"                                                 ,
"gimmeTools"                                          ,
"indirect"                                            ,
"labstatR"                                            ,
"minval"                                              ,
"phyreg"                                              ,
"PreciseSums"                                         ,
"rtip"                                                ,
"sokoban"                                             ,
"stratifyR"                                           ,
"bioimagetools"                                       ,
"di"                                                  ,
"FindIt"                                              ,
"GroupSeq"                                            ,
"haplo.stats"                                         ,
"irrNA"                                               ,
"lubridate"                                           ,
"RSurvey"                                             ,
"snpReady"                                            ,
"sparsepca"                                           ,
"splitFeas"                                           ,
"webdriver"                                           ,
"fasta"                                               ,
"FeatureHashing"                                      ,
"fluoSurv"                                            ,
"GLMsData"                                            ,
"mispr"                                               ,
"mknapsack"                                           ,
"multisensi"                                          ,
"mycor"                                               ,
"nmw"                                                 ,
"RSAlgaeR"                                            ,
"AnnotationBustR"                                     ,
"aws.transcribe"                                      ,
"esvis"                                               ,
"IGG"                                                 ,
"kdecopula"                                           ,
"loggit"                                              ,
"mapmisc"                                             ,
"MargCond"                                            ,
"MBSP"                                                ,
"mclcar"                                              ,
"mstate"                                              ,
"MultiVarMI"                                          ,
"mvPot"                                               ,
"OptimClassifier"                                     ,
"RiverBuilder"                                        ,
"shuffleCI"                                           ,
"textstem"                                            ,
"equate"                                              ,
"ggconf"                                              ,
"liureg"                                              ,
"lmSupport"                                           ,
"MixAll"                                              ,
"nlnet"                                               ,
"bnpmr"                                               ,
"webchem"                                             ,
"accelmissing"                                        ,
"ARTP"                                                ,
"BayesRS"                                             ,
"costat"                                              ,
"CustomerScoringMetrics"                              ,
"dggridR"                                             ,
"eply"                                                ,
"flam"                                                ,
"flexmet"                                             ,
"gapfill"                                             ,
"gmfd"                                                ,
"hbmem"                                               ,
"mueRelativeRisk"                                     ,
"rtop"                                                ,
"RWildbook"                                           ,
"selectr"                                             ,
"ANOVA.TFNs"                                          ,
"BayesESS"                                            ,
"bsplus"                                              ,
"CISE"                                                ,
"corrDNA"                                             ,
"DTDA.ni"                                             ,
"GAMens"                                              ,
"irtDemo"                                             ,
"numbersBR"                                           ,
"prepplot"                                            ,
"qtlhot"                                              ,
"qtlnet"                                              ,
"SnakesAndLaddersAnalysis"                            ,
"subplex"                                             ,
"TVsMiss"                                             ,
"XLConnect"                                           ,
"XLConnectJars"                                       ,
"apTreeshape"                                         ,
"data.world"                                          ,
"EffectStars"                                         ,
"EffectStars2"                                        ,
"ForestTools"                                         ,
"HistData"                                            ,
"humanize"                                            ,
"powerbydesign"                                       ,
"scagnostics"                                         ,
"systemfit"                                           ,
"archiDART"                                           ,
"bio3d"                                               ,
"CollessLike"                                         ,
"fmsb"                                                ,
"heplots"                                             ,
"maps"                                                ,
"MortalityTables"                                     ,
"pollstR"                                             ,
"SSrat"                                               ,
"Bivariate.Pareto"                                    ,
"dfoptim"                                             ,
"GFGM.copula"                                         ,
"SpatMCA"                                             ,
"usfertilizer"                                        ,
"AssetPricing"                                        ,
"gamsel"                                              ,
"gb"                                                  ,
"geecure"                                             ,
"ICSShiny"                                            ,
"metaplus"                                            ,
"rGroovy"                                             ,
"sparseFLMM"                                          ,
"growfunctions"                                       ,
"ads"                                                 ,
"mapdata"                                             ,
"marked"                                              ,
"mixreg"                                              ,
"mstR"                                                ,
"normalr"                                             ,
"radiomics"                                           ,
"spcosa"                                              ,
"bindrcpp"                                            ,
"jdx"                                                 ,
"lcyanalysis"                                         ,
"LTRCtrees"                                           ,
"mapproj"                                             ,
"phantom"                                             ,
"RQDA"                                                ,
"tokenizers"                                          ,
"USAboundaries"                                       ,
"viridis"                                             ,
"wellknown"                                           ,
"productivity"                                        ,
"SubTite"                                             ,
"VennDiagram"                                         ,
"ARHT"                                                ,
"bairt"                                               ,
"config"                                              ,
"constellation"                                       ,
"dfmeta"                                              ,
"dfped"                                               ,
"dynprog"                                             ,
"ggdag"                                               ,
"saeRobust"                                           ,
"semPower"                                            ,
"shinycustomloader"                                   ,
"sperrorest"                                          ,
"ssc"                                                 ,
"trustOptim"                                          ,
"datr"                                                ,
"degreenet"                                           ,
"hts"                                                 ,
"phenomap"                                            ,
"ramps"                                               ,
"sparseMVN"                                           ,
"plogr"                                               ,
"randomForest"                                        ,
"tabuSearch"                                          ,
"DMRnet"                                              ,
"h2o4gpu"                                             ,
"ioncopy"                                             ,
"itcSegment"                                          ,
"miLineage"                                           ,
"NormalizeMets"                                       ,
"polySegratioMM"                                      ,
"Rclean"                                              ,
"RunuranGUI"                                          ,
"sfdct"                                               ,
"swagger"                                             ,
"SYNCSA"                                              ,
"ZIBBSeqDiscovery"                                    ,
"eaf"                                                 ,
"ensembleMOS"                                         ,
"fasterize"                                           ,
"GPGame"                                              ,
"hsm"                                                 ,
"KappaGUI"                                            ,
"MCPAN"                                               ,
"MED"                                                 ,
"Momocs"                                              ,
"multilevelPSA"                                       ,
"polySegratio"                                        ,
"R3port"                                              ,
"rreg"                                                ,
"SetTest"                                             ,
"SympluR"                                             ,
"AdapSamp"                                            ,
"crqanlp"                                             ,
"horserule"                                           ,
"incR"                                                ,
"mapr"                                                ,
"NEArender"                                           ,
"purging"                                             ,
"rforensicbatwing"                                    ,
"rjazz"                                               ,
"RNRCS"                                               ,
"SPINA"                                               ,
"TFisher"                                             ,
"catdap"                                              ,
"eventdataR"                                          ,
"evir"                                                ,
"fastmaRching"                                        ,
"fsthet"                                              ,
"genlogis"                                            ,
"multipol"                                            ,
"Oarray"                                              ,
"PHYLOGR"                                             ,
"PrivateLR"                                           ,
"rarhsmm"                                             ,
"rmeta"                                               ,
"spnn"                                                ,
"CIEE"                                                ,
"cotrend"                                             ,
"DGVM3D"                                              ,
"dyn"                                                 ,
"klaR"                                                ,
"ppsbm"                                               ,
"ROlogit"                                             ,
"RSSampling"                                          ,
"sylly.en"                                            ,
"TwoRegression"                                       ,
"bcgam"                                               ,
"denstrip"                                            ,
"g2f"                                                 ,
"hierNet"                                             ,
"LearnBayes"                                          ,
"msme"                                                ,
"QuantTools"                                          ,
"RcppSMC"                                             ,
"Rgb"                                                 ,
"smatr"                                               ,
"WaveletComp"                                         ,
"XR"                                                  ,
"dispmod"                                             ,
"OPDOE"                                               ,
"orclus"                                              ,
"popdemo"                                             ,
"rootWishart"                                         ,
"beast"                                               ,
"deltaPlotR"                                          ,
"EthSEQ"                                              ,
"fitPoly"                                             ,
"glogis"                                              ,
"gsubfn"                                              ,
"lfactors"                                            ,
"mlbstats"                                            ,
"NetworkExtinction"                                   ,
"petitr"                                              ,
"QRegVCM"                                             ,
"sphet"                                               ,
"synbreed"                                            ,
"esaps"                                               ,
"gmm"                                                 ,
"paramGUI"                                            ,
"pastecs"                                             ,
"RcppClassicExamples"                                 ,
"wheatmap"                                            ,
"withr"                                               ,
"adaptiveGPCA"                                        ,
"corregp"                                             ,
"funnelR"                                             ,
"genogeographer"                                      ,
"gfmR"                                                ,
"ggQQunif"                                            ,
"gim"                                                 ,
"higrad"                                              ,
"hmmm"                                                ,
"maxadjAUC"                                           ,
"ordinalRR"                                           ,
"RH2"                                                 ,
"ridittools"                                          ,
"scatterplot3d"                                       ,
"SensMixed"                                           ,
"tidyboot"                                            ,
"vstsr"                                               ,
"wordbankr"                                           ,
"zipfextR"                                            ,
"abnormality"                                         ,
"AnglerCreelSurveySimulation"                         ,
"baystability"                                        ,
"bindr"                                               ,
"BKPC"                                                ,
"Dominance"                                           ,
"fcr"                                                 ,
"findR"                                               ,
"genBart"                                             ,
"Modelcharts"                                         ,
"readit"                                              ,
"RM2006"                                              ,
"winRatioAnalysis"                                    ,
"bdots"                                               ,
"inTrees"                                             ,
"pgirmess"                                            ,
"TimeVTree"                                           ,
"cmm"                                                 ,
"gfer"                                                ,
"widyr"                                               ,
"ICSNP"                                               ,
"properties"                                          ,
"Rcriticor"                                           ,
"relaimpo"                                            ,
"rmonad"                                              ,
"circglmbayes"                                        ,
"cronR"                                               ,
"EpiILM"                                              ,
"FASeg"                                               ,
"glm.deploy"                                          ,
"IDetect"                                             ,
"joineRmeta"                                          ,
"micompr"                                             ,
"PP3"                                                 ,
"wgeesel"                                             ,
"PartCensReg"                                         ,
"xpose4"                                              ,
"gender"                                              ,
"LSRS"                                                ,
"mvmeta"                                              ,
"ppcSpatial"                                          ,
"rodeo"                                               ,
"ACMEeqtl"                                            ,
"ahp"                                                 ,
"HIMA"                                                ,
"ILS"                                                 ,
"KraljicMatrix"                                       ,
"nls.multstart"                                       ,
"NMF"                                                 ,
"partialAR"                                           ,
"robets"                                              ,
"slickR"                                              ,
"timsac"                                              ,
"adfExplorer"                                         ,
"DAP"                                                 ,
"ecospace"                                            ,
"elections"                                           ,
"forestr"                                             ,
"gepaf"                                               ,
"NormExpression"                                      ,
"pbm"                                                 ,
"qdapDictionaries"                                    ,
"rgr"                                                 ,
"soundecology"                                        ,
"tidyimpute"                                          ,
"vowels"                                              ,
"adeba"                                               ,
"bartMachineJARs"                                     ,
"coxphf"                                              ,
"ICS"                                                 ,
"MNM"                                                 ,
"pwr"                                                 ,
"RcmdrPlugin.aRnova"                                  ,
"comato"                                              ,
"LDRTools"                                            ,
"rbiouml"                                             ,
"rpostgisLT"                                          ,
"fauxpas"                                             ,
"formula.tools"                                       ,
"gWidgetsRGtk2"                                       ,
"SCperf"                                              ,
"TrafficBDE"                                          ,
"TTAinterfaceTrendAnalysis"                           ,
"ldhmm"                                               ,
"rmcorr"                                              ,
"SpatEntropy"                                         ,
"bpnreg"                                              ,
"depend.truncation"                                   ,
"filematrix"                                          ,
"icRSF"                                               ,
"idmTPreg"                                            ,
"projector"                                           ,
"truncnorm"                                           ,
"Ac3net"                                              ,
"carSurv"                                             ,
"fuzzywuzzyR"                                         ,
"KGode"                                               ,
"mctest"                                              ,
"mmcm"                                                ,
"ph2bayes"                                            ,
"photobiologySensors"                                 ,
"PK"                                                  ,
"rospca"                                              ,
"SASxport"                                            ,
"EDMeasure"                                           ,
"nzpullover"                                          ,
"Rcssplot"                                            ,
"bea.R"                                               ,
"clespr"                                              ,
"dynamo"                                              ,
"intamapInteractive"                                  ,
"multfisher"                                          ,
"neat"                                                ,
"PropCIs"                                             ,
"tkRplotR"                                            ,
"braQCA"                                              ,
"colmozzie"                                           ,
"DGM"                                                 ,
"irtreliability"                                      ,
"lassopv"                                             ,
"RCRnorm"                                             ,
"VTrack"                                              ,
"wbsts"                                               ,
"chunkR"                                              ,
"FlexGAM"                                             ,
"ivprobit"                                            ,
"oasis"                                               ,
"phyloTop"                                            ,
"RcmdrPlugin.KMggplot2"                               ,
"rroad"                                               ,
"timeDate"                                            ,
"VeryLargeIntegers"                                   ,
"CircMLE"                                             ,
"gvcR"                                                ,
"idx2r"                                               ,
"InterVA4"                                            ,
"knitrProgressBar"                                    ,
"LLM"                                                 ,
"measuRing"                                           ,
"meetupapi"                                           ,
"naivereg"                                            ,
"reinsureR"                                           ,
"sNPLS"                                               ,
"SpatPCA"                                             ,
"archdata"                                            ,
"colorhcplot"                                         ,
"confSAM"                                             ,
"fmriqa"                                              ,
"IROmiss"                                             ,
"biosignalEMG"                                        ,
"codingMatrices"                                      ,
"githubinstall"                                       ,
"imbalance"                                           ,
"likelihoodAsy"                                       ,
"PottsUtils"                                          ,
"pryr"                                                ,
"ratios"                                              ,
"adklakedata"                                         ,
"cnmlcd"                                              ,
"miscF"                                               ,
"MST"                                                 ,
"PakPC2017"                                           ,
"recurse"                                             ,
"spider"                                              ,
"clust.bin.pair"                                      ,
"micropan"                                            ,
"nscancor"                                            ,
"ratesci"                                             ,
"rmetasim"                                            ,
"teigen"                                              ,
"vcrpart"                                             ,
"CorrectedFDR"                                        ,
"EMMIXcskew"                                          ,
"MDMR"                                                ,
"monitoR"                                             ,
"neldermead"                                          ,
"PPtreeViz"                                           ,
"spTest"                                              ,
"ssrm.logmer"                                         ,
"stpp"                                                ,
"TurtleGraphics"                                      ,
"EMMIXskew"                                           ,
"epinet"                                              ,
"fishdata"                                            ,
"heavy"                                               ,
"inferr"                                              ,
"librarysnapshot"                                     ,
"mixture"                                             ,
"mopa"                                                ,
"optimsimplex"                                        ,
"pdmod"                                               ,
"RmecabKo"                                            ,
"SIS"                                                 ,
"TexExamRandomizer"                                   ,
"basicspace"                                          ,
"gamm4.test"                                          ,
"gglorenz"                                            ,
"GUniFrac"                                            ,
"inca"                                                ,
"Lavash"                                              ,
"npmlda"                                              ,
"Pomic"                                               ,
"scoring"                                             ,
"shinyalert"                                          ,
"shinyKGode"                                          ,
"wnominate"                                           ,
"distrom"                                             ,
"estimability"                                        ,
"HDoutliers"                                          ,
"lakemorpho"                                          ,
"pheno2geno"                                          ,
"rphast"                                              ,
"snappier"                                            ,
"textir"                                              ,
"dataonderivatives"                                   ,
"support.BWS"                                         ,
"BSPADATA"                                            ,
"Dykstra"                                             ,
"facilitation"                                        ,
"likeLTD"                                             ,
"LSDinterface"                                        ,
"NCutYX"                                              ,
"npcopTest"                                           ,
"oxcAAR"                                              ,
"sme"                                                 ,
"vote"                                                ,
"adepro"                                              ,
"BIGDAWG"                                             ,
"fdadensity"                                          ,
"geozoning"                                           ,
"micromap"                                            ,
"mvoutlier"                                           ,
"oc"                                                  ,
"qcQpcr"                                              ,
"specklestar"                                         ,
"coefplot"                                            ,
"glarma"                                              ,
"glmtlp"                                              ,
"jmotif"                                              ,
"LowRankQP"                                           ,
"powerMediation"                                      ,
"powerSurvEpi"                                        ,
"rosqp"                                               ,
"shape"                                               ,
"valuer"                                              ,
"WindCurves"                                          ,
"DACF"                                                ,
"detectRUNS"                                          ,
"DWDLargeR"                                           ,
"ecespa"                                              ,
"googlePrintr"                                        ,
"lira"                                                ,
"valection"                                           ,
"ggbuildr"                                            ,
"lmodel2"                                             ,
"MBC"                                                 ,
"population"                                          ,
"quantoptr"                                           ,
"RaPKod"                                              ,
"survivalsvm"                                         ,
"aVirtualTwins"                                       ,
"eda4treeR"                                           ,
"gk"                                                  ,
"rmdshower"                                           ,
"fuzzyforest"                                         ,
"pcev"                                                ,
"adegenet"                                            ,
"BNN"                                                 ,
"ICSOutlier"                                          ,
"miic"                                                ,
"pinnacle.API"                                        ,
"pipeGS"                                              ,
"pmclust"                                             ,
"wrswoR"                                              ,
"Zseq"                                                ,
"ald"                                                 ,
"BsMD"                                                ,
"cifti"                                               ,
"d3heatmap"                                           ,
"gifti"                                               ,
"lqr"                                                 ,
"modmarg"                                             ,
"opticut"                                             ,
"topmodel"                                            ,
"viridisLite"                                         ,
"denoiSeq"                                            ,
"Massign"                                             ,
"mvnfast"                                             ,
"OOR"                                                 ,
"RCrypto"                                             ,
"RTriangle"                                           ,
"SiMRiv"                                              ,
"asciiruler"                                          ,
"gnorm"                                               ,
"KRIG"                                                ,
"mr.raps"                                             ,
"profmem"                                             ,
"RBMRB"                                               ,
"relatable"                                           ,
"robustBLME"                                          ,
"rpivotTable"                                         ,
"rtfbs"                                               ,
"scdensity"                                           ,
"String2AdjMatrix"                                    ,
"atsd"                                                ,
"bmotif"                                              ,
"cherry"                                              ,
"cometExactTest"                                      ,
"dalmatian"                                           ,
"networkR"                                            ,
"NIRStat"                                             ,
"subselect"                                           ,
"timetools"                                           ,
"waver"                                               ,
"clustRcompaR"                                        ,
"glmpath"                                             ,
"rrBLUP"                                              ,
"stepPlr"                                             ,
"stm"                                                 ,
"zeallot"                                             ,
"bigtcr"                                              ,
"aidar"                                               ,
"bcrypt"                                              ,
"gutenbergr"                                          ,
"hypergea"                                            ,
"ic.infer"                                            ,
"kokudosuuchi"                                        ,
"LSD"                                                 ,
"matchingR"                                           ,
"mogavs"                                              ,
"prrd"                                                ,
"R2DT"                                                ,
"waterfalls"                                          ,
"aggregation"                                         ,
"aucm"                                                ,
"censusr"                                             ,
"crimCV"                                              ,
"DBItest"                                             ,
"dendroextras"                                        ,
"diagis"                                              ,
"ETLUtils"                                            ,
"gradDescent"                                         ,
"heims"                                               ,
"iotools"                                             ,
"lazyWeave"                                           ,
"multiselect"                                         ,
"datapasta"                                           ,
"devFunc"                                             ,
"geno2proteo"                                         ,
"highlightHTML"                                       ,
"letsR"                                               ,
"MPR.genotyping"                                      ,
"neverhpfilter"                                       ,
"pgam"                                                ,
"RcppGreedySetCover"                                  ,
"RInSp"                                               ,
"santaR"                                              ,
"scanstatistics"                                      ,
"showimage"                                           ,
"strandCet"                                           ,
"thief"                                               ,
"TSS.RESTREND"                                        ,
"MaxentVariableSelection"                             ,
"MRS"                                                 ,
"mtsdi"                                               ,
"RDocumentation"                                      ,
"zoon"                                                ,
"Bioi"                                                ,
"enetLTS"                                             ,
"gLRTH"                                               ,
"pals"                                                ,
"PAmeasures"                                          ,
"CreditRisk"                                          ,
"geophys"                                             ,
"LinkageMapView"                                      ,
"listenv"                                             ,
"nofrills"                                            ,
"rMR"                                                 ,
"rSCA"                                                ,
"tropAlgebra"                                         ,
"dat"                                                 ,
"ewoc"                                                ,
"LoopAnalyst"                                         ,
"RClickhouse"                                         ,
"Rwave"                                               ,
"adabag"                                              ,
"descr"                                               ,
"dynatopmodel"                                        ,
"glamlasso"                                           ,
"lbreg"                                               ,
"LN0SCIs"                                             ,
"longRPart2"                                          ,
"pivotaltrackR"                                       ,
"PSW"                                                 ,
"RNAseqNet"                                           ,
"dtree"                                               ,
"ensembleBMA"                                         ,
"GEOmap"                                              ,
"gkmSVM"                                              ,
"PEIP"                                                ,
"RFOC"                                                ,
"RPMG"                                                ,
"RTOMO"                                               ,
"SDT"                                                 ,
"SMMA"                                                ,
"TukeyRegion"                                         ,
"vecsets"                                             ,
"zoeppritz"                                           ,
"bbw"                                                 ,
"GenSA"                                               ,
"mau"                                                 ,
"dawai"                                               ,
"LPM"                                                 ,
"mazeinda"                                            ,
"natural"                                             ,
"opencage"                                            ,
"optimus"                                             ,
"palmtree"                                            ,
"randomLCA"                                           ,
"toolmaRk"                                            ,
"GoogleKnowledgeGraphR"                               ,
"io"                                                  ,
"RPPanalyzer"                                         ,
"adaptMCMC"                                           ,
"COMBAT"                                              ,
"cordillera"                                          ,
"Fragman"                                             ,
"pcaPP"                                               ,
"photobiologyLEDs"                                    ,
"Emcdf"                                               ,
"hkclustering"                                        ,
"MatrixEQTL"                                          ,
"PGRdup"                                              ,
"stubthat"                                            ,
"moult"                                               ,
"ngspatial"                                           ,
"preText"                                             ,
"qCBA"                                                ,
"rSQM"                                                ,
"AlleleRetain"                                        ,
"bayesSurv"                                           ,
"bigmemory"                                           ,
"DiceView"                                            ,
"DirectedClustering"                                  ,
"kexpmv"                                              ,
"QCApro"                                              ,
"text2vec"                                            ,
"brant"                                               ,
"changepointsVar"                                     ,
"ExpDE"                                               ,
"FinancialInstrument"                                 ,
"IATScore"                                            ,
"influxdbr"                                           ,
"lba"                                                 ,
"RandPro"                                             ,
"rFSA"                                                ,
"tdr"                                                 ,
"ATR"                                                 ,
"bullwhipgame"                                        ,
"easySVG"                                             ,
"formulize"                                           ,
"MatchIt"                                             ,
"osmose"                                              ,
"Rbgs"                                                ,
"RGBM"                                                ,
"rlme"                                                ,
"univOutl"                                            ,
"constants"                                           ,
"dprint"                                              ,
"factoptd"                                            ,
"mistat"                                              ,
"sgee"                                                ,
"shinyjs"                                             ,
"ExplainPrediction"                                   ,
"fingerprint"                                         ,
"grove"                                               ,
"date"                                                ,
"DRR"                                                 ,
"fmbasics"                                            ,
"qlcData"                                             ,
"SciViews"                                            ,
"APPEstimation"                                       ,
"bda"                                                 ,
"binst"                                               ,
"coindeskr"                                           ,
"diffusion"                                           ,
"DNetFinder"                                          ,
"ellipse"                                             ,
"fdrDiscreteNull"                                     ,
"MAPA"                                                ,
"MatManlyMix"                                         ,
"mra"                                                 ,
"paramlink"                                           ,
"probout"                                             ,
"profile"                                             ,
"rapidraker"                                          ,
"relSim"                                              ,
"remoter"                                             ,
"unitedR"                                             ,
"walmartAPI"                                          ,
"washdata"                                            ,
"wavefunction"                                        ,
"afCEC"                                               ,
"DoseFinding"                                         ,
"fmdates"                                             ,
"gWidgets2RGtk2"                                      ,
"ipft"                                                ,
"LCAvarsel"                                           ,
"memo"                                                ,
"PCMRS"                                               ,
"R.cache"                                             ,
"SparseDC"                                            ,
"Biolinv"                                             ,
"discgolf"                                            ,
"ExtremeBounds"                                       ,
"httping"                                             ,
"mekko"                                               ,
"miRNAss"                                             ,
"mixlm"                                               ,
"oaxaca"                                              ,
"qualvar"                                             ,
"rprojroot"                                           ,
"Rsampletrees"                                        ,
"sofa"                                                ,
"wbstats"                                             ,
"wordcloud2"                                          ,
"addreg"                                              ,
"CytobankBridgeR"                                     ,
"conformalClassification"                             ,
"pKSEA"                                               ,
"spectacles"                                          ,
"alphashape3d"                                        ,
"bdsmatrix"                                           ,
"BullsEyeR"                                           ,
"fdcov"                                               ,
"fractal"                                             ,
"grapherator"                                         ,
"PBIBD"                                               ,
"samplesizeCMH"                                       ,
"seaaroundus"                                         ,
"sparseEigen"                                         ,
"swatches"                                            ,
"TrioSGL"                                             ,
"datoramar"                                           ,
"kgc"                                                 ,
"nparACT"                                             ,
"OGI"                                                 ,
"SimCop"                                              ,
"theseus"                                             ,
"widgetframe"                                         ,
"GeDS"                                                ,
"PathSelectMP"                                        ,
"quantregForest"                                      ,
"reda"                                                ,
"rhmmer"                                              ,
"rhymer"                                              ,
"ZOIP"                                                ,
"adephylo"                                            ,
"Canopy"                                              ,
"kimisc"                                              ,
"SpatialBall"                                         ,
"Umatrix"                                             ,
"AmesHousing"                                         ,
"CoClust"                                             ,
"FreeSortR"                                           ,
"mlapi"                                               ,
"PKPDmisc"                                            ,
"SegCorr"                                             ,
"stringformattr"                                      ,
"SyncRNG"                                             ,
"cleanEHR"                                            ,
"gofastr"                                             ,
"BClustLonG"                                          ,
"BisRNA"                                              ,
"cit"                                                 ,
"clustEff"                                            ,
"EMSC"                                                ,
"GMAC"                                                ,
"hypercube"                                           ,
"irtoys"                                              ,
"plsVarSel"                                           ,
"quantregRanger"                                      ,
"clickstream"                                         ,
"PooledMeanGroup"                                     ,
"RSNPset"                                             ,
"spAddins"                                            ,
"spatgraphs"                                          ,
"sRDA"                                                ,
"submax"                                              ,
"syuzhet"                                             ,
"tweedie"                                             ,
"binaryLogic"                                         ,
"hysteresis"                                          ,
"POPdemog"                                            ,
"SensoMineR"                                          ,
"control"                                             ,
"corpus"                                              ,
"mazealls"                                            ,
"NanoStringNorm"                                      ,
"secrlinear"                                          ,
"sets"                                                ,
"XiMpLe"                                              ,
"ECharts2Shiny"                                       ,
"fitteR"                                              ,
"junr"                                                ,
"OpenMPController"                                    ,
"reshape2"                                            ,
"snplist"                                             ,
"survivalMPL"                                         ,
"dynCorr"                                             ,
"ESTER"                                               ,
"phylotools"                                          ,
"refset"                                              ,
"parsemsf"                                            ,
"randomcoloR"                                         ,
"snowflakes"                                          ,
"WARN"                                                ,
"xray"                                                ,
"controlTest"                                         ,
"DOBAD"                                               ,
"MMDai"                                               ,
"CPMCGLM"                                             ,
"earlyR"                                              ,
"gTests"                                              ,
"Inventorymodel"                                      ,
"ISM"                                                 ,
"LassoSIR"                                            ,
"MetamapsDB"                                          ,
"PSLM2015"                                            ,
"TriMatch"                                            ,
"vcd"                                                 ,
"wmtsa"                                               ,
"CaDENCE"                                             ,
"deGradInfer"                                         ,
"GEVcdn"                                              ,
"monmlp"                                              ,
"plotrr"                                              ,
"R2BayesX"                                            ,
"scatr"                                               ,
"sptm"                                                ,
"x12"                                                 ,
"BayesVarSel"                                         ,
"CollapseLevels"                                      ,
"combiter"                                            ,
"easyVerification"                                    ,
"flora"                                               ,
"GRS.test"                                            ,
"mazeGen"                                             ,
"mdw"                                                 ,
"mutoss"                                              ,
"mutossGUI"                                           ,
"networkGen"                                          ,
"OSTSC"                                               ,
"EAinference"                                         ,
"ROI.models.globalOptTests"                           ,
"harrietr"                                            ,
"lrgs"                                                ,
"velox"                                               ,
"whereport"                                           ,
"adapr"                                               ,
"BatchExperiments"                                    ,
"goftte"                                              ,
"npregfast"                                           ,
"passport"                                            ,
"SASmarkdown"                                         ,
"lmomPi"                                              ,
"StrainRanking"                                       ,
"streambugs"                                          ,
"Dforest"                                             ,
"Sim.PLFN"                                            ,
"tipr"                                                ,
"xltabr"                                              ,
"ceg"                                                 ,
"EvolutionaryGames"                                   ,
"odk"                                                 ,
"phase1RMD"                                           ,
"simEd"                                               ,
"skeleSim"                                            ,
"EKMCMC"                                              ,
"RAPTOR"                                              ,
"shinyaframe"                                         ,
"wordnet"                                             ,
"Aoptbdtvc"                                           ,
"dixon"                                               ,
"galts"                                               ,
"inferference"                                        ,
"replicatedpp2w"                                      ,
"tgram"                                               ,
"tscount"                                             ,
"adjustedcranlogs"                                    ,
"CopulaDTA"                                           ,
"CorporaCoCo"                                         ,
"distcrete"                                           ,
"IDCard"                                              ,
"powdist"                                             ,
"domino"                                              ,
"fpmoutliers"                                         ,
"KMgene"                                              ,
"LCF"                                                 ,
"ManifoldOptim"                                       ,
"SELF"                                                ,
"biofiles"                                            ,
"epicontacts"                                         ,
"multichull"                                          ,
"ngram"                                               ,
"RcppHMM"                                             ,
"RcppTN"                                              ,
"rsed"                                                ,
"bang"                                                ,
"basad"                                               ,
"carx"                                                ,
"fImport"                                             ,
"gcdnet"                                              ,
"LaF"                                                 ,
"longROC"                                             ,
"tm.plugin.factiva"                                   ,
"wicket"                                              ,
"easyAHP"                                             ,
"flowDiv"                                             ,
"xslt"                                                ,
"ArchaeoChron"                                        ,
"fAsianOptions"                                       ,
"fBasics"                                             ,
"fExoticOptions"                                      ,
"fExtremes"                                           ,
"gamreg"                                              ,
"logihist"                                            ,
"mpmcorrelogram"                                      ,
"ores"                                                ,
"proPubBills"                                         ,
"Relatedness"                                         ,
"revgeo"                                              ,
"RHawkes"                                             ,
"sppmix"                                              ,
"timeSeries"                                          ,
"dequer"                                              ,
"epr"                                                 ,
"fAssets"                                             ,
"fCopulae"                                            ,
"flatr"                                               ,
"fMultivar"                                           ,
"fNonlinear"                                          ,
"fOptions"                                            ,
"fPortfolio"                                          ,
"fUnitRoots"                                          ,
"hashmap"                                             ,
"MixedPsy"                                            ,
"modest"                                              ,
"Myrrix"                                              ,
"Myrrixjars"                                          ,
"PtProcess"                                           ,
"SequentialDesign"                                    ,
"thregI"                                              ,
"easynls"                                             ,
"EurosarcBayes"                                       ,
"fBonds"                                              ,
"fRegression"                                         ,
"fTrading"                                            ,
"icensBKL"                                            ,
"IQCC"                                                ,
"rpicosat"                                            ,
"ICCbin"                                              ,
"L1pack"                                              ,
"likelihoodExplore"                                   ,
"oaqc"                                                ,
"OptimalTiming"                                       ,
"subsamp"                                             ,
"tidyverse"                                           ,
"auctestr"                                            ,
"bootcluster"                                         ,
"Dasst"                                               ,
"dparser"                                             ,
"OneArmPhaseTwoStudy"                                 ,
"QuantumClone"                                        ,
"rethinker"                                           ,
"StakeholderAnalysis"                                 ,
"ToolsForCoDa"                                        ,
"ggplotAssist"                                        ,
"openNLPdata"                                         ,
"qboxplot"                                            ,
"rcbalance"                                           ,
"RKEAjars"                                            ,
"snnR"                                                ,
"ThankYouStars"                                       ,
"BatchMap"                                            ,
"anocva"                                              ,
"ctmle"                                               ,
"memuse"                                              ,
"PdPDB"                                               ,
"quokar"                                              ,
"rGoodData"                                           ,
"rLDCP"                                               ,
"rODE"                                                ,
"sound"                                               ,
"StAMPP"                                              ,
"bigtime"                                             ,
"famSKATRC"                                           ,
"MOST"                                                ,
"predkmeans"                                          ,
"wNNSel"                                              ,
"condusco"                                            ,
"deepboost"                                           ,
"FHtest"                                              ,
"lsl"                                                 ,
"RcmdrPlugin.GWRM"                                    ,
"APCanalysis"                                         ,
"epxToR"                                              ,
"hues"                                                ,
"pcnetmeta"                                           ,
"Rsymphony"                                           ,
"UniIsoRegression"                                    ,
"ABCoptim"                                            ,
"dcminfo"                                             ,
"Diderot"                                             ,
"goft"                                                ,
"googlePublicData"                                    ,
"pCalibrate"                                          ,
"approxmatch"                                         ,
"ArCo"                                                ,
"extremeStat"                                         ,
"lplyr"                                               ,
"Pstat"                                               ,
"inaparc"                                             ,
"AnnuityRIR"                                          ,
"convoSPAT"                                           ,
"PDN"                                                 ,
"vmd"                                                 ,
"DatAssim"                                            ,
"equateMultiple"                                      ,
"l2boost"                                             ,
"mda"                                                 ,
"mobsim"                                              ,
"nopp"                                                ,
"pglm"                                                ,
"RJSplot"                                             ,
"SLDAssay"                                            ,
"slowraker"                                           ,
"Taxonstand"                                          ,
"HiddenMarkov"                                        ,
"infraFDTD.assist"                                    ,
"PanJen"                                              ,
"unittest"                                            ,
"gapminder"                                           ,
"lpme"                                                ,
"metricTester"                                        ,
"wec"                                                 ,
"abe"                                                 ,
"bbmle"                                               ,
"conover.test"                                        ,
"dynfrail"                                            ,
"MBHdesign"                                           ,
"phylocanvas"                                         ,
"R2admb"                                              ,
"Rduino"                                              ,
"rsurfer"                                             ,
"episode"                                             ,
"rgen"                                                ,
"rpg"                                                 ,
"spgwr"                                               ,
"coppeCosenzaR"                                       ,
"nets"                                                ,
"distdrawr"                                           ,
"dunn.test"                                           ,
"pixels"                                              ,
"rcreds"                                              ,
"vfcp"                                                ,
"countyfloods"                                        ,
"epitools"                                            ,
"FinAna"                                              ,
"oro.nifti"                                           ,
"routr"                                               ,
"actogrammr"                                          ,
"asus"                                                ,
"gmeta"                                               ,
"multitaper"                                          ,
"OPI"                                                 ,
"WWR"                                                 ,
"cbar"                                                ,
"densityClust"                                        ,
"jsTree"                                              ,
"lexiconPT"                                           ,
"MOEADr"                                              ,
"MoLE"                                                ,
"nspmix"                                              ,
"optimization"                                        ,
"paramtest"                                           ,
"spTDyn"                                              ,
"XRPython"                                            ,
"zebu"                                                ,
"DBEST"                                               ,
"EfficientMaxEigenpair"                               ,
"lsei"                                                ,
"PCAmixdata"                                          ,
"WaveLetLongMemory"                                   ,
"debugme"                                             ,
"docuSignr"                                           ,
"modcmfitr"                                           ,
"oii"                                                 ,
"PCA4you"                                             ,
"ROI.plugin.msbinlp"                                  ,
"sankey"                                              ,
"hogsvdR"                                             ,
"RcmdrPlugin.survival"                                ,
"skyscapeR"                                           ,
"accrual"                                             ,
"DOvalidation"                                        ,
"EHR"                                                 ,
"genomic.autocorr"                                    ,
"ISLR"                                                ,
"mvMonitoring"                                        ,
"OnAge"                                               ,
"ROI.plugin.cplex"                                    ,
"ROI.plugin.glpk"                                     ,
"CausalGAM"                                           ,
"EasyMx"                                              ,
"EpiContactTrace"                                     ,
"gwrpvr"                                              ,
"htm2txt"                                             ,
"mafs"                                                ,
"monoreg"                                             ,
"rex"                                                 ,
"rslurm"                                              ,
"divo"                                                ,
"FlowRegEnvCost"                                      ,
"geoRglm"                                             ,
"GreedySBTM"                                          ,
"HSAR"                                                ,
"LZeroSpikeInference"                                 ,
"onemap"                                              ,
"satellite"                                           ,
"swa"                                                 ,
"dbstats"                                             ,
"httpcache"                                           ,
"ivmodel"                                             ,
"mregions"                                            ,
"reddPrec"                                            ,
"snht"                                                ,
"vennplot"                                            ,
"copulaData"                                          ,
"corrplot"                                            ,
"forestinventory"                                     ,
"GSMX"                                                ,
"mvdalab"                                             ,
"rmsfuns"                                             ,
"SelvarMix"                                           ,
"SILGGM"                                              ,
"TSclust"                                             ,
"Plasmode"                                            ,
"AssocTests"                                          ,
"collectArgs"                                         ,
"covequal"                                            ,
"npsurv"                                              ,
"oshka"                                               ,
"dave"                                                ,
"queueing"                                            ,
"rucrdtw"                                             ,
"brainR"                                              ,
"curstatCI"                                           ,
"icmm"                                                ,
"incgraph"                                            ,
"AmyloGram"                                           ,
"dblr"                                                ,
"SDMPlay"                                             ,
"triversity"                                          ,
"XKCDdata"                                            ,
"CCTpack"                                             ,
"colt"                                                ,
"MCI"                                                 ,
"pscl"                                                ,
"rakeR"                                               ,
"brea"                                                ,
"CatPredi"                                            ,
"colf"                                                ,
"mada"                                                ,
"anominate"                                           ,
"mozzie"                                              ,
"RNetCDF"                                             ,
"D3partitionR"                                        ,
"editData"                                            ,
"RPEnsemble"                                          ,
"SQUAREM"                                             ,
"BaTFLED3D"                                           ,
"BLRPM"                                               ,
"BrownDog"                                            ,
"DWLasso"                                             ,
"genridge"                                            ,
"IIS"                                                 ,
"MareyMap"                                            ,
"oro.pet"                                             ,
"shelltrace"                                          ,
"DataEntry"                                           ,
"diffMeanVar"                                         ,
"INCATome"                                            ,
"matchbook"                                           ,
"birtr"                                               ,
"highlight"                                           ,
"IDF"                                                 ,
"kader"                                               ,
"rma.exact"                                           ,
"Rssa"                                                ,
"smpic"                                               ,
"Sunclarco"                                           ,
"confinterpret"                                       ,
"ecd"                                                 ,
"fedreporter"                                         ,
"additiveDEA"                                         ,
"ConSpline"                                           ,
"DoubleCone"                                          ,
"EnergyOnlineCPM"                                     ,
"exactci"                                             ,
"ggallin"                                             ,
"nzelect"                                             ,
"SigTree"                                             ,
"SPYvsSPY"                                            ,
"TSCS"                                                ,
"msaR"                                                ,
"realestateDK"                                        ,
"msu"                                                 ,
"SimTimeVar"                                          ,
"statsDK"                                             ,
"apercu"                                              ,
"CHMM"                                                ,
"htmltidy"                                            ,
"MFKnockoffs"                                         ,
"mmpp"                                                ,
"RCPmod"                                              ,
"rdrop2"                                              ,
"vcdExtra"                                            ,
"breakfast"                                           ,
"chopthin"                                            ,
"enaR"                                                ,
"gesca"                                               ,
"glacierSMBM"                                         ,
"isnullptr"                                           ,
"pifpaf"                                              ,
"quickReg"                                            ,
"repfdr"                                              ,
"colourpicker"                                        ,
"ecoseries"                                           ,
"etl"                                                 ,
"importar"                                            ,
"kfda"                                                ,
"LFDREmpiricalBayes"                                  ,
"MNP"                                                 ,
"sampSurf"                                            ,
"BNPMIXcluster"                                       ,
"d3plus"                                              ,
"Rpdb"                                                ,
"TreeSimGM"                                           ,
"wrswoR.benchmark"                                    ,
"wsrf"                                                ,
"ICBayes"                                             ,
"MatrixCorrelation"                                   ,
"apex"                                                ,
"wikipediatrend"                                      ,
"leabRa"                                              ,
"mosaicModel"                                         ,
"OBMbpkg"                                             ,
"rase"                                                ,
"RcmdrPlugin.sutteForecastR"                          ,
"regexSelect"                                         ,
"WikidataR"                                           ,
"AbsFilterGSEA"                                       ,
"brazilmaps"                                          ,
"magicLamp"                                           ,
"QLearning"                                           ,
"cptec"                                               ,
"darksky"                                             ,
"FDRSeg"                                              ,
"prettymapr"                                          ,
"ROI.plugin.clp"                                      ,
"SOIL"                                                ,
"binaryGP"                                            ,
"BMT"                                                 ,
"candisc"                                             ,
"COCONUT"                                             ,
"curvecomp"                                           ,
"EQUIVNONINF"                                         ,
"fusionclust"                                         ,
"gma"                                                 ,
"IGP"                                                 ,
"kpeaks"                                              ,
"lgtdl"                                               ,
"SiER"                                                ,
"sure"                                                ,
"egcm"                                                ,
"leafSTAR"                                            ,
"mcMST"                                               ,
"prefmod"                                             ,
"scaleboot"                                           ,
"tsgui"                                               ,
"rbtt"                                                ,
"SampleSize4ClinicalTrials"                           ,
"crayon"                                              ,
"DES"                                                 ,
"gensphere"                                           ,
"MsdeParEst"                                          ,
"mvmesh"                                              ,
"PKI"                                                 ,
"SphericalCubature"                                   ,
"wildcard"                                            ,
"AurieLSHGaussian"                                    ,
"CATTexact"                                           ,
"fastcox"                                             ,
"gglasso"                                             ,
"Gmedian"                                             ,
"gradientPickerD3"                                    ,
"linemap"                                             ,
"veccompare"                                          ,
"alineR"                                              ,
"CausalImpact"                                        ,
"causalMGM"                                           ,
"ADPF"                                                ,
"asdreader"                                           ,
"auto.pca"                                            ,
"BSSasymp"                                            ,
"DCD"                                                 ,
"dosresmeta"                                          ,
"fICA"                                                ,
"gains"                                               ,
"medmod"                                              ,
"sparsebn"                                            ,
"BayesCR"                                             ,
"bmp"                                                 ,
"GauPro"                                              ,
"gmt"                                                 ,
"mvna"                                                ,
"nos"                                                 ,
"opendotaR"                                           ,
"showtextdb"                                          ,
"sutteForecastR"                                      ,
"TideCurves"                                          ,
"TideTables"                                          ,
"discretecdAlgorithm"                                 ,
"MTSYS"                                               ,
"GeoMongo"                                            ,
"gridExtra"                                           ,
"jointDiag"                                           ,
"RTaxometrics"                                        ,
"openintro"                                           ,
"SMM"                                                 ,
"AWR"                                                 ,
"backShift"                                           ,
"Bios2cor"                                            ,
"mapReasy"                                            ,
"modelwordcloud"                                      ,
"multilaterals"                                       ,
"PakPMICS2014Ch"                                      ,
"PakPMICS2014HH"                                      ,
"PakPMICS2014HL"                                      ,
"PakPMICS2014Wm"                                      ,
"unjoin"                                              ,
"csabounds"                                           ,
"Ohit"                                                ,
"survPresmooth"                                       ,
"TNC"                                                 ,
"UdderQuarterInfectionData"                           ,
"clusterPower"                                        ,
"FuzzyNumbers.Ext.2"                                  ,
"RcextTools"                                          ,
"VDSPCalibration"                                     ,
"billboard"                                           ,
"BosonSampling"                                       ,
"cbanalysis"                                          ,
"dplyrAssist"                                         ,
"freqdom.fda"                                         ,
"re2r"                                                ,
"RGeode"                                              ,
"doubcens"                                            ,
"pcdpca"                                              ,
"ShinyImage"                                          ,
"AlphaVantageClient"                                  ,
"BinarybalancedCut"                                   ,
"Fuzzy.p.value"                                       ,
"recosystem"                                          ,
"afpt"                                                ,
"cr17"                                                ,
"DelayedEffect.Design"                                ,
"edrGraphicalTools"                                   ,
"freqdom"                                             ,
"Inflation"                                           ,
"LGEWIS"                                              ,
"R2SWF"                                               ,
"rDotNet"                                             ,
"itsadug"                                             ,
"npphen"                                              ,
"wktmo"                                               ,
"cenGAM"                                              ,
"deltar"                                              ,
"h5"                                                  ,
"PIGE"                                                ,
"plotfunctions"                                       ,
"seedCCA"                                             ,
"datapack"                                            ,
"ebmc"                                                ,
"FPV"                                                 ,
"LowWAFOMSobol"                                       ,
"mcompanion"                                          ,
"MTE"                                                 ,
"nomogramEx"                                          ,
"PeriodicTable"                                       ,
"sensitivitymult"                                     ,
"sgPLS"                                               ,
"statquotes"                                          ,
"caffsim"                                             ,
"ccmm"                                                ,
"dmutate"                                             ,
"holdem"                                              ,
"plot3D"                                              ,
"PSIMEX"                                              ,
"qicharts"                                            ,
"SpatialEpiApp"                                       ,
"LowWAFOMNX"                                          ,
"prcr"                                                ,
"accSDA"                                              ,
"Ecohydmod"                                           ,
"ecoreg"                                              ,
"mExplorer"                                           ,
"multdyn"                                             ,
"noncomplyR"                                          ,
"Strategy"                                            ,
"ActiveDriver"                                        ,
"CpGFilter"                                           ,
"macc"                                                ,
"SCRSELECT"                                           ,
"ConfigParser"                                        ,
"factoextra"                                          ,
"R2DGC"                                               ,
"SLICER"                                              ,
"STARTdesign"                                         ,
"Wrapped"                                             ,
"zic"                                                 ,
"BayesBD"                                             ,
"BeviMed"                                             ,
"InfoTrad"                                            ,
"ircor"                                               ,
"GRCdata"                                             ,
"happybiRthday"                                       ,
"HSAUR2"                                              ,
"alleHap"                                             ,
"ccRemover"                                           ,
"HSAUR"                                               ,
"mHG"                                                 ,
"sasMap"                                              ,
"wppExplorer"                                         ,
"CompDist"                                            ,
"ecoengine"                                           ,
"FuzzyLP"                                             ,
"outbreaker"                                          ,
"zipfR"                                               ,
"AssayCorrector"                                      ,
"diagram"                                             ,
"xmeta"                                               ,
"GrpString"                                           ,
"PeakSegDP"                                           ,
"ReacTran"                                            ,
"samon"                                               ,
"SorptionAnalysis"                                    ,
"CIplot"                                              ,
"ConfoundedMeta"                                      ,
"jtGWAS"                                              ,
"limSolve"                                            ,
"lmmen"                                               ,
"MPLikelihoodWB"                                      ,
"robustrao"                                           ,
"smoof"                                               ,
"OrgMassSpecR"                                        ,
"rdpla"                                               ,
"threejs"                                             ,
"ClustOfVar"                                          ,
"dataMeta"                                            ,
"IAbin"                                               ,
"rasterList"                                          ,
"rcv"                                                 ,
"TreatmentSelection"                                  ,
"BCEE"                                                ,
"SALTSampler"                                         ,
"vocaldia"                                            ,
"gsw"                                                 ,
"IPMRF"                                               ,
"MBSGS"                                               ,
"phylocurve"                                          ,
"poker"                                               ,
"PsyControl"                                          ,
"EbayesThresh"                                        ,
"hydroGOF"                                            ,
"kml3d"                                               ,
"npbr"                                                ,
"plaqr"                                               ,
"skmeans"                                             ,
"spatialkernel"                                       ,
"blink"                                               ,
"default"                                             ,
"dlstats"                                             ,
"ggbeeswarm"                                          ,
"hydroTSM"                                            ,
"multiCA"                                             ,
"permGS"                                              ,
"tsPI"                                                ,
"usl"                                                 ,
"restfulr"                                            ,
"ggversa"                                             ,
"italy"                                               ,
"iterLap"                                             ,
"MCPMod"                                              ,
"r2glmm"                                              ,
"equaltestMI"                                         ,
"GFA"                                                 ,
"MCS"                                                 ,
"soilcarbon"                                          ,
"banR"                                                ,
"echogram"                                            ,
"EMAtools"                                            ,
"glmBfp"                                              ,
"IntegratedJM"                                        ,
"lmmpar"                                              ,
"multistate"                                          ,
"SKAT"                                                ,
"svenssonm"                                           ,
"ambhasGW"                                            ,
"GreedyEPL"                                           ,
"MixtureRegLTIC"                                      ,
"NPflow"                                              ,
"rgeolocate"                                          ,
"ThermIndex"                                          ,
"DoEstRare"                                           ,
"eco"                                                 ,
"fasjem"                                              ,
"kpmt"                                                ,
"structree"                                           ,
"GeNetIt"                                             ,
"GWRM"                                                ,
"LindenmayeR"                                         ,
"nonlinearICP"                                        ,
"RODBCext"                                            ,
"BSDA"                                                ,
"phybreak"                                            ,
"pseudo"                                              ,
"MasterBayes"                                         ,
"rAverage"                                            ,
"RkMetrics"                                           ,
"soilwater"                                           ,
"CDVineCopulaConditional"                             ,
"HiveR"                                               ,
"js"                                                  ,
"mmtsne"                                              ,
"fastR"                                               ,
"lsplsGlm"                                            ,
"mosaicCalc"                                          ,
"optiSolve"                                           ,
"exCon"                                               ,
"iRF"                                                 ,
"maxTPR"                                              ,
"SpecHelpers"                                         ,
"table1xls"                                           ,
"unrepx"                                              ,
"bayeslongitudinal"                                   ,
"CerioliOutlierDetection"                             ,
"concaveman"                                          ,
"gamm4"                                               ,
"ggseqlogo"                                           ,
"MRTSampleSize"                                       ,
"PowerNormal"                                         ,
"qtlmt"                                               ,
"RxnSim"                                              ,
"seqICP"                                              ,
"sidrar"                                              ,
"WhatIf"                                              ,
"EFS"                                                 ,
"GGEBiplots"                                          ,
"gIPFrm"                                              ,
"l0ara"                                               ,
"pcensmix"                                            ,
"aRpsDCA"                                             ,
"c2c"                                                 ,
"geoGAM"                                              ,
"dsmodels"                                            ,
"faoutlier"                                           ,
"prisonbrief"                                         ,
"vcov"                                                ,
"getPass"                                             ,
"IMPACT"                                              ,
"DCA"                                                 ,
"pcaL1"                                               ,
"BayesianGLasso"                                      ,
"corHMM"                                              ,
"Counterfactual"                                      ,
"FastSF"                                              ,
"ivregEX"                                             ,
"OBRE"                                                ,
"refimpact"                                           ,
"sfinx"                                               ,
"stabs"                                               ,
"diffpriv"                                            ,
"riverdist"                                           ,
"rodham"                                              ,
"social"                                              ,
"brlrmr"                                              ,
"Census2016"                                          ,
"cnbdistr"                                            ,
"ercv"                                                ,
"fragilityindex"                                      ,
"LCox"                                                ,
"PearsonDS"                                           ,
"sergeant"                                            ,
"gh"                                                  ,
"senstrat"                                            ,
"vembedr"                                             ,
"centiserve"                                          ,
"DrImpute"                                            ,
"randomForestExplainer"                               ,
"spongecake"                                          ,
"tetraclasse"                                         ,
"TSF"                                                 ,
"ClustGeo"                                            ,
"geotoolsR"                                           ,
"grt"                                                 ,
"ecodist"                                             ,
"ICEbox"                                              ,
"interfr"                                             ,
"mountainplot"                                        ,
"pagenum"                                             ,
"UBL"                                                 ,
"facebook.S4"                                         ,
"JOUSBoost"                                           ,
"MAVIS"                                               ,
"rLiDAR"                                              ,
"SmoothHazard"                                        ,
"multgee"                                             ,
"NameNeedle"                                          ,
"onion"                                               ,
"qcc"                                                 ,
"roots"                                               ,
"VRPM"                                                ,
"AutoDeskR"                                           ,
"dGAselID"                                            ,
"ecr"                                                 ,
"KRLS"                                                ,
"MitISEM"                                             ,
"netcom"                                              ,
"OAIHarvester"                                        ,
"oompaData"                                           ,
"varSelRF"                                            ,
"CRTgeeDR"                                            ,
"decomposedPSF"                                       ,
"fcm"                                                 ,
"postlightmercury"                                    ,
"fcuk"                                                ,
"ggplotgui"                                           ,
"GofKmt"                                              ,
"l1kdeconv"                                           ,
"mixlink"                                             ,
"waccR"                                               ,
"LCMCR"                                               ,
"partitions"                                          ,
"SobolSequence"                                       ,
"subniche"                                            ,
"textile"                                             ,
"Autoplotprotein"                                     ,
"correlbinom"                                         ,
"ggChernoff"                                          ,
"mandelbrot"                                          ,
"Robocoap"                                            ,
"assocInd"                                            ,
"awspack"                                             ,
"bingat"                                              ,
"catSurv"                                             ,
"coxphSGD"                                            ,
"HMPTrees"                                            ,
"landscapeR"                                          ,
"midas"                                               ,
"moko"                                                ,
"numGen"                                              ,
"aws.cloudtrail"                                      ,
"aws.sns"                                             ,
"aws.sqs"                                             ,
"BayesBinMix"                                         ,
"cepR"                                                ,
"cleangeo"                                            ,
"elhmc"                                               ,
"GameTheory"                                          ,
"knitcitations"                                       ,
"mcmcse"                                              ,
"MDplot"                                              ,
"plotSEMM"                                            ,
"SSM"                                                 ,
"wql"                                                 ,
"aurelius"                                            ,
"CovSelHigh"                                          ,
"ecolottery"                                          ,
"mnis"                                                ,
"sos"                                                 ,
"autoBagging"                                         ,
"aws.lambda"                                          ,
"geoSpectral"                                         ,
"hurdlr"                                              ,
"kgschart"                                            ,
"quickregression"                                     ,
"twang"                                               ,
"aws.iam"                                             ,
"bsam"                                                ,
"chunked"                                             ,
"confidence"                                          ,
"fence"                                               ,
"Lock5Data"                                           ,
"MicroMacroMultilevel"                                ,
"SkyWatchr"                                           ,
"BiBitR"                                              ,
"bibtex"                                              ,
"hhh4contacts"                                        ,
"mcPAFit"                                             ,
"OpasnetUtils"                                        ,
"scpm"                                                ,
"StratifiedRF"                                        ,
"BradleyTerryScalable"                                ,
"catmap"                                              ,
"circular"                                            ,
"gsloid"                                              ,
"kazaam"                                              ,
"My.stepwise"                                         ,
"pinnacle.data"                                       ,
"treethresh"                                          ,
"ExpRep"                                              ,
"goldi"                                               ,
"psbcGroup"                                           ,
"rkafka"                                              ,
"SFtools"                                             ,
"sparsio"                                             ,
"sqldf"                                               ,
"BRugs"                                               ,
"FSTpackage"                                          ,
"gmediation"                                          ,
"NCSampling"                                          ,
"Opportunistic"                                       ,
"otrimle"                                             ,
"BayesTwin"                                           ,
"CRANsearcher"                                        ,
"intRvals"                                            ,
"plater"                                              ,
"pltesim"                                             ,
"rattle.data"                                         ,
"twfy"                                                ,
"TPEA"                                                ,
"usdm"                                                ,
"aSPU"                                                ,
"MultiSkew"                                           ,
"RPostgreSQL"                                         ,
"BCgee"                                               ,
"corset"                                              ,
"fsia"                                                ,
"m2r"                                                 ,
"mljar"                                               ,
"nlsrk"                                               ,
"surv2sampleComp"                                     ,
"MOQA"                                                ,
"rMouse"                                              ,
"rrcov3way"                                           ,
"swapClass"                                           ,
"ftDK"                                                ,
"rstream"                                             ,
"anchoredDistr"                                       ,
"mssqlR"                                              ,
"POMaSPU"                                             ,
"qdapTools"                                           ,
"rkafkajars"                                          ,
"RLeafAngle"                                          ,
"taxizedb"                                            ,
"ukbabynames"                                         ,
"clinPK"                                              ,
"PCovR"                                               ,
"PeakError"                                           ,
"Rdroolsjars"                                         ,
"sciplot"                                             ,
"sfadv"                                               ,
"unvotes"                                             ,
"processmonitR"                                       ,
"ECFsup"                                              ,
"secret"                                              ,
"snowFT"                                              ,
"CityWaterBalance"                                    ,
"eiCompare"                                           ,
"phenopix"                                            ,
"pipe.design"                                         ,
"baseballDBR"                                         ,
"dataverse"                                           ,
"GeoRange"                                            ,
"mccr"                                                ,
"pleiades"                                            ,
"TANDEM"                                              ,
"zooaRchGUI"                                          ,
"flacco"                                              ,
"nscprepr"                                            ,
"ZIBseq"                                              ,
"d3Tree"                                              ,
"MfUSampler"                                          ,
"muRL"                                                ,
"UNF"                                                 ,
"argon2"                                              ,
"FAmle"                                               ,
"mix"                                                 ,
"patternator"                                         ,
"polyapost"                                           ,
"pspline"                                             ,
"qualV"                                               ,
"SanFranBeachWater"                                   ,
"standardize"                                         ,
"featurizer"                                          ,
"orderstats"                                          ,
"pse"                                                 ,
"rcdklibs"                                            ,
"colorpatch"                                          ,
"GIGrvg"                                              ,
"janeaustenr"                                         ,
"RNetLogo"                                            ,
"soptdmaeA"                                           ,
"permPATH"                                            ,
"DLASSO"                                              ,
"listWithDefaults"                                    ,
"milr"                                                ,
"mixEMM"                                              ,
"msmtools"                                            ,
"Risk"                                                ,
"xsp"                                                 ,
"gqlr"                                                ,
"hapsim"                                              ,
"influence.ME"                                        ,
"iNOTE"                                               ,
"SPEI"                                                ,
"ZeligChoice"                                         ,
"ZeligEI"                                             ,
"BAYESDEF"                                            ,
"covfefe"                                             ,
"gamlssbssn"                                          ,
"gdata"                                               ,
"gghalfnorm"                                          ,
"gquad"                                               ,
"HDCI"                                                ,
"iadf"                                                ,
"MultiLCIRT"                                          ,
"nomclust"                                            ,
"piton"                                               ,
"pwrAB"                                               ,
"RISmed"                                              ,
"rpst"                                                ,
"themetagenomics"                                     ,
"capn"                                                ,
"netassoc"                                            ,
"pvclass"                                             ,
"ck37r"                                               ,
"eesim"                                               ,
"KRMM"                                                ,
"NB.MClust"                                           ,
"CADFtest"                                            ,
"glmbb"                                               ,
"hypersampleplan"                                     ,
"logOfGamma"                                          ,
"PHENIX"                                              ,
"webglobe"                                            ,
"kerasR"                                              ,
"QPBoot"                                              ,
"baitmet"                                             ,
"difconet"                                            ,
"ElastH"                                              ,
"genpathmox"                                          ,
"ids"                                                 ,
"ImaginR"                                             ,
"kernDeepStackNet"                                    ,
"logistic4p"                                          ,
"pencopulaCond"                                       ,
"rangeBuilder"                                        ,
"cocoreg"                                             ,
"CpGassoc"                                            ,
"forestFloor"                                         ,
 "mdpeer"                                             ,
 "sisVIVE"                                            ,
 "DDM"                                                ,
 "dfCompare"                                          ,
 "multivator"                                         ,
 "phenex"                                             ,
 "phrasemachine"                                      ,
 "RGoogleFit"                                         ,
 "rLTP"                                               ,
 "robumeta"                                           ,
 "squash"                                             ,
 "here"                                               ,
 "SmartSVA"                                           ,
 "didrooRFM"                                          ,
 "MIICD"                                              ,
 "polypoly"                                           ,
 "SeerMapper"                                         ,
 "stratvns"                                           ,
 "tablaxlsx"                                          ,
 "knotR"                                              ,
 "ldbod"                                              ,
 "stratbr"                                            ,
 "tictactoe"                                          ,
 "amen"                                               ,
 "apng"                                               ,
 "biotools"                                           ,
 "RAMP"                                               ,
 "rdoxygen"                                           ,
 "Rfacebook"                                          ,
 "soilphysics"                                        ,
 "timelineR"                                          ,
 "TriadSim"                                           ,
 "lans2r"                                             ,
 "wally"                                              ,
 "x.ent"                                              ,
 "FLLat"                                              ,
 "multisom"                                           ,
 "RHPCBenchmark"                                      ,
 "RNAstructureModuleMiner"                            ,
 "rpatrec"                                            ,
 "samplingbook"                                       ,
 "SubVis"                                             ,
 "xLLiM"                                              ,
 "Rd2md"                                              ,
 "sotu"                                               ,
 "clisymbols"                                         ,
 "IgorR"                                              ,
 "NPMOD"                                              ,
 "plfMA"                                              ,
 "rngWELL"                                            ,
 "AHMbook"                                            ,
 "bgeva"                                              ,
 "CALF"                                               ,
 "CATT"                                               ,
 "checkarg"                                           ,
 "covRobust"                                          ,
 "numKM"                                              ,
 "poisbinom"                                          ,
 "printr"                                             ,
 "afc"                                                ,
 "complexplus"                                        ,
 "flexrsurv"                                          ,
 "GESE"                                               ,
 "GUIgems"                                            ,
 "powerplus"                                          ,
 "REDCapR"                                            ,
 "SSRMST"                                             ,
 "wCorr"                                              ,
 "easyNCDF"                                           ,
 "ROI.plugin.ipop"                                    ,
 "ROI.plugin.quadprog"                                ,
 "ROI.plugin.symphony"                                ,
 "SemiParSampleSel"                                   ,
 "zeligverse"                                         ,
 "distcomp"                                           ,
 "ExcessMass"                                         ,
 "FatTailsR"                                          ,
 "PROscorer"                                          ,
 "rif"                                                ,
 "sotkanet"                                           ,
 "coreSim"                                            ,
 "PairwiseD"                                          ,
 "PROscorerTools"                                     ,
 "RBPcurve"                                           ,
 "simPH"                                              ,
 "slim"                                               ,
 "gomms"                                              ,
 "learnrbook"                                         ,
 "diffMeshGP"                                         ,
 "EpistemicGameTheory"                                ,
 "fastTextR"                                          ,
 "globe"                                              ,
 "iGSEA"                                              ,
 "jocre"                                              ,
 "lifelogr"                                           ,
 "rel"                                                ,
 "shinycssloaders"                                    ,
 "sinib"                                              ,
 "usedist"                                            ,
 "iECAT"                                              ,
 "lhmixr"                                             ,
 "pawls"                                              ,
 "RGCCA"                                              ,
 "TransModel"                                         ,
 "bnnSurvival"                                        ,
 "CADStat"                                            ,
 "lindia"                                             ,
 "lmmot"                                              ,
 "lsasim"                                             ,
 "nephro"                                             ,
 "optband"                                            ,
 "qrsvm"                                              ,
 "TSdbi"                                              ,
 "TSsql"                                              ,
 "AcuityView"                                         ,
 "DIFlasso"                                           ,
 "plotprotein"                                        ,
 "sourceR"                                            ,
 "TestFunctions"                                      ,
 "yhatr"                                              ,
 "bit64"                                              ,
 "clustMD"                                            ,
 "IndianTaxCalc"                                      ,
 "MaxSkew"                                            ,
 "RPEXE.RPEXT"                                        ,
 "ACEt"                                               ,
 "aoos"                                               ,
 "chi"                                                ,
 "invgamma"                                           ,
 "efreadr"                                            ,
 "glmmLasso"                                          ,
 "PROreg"                                             ,
 "pwr2"                                               ,
 "cowbell"                                            ,
 "OneR"                                               ,
 "biglasso"                                           ,
 "compHclust"                                         ,
 "DataGraph"                                          ,
 "idar"                                               ,
 "LDcorSV"                                            ,
 "PBD"                                                ,
 "seasonalview"                                       ,
 "TideHarmonics"                                      ,
 "m2b"                                                ,
 "matrixpls"                                          ,
 "proportion"                                         ,
 "qrmix"                                              ,
 "discord"                                            ,
 "dsrTest"                                            ,
 "KSEAapp"                                            ,
 "mut"                                                ,
 "ncdump"                                             ,
 "onehot"                                             ,
 "anacor"                                             ,
 "angstroms"                                          ,
 "cfa"                                                ,
 "doMPI"                                              ,
 "ibr"                                                ,
 "MHTmult"                                            ,
 "MultivariateRandomForest"                           ,
 "nlsem"                                              ,
 "virustotal"                                         ,
 "cubfits"                                            ,
 "DDRTree"                                            ,
 "pafdR"                                              ,
 "SPADAR"                                             ,
 "ChocoLattes"                                        ,
 "dataseries"                                         ,
 "MixSim"                                             ,
 "mockr"                                              ,
 "pim"                                                ,
 "casebase"                                           ,
 "ConsRank"                                           ,
 "htmltools"                                          ,
 "PerMallows"                                         ,
 "waterData"                                          ,
 "WikidataQueryServiceR"                              ,
 "aSPC"                                               ,
 "genie"                                              ,
 "lm.br"                                              ,
 "msos"                                               ,
 "mstherm"                                            ,
 "NonpModelCheck"                                     ,
 "visualize"                                          ,
 "bulletr"                                            ,
 "coxphMIC"                                           ,
 "md.log"                                             ,
 "plink"                                              ,
 "censCov"                                            ,
 "RandMeta"                                           ,
 "rebus"                                              ,
 "rebus.base"                                         ,
 "simest"                                             ,
 "ajv"                                                ,
 "BarBorGradient"                                     ,
 "ider"                                               ,
 "lamW"                                               ,
 "nonpar"                                             ,
 "PGEE"                                               ,
 "ring"                                               ,
 "sem"                                                ,
 "npROCRegression"                                    ,
 "plspm"                                              ,
 "PSF"                                                ,
 "SparseM"                                            ,
 "readBrukerFlexData"                                 ,
 "startR"                                             ,
 "chemmodlab"                                         ,
 "CrossScreening"                                     ,
 "DMRMark"                                            ,
 "KENDL"                                              ,
 "memoise"                                            ,
 "regress"                                            ,
 "RpeakChrom"                                         ,
 "simMP"                                              ,
 "sinaplot"                                           ,
 "LOST"                                               ,
 "validann"                                           ,
 "paleomorph"                                         ,
 "PCADSC"                                             ,
 "rAltmetric"                                         ,
 "ss3sim"                                             ,
 "yarrr"                                              ,
 "acnr"                                               ,
 "icosa"                                              ,
 "longCatEDA"                                         ,
 "Rpoppler"                                           ,
 "TSGSIS"                                             ,
 "uqr"                                                ,
 "WLreg"                                              ,
 "icamix"                                             ,
 "largeList"                                          ,
 "mro"                                                ,
 "rotationForest"                                     ,
 "splancs"                                            ,
 "captr"                                              ,
 "complexity"                                         ,
 "ConvergenceConcepts"                                ,
 "tseriesEntropy"                                     ,
 "bayesCL"                                            ,
 "nortestARMA"                                        ,
 "scidb"                                              ,
 "SeerMapper2010East"                                 ,
 "SeerMapper2010West"                                 ,
 "StatPerMeCo"                                        ,
 "thankr"                                             ,
 "DAISIE"                                             ,
 "fitbitScraper"                                      ,
 "fpest"                                              ,
 "R2ucare"                                            ,
 "SeerMapper2010Regs"                                 ,
 "Tnseq"                                              ,
 "tubern"                                             ,
 "ANOM"                                               ,
 "clarifai"                                           ,
 "CompQuadForm"                                       ,
 "marcher"                                            ,
 "optrcdmaeAT"                                        ,
 "OSMscale"                                           ,
 "r.jive"                                             ,
 "rbhl"                                               ,
 "SeerMapperEast"                                     ,
 "SeerMapperWest"                                     ,
 "stddiff"                                            ,
 "GADAG"                                              ,
 "strataG"                                            ,
 "D3GB"                                               ,
 "lspline"                                            ,
 "partools"                                           ,
 "SeerMapperRegs"                                     ,
 "tlm"                                                ,
 "Grace"                                              ,
 "qdapRegex"                                          ,
 "simpleRCache"                                       ,
 "skeletor"                                           ,
 "tpAUC"                                              ,
 "duckduckr"                                          ,
 "IMTest"                                             ,
 "IRTShiny"                                           ,
 "lavaan.shiny"                                       ,
 "micEconIndex"                                       ,
 "secure"                                             ,
 "classifierplots"                                    ,
 "CoFRA"                                              ,
 "eshrink"                                            ,
 "fit.models"                                         ,
 "mmod"                                               ,
 "zfa"                                                ,
 "jetset"                                             ,
 "stablespec"                                         ,
 "diveRsity"                                          ,
 "HYRISK"                                             ,
 "LassoBacktracking"                                  ,
 "sensitivityfull"                                    ,
 "TSMN"                                               ,
 "afmToolkit"                                         ,
 "downsize"                                           ,
 "FSInteract"                                         ,
 "goftest"                                            ,
 "groupsubsetselection"                               ,
 "HMR"                                                ,
 "nlshelper"                                          ,
 "RCEIM"                                              ,
 "RPtests"                                            ,
 "RsimMosaic"                                         ,
 "scuba"                                              ,
 "vesselr"                                            ,
 "xbreed"                                             ,
 "xyz"                                                ,
 "medicare"                                           ,
 "miscor"                                             ,
 "corpcor"                                            ,
 "grapes"                                             ,
 "hgm"                                                ,
 "InterfaceqPCR"                                      ,
 "sss"                                                ,
 "tnam"                                               ,
 "aptg"                                               ,
 "cobs"                                               ,
 "datautils"                                          ,
 "focusedMDS"                                         ,
 "gRim"                                               ,
 "lava.tobit"                                         ,
 "NADA"                                               ,
 "PCGSE"                                              ,
 "RXKCD"                                              ,
 "xergm.common"                                       ,
 "optim.functions"                                    ,
 "smnet"                                              ,
 "sodium"                                             ,
 "BLModel"                                            ,
 "care"                                               ,
 "fam2r"                                              ,
 "metavcov"                                           ,
 "moezipfR"                                           ,
 "odeintr"                                            ,
 "tilegramsR"                                         ,
 "tsqn"                                               ,
 "itan"                                               ,
 "mmeta"                                              ,
 "sejmRP"                                             ,
 "shinyHeatmaply"                                     ,
 "XBRL"                                               ,
 "dc3net"                                             ,
 "pooh"                                               ,
 "qrcm"                                               ,
 "ump"                                                ,
 "aster2"                                             ,
 "frair"                                              ,
 "gems"                                               ,
 "MVLM"                                               ,
 "NetworkRiskMeasures"                                ,
 "fftwtools"                                          ,
 "slimrec"                                            ,
 "TeachBayes"                                         ,
 "bkmr"                                               ,
 "docstring"                                          ,
 "FuzzyR"                                             ,
 "IPWsurvival"                                        ,
 "metagear"                                           ,
 "PCFAM"                                              ,
 "PortfolioEffectHFT"                                 ,
 "rriskDistributions"                                 ,
 "affluenceIndex"                                     ,
 "diverse"                                            ,
 "simctest"                                           ,
 "summariser"                                         ,
 "CVR"                                                ,
 "neuropsychology"                                    ,
 "pathmapping"                                        ,
 "vipor"                                              ,
 "xlsimple"                                           ,
 "addhazard"                                          ,
 "averisk"                                            ,
 "censReg"                                            ,
 "distory"                                            ,
 "ega"                                                ,
 "FMC"                                                ,
 "orgutils"                                           ,
 "qrLMM"                                              ,
 "qrNLMM"                                             ,
 "rnaturalearth"                                      ,
 "rolr"                                               ,
 "disposables"                                        ,
 "EstimateGroupNetwork"                               ,
 "evclass"                                            ,
 "LW1949"                                             ,
 "Rvoterdistance"                                     ,
 "sadists"                                            ,
 "sticky"                                             ,
 "fasteraster"                                        ,
 "shinyShortcut"                                      ,
 "networkD3"                                          ,
 "OSCV"                                               ,
 "pcrsim"                                             ,
 "PDQutils"                                           ,
 "adhoc"                                              ,
 "BASS"                                               ,
 "chemometrics"                                       ,
 "frontier"                                           ,
 "BCSub"                                              ,
 "cir"                                                ,
 "CNull"                                              ,
 "FlexDir"                                            ,
 "micEcon"                                            ,
 "micEconAids"                                        ,
 "qqman"                                              ,
 "RankingProject"                                     ,
 "rolypoly"                                           ,
 "dynaTree"                                           ,
 "fuzzyRankTests"                                     ,
 "gee4"                                               ,
 "pbkrtest"                                           ,
 "clusternomics"                                      ,
 "copCAR"                                             ,
 "gatepoints"                                         ,
 "linbin"                                             ,
 "outreg"                                             ,
 "simboot"                                            ,
 "ABCanalysis"                                        ,
 "collpcm"                                            ,
 "gazepath"                                           ,
 "optCluster"                                         ,
 "ProfessR"                                           ,
 "radmixture"                                         ,
 "revealjs"                                           ,
 "WhopGenome"                                         ,
 "FastImputation"                                     ,
 "hydrogeo"                                           ,
 "cmpprocess"                                         ,
 "iNextPD"                                            ,
 "stratification"                                     ,
 "BBmisc"                                             ,
 "DREGAR"                                             ,
 "mixtools"                                           ,
 "rinat"                                              ,
 "SigOptR"                                            ,
 "CONS"                                               ,
 "DAAGbio"                                            ,
 "emon"                                               ,
 "gnumeric"                                           ,
 "iSDM"                                               ,
 "SymTS"                                              ,
 "Ultimixt"                                           ,
 "VertexSort"                                         ,
 "BCellMA"                                            ,
 "binomSamSize"                                       ,
 "CatEncoders"                                        ,
 "MBA"                                                ,
 "NHLData"                                            ,
 "prop.comb.RR"                                       ,
 "readOffice"                                         ,
 "STMedianPolish"                                     ,
 "ALSM"                                               ,
 "cpr"                                                ,
 "GDAtools"                                           ,
 "otinference"                                        ,
 "remindR"                                            ,
 "rkt"                                                ,
 "SpecsVerification"                                  ,
 "DySeq"                                              ,
 "edarf"                                              ,
 "funchir"                                            ,
 "gt4ireval"                                          ,
 "mc2d"                                               ,
 "MLID"                                               ,
 "twilio"                                             ,
 "VarED"                                              ,
 "CPsurv"                                             ,
 "SubgrpID"                                           ,
 "CensSpatial"                                        ,
 "cghRA"                                              ,
 "Demerelate"                                         ,
 "Dpit"                                               ,
 "MDPtoolbox"                                         ,
 "texreg"                                             ,
 "cropdatape"                                         ,
 "DiversityOccupancy"                                 ,
 "erp.easy"                                           ,
 "gaiah"                                              ,
 "isa2"                                               ,
 "keypress"                                           ,
 "maxstat"                                            ,
 "pingr"                                              ,
 "sand"                                               ,
 "WPKDE"                                              ,
 "dotenv"                                             ,
 "pssm"                                               ,
 "R4CouchDB"                                          ,
 "DiffNet"                                            ,
 "GLDreg"                                             ,
 "operator.tools"                                     ,
 "rcc"                                                ,
 "RPMM"                                               ,
 "bgmm"                                               ,
 "diffr"                                              ,
 "dirmcmc"                                            ,
 "extdplyr"                                           ,
 "fontHind"                                           ,
 "fontMPlus"                                          ,
 "gbts"                                               ,
 "leafletCN"                                          ,
 "misreport"                                          ,
 "qap"                                                ,
 "AWR.Kinesis"                                        ,
 "IDPSurvival"                                        ,
 "betacal"                                            ,
 "recexcavAAR"                                        ,
 "ccafs"                                              ,
 "europop"                                            ,
 "falconx"                                            ,
 "hasseDiagram"                                       ,
 "miscset"                                            ,
 "som.nn"                                             ,
 "TAR"                                                ,
 "EnviroPRA"                                          ,
 "GsymPoint"                                          ,
 "InvasionCorrection"                                 ,
 "naptime"                                            ,
 "prefeR"                                             ,
 "robustarima"                                        ,
 "BayesSpec"                                          ,
 "BCC1997"                                            ,
 "bsearchtools"                                       ,
 "choroplethrAdmin1"                                  ,
 "DecorateR"                                          ,
 "idendr0"                                            ,
 "Mangrove"                                           ,
 "R2OpenBUGS"                                         ,
 "RWiener"                                            ,
 "lclGWAS"                                            ,
 "mle.tools"                                          ,
 "rnaturalearthdata"                                  ,
 "stdvectors"                                         ,
 "AWR.KMS"                                            ,
 "BayesFM"                                            ,
 "CEoptim"                                            ,
 "colorr"                                             ,
 "fileplyr"                                           ,
 "genemodel"                                          ,
 "GSparO"                                             ,
 "OxyBS"                                              ,
 "pawacc"                                             ,
 "survRM2"                                            ,
 "W3CMarkupValidator"                                 ,
 "ROCt"                                               ,
 "upmfit"                                             ,
 "AMCP"                                               ,
 "helsinki"                                           ,
 "mixtox"                                             ,
 "yorkr"                                              ,
 "BDWreg"                                             ,
 "educineq"                                           ,
 "EMMLi"                                              ,
 "keyringr"                                           ,
 "marinespeed"                                        ,
 "pca3d"                                              ,
 "riverplot"                                          ,
 "hqreg"                                              ,
 "mdscore"                                            ,
 "opusminer"                                          ,
 "ridigbio"                                           ,
 "rope"                                               ,
 "spdownscale"                                        ,
 "condir"                                             ,
 "ggalt"                                              ,
 "warpMix"                                            ,
 "dgodata"                                            ,
 "eefAnalytics"                                       ,
 "emIRT"                                              ,
 "paperplanes"                                        ,
 "rapiclient"                                         ,
 "cope"                                               ,
 "LiblineaR"                                          ,
 "dynBiplotGUI"                                       ,
 "fma"                                                ,
 "HRQoL"                                              ,
 "imputeMulti"                                        ,
 "purge"                                              ,
 "DNAtools"                                           ,
 "maxnet"                                             ,
 "RGENERATE"                                          ,
 "RGENERATEPREC"                                      ,
 "RMAWGEN"                                            ,
 "StVAR"                                              ,
 "optbdmaeAT"                                         ,
 "seqMeta"                                            ,
 "BayesCombo"                                         ,
 "FuzzyStatTra"                                       ,
 "OCA"                                                ,
 "arabicStemR"                                        ,
 "GPB"                                                ,
 "pMineR"                                             ,
 "cbird"                                              ,
 "CompareTests"                                       ,
 "mdftracks"                                          ,
 "QuantileGradeR"                                     ,
 "samplingDataCRT"                                    ,
 "SMLoutliers"                                        ,
 "supportInt"                                         ,
 "treebase"                                           ,
 "AdMit"                                              ,
 "mbrglm"                                             ,
 "random"                                             ,
 "WikipediR"                                          ,
 "bayesGARCH"                                         ,
 "flippant"                                           ,
 "samplesizelogisticcasecontrol"                      ,
 "superheat"                                          ,
 "tolerance"                                          ,
 "BAMMtools"                                          ,
 "PST"                                                ,
 "mpe"                                                ,
 "powerAnalysis"                                      ,
 "readHAC"                                            ,
 "demoKde"                                            ,
 "EnsCat"                                             ,
 "fontBitstreamVera"                                  ,
 "fontquiver"                                         ,
 "ShinyTester"                                        ,
 "sonify"                                             ,
 "ssmsn"                                              ,
 "blendedLink"                                        ,
 "choroplethrMaps"                                    ,
 "edstan"                                             ,
 "iBST"                                               ,
 "bunchr"                                             ,
 "flood"                                              ,
 "gunsales"                                           ,
 "inspectr"                                           ,
 "observer"                                           ,
 "poio"                                               ,
 "TTCA"                                               ,
 "alphabetr"                                          ,
 "fastmatch"                                          ,
 "gbp"                                                ,
 "iosmooth"                                           ,
 "tracer"                                             ,
 "bayesQR"                                            ,
 "cooccurNet"                                         ,
 "landsat8"                                           ,
 "sparsestep"                                         ,
 "BayesPieceHazSelect"                                ,
 "BiSEp"                                              ,
 "eclust"                                             ,
 "nparsurv"                                           ,
 "Rcereal"                                            ,
 "givitiR"                                            ,
 "mutSignatures"                                      ,
 "NORMA"                                              ,
 "powerEQTL"                                          ,
 "rKIN"                                               ,
 "simpleSetup"                                        ,
 "chi2x3way"                                          ,
 "errint"                                             ,
 "mads"                                               ,
 "RNCEP"                                              ,
 "skm"                                                ,
 "ALDqr"                                              ,
 "ICV"                                                ,
 "intrval"                                            ,
 "kpcalg"                                             ,
 "maxlike"                                            ,
 "SOFIA"                                              ,
 "tea"                                                ,
 "teda"                                               ,
 "chords"                                             ,
 "rcure"                                              ,
 "testforDEP"                                         ,
 "bytescircle"                                        ,
 "rUnemploymentData"                                  ,
 "rgho"                                               ,
 "bayesianETAS"                                       ,
 "cclust"                                             ,
 "mangoTraining"                                      ,
 "microclass"                                         ,
 "rCAT"                                               ,
 "rorutadis"                                          ,
 "cumstats"                                           ,
 "peakRAM"                                            ,
 "maxmatching"                                        ,
 "mhsmm"                                              ,
 "PoSI"                                               ,
 "dfphase1"                                           ,
 "InfiniumPurify"                                     ,
 "isva"                                               ,
 "marima"                                             ,
 "PhyloMeasures"                                      ,
 "rwars"                                              ,
 "GeneClusterNet"                                     ,
 "GORCure"                                            ,
 "ICGOR"                                              ,
 "SADEG"                                              ,
 "sparkwarc"                                          ,
 "TRADER"                                             ,
 "boxr"                                               ,
 "EBrank"                                             ,
 "pxR"                                                ,
 "RChronoModel"                                       ,
 "rld"                                                ,
 "esmisc"                                             ,
 "QRank"                                              ,
 "RND"                                                ,
 "leaps"                                              ,
 "qfasar"                                             ,
 "regclass"                                           ,
 "SAVE"                                               ,
 "USGSstates2k"                                       ,
 "VGAMdata"                                           ,
 "BoolFilter"                                         ,
 "dismo"                                              ,
 "esaddle"                                            ,
 "npmv"                                               ,
 "rjsonapi"                                           ,
 "speedglm"                                           ,
 "lowmemtkmeans"                                      ,
 "caseMatch"                                          ,
 "depth"                                              ,
 "MAMSE"                                              ,
 "panelaggregation"                                   ,
 "sysid"                                              ,
 "waffle"                                             ,
 "biogram"                                            ,
 "DFIT"                                               ,
 "eHOF"                                               ,
 "fabCI"                                              ,
 "ltxsparklines"                                      ,
 "pamctdp"                                            ,
 "Plasmidprofiler"                                    ,
 "semver"                                             ,
 "crisp"                                              ,
 "etasFLP"                                            ,
 "FixSeqMTP"                                          ,
 "hail"                                               ,
 "MRCE"                                               ,
 "partDSA"                                            ,
 "requireR"                                           ,
 "RWinEdt"                                            ,
 "TBSSurvival"                                        ,
 "BayesPiecewiseICAR"                                 ,
 "dSVA"                                               ,
 "minimalRSD"                                         ,
 "pwt8"                                               ,
 "Redmonder"                                          ,
 "treemap"                                            ,
 "tsdecomp"                                           ,
 "TSeriesMMA"                                         ,
 "colr"                                               ,
 "openadds"                                           ,
 "QCAtools"                                           ,
 "reader"                                             ,
 "rebus.unicode"                                      ,
 "rneos"                                              ,
 "abundant"                                           ,
 "awsjavasdk"                                         ,
 "assertive"                                          ,
 "censys"                                             ,
 "futureheatwaves"                                    ,
 "grpregOverlap"                                      ,
 "hds"                                                ,
 "likert"                                             ,
 "assertive.base"                                     ,
 "assertive.properties"                               ,
 "assertive.reflection"                               ,
 "assertive.sets"                                     ,
 "assertive.types"                                    ,
 "bootruin"                                           ,
 "bossMaps"                                           ,
 "finiteruinprob"                                     ,
 "GCPM"                                               ,
 "gsheet"                                             ,
 "RcppMLPACK"                                         ,
 "uniah"                                              ,
 "fortunes"                                           ,
 "htmltab"                                            ,
 "attrCUSUM"                                          ,
 "nplr"                                               ,
 "acrt"                                               ,
 "bayesloglin"                                        ,
 "Bclim"                                              ,
 "DStree"                                             ,
 "mRm"                                                ,
 "restrictedMVN"                                      ,
 "rpubchem"                                           ,
 "SCBmeanfd"                                          ,
 "spark.sas7bdat"                                     ,
 "UStatBookABSC"                                      ,
 "DISTRIB"                                            ,
 "factorcpt"                                          ,
 "tilting"                                            ,
 "findviews"                                          ,
 "ggguitar"                                           ,
 "samplesize"                                         ,
 "gemtc"                                              ,
 "Rarity"                                             ,
 "ztype"                                              ,
 "concreg"                                            ,
 "heritability"                                       ,
 "lavaan.survey"                                      ,
 "rowr"                                               ,
 "sampling"                                           ,
 "condSURV"                                           ,
 "crosstalk"                                          ,
 "ms.sev"                                             ,
 "pgee.mixed"                                         ,
 "rocNIT"                                             ,
 "SpaTimeClus"                                        ,
 "akima"                                              ,
 "aws.ses"                                            ,
 "OData"                                              ,
 "radarchart"                                         ,
 "scdhlm"                                             ,
 "SPPcomb"                                            ,
 "stmgui"                                             ,
 "bcRep"                                              ,
 "DEoptim"                                            ,
 "pipeliner"                                          ,
 "DirectStandardisation"                              ,
 "longurl"                                            ,
 "magicfor"                                           ,
 "tigerstats"                                         ,
 "MTDrh"                                              ,
 "MWRidge"                                            ,
 "PepSAVIms"                                          ,
 "PGM2"                                               ,
 "ramify"                                             ,
 "acc"                                                ,
 "binseqtest"                                         ,
 "Ecdat"                                              ,
 "FinancialMath"                                      ,
 "polyfreqs"                                          ,
 "sjdbc"                                              ,
 "tripack"                                            ,
 "coprimary"                                          ,
 "KoNLP"                                              ,
 "MvBinary"                                           ,
 "skellam"                                            ,
 "tibbrConnector"                                     ,
 "tmvnsim"                                            ,
 "activpalProcessing"                                 ,
 "BTYDplus"                                           ,
 "wux"                                                ,
 "admixturegraph"                                     ,
 "bpp"                                                ,
 "fanc"                                               ,
 "GrapheR"                                            ,
 "MODISSnow"                                          ,
 "patchSynctex"                                       ,
 "poptrend"                                           ,
 "prclust"                                            ,
 "sparkTable"                                         ,
 "FRAPO"                                              ,
 "ampd"                                               ,
 "CollapsABEL"                                        ,
 "DNLC"                                               ,
 "errorizer"                                          ,
 "pointRes"                                           ,
 "reinstallr"                                         ,
 "desiR"                                              ,
 "rtrends"                                            ,
 "GLMaSPU"                                            ,
 "INSPIRE"                                            ,
 "isocir"                                             ,
 "magree"                                             ,
 "MixedPoisson"                                       ,
 "rrr"                                                ,
 "ShapeChange"                                        ,
 "aws.polly"                                          ,
 "geomnet"                                            ,
 "gofMC"                                              ,
 "internetarchive"                                    ,
 "micromapST"                                         ,
 "oz"                                                 ,
 "Radviz"                                             ,
 "DCM"                                                ,
 "madsim"                                             ,
 "seeclickfixr"                                       ,
 "Branching"                                          ,
 "bvpSolve"                                           ,
 "rccmisc"                                            ,
 "robustsae"                                          ,
 "rootSolve"                                          ,
 "SafeQuant"                                          ,
 "AmmoniaConcentration"                               ,
 "anchors"                                            ,
 "bridger2"                                           ,
 "Causata"                                            ,
 "diptest"                                            ,
 "DrBats"                                             ,
 "easypackages"                                       ,
 "ezec"                                               ,
 "FAwR"                                               ,
 "flows"                                              ,
 "gafit"                                              ,
 "gapmap"                                             ,
 "ggparallel"                                         ,
 "IHSEP"                                              ,
 "intergraph"                                         ,
 "labstats"                                           ,
 "melviewr"                                           ,
 "MultiBD"                                            ,
 "mvngGrAd"                                           ,
 "mvtboost"                                           ,
 "networkreporting"                                   ,
 "PANDA"                                              ,
 "rt3"                                                ,
 "tswge"                                              ,
 "cmaesr"                                             ,
 "lazyData"                                           ,
 "phuassess"                                          ,
 "SISIR"                                              ,
 "rhnerm"                                             ,
 "blockTools"                                         ,
 "ClimDown"                                           ,
 "cshapes"                                            ,
 "pinfsc50"                                           ,
 "vkR"                                                ,
 "apc"                                                ,
 "freqdist"                                           ,
 "gbs2ploidy"                                         ,
 "rxSeq"                                              ,
 "camsRad"                                            ,
 "eMLEloglin"                                         ,
 "MotilityLab"                                        ,
 "listdtr"                                            ,
 "optR"                                               ,
 "Przewodnik"                                         ,
 "RNHANES"                                            ,
 "testassay"                                          ,
 "tsdisagg2"                                          ,
 "bc3net"                                             ,
 "CTM"                                                ,
 "CVD"                                                ,
 "nice"                                               ,
 "textreuse"                                          ,
 "GeomComb"                                           ,
 "notifyme"                                           ,
 "omics"                                              ,
 "DALY"                                               ,
 "FITSio"                                             ,
 "xVA"                                                ,
 "aslib"                                              ,
 "dynetNLAResistance"                                 ,
 "miscTools"                                          ,
 "multicool"                                          ,
 "rDEA"                                               ,
 "rIsing"                                             ,
 "rlo"                                                ,
 "survAWKMT2"                                         ,
 "AnalyzeTS"                                          ,
 "quarrint"                                           ,
 "RDStreeboot"                                        ,
 "CombinS"                                            ,
 "epistasis"                                          ,
 "FBFsearch"                                          ,
 "gridsampler"                                        ,
 "ph2mult"                                            ,
 "pumilioR"                                           ,
 "pushoverr"                                          ,
 "SpaCCr"                                             ,
 "strat"                                              ,
 "bayesDem"                                           ,
 "mbclusterwise"                                      ,
 "ockc"                                               ,
 "ParallelTree"                                       ,
 "Rdtq"                                               ,
 "SeleMix"                                            ,
 "ebdbNet"                                            ,
 "HAC"                                                ,
 "NFP"                                                ,
 "relabeLoadings"                                     ,
 "ADCT"                                               ,
 "grpSLOPE"                                           ,
 "minpack.lm"                                         ,
 "climbeR"                                            ,
 "DEoptimR"                                           ,
 "nhstplot"                                           ,
 "meboot"                                             ,
 "DGCA"                                               ,
 "gwfa"                                               ,
 "hit"                                                ,
 "NegBinBetaBinreg"                                   ,
 "tinsel"                                             ,
 "udunits2"                                           ,
 "ibmcraftr"                                          ,
 "manhattanly"                                        ,
 "apdesign"                                           ,
 "colormap"                                           ,
 "fdrci"                                              ,
 "WEE"                                                ,
 "httpcode"                                           ,
 "localgauss"                                         ,
 "locits"                                             ,
 "miscFuncs"                                          ,
 "peacots"                                            ,
 "PhyInformR"                                         ,
 "pkgKitten"                                          ,
 "refund.shiny"                                       ,
 "CSclone"                                            ,
 "flock"                                              ,
 "jointNmix"                                          ,
 "LEANR"                                              ,
 "Planesmuestra"                                      ,
 "ptest"                                              ,
 "qiitr"                                              ,
 "quantification"                                     ,
 "RcmdrPlugin.TeachingDemos"                          ,
 "sparkline"                                          ,
 "XRSCC"                                              ,
 "obAnalytics"                                        ,
 "rrcovHD"                                            ,
 "cthreshER"                                          ,
 "exteriorMatch"                                      ,
 "RSIP"                                               ,
 "HARtools"                                           ,
 "MCMC.qpcr"                                          ,
 "msaFACE"                                            ,
 "MSGLasso"                                           ,
 "pirate"                                             ,
 "rehh.data"                                          ,
 "wvtool"                                             ,
 "CollocInfer"                                        ,
 "diffrprojectswidget"                                ,
 "OptimalDesign"                                      ,
 "Rga4gh"                                             ,
 "SparseTSCGM"                                        ,
 "starmie"                                            ,
 "varband"                                            ,
 "diffrprojects"                                      ,
 "pch"                                                ,
 "dams"                                               ,
 "ei"                                                 ,
 "gettz"                                              ,
 "varSel"                                             ,
 "ordcrm"                                             ,
 "packagedocs"                                        ,
 "RLRsim"                                             ,
 "ADDT"                                               ,
 "MiRAnorm"                                           ,
 "PhViD"                                              ,
 "ConnMatTools"                                       ,
 "ez"                                                 ,
 "BMRV"                                               ,
 "dtables"                                            ,
 "stringb"                                            ,
 "finreportr"                                         ,
 "ManyTests"                                          ,
 "rRAP"                                               ,
 "AdjBQR"                                             ,
 "gtheory"                                            ,
 "InformationValue"                                   ,
 "mhtboot"                                            ,
 "RDota2"                                             ,
 "Rlinkedin"                                          ,
 "acepack"                                            ,
 "endogenous"                                         ,
 "ggswissmaps"                                        ,
 "IMFData"                                            ,
 "ltbayes"                                            ,
 "proto"                                              ,
 "Ramble"                                             ,
 "SpeciesMix"                                         ,
 "spftir"                                             ,
 "triangulation"                                      ,
 "MIDN"                                               ,
 "ontologyPlot"                                       ,
 "ontologySimilarity"                                 ,
 "PatternClass"                                       ,
 "ShapeSelectForest"                                  ,
 "spellcheckr"                                        ,
 "fizzbuzzR"                                          ,
 "RStata"                                             ,
 "countyweather"                                      ,
 "fasttime"                                           ,
 "iRafNet"                                            ,
 "IRATER"                                             ,
 "ROracle"                                            ,
 "ShapePattern"                                       ,
 "DDHFm"                                              ,
 "feedeR"                                             ,
 "GeneticSubsetter"                                   ,
 "gym"                                                ,
 "homeR"                                              ,
 "JRF"                                                ,
 "pbdDEMO"                                            ,
 "sns"                                                ,
 "graphicsQC"                                         ,
 "bacr"                                               ,
 "ClickClust"                                         ,
 "rchallenge"                                         ,
 "stsm"                                               ,
 "countytimezones"                                    ,
 "Exact"                                              ,
 "RI2by2"                                             ,
 "Rknots"                                             ,
 "scmamp"                                             ,
 "ABC.RAP"                                            ,
 "aplore3"                                            ,
 "Barnard"                                            ,
 "corlink"                                            ,
 "ggloop"                                             ,
 "MullerPlot"                                         ,
 "Rsurrogate"                                         ,
 "SafeBayes"                                          ,
 "validatejsonr"                                      ,
 "wavethresh"                                         ,
 "COUNT"                                              ,
 "gamlss.add"                                         ,
 "PCSinR"                                             ,
 "BarcodingR"                                         ,
 "cryst"                                              ,
 "NPCD"                                               ,
 "sirad"                                              ,
 "TDPanalysis"                                        ,
 "vdg"                                                ,
 "VIMGUI"                                             ,
 "gRain"                                              ,
 "GraphFactor"                                        ,
 "metafuse"                                           ,
 "nmfgpu4R"                                           ,
 "pageviews"                                          ,
 "betategarch"                                        ,
 "KnapsackSampling"                                   ,
 "ssym"                                               ,
 "ADPclust"                                           ,
 "fontLiberation"                                     ,
 "mnormt"                                             ,
 "RSarules"                                           ,
 "SimReg"                                             ,
 "types"                                              ,
 "clues"                                              ,
 "dbfaker"                                            ,
 "graphscan"                                          ,
 "highmean"                                           ,
 "in2extRemes"                                        ,
 "ISR3"                                               ,
 "tutorial"                                           ,
 "convertr"                                           ,
 "DMwR2"                                              ,
 "ENiRG"                                              ,
 "MetaAnalyser"                                       ,
 "performanceEstimation"                              ,
 "daewr"                                              ,
 "MethodCompare"                                      ,
 "mpr"                                                ,
 "rbokeh"                                             ,
 "exp2flux"                                           ,
 "lokern"                                             ,
 "regspec"                                            ,
 "haploReconstruct"                                   ,
 "mcprofile"                                          ,
 "rgw"                                                ,
 "QQperm"                                             ,
 "reldist"                                            ,
 "selectiongain"                                      ,
 "valorate"                                           ,
 "extremogram"                                        ,
 "ptwikiwords"                                        ,
 "machina"                                            ,
 "RSeed"                                              ,
 "sharpeRratio"                                       ,
 "tsSelect"                                           ,
 "RAMpath"                                            ,
 "sAIC"                                               ,
 "SAPP"                                               ,
 "changepoint"                                        ,
 "ordDisp"                                            ,
 "spcr"                                               ,
 "tigerhitteR"                                        ,
 "EBASS"                                              ,
 "fixedTimeEvents"                                    ,
 "expandFunctions"                                    ,
 "LANDD"                                              ,
 "linkR"                                              ,
 "sBIC"                                               ,
 "seleniumPipes"                                      ,
 "supervisedPRIM"                                     ,
 "sublime"                                            ,
 "BigSEM"                                             ,
 "DPBBM"                                              ,
 "spcadjust"                                          ,
 "AncestryMapper"                                     ,
 "curry"                                              ,
 "gcbd"                                               ,
 "papayar"                                            ,
 "nontarget"                                          ,
 "RSauceLabs"                                         ,
 "BioFTF"                                             ,
 "CAMAN"                                              ,
 "dataframes2xls"                                     ,
 "GUILDS"                                             ,
 "malani"                                             ,
 "momentchi2"                                         ,
 "PersomicsArray"                                     ,
 "svgPanZoom"                                         ,
 "textmining"                                         ,
 "ensembleR"                                          ,
 "geepack"                                            ,
 "PANICr"                                             ,
 "QRAGadget"                                          ,
 "Rdice"                                              ,
 "fishMod"                                            ,
 "hkevp"                                              ,
 "pbdPROF"                                            ,
 "prepdat"                                            ,
 "SuppDists"                                          ,
 "trelloR"                                            ,
 "contrast"                                           ,
 "desirability"                                       ,
 "FuzzyMCDM"                                          ,
 "iemisctext"                                         ,
 "MCDM"                                               ,
 "onlinePCA"                                          ,
 "rcanvec"                                            ,
 "RcmdrPlugin.PcaRobust"                              ,
 "sparseLDA"                                          ,
 "spikes"                                             ,
 "strategicplayers"                                   ,
 "topologyGSA"                                        ,
 "BSGW"                                               ,
 "coreNLP"                                            ,
 "dCovTS"                                             ,
 "simPATHy"                                           ,
 "STEPCAM"                                            ,
 "textgRid"                                           ,
 "aspi"                                               ,
 "coenoflex"                                          ,
 "GMDH"                                               ,
 "hdlm"                                               ,
 "optpart"                                            ,
 "htdp"                                               ,
 "joinXL"                                             ,
 "Tlasso"                                             ,
 "UCR.ColumnNames"                                    ,
 "fitdc"                                              ,
 "brm"                                                ,
 "BTR"                                                ,
 "PortfolioEffectEstim"                               ,
 "remote"                                             ,
 "telegram"                                           ,
 "AEDForecasting"                                     ,
 "ezknitr"                                            ,
 "fishkirkko2015"                                     ,
 "qrencoder"                                          ,
 "read.dbc"                                           ,
 "CUMP"                                               ,
 "DiceOptim"                                          ,
 "MetaboQC"                                           ,
 "PEGroupTesting"                                     ,
 "RRedshiftSQL"                                       ,
 "sonar"                                              ,
 "DamiaNN"                                            ,
 "EDR"                                                ,
 "EnsemblePCReg"                                      ,
 "EnsemblePenReg"                                     ,
 "pipefittr"                                          ,
 "qqvases"                                            ,
 "rBayesianOptimization"                              ,
 "regexPipes"                                         ,
 "SmartSifter"                                        ,
 "StepwiseTest"                                       ,
 "sweidnumbr"                                         ,
 "altmeta"                                            ,
 "EnsembleBase"                                       ,
 "EnsembleCV"                                         ,
 "fastcmh"                                            ,
 "LEAP"                                               ,
 "lifecourse"                                         ,
 "whoapi"                                             ,
 "BootMRMR"                                           ,
 "deadband"                                           ,
 "GWsignif"                                           ,
 "stabledist"                                         ,
 "StreamMetabolism"                                   ,
 "stripless"                                          ,
 "ARCensReg"                                          ,
 "crp.CSFP"                                           ,
 "elasticIsing"                                       ,
 "Ritc"                                               ,
 "cyclocomp"                                          ,
 "qha"                                                ,
 "rasterImage"                                        ,
 "riem"                                               ,
 "alluvial"                                           ,
 "alphaOutlier"                                       ,
 "ecdfHT"                                             ,
 "FME"                                                ,
 "inbreedR"                                           ,
 "pcrcoal"                                            ,
 "softmaxreg"                                         ,
 "BayesMixSurv"                                       ,
 "etseed"                                             ,
 "GSAQ"                                               ,
 "RegressionFactory"                                  ,
 "EMA"                                                ,
 "ggRandomForests"                                    ,
 "IsingFit"                                           ,
 "AquaEnv"                                            ,
 "dhga"                                               ,
 "imputeMissings"                                     ,
 "plotwidgets"                                        ,
 "SpadeR"                                             ,
 "urca"                                               ,
 "madr"                                               ,
 "rrcovNA"                                            ,
 "advclust"                                           ,
 "evclust"                                            ,
 "RcmdrPlugin.FuzzyClust"                             ,
 "misclassGLM"                                        ,
 "musica"                                             ,
 "psData"                                             ,
 "reutils"                                            ,
 "ROCS"                                               ,
 "spatialClust"                                       ,
 "FPCA2D"                                             ,
 "rminer"                                             ,
 "splus2R"                                            ,
 "ACSNMineR"                                          ,
 "versions"                                           ,
 "fpa"                                                ,
 "paintmap"                                           ,
 "rslp"                                               ,
 "T2EQ"                                               ,
 "TLdating"                                           ,
 "VIFCP"                                              ,
 "xlutils3"                                           ,
 "drc"                                                ,
 "indicspecies"                                       ,
 "naturalsort"                                        ,
 "PASenseWear"                                        ,
 "represtools"                                        ,
 "RSSOP"                                              ,
 "sharpshootR"                                        ,
 "svmpath"                                            ,
 "truncdist"                                          ,
 "censusGeography"                                    ,
 "gplm"                                               ,
 "MetaheuristicFPA"                                   ,
 "spselect"                                           ,
 "dynpanel"                                           ,
 "intubate"                                           ,
 "REPPlab"                                            ,
 "RSKC"                                               ,
 "slp"                                                ,
 "digitize"                                           ,
 "edfun"                                              ,
 "Rtextrankr"                                         ,
 "dagitty"                                            ,
 "DiffusionRgqd"                                      ,
 "grr"                                                ,
 "MicroDatosEs"                                       ,
 "Rexperigen"                                         ,
 "rglwidget"                                          ,
 "rpnf"                                               ,
 "rtson"                                              ,
 "tframePlus"                                         ,
 "TSdata"                                             ,
 "TSmisc"                                             ,
 "TSsdmx"                                             ,
 "parma"                                              ,
 "phonR"                                              ,
 "sfc"                                                ,
 "accrued"                                            ,
 "dglm"                                               ,
 "DiffusionRimp"                                      ,
 "evidenceFactors"                                    ,
 "funreg"                                             ,
 "lfstat"                                             ,
 "makeFlow"                                           ,
 "swfscMisc"                                          ,
 "tm.plugin.europresse"                               ,
 "liqueueR"                                           ,
 "SSRA"                                               ,
 "timelineS"                                          ,
 "PDM"                                                ,
 "ph2bye"                                             ,
 "vows"                                               ,
 "aemo"                                               ,
 "ctqr"                                               ,
 "datastepr"                                          ,
 "ggghost"                                            ,
 "NMI"                                                ,
 "predictionInterval"                                 ,
 "Rquake"                                             ,
 "okcupiddata"                                        ,
 "rportfolios"                                        ,
 "SPIAssay"                                           ,
 "ucminf"                                             ,
 "banxicoR"                                           ,
 "decompr"                                            ,
 "ghyp"                                               ,
 "gMWT"                                               ,
 "instaR"                                             ,
 "opera"                                              ,
 "optimr"                                             ,
 "rpinterest"                                         ,
 "rr"                                                 ,
 "RRate"                                              ,
 "DiffusionRjgqd"                                     ,
 "Rfit"                                               ,
 "conclust"                                           ,
 "mp"                                                 ,
 "AcrossTic"                                          ,
 "BurStMisc"                                          ,
 "ggThemeAssist"                                      ,
 "bundesligR"                                         ,
 "GlobalFit"                                          ,
 "listless"                                           ,
 "mixexp"                                             ,
 "MultiGHQuad"                                        ,
 "rich"                                               ,
 "estatapi"                                           ,
 "InformativeCensoring"                               ,
 "NISTunits"                                          ,
 "segmag"                                             ,
 "StagedChoiceSplineMix"                              ,
 "bgmfiles"                                           ,
 "ggpolypath"                                         ,
 "machQA"                                             ,
 "GLMMRR"                                             ,
 "ie2misc"                                            ,
 "MWLasso"                                            ,
 "ssmn"                                               ,
 "vaersNDvax"                                         ,
 "blkbox"                                             ,
 "EnvNicheR"                                          ,
 "optextras"                                          ,
 "sna"                                                ,
 "Rtauchen"                                           ,
 "MLCIRTwithin"                                       ,
 "formattable"                                        ,
 "uaparserjs"                                         ,
 "belex"                                              ,
 "ClimClass"                                          ,
 "CorrMixed"                                          ,
 "frmhet"                                             ,
 "GEEaSPU"                                            ,
 "geneNetBP"                                          ,
 "metaplotr"                                          ,
 "multilevel"                                         ,
 "rasterKernelEstimates"                              ,
 "rmsfact"                                            ,
 "SeqAlloc"                                           ,
 "triebeard"                                          ,
 "frmpd"                                              ,
 "gaussfacts"                                         ,
 "resumer"                                            ,
 "simplexreg"                                         ,
 "hdpca"                                              ,
 "lmPerm"                                             ,
 "rstack"                                             ,
 "AntAngioCOOL"                                       ,
 "binford"                                            ,
 "denovolyzeR"                                        ,
 "IndTestPP"                                          ,
 "ttbbeer"                                            ,
 "KSD"                                                ,
 "proccalibrad"                                       ,
 "BetaBit"                                            ,
 "RCMIP5"                                             ,
 "TCGAretriever"                                      ,
 "time2event"                                         ,
 "WPC"                                                ,
 "AhoCorasickTrie"                                    ,
 "deconstructSigs"                                    ,
 "FinCal"                                             ,
 "microcontax"                                        ,
 "noise"                                              ,
 "presens"                                            ,
 "sbrl"                                               ,
 "SPreFuGED"                                          ,
 "files"                                              ,
 "orca"                                               ,
 "sybilDynFBA"                                        ,
 "vines"                                              ,
 "birk"                                               ,
 "cellranger"                                         ,
 "EvaluationMeasures"                                 ,
 "iDOS"                                               ,
 "JAGUAR"                                             ,
 "palr"                                               ,
 "sEparaTe"                                           ,
 "SimSCRPiecewise"                                    ,
 "SNPMClust"                                          ,
 "StableEstim"                                        ,
 "AmostraBrasil"                                      ,
 "denoiseR"                                           ,
 "LARF"                                               ,
 "lightsout"                                          ,
 "MonoPhy"                                            ,
 "preprosim"                                          ,
 "random.polychor.pa"                                 ,
 "SizeEstimation"                                     ,
 "bigReg"                                             ,
 "HK80"                                               ,
 "GameTheoryAllocation"                               ,
 "lmenssp"                                            ,
 "mcparallelDo"                                       ,
 "mod09nrt"                                           ,
 "qat"                                                ,
 "RSVGTipsDevice"                                     ,
 "stmCorrViz"                                         ,
 "fastSOM"                                            ,
 "pullword"                                           ,
 "track"                                              ,
 "DIFboost"                                           ,
 "Holidays"                                           ,
 "icdGLM"                                             ,
 "iemiscdata"                                         ,
 "IPtoCountry"                                        ,
 "optDesignSlopeInt"                                  ,
 "scriptests"                                         ,
 "TimeWarp"                                           ,
 "abind"                                              ,
 "clinsig"                                            ,
 "TripleR"                                            ,
 "ChIPtest"                                           ,
 "mvQuad"                                             ,
 "qcr"                                                ,
 "scientoText"                                        ,
 "slackr"                                             ,
 "RForcecom"                                          ,
 "BoardGames"                                         ,
 "DWreg"                                              ,
 "iRegression"                                        ,
 "SecKW"                                              ,
 "SinIW"                                              ,
 "CosW"                                               ,
 "precintcon"                                         ,
 "TanB"                                               ,
 "ACDm"                                               ,
 "EconDemand"                                         ,
 "MRMR"                                               ,
 "QVM"                                                ,
 "sbfc"                                               ,
 "modEvA"                                             ,
 "RStripe"                                            ,
 "sophisthse"                                         ,
 "tsne"                                               ,
 "artfima"                                            ,
 "logcondens"                                         ,
 "matrixLaplacian"                                    ,
 "MCMC4Extremes"                                      ,
 "midrangeMCP"                                        ,
 "mlma"                                               ,
 "Peacock.test"                                       ,
 "rDNAse"                                             ,
 "monographaR"                                        ,
 "rcicr"                                              ,
 "smoothtail"                                         ,
 "install.load"                                       ,
 "lmem.qtler"                                         ,
 "simulator"                                          ,
 "gnmf"                                               ,
 "Homeric"                                            ,
 "rmdHelpers"                                         ,
 "CIFsmry"                                            ,
 "crskdiag"                                           ,
 "futile.logger"                                      ,
 "tempdisagg"                                         ,
 "gwdegree"                                           ,
 "preproviz"                                          ,
 "DCchoice"                                           ,
 "STB"                                                ,
 "RCriteo"                                            ,
 "Rtnmin"                                             ,
 "SpectralMap"                                        ,
 "vietnamcode"                                        ,
 "BayesTree"                                          ,
 "EXRQ"                                               ,
 "linear.tools"                                       ,
 "MLPUGS"                                             ,
 "som"                                                ,
 "genomicper"                                         ,
 "PACBO"                                              ,
 "batchmeans"                                         ,
 "BayesTreePrior"                                     ,
 "Cite"                                               ,
 "csp"                                                ,
 "etrunct"                                            ,
 "gramEvol"                                           ,
 "pfa"                                                ,
 "Renext"                                             ,
 "RenextGUI"                                          ,
 "wfindr"                                             ,
 "fmrs"                                               ,
 "InspectChangepoint"                                 ,
 "isingLenzMC"                                        ,
 "modygliani"                                         ,
 "productplots"                                       ,
 "QCSimulator"                                        ,
 "randNames"                                          ,
 "Comp2ROC"                                           ,
 "hypothesisr"                                        ,
 "rccdates"                                           ,
 "RcmdrPlugin.EcoVirtual"                             ,
 "BootWPTOS"                                          ,
 "metaheur"                                           ,
 "mise"                                               ,
 "Segmentor3IsBack"                                   ,
 "correctedAUC"                                       ,
 "CP"                                                 ,
 "DrillR"                                             ,
 "POET"                                               ,
 "survSNP"                                            ,
 "UNCLES"                                             ,
 "pGMGM"                                              ,
 "IntegrateBs"                                        ,
 "ldamatch"                                           ,
 "meteR"                                              ,
 "SeqMADE"                                            ,
 "couchDB"                                            ,
 "preprocomb"                                         ,
 "BACCT"                                              ,
 "NoiseFiltersR"                                      ,
 "babel"                                              ,
 "boa"                                                ,
 "factorplot"                                         ,
 "GAabbreviate"                                       ,
 "LakeMetabolizer"                                    ,
 "LindleyR"                                           ,
 "mscstexta4r"                                        ,
 "R2HTML"                                             ,
 "RMThreshold"                                        ,
 "shinyDND"                                           ,
 "pearson7"                                           ,
 "REPTILE"                                            ,
 "DoTC"                                               ,
 "PATHChange"                                         ,
 "gmDatabase"                                         ,
 "Rphylopars"                                         ,
 "SPIGA"                                              ,
 "cssTools"                                           ,
 "IsotopeR"                                           ,
 "mscsweblm4r"                                        ,
 "bpcp"                                               ,
 "cointmonitoR"                                       ,
 "cointReg"                                           ,
 "perccal"                                            ,
 "TEEReg"                                             ,
 "AUtests"                                            ,
 "engsoccerdata"                                      ,
 "Rcplex"                                             ,
 "HDDesign"                                           ,
 "PARSE"                                              ,
 "ibm"                                                ,
 "kehra"                                              ,
 "sesem"                                              ,
 "cape"                                               ,
 "cloudUtil"                                          ,
 "dam"                                                ,
 "JointModel"                                         ,
 "spaa"                                               ,
 "TRAMPR"                                             ,
 "MGSDA"                                              ,
 "plyr"                                               ,
 "Rbent"                                              ,
 "BaPreStoPro"                                        ,
 "pop"                                                ,
 "Rsampling"                                          ,
 "bioplots"                                           ,
 "DAKS"                                               ,
 "DisimForMixed"                                      ,
 "enviPick"                                           ,
 "fechner"                                            ,
 "heatmapFit"                                         ,
 "muckrock"                                           ,
 "PreKnitPostHTMLRender"                              ,
 "RClone"                                             ,
 "shinytoastr"                                        ,
 "uniReg"                                             ,
 "iGasso"                                             ,
 "ROpenFIGI"                                          ,
 "bapred"                                             ,
 "clikcorr"                                           ,
 "flars"                                              ,
 "hkex.api"                                           ,
 "bastah"                                             ,
 "Dark"                                               ,
 "MSQC"                                               ,
 "sybilcycleFreeFlux"                                 ,
 "BHH2"                                               ,
 "Fahrmeir"                                           ,
 "HandTill2001"                                       ,
 "maSAE"                                              ,
 "PolyTrend"                                          ,
 "ClustMMDD"                                          ,
 "DODR"                                               ,
 "qrmdata"                                            ,
 "arrayhelpers"                                       ,
 "cocor"                                              ,
 "softclassval"                                       ,
 "GK2011"                                             ,
 "CHFF"                                               ,
 "Davies"                                             ,
 "ddst"                                               ,
 "fakeR"                                              ,
 "forecTheta"                                         ,
 "HTSCluster"                                         ,
 "replicationInterval"                                ,
 "spfrontier"                                         ,
 "kantorovich"                                        ,
 "bayesbio"                                           ,
 "bitrugs"                                            ,
 "ChangepointTesting"                                 ,
 "GiniWegNeg"                                         ,
 "uskewFactors"                                       ,
 "asd"                                                ,
 "prof.tree"                                          ,
 "VDAP"                                               ,
 "sapa"                                               ,
 "AdequacyModel"                                      ,
 "crtests"                                            ,
 "MonoInc"                                            ,
 "coenocliner"                                        ,
 "CUSUMdesign"                                        ,
 "LSPFP"                                              ,
 "MetProc"                                            ,
 "weathermetrics"                                     ,
 "BaM"                                                ,
 "gamlss.mx"                                          ,
 "gamlss.util"                                        ,
 "hht"                                                ,
 "saasCNV"                                            ,
 "clusterhap"                                         ,
 "pRSR"                                               ,
 "SimplicialCubature"                                 ,
 "cystiSim"                                           ,
 "equivalence"                                        ,
 "lrmest"                                             ,
 "rollbar"                                            ,
 "ada"                                                ,
 "DengueRT"                                           ,
 "HMVD"                                               ,
 "MLmetrics"                                          ,
 "RFormatter"                                         ,
 "rjstat"                                             ,
 "lambda.tools"                                       ,
 "SetRank"                                            ,
 "assertive.datetimes"                                ,
 "assertive.files"                                    ,
 "assertive.strings"                                  ,
 "base64"                                             ,
 "CircNNTSR"                                          ,
 "FeedbackTS"                                         ,
 "rankhazard"                                         ,
 "RcmdrPlugin.RMTCJags"                               ,
 "subdetect"                                          ,
 "assertive.numbers"                                  ,
 "olctools"                                           ,
 "concatenate"                                        ,
 "geozoo"                                             ,
 "nasadata"                                           ,
 "PenCoxFrail"                                        ,
 "multiwayvcov"                                       ,
 "sld"                                                ,
 "surveybootstrap"                                    ,
 "qmap"                                               ,
 "recommenderlabJester"                               ,
 "SCI"                                                ,
 "imputePSF"                                          ,
 "netcoh"                                             ,
 "rasclass"                                           ,
 "shuffle"                                            ,
 "TippingPoint"                                       ,
 "DiscreteInverseWeibull"                             ,
 "DiscreteLaplace"                                    ,
 "ljr"                                                ,
 "lvplot"                                             ,
 "StressStrength"                                     ,
 "IAT"                                                ,
 "plac"                                               ,
 "ClusteredMutations"                                 ,
 "fastclime"                                          ,
 "googleformr"                                        ,
 "ClusterRankTest"                                    ,
 "CDROM"                                              ,
 "ggdendro"                                           ,
 "NAEPprimer"                                         ,
 "crsnls"                                             ,
 "nat.nblast"                                         ,
 "REREFACT"                                           ,
 "RZabbix"                                            ,
 "VecStatGraphs2D"                                    ,
 "beeswarm"                                           ,
 "FinCovRegularization"                               ,
 "genomeplot"                                         ,
 "MissingDataGUI"                                     ,
 "Rcsdp"                                              ,
 "humaniformat"                                       ,
 "mlmc"                                               ,
 "Rdsdp"                                              ,
 "TELP"                                               ,
 "DrugClust"                                          ,
 "fdaPDE"                                             ,
 "IncucyteDRC"                                        ,
 "aimPlot"                                            ,
 "bridgedist"                                         ,
 "edf"                                                ,
 "hextri"                                             ,
 "uHMM"                                               ,
 "xkcdcolors"                                         ,
 "EEM"                                                ,
 "eyelinker"                                          ,
 "falcon"                                             ,
 "rematch"                                            ,
 "biotic"                                             ,
 "rnpn"                                               ,
 "capushe"                                            ,
 "DidacticBoost"                                      ,
 "disparityfilter"                                    ,
 "envlpaster"                                         ,
 "flowr"                                              ,
 "funr"                                               ,
 "multipleNCC"                                        ,
 "nbpMatching"                                        ,
 "NlsyLinks"                                          ,
 "params"                                             ,
 "refnr"                                              ,
 "StroupGLMM"                                         ,
 "GoodmanKruskal"                                     ,
 "sgof"                                               ,
 "convertGraph"                                       ,
 "DOT"                                                ,
 "keyplayer"                                          ,
 "solaR"                                              ,
 "sprex"                                              ,
 "pact"                                               ,
 "tth"                                                ,
 "reservoir"                                          ,
 "DataCombine"                                        ,
 "ebGenotyping"                                       ,
 "sde"                                                ,
 "stcov"                                              ,
 "asaur"                                              ,
 "gencve"                                             ,
 "hierarchicalSets"                                   ,
 "nlshrink"                                           ,
 "progenyClust"                                       ,
 "rYoutheria"                                         ,
 "ELT"                                                ,
 "PolyPatEx"                                          ,
 "analogueExtra"                                      ,
 "etma"                                               ,
 "GiNA"                                               ,
 "greport"                                            ,
 "mgarchBEKK"                                         ,
 "Sofi"                                               ,
 "DiallelAnalysisR"                                   ,
 "gds"                                                ,
 "Information"                                        ,
 "investr"                                            ,
 "mateable"                                           ,
 "rrefine"                                            ,
 "biogeo"                                             ,
 "braidReports"                                       ,
 "RGA"                                                ,
 "AdvBinomApps"                                       ,
 "hypergeo"                                           ,
 "MixtureInf"                                         ,
 "palettetown"                                        ,
 "fbRads"                                             ,
 "haplotyper"                                         ,
 "MixRF"                                              ,
 "PhysActBedRest"                                     ,
 "pop.wolf"                                           ,
 "decision"                                           ,
 "gelnet"                                             ,
 "MHTrajectoryR"                                      ,
 "vhica"                                              ,
 "argparser"                                          ,
 "bpa"                                                ,
 "cardioModel"                                        ,
 "emov"                                               ,
 "iopsych"                                            ,
 "LexisPlotR"                                         ,
 "pipeR"                                              ,
 "rlist"                                              ,
 "cancerTiming"                                       ,
 "diskImageR"                                         ,
 "mistral"                                            ,
 "mvcluster"                                          ,
 "pergola"                                            ,
 "RevEcoR"                                            ,
 "rusda"                                              ,
 "waterfall"                                          ,
 "WiSEBoot"                                           ,
 "agrmt"                                              ,
 "WHO"                                                ,
 "ArgumentCheck"                                      ,
 "cancerGI"                                           ,
 "DIRECT"                                             ,
 "FUNTA"                                              ,
 "leafletR"                                           ,
 "quantreg.nonpar"                                    ,
 "soql"                                               ,
 "Biograph"                                           ,
 "calACS"                                             ,
 "collUtils"                                          ,
 "processcontrol"                                     ,
 "bdpopt"                                             ,
 "breakaway"                                          ,
 "EDISON"                                             ,
 "FMsmsnReg"                                          ,
 "ForeCA"                                             ,
 "GOplot"                                             ,
 "relimp"                                             ,
 "setter"                                             ,
 "TDboost"                                            ,
 "tdROC"                                              ,
 "gsbDesign"                                          ,
 "imguR"                                              ,
 "LambertW"                                           ,
 "randstr"                                            ,
 "robustETM"                                          ,
 "svmadmm"                                            ,
 "tttplot"                                            ,
 "needs"                                              ,
 "NPC"                                                ,
 "quantspec"                                          ,
 "rappdirs"                                           ,
 "SmarterPoland"                                      ,
 "RobPer"                                             ,
 "betalink"                                           ,
 "kknn"                                               ,
 "NEpiC"                                              ,
 "seqtest"                                            ,
 "sSDR"                                               ,
 "tailDepFun"                                         ,
 "ZillowR"                                            ,
 "censorcopula"                                       ,
 "DataClean"                                          ,
 "ggnetwork"                                          ,
 "Newdistns"                                          ,
 "nlWaldTest"                                         ,
 "DNAseqtest"                                         ,
 "wkb"                                                ,
 "directPA"                                           ,
 "WrightMap"                                          ,
 "clere"                                              ,
 "password"                                           ,
 "vegalite"                                           ,
 "getMet"                                             ,
 "MonetDB.R"                                          ,
 "TwoStepCLogit"                                      ,
 "VSE"                                                ,
 "commentr"                                           ,
 "edgebundleR"                                        ,
 "interpretR"                                         ,
 "rfml"                                               ,
 "ecb"                                                ,
 "MiSPU"                                              ,
 "QRM"                                                ,
 "rechonest"                                          ,
 "vottrans"                                           ,
 "bdscale"                                            ,
 "bssn"                                               ,
 "housingData"                                        ,
 "RFgroove"                                           ,
 "rscimark"                                           ,
 "TwoPhaseInd"                                        ,
 "dendrometeR"                                        ,
 "edgeCorr"                                           ,
 "ROpenWeatherMap"                                    ,
 "DEMOVA"                                             ,
 "NBDdirichlet"                                       ,
 "RADanalysis"                                        ,
 "DandEFA"                                            ,
 "iWISA"                                              ,
 "logisticPCA"                                        ,
 "rdd"                                                ,
 "RMediation"                                         ,
 "RODBCDBI"                                           ,
 "cocron"                                             ,
 "dcmle"                                              ,
 "fermicatsR"                                         ,
 "lazysql"                                            ,
 "mdsdt"                                              ,
 "Pijavski"                                           ,
 "PVAClone"                                           ,
 "Dowd"                                               ,
 "dynamicTreeCut"                                     ,
 "DCODE"                                              ,
 "ngramrr"                                            ,
 "nnlasso"                                            ,
 "osd"                                                ,
 "rARPACK"                                            ,
 "sparsereg"                                          ,
 "tsintermittent"                                     ,
 "BBEST"                                              ,
 "carcass"                                            ,
 "digitalPCR"                                         ,
 "NIPTeR"                                             ,
 "regsel"                                             ,
 "shrink"                                             ,
 "braidrm"                                            ,
 "OmicKriging"                                        ,
 "ProbitSpatial"                                      ,
 "scrubr"                                             ,
 "ccaPP"                                              ,
 "dpmr"                                               ,
 "lmms"                                               ,
 "modes"                                              ,
 "jrich"                                              ,
 "kmlShape"                                           ,
 "palinsol"                                           ,
 "cdom"                                               ,
 "hda"                                                ,
 "Mondrian"                                           ,
 "nlmrt"                                              ,
 "resemble"                                           ,
 "karaoke"                                            ,
 "lettercase"                                         ,
 "lordif"                                             ,
 "ProteinDescriptors"                                 ,
 "RGraphics"                                          ,
 "lxb"                                                ,
 "MANCIE"                                             ,
 "Rearrangement"                                      ,
 "spiders"                                            ,
 "OTRselect"                                          ,
 "LRTH"                                               ,
 "pequod"                                             ,
 "planar"                                             ,
 "PWD"                                                ,
 "RCA"                                                ,
 "RFmarkerDetector"                                   ,
 "RYandexTranslate"                                   ,
 "tmpm"                                               ,
 "Weighted.Desc.Stat"                                 ,
 "COUSCOus"                                           ,
 "fastAdaboost"                                       ,
 "TestScorer"                                         ,
 "varian"                                             ,
 "repolr"                                             ,
 "MRH"                                                ,
 "apt"                                                ,
 "CryptRndTest"                                       ,
 "erer"                                               ,
 "Evomorph"                                           ,
 "livechatR"                                          ,
 "luzlogr"                                            ,
 "PBImisc"                                            ,
 "perARMA"                                            ,
 "qqtest"                                             ,
 "wfg"                                                ,
 "CONDOP"                                             ,
 "GiRaF"                                              ,
 "qualityTools"                                       ,
 "linERR"                                             ,
 "permGPU"                                            ,
 "sensory"                                            ,
 "lrequire"                                           ,
 "MuViCP"                                             ,
 "spnet"                                              ,
 "sprm"                                               ,
 "ChannelAttributionApp"                              ,
 "md"                                                 ,
 "mptools"                                            ,
 "vudc"                                               ,
 "clttools"                                           ,
 "geoaxe"                                             ,
 "rfisheries"                                         ,
 "biganalytics"                                       ,
 "bigtabulate"                                        ,
 "Familias"                                           ,
 "microbats"                                          ,
 "minimap"                                            ,
 "openNLP"                                            ,
 "calibrar"                                           ,
 "GGEBiplotGUI"                                       ,
 "scenario"                                           ,
 "designGLMM"                                         ,
 "kml"                                                ,
 "longitudinalData"                                   ,
 "R.methodsS3"                                        ,
 "faraway"                                            ,
 "fractional"                                         ,
 "noncompliance"                                      ,
 "pdftables"                                          ,
 "geotech"                                            ,
 "GrammR"                                             ,
 "ipfp"                                               ,
 "BonEV"                                              ,
 "TransP"                                             ,
 "FCMapper"                                           ,
 "MCMC.OTU"                                           ,
 "TeachingDemos"                                      ,
 "WaverR"                                             ,
 "BGPhazard"                                          ,
 "ionr"                                               ,
 "spacom"                                             ,
 "starma"                                             ,
 "titrationCurves"                                    ,
 "EBMAforecast"                                       ,
 "exploreR"                                           ,
 "mvnTest"                                            ,
 "cooccur"                                            ,
 "FMP"                                                ,
 "latticeExtra"                                       ,
 "loe"                                                ,
 "soc.ca"                                             ,
 "Watersheds"                                         ,
 "ClusterStability"                                   ,
 "guess"                                              ,
 "munfold"                                            ,
 "spgrass6"                                           ,
 "bayesDccGarch"                                      ,
 "Bmix"                                               ,
 "caretEnsemble"                                      ,
 "midastouch"                                         ,
 "MixedDataImpute"                                    ,
 "mpbart"                                             ,
 "repmis"                                             ,
 "RLogicalOps"                                        ,
 "tgp"                                                ,
 "DRaWR"                                              ,
 "rdian"                                              ,
 "TEQR"                                               ,
 "WikipediaR"                                         ,
 "ABCp2"                                              ,
 "ABHgenotypeR"                                       ,
 "DiagrammeRsvg"                                      ,
 "elexr"                                              ,
 "epandist"                                           ,
 "IFP"                                                ,
 "Sky"                                                ,
 "tumgr"                                              ,
 "rworldmap"                                          ,
 "sgeostat"                                           ,
 "FastGP"                                             ,
 "GGMridge"                                           ,
 "graticule"                                          ,
 "nnet"                                               ,
 "patPRO"                                             ,
 "quantileDA"                                         ,
 "acid"                                               ,
 "cmprskQR"                                           ,
 "exreport"                                           ,
 "gsDesign"                                           ,
 "murphydiagram"                                      ,
 "scorer"                                             ,
 "psoptim"                                            ,
 "regpro"                                             ,
 "tripEstimation"                                     ,
 "asymLD"                                             ,
 "cepp"                                               ,
 "EBglmnet"                                           ,
 "StockChina"                                         ,
 "wildpoker"                                          ,
 "zooaRch"                                            ,
 "bqtl"                                               ,
 "IMP"                                                ,
 "sscor"                                              ,
 "LaplaceDeconv"                                      ,
 "wBoot"                                              ,
 "distance.sample.size"                               ,
 "SteinIV"                                            ,
 "surveyoutliers"                                     ,
 "medicalrisk"                                        ,
 "VertexSimilarity"                                   ,
 "AsynchLong"                                         ,
 "fSRM"                                               ,
 "ramsvm"                                             ,
 "webreadr"                                           ,
 "DYM"                                                ,
 "netgen"                                             ,
 "FreqProf"                                           ,
 "sclero"                                             ,
 "corclass"                                           ,
 "extremevalues"                                      ,
 "miCoPTCM"                                           ,
 "AmpliconDuo"                                        ,
 "igraphinshiny"                                      ,
 "pAnalysis"                                          ,
 "SNscan"                                             ,
 "SubpathwayLNCE"                                     ,
 "assortnet"                                          ,
 "CepLDA"                                             ,
 "cycleRtools"                                        ,
 "OceanView"                                          ,
 "plot3Drgl"                                          ,
 "rpcdsearch"                                         ,
 "freqparcoord"                                       ,
 "OriGen"                                             ,
 "Actigraphy"                                         ,
 "FDRsampsize"                                        ,
 "HKprocess"                                          ,
 "PASWR2"                                             ,
 "sharx"                                              ,
 "BinaryEMVS"                                         ,
 "breakpoint"                                         ,
 "dChipIO"                                            ,
 "MASSTIMATE"                                         ,
 "proftools"                                          ,
 "svs"                                                ,
 "CircOutlier"                                        ,
 "intercure"                                          ,
 "mefa"                                               ,
 "msltrend"                                           ,
 "networkDynamicData"                                 ,
 "rodd"                                               ,
 "SALES"                                              ,
 "concordance"                                        ,
 "gSEM"                                               ,
 "mmppr"                                              ,
 "pRF"                                                ,
 "rlm"                                                ,
 "bmeta"                                              ,
 "fwi.fbp"                                            ,
 "MakefileR"                                          ,
 "robustHD"                                           ,
 "S2sls"                                              ,
 "vrcp"                                               ,
 "glmvsd"                                             ,
 "gyriq"                                              ,
 "neuroim"                                            ,
 "solarPos"                                           ,
 "RcmdrPlugin.FactoMineR"                             ,
 "rotations"                                          ,
 "SimRAD"                                             ,
 "stlplus"                                            ,
 "cds"                                                ,
 "hiddenf"                                            ,
 "RImagePalette"                                      ,
 "cg"                                                 ,
 "hot.deck"                                           ,
 "LiblineaR.ACF"                                      ,
 "qtlc"                                               ,
 "grpss"                                              ,
 "icensmis"                                           ,
 "request"                                            ,
 "tempcyclesdata"                                     ,
 "hwde"                                               ,
 "DTR"                                                ,
 "ParallelPC"                                         ,
 "ifctools"                                           ,
 "plspm.formula"                                      ,
 "robreg3S"                                           ,
 "sdmvspecies"                                        ,
 "Deducer"                                            ,
 "TOC"                                                ,
 "xtal"                                               ,
 "BiasedUrn"                                          ,
 "qiimer"                                             ,
 "Rsolnp"                                             ,
 "MultAlloc"                                          ,
 "validateRS"                                         ,
 "MediaK"                                             ,
 "multifwf"                                           ,
 "SchemaOnRead"                                       ,
 "shock"                                              ,
 "XNomial"                                            ,
 "batteryreduction"                                   ,
 "BlandAltmanLeh"                                     ,
 "r4ss"                                               ,
 "federalregister"                                    ,
 "Lock5withR"                                         ,
 "RPublica"                                           ,
 "clogitboost"                                        ,
 "ltsa"                                               ,
 "metricsgraphics"                                    ,
 "nnetpredint"                                        ,
 "ProbYX"                                             ,
 "statebins"                                          ,
 "VARsignR"                                           ,
 "depth.plot"                                         ,
 "dynsim"                                             ,
 "FWDselect"                                          ,
 "bodenmiller"                                        ,
 "granovaGG"                                          ,
 "CCAGFA"                                             ,
 "dse"                                                ,
 "gpDDE"                                              ,
 "keep"                                               ,
 "pamm"                                               ,
 "rebus.datetimes"                                    ,
 "rebus.numbers"                                      ,
 "tfplot"                                             ,
 "treecm"                                             ,
 "dendsort"                                           ,
 "exif"                                               ,
 "hellno"                                             ,
 "hpoPlot"                                            ,
 "LightningR"                                         ,
 "vita"                                               ,
 "CITAN"                                              ,
 "solarius"                                           ,
 "ADMMnet"                                            ,
 "Frames2"                                            ,
 "HLMdiag"                                            ,
 "rbvs"                                               ,
 "RXMCDA"                                             ,
 "SPmlficmcm"                                         ,
 "RANKS"                                              ,
 "simplegraph"                                        ,
 "maddison"                                           ,
 "NetPreProc"                                         ,
 "orsifronts"                                         ,
 "pitchRx"                                            ,
 "riskR"                                              ,
 "rwfec"                                              ,
 "sensitivity2x2xk"                                   ,
 "MNS"                                                ,
 "PAGWAS"                                             ,
 "G2Sd"                                               ,
 "PIGShift"                                           ,
 "caRpools"                                           ,
 "sleekts"                                            ,
 "acp"                                                ,
 "hierfstat"                                          ,
 "npIntFactRep"                                       ,
 "ordiBreadth"                                        ,
 "peakPick"                                           ,
 "EpiDynamics"                                        ,
 "pedometrics"                                        ,
 "ppcor"                                              ,
 "ryouready"                                          ,
 "tggd"                                               ,
 "features"                                           ,
 "kriens"                                             ,
 "QCSIS"                                              ,
 "causaldrf"                                          ,
 "dsample"                                            ,
 "injectoR"                                           ,
 "latex2exp"                                          ,
 "lodGWAS"                                            ,
 "oaColors"                                           ,
 "oaPlots"                                            ,
 "proton"                                             ,
 "mcemGLM"                                            ,
 "ELMR"                                               ,
 "GroupTest"                                          ,
 "bvarsv"                                             ,
 "MMWRweek"                                           ,
 "apricom"                                            ,
 "BMS"                                                ,
 "pgnorm"                                             ,
 "UncerIn2"                                           ,
 "yummlyr"                                            ,
 "AggregateR"                                         ,
 "flexPM"                                             ,
 "ForecastCombinations"                               ,
 "SimSeq"                                             ,
 "labelrank"                                          ,
 "lda"                                                ,
 "tracheideR"                                         ,
 "FACTMLE"                                            ,
 "glmx"                                               ,
 "s4vd"                                               ,
 "ConConPiWiFun"                                      ,
 "JGEE"                                               ,
 "jvnVaR"                                             ,
 "rapport"                                            ,
 "tnet"                                               ,
 "Traitspace"                                         ,
 "phylometrics"                                       ,
 "gridDebug"                                          ,
 "nCDunnett"                                          ,
 "landest"                                            ,
 "rPython"                                            ,
 "sms"                                                ,
 "REdaS"                                              ,
 "TopKLists"                                          ,
 "ptycho"                                             ,
 "rnetcarto"                                          ,
 "gbm2sas"                                            ,
 "qtpaint"                                            ,
 "steadyICA"                                          ,
 "TRD"                                                ,
 "FastKM"                                             ,
 "ade4TkGUI"                                          ,
 "CovSel"                                             ,
 "genSurv"                                            ,
 "gvc"                                                ,
 "smoothHR"                                           ,
 "YPmodel"                                            ,
 "BaSTA"                                              ,
 "LRcontrast"                                         ,
 "pampe"                                              ,
 "personograph"                                       ,
 "phia"                                               ,
 "SPCALDA"                                            ,
 "BayesMAMS"                                          ,
 "easypower"                                          ,
 "rucm"                                               ,
 "rchess"                                             ,
 "survJamda"                                          ,
 "tolBasis"                                           ,
 "relen"                                              ,
 "wPerm"                                              ,
 "merror"                                             ,
 "Phxnlme"                                            ,
 "RRTCS"                                              ,
 "SLOPE"                                              ,
 "mvProbit"                                           ,
 "Rcolombos"                                          ,
 "RGoogleAnalyticsPremium"                            ,
 "DataLoader"                                         ,
 "NetSwan"                                            ,
 "nlrr"                                               ,
 "PortRisk"                                           ,
 "AlgebraicHaploPackage"                              ,
 "corkscrew"                                          ,
 "kedd"                                               ,
 "smss"                                               ,
 "TSTr"                                               ,
 "CDVine"                                             ,
 "RNaviCell"                                          ,
 "aroma.cn"                                           ,
 "RESS"                                               ,
 "TESS"                                               ,
 "ACNE"                                               ,
 "calmate"                                            ,
 "settings"                                           ,
 "batman"                                             ,
 "feature"                                            ,
 "TDCor"                                              ,
 "MixedTS"                                            ,
 "MPCI"                                               ,
 "AdapEnetClass"                                      ,
 "CoxPlus"                                            ,
 "jackknifeKME"                                       ,
 "LDAvis"                                             ,
 "PanelCount"                                         ,
 "hotspot"                                            ,
 "imputeYn"                                           ,
 "poisson"                                            ,
 "pvclust"                                            ,
 "dgmb"                                               ,
 "HotDeckImputation"                                  ,
 "TCGA2STAT"                                          ,
 "uptimeRobot"                                        ,
 "weightedScores"                                     ,
 "MultiRR"                                            ,
 "eventInterval"                                      ,
 "GSSE"                                               ,
 "crop"                                               ,
 "diezeit"                                            ,
 "HiDimDA"                                            ,
 "ksrlive"                                            ,
 "RcmdrPlugin.Export"                                 ,
 "SimHaz"                                             ,
 "DiscreteWeibull"                                    ,
 "BcDiag"                                             ,
 "CHsharp"                                            ,
 "covBM"                                              ,
 "ecotoxicology"                                      ,
 "ReliabilityTheory"                                  ,
 "snowfall"                                           ,
 "TLBC"                                               ,
 "FCGR"                                               ,
 "lift"                                               ,
 "SyncMove"                                           ,
 "wingui"                                             ,
 "apsimr"                                             ,
 "compendiumdb"                                       ,
 "deformula"                                          ,
 "geesmv"                                             ,
 "subspace"                                           ,
 "gitter"                                             ,
 "MVT"                                                ,
 "HDGLM"                                              ,
 "sisal"                                              ,
 "FIACH"                                              ,
 "fastdigest"                                         ,
 "SurvRegCensCov"                                     ,
 "EBEN"                                               ,
 "fanplot"                                            ,
 "elasso"                                             ,
 "conicfit"                                           ,
 "diagonals"                                          ,
 "geofd"                                              ,
 "smallarea"                                          ,
 "wqs"                                                ,
 "MetaPath"                                           ,
 "GeoBoxplot"                                         ,
 "SphericalK"                                         ,
 "fail"                                               ,
 "RcmdrPlugin.EBM"                                    ,
 "sanon"                                              ,
 "snpEnrichment"                                      ,
 "kernelFactory"                                      ,
 "learNN"                                             ,
 "pdc"                                                ,
 "pwrRasch"                                           ,
 "rleafmap"                                           ,
 "SOPIE"                                              ,
 "plotpc"                                             ,
 "twostageTE"                                         ,
 "BCBCSF"                                             ,
 "nivm"                                               ,
 "TTS"                                                ,
 "weightTAPSPACK"                                     ,
 "drsmooth"                                           ,
 "nanop"                                              ,
 "Skillings.Mack"                                     ,
 "STAND"                                              ,
 "drLumi"                                             ,
 "meteo"                                              ,
 "NostalgiR"                                          ,
 "packagetrackr"                                      ,
 "Rip46"                                              ,
 "cord"                                               ,
 "rcrypt"                                             ,
 "crimelinkage"                                       ,
 "ImportExport"                                       ,
 "kelvin"                                             ,
 "GAR"                                                ,
 "GRTo"                                               ,
 "rlecuyer"                                           ,
 "Rtwalk"                                             ,
 "backtest"                                           ,
 "describer"                                          ,
 "multirich"                                          ,
 "pro"                                                ,
 "Rlof"                                               ,
 "spatialprobit"                                      ,
 "discreteRV"                                         ,
 "JPEN"                                               ,
 "peptider"                                           ,
 "readMzXmlData"                                      ,
 "TestingSimilarity"                                  ,
 "bclust"                                             ,
 "lgarch"                                             ,
 "OptGS"                                              ,
 "AutoregressionMDE"                                  ,
 "LinearRegressionMDE"                                ,
 "ESKNN"                                              ,
 "LGRF"                                               ,
 "GenOrd"                                             ,
 "hashids"                                            ,
 "hindexcalculator"                                   ,
 "regRSM"                                             ,
 "spatialfil"                                         ,
 "xseq"                                               ,
 "acmeR"                                              ,
 "ThreeGroups"                                        ,
 "tm.plugin.dc"                                       ,
 "mfp"                                                ,
 "onls"                                               ,
 "pla"                                                ,
 "rprintf"                                            ,
 "StatRank"                                           ,
 "ftsspec"                                            ,
 "prim"                                               ,
 "weatherr"                                           ,
 "BioMark"                                            ,
 "brr"                                                ,
 "FAdist"                                             ,
 "mlsjunkgen"                                         ,
 "phonenumber"                                        ,
 "spinyReg"                                           ,
 "ThreeWay"                                           ,
 "evobiR"                                             ,
 "GeoLight"                                           ,
 "ACSWR"                                              ,
 "PabonLasso"                                         ,
 "RoughSets"                                          ,
 "DoubleExpSeq"                                       ,
 "edesign"                                            ,
 "IASD"                                               ,
 "ICC.Sample.Size"                                    ,
 "sgt"                                                ,
 "influenceR"                                         ,
 "invLT"                                              ,
 "LifeHist"                                           ,
 "EasyABC"                                            ,
 "functools"                                          ,
 "IalsaSynthesis"                                     ,
 "ParetoPosStable"                                    ,
 "valottery"                                          ,
 "ash"                                                ,
 "classGraph"                                         ,
 "eel"                                                ,
 "abodOutlier"                                        ,
 "titanic"                                            ,
 "ZRA"                                                ,
 "spatial"                                            ,
 "dml"                                                ,
 "ALTopt"                                             ,
 "cacIRT"                                             ,
 "detector"                                           ,
 "erboost"                                            ,
 "KERE"                                               ,
 "kin.cohort"                                         ,
 "lookupTable"                                        ,
 "tmvtnorm"                                           ,
 "trimr"                                              ,
 "intervals"                                          ,
 "support.CEs"                                        ,
 "FastBandChol"                                       ,
 "generator"                                          ,
 "SDDE"                                               ,
 "bimixt"                                             ,
 "compare"                                            ,
 "contoureR"                                          ,
 "fptdApprox"                                         ,
 "ggplot2movies"                                      ,
 "ONETr"                                              ,
 "optismixture"                                       ,
 "AutoModel"                                          ,
 "GUIProfiler"                                        ,
 "qrcode"                                             ,
 "R2jags"                                             ,
 "snn"                                                ,
 "stocc"                                              ,
 "MatrixModels"                                       ,
 "ifs"                                                ,
 "NNTbiomarker"                                       ,
 "rsggm"                                              ,
 "GEVStableGarch"                                     ,
 "weirs"                                              ,
 "LifeTables"                                         ,
 "MLDS"                                               ,
 "REGENT"                                             ,
 "spsi"                                               ,
 "STI"                                                ,
 "ipw"                                                ,
 "GeneralOaxaca"                                      ,
 "LotkasLaw"                                          ,
 "mixedMem"                                           ,
 "A3"                                                 ,
 "bootES"                                             ,
 "diffIRT"                                            ,
 "ART"                                                ,
 "geospt"                                             ,
 "QuantifQuantile"                                    ,
 "MDimNormn"                                          ,
 "MethylCapSig"                                       ,
 "mmc"                                                ,
 "praise"                                             ,
 "RateDistortion"                                     ,
 "restlos"                                            ,
 "cusp"                                               ,
 "MixMAP"                                             ,
 "ocomposition"                                       ,
 "TapeR"                                              ,
 "threg"                                              ,
 "perspectev"                                         ,
 "rafalib"                                            ,
 "threewords"                                         ,
 "BMhyd"                                              ,
 "CTTShiny"                                           ,
 "hashr"                                              ,
 "LFDR.MLE"                                           ,
 "credule"                                            ,
 "PCA4TS"                                             ,
 "sdwd"                                               ,
 "W2CWM2C"                                            ,
 "BenfordTests"                                       ,
 "dr"                                                 ,
 "yakmoR"                                             ,
 "ALS"                                                ,
 "taber"                                              ,
 "transcribeR"                                        ,
 "fast"                                               ,
 "GeneNet"                                            ,
 "Harvest.Tree"                                       ,
 "IsoGene"                                            ,
 "MRQoL"                                              ,
 "multcompView"                                       ,
 "nlstools"                                           ,
 "ORCME"                                              ,
 "OTE"                                                ,
 "phonTools"                                          ,
 "rpca"                                               ,
 "nortest"                                            ,
 "R2WinBUGS"                                          ,
 "signal"                                             ,
 "ClamR"                                              ,
 "HistogramTools"                                     ,
 "Pade"                                               ,
 "pco"                                                ,
 "Rgnuplot"                                           ,
 "RRNA"                                               ,
 "twitteR"                                            ,
 "wiod"                                               ,
 "base64enc"                                          ,
 "cpm"                                                ,
 "MVA"                                                ,
 "QFRM"                                               ,
 "RSclient"                                           ,
 "uuid"                                               ,
 "woe"                                                ,
 "dcGOR"                                              ,
 "momr"                                               ,
 "phyext2"                                            ,
 "qrage"                                              ,
 "rddtools"                                           ,
 "isotone"                                            ,
 "modMax"                                             ,
 "rEMM"                                               ,
 "RKlout"                                             ,
 "BayesComm"                                          ,
 "EstHer"                                             ,
 "mixPHM"                                             ,
 "RaschSampler"                                       ,
 "saturnin"                                           ,
 "SparseFactorAnalysis"                               ,
 "EMMREML"                                            ,
 "favnums"                                            ,
 "hybridHclust"                                       ,
 "PlotPrjNetworks"                                    ,
 "far"                                                ,
 "nbconvertR"                                         ,
 "RTDE"                                               ,
 "SeqGrapheR"                                         ,
 "anoint"                                             ,
 "glmgraph"                                           ,
 "nordklimdata1"                                      ,
 "thermocouple"                                       ,
 "kappalab"                                           ,
 "lpbrim"                                             ,
 "RAHRS"                                              ,
 "EDFIR"                                              ,
 "gamlss.demo"                                        ,
 "RSpincalc"                                          ,
 "captioner"                                          ,
 "efflog"                                             ,
 "ibelief"                                            ,
 "deducorrect"                                        ,
 "kfigr"                                              ,
 "locfdr"                                             ,
 "ROMIplot"                                           ,
 "ROptimizely"                                        ,
 "verification"                                       ,
 "decode"                                             ,
 "HBglm"                                              ,
 "recommenderlabBX"                                   ,
 "SoundexBR"                                          ,
 "dynpred"                                            ,
 "EstCRM"                                             ,
 "igraphdata"                                         ,
 "prettyunits"                                        ,
 "appnn"                                              ,
 "fame"                                               ,
 "MatchLinReg"                                        ,
 "WMCapacity"                                         ,
 "operators"                                          ,
 "penalizedLDA"                                       ,
 "recoder"                                            ,
 "RJaCGH"                                             ,
 "scout"                                              ,
 "selectspm"                                          ,
 "Biodem"                                             ,
 "downloader"                                         ,
 "penDvine"                                           ,
 "allanvar"                                           ,
 "aTSA"                                               ,
 "bigGP"                                              ,
 "binda"                                              ,
 "cccrm"                                              ,
 "crossval"                                           ,
 "fdrtool"                                            ,
 "GSM"                                                ,
 "HWEintrinsic"                                       ,
 "LogisticDx"                                         ,
 "longitudinal"                                       ,
 "mederrRank"                                         ,
 "PopVar"                                             ,
 "sda"                                                ,
 "spatialCovariance"                                  ,
 "sperich"                                            ,
 "st"                                                 ,
 "StatDataML"                                         ,
 "tailloss"                                           ,
 "tspmeta"                                            ,
 "vardiag"                                            ,
 "wskm"                                               ,
 "futile.any"                                         ,
 "SocialPosition"                                     ,
 "iBATCGH"                                            ,
 "baseline"                                           ,
 "gromovlab"                                          ,
 "rredis"                                             ,
 "forams"                                             ,
 "Interatrix"                                         ,
 "nat.utils"                                          ,
 "OrdFacReg"                                          ,
 "reporttools"                                        ,
 "trust"                                              ,
 "abd"                                                ,
 "hive"                                               ,
 "logcondiscr"                                        ,
 "modehunt"                                           ,
 "powerGWASinteraction"                               ,
 "rseedcalc"                                          ,
 "selectMeta"                                         ,
 "spd"                                                ,
 "tagcloud"                                           ,
 "arulesNBMiner"                                      ,
 "autoencoder"                                        ,
 "gsscopu"                                            ,
 "NHANES"                                             ,
 "SRCS"                                               ,
 "translation.ko"                                     ,
 "bayesmix"                                           ,
 "CompR"                                              ,
 "MiRSEA"                                             ,
 "spectralGP"                                         ,
 "Maeswrap"                                           ,
 "gee"                                                ,
 "KernSmooth"                                         ,
 "cvxbiclustr"                                        ,
 "knitLatex"                                          ,
 "LncMod"                                             ,
 "asypow"                                             ,
 "parallelML"                                         ,
 "parallelSVM"                                        ,
 "SenSrivastava"                                      ,
 "sorvi"                                              ,
 "TSMining"                                           ,
 "unbalanced"                                         ,
 "learnstats"                                         ,
 "XMRF"                                               ,
 "BSGS"                                               ,
 "ColorPalette"                                       ,
 "geospacom"                                          ,
 "pscore"                                             ,
 "runittotestthat"                                    ,
 "geosptdb"                                           ,
 "pcaBootPlot"                                        ,
 "selfea"                                             ,
 "ssanv"                                              ,
 "import"                                             ,
 "primes"                                             ,
 "rAvis"                                              ,
 "sbmSDP"                                             ,
 "cncaGUI"                                            ,
 "FAMILY"                                             ,
 "crrp"                                               ,
 "multibiplotGUI"                                     ,
 "Sejong"                                             ,
 "GPLTR"                                              ,
 "james.analysis"                                     ,
 "ICC"                                                ,
 "CommT"                                              ,
 "gridGraphviz"                                       ,
 "TMDb"                                               ,
 "BaBooN"                                             ,
 "DiceEval"                                           ,
 "kolmim"                                             ,
 "MSIseq"                                             ,
 "rfigshare"                                          ,
 "sgRSEA"                                             ,
 "blme"                                               ,
 "hierband"                                           ,
 "nodeHarvest"                                        ,
 "strum"                                              ,
 "EntropyExplorer"                                    ,
 "LTPDvar"                                            ,
 "P2C2M"                                              ,
 "ssfa"                                               ,
 "DNMF"                                               ,
 "patchDVI"                                           ,
 "RcppFaddeeva"                                       ,
 "rknn"                                               ,
 "ISwR"                                               ,
 "BayesGESM"                                          ,
 "strucchange"                                        ,
 "seismic"                                            ,
 "kineticF"                                           ,
 "delt"                                               ,
 "EnviroStat"                                         ,
 "kwb.hantush"                                        ,
 "score"                                              ,
 "surveyeditor"                                       ,
 "XHWE"                                               ,
 "bootSVD"                                            ,
 "cccd"                                               ,
 "evolvability"                                       ,
 "FAOSTAT"                                            ,
 "JBTools"                                            ,
 "spanel"                                             ,
 "spectral.methods"                                   ,
 "dkDNA"                                              ,
 "coefficientalpha"                                   ,
 "TDAmapper"                                          ,
 "hybridEnsemble"                                     ,
 "RcellData"                                          ,
 "signmedian.test"                                    ,
 "dostats"                                            ,
 "esaBcv"                                             ,
 "fdakma"                                             ,
 "ncdf.tools"                                         ,
 "rprime"                                             ,
 "RSPS"                                               ,
 "loopr"                                              ,
 "packS4"                                             ,
 "BACA"                                               ,
 "empiricalFDR.DESeq2"                                ,
 "reval"                                              ,
 "restimizeapi"                                       ,
 "bcv"                                                ,
 "dataview"                                           ,
 "denpro"                                             ,
 "frbs"                                               ,
 "tmle.npvi"                                          ,
 "LDPD"                                               ,
 "ordPens"                                            ,
 "R0"                                                 ,
 "riceware"                                           ,
 "xtermStyle"                                         ,
 "ARTIVA"                                             ,
 "bigml"                                              ,
 "CausalFX"                                           ,
 "clustertend"                                        ,
 "shapeR"                                             ,
 "SubpathwayGMir"                                     ,
 "TickExec"                                           ,
 "VIGoR"                                              ,
 "greyzoneSurv"                                       ,
 "QCAfalsePositive"                                   ,
 "rivervis"                                           ,
 "MInt"                                               ,
 "supcluster"                                         ,
 "TTmoment"                                           ,
 "NEff"                                               ,
 "pmr"                                                ,
 "rsem"                                               ,
 "geoscale"                                           ,
 "MCTM"                                               ,
 "DSviaDRM"                                           ,
 "fastHICA"                                           ,
 "stagePop"                                           ,
 "boilerpipeR"                                        ,
 "GB2"                                                ,
 "hapassoc"                                           ,
 "mglmn"                                              ,
 "probFDA"                                            ,
 "tm.plugin.webmining"                                ,
 "csn"                                                ,
 "aprean3"                                            ,
 "asVPC"                                              ,
 "lsa"                                                ,
 "dummy"                                              ,
 "fishmove"                                           ,
 "RImageJROI"                                         ,
 "Scale"                                              ,
 "abc"                                                ,
 "abc.data"                                           ,
 "hiertest"                                           ,
 "LocFDRPois"                                         ,
 "optifunset"                                         ,
 "reGenotyper"                                        ,
 "structSSI"                                          ,
 "clustering.sc.dp"                                   ,
 "parboost"                                           ,
 "EW"                                                 ,
 "coxinterval"                                        ,
 "EvalEst"                                            ,
 "GDAdata"                                            ,
 "ig.vancouver.2014.topcolour"                        ,
 "mixtNB"                                             ,
 "muir"                                               ,
 "survJamda.data"                                     ,
 "CDNmoney"                                           ,
 "TScompare"                                          ,
 "tsfa"                                               ,
 "uniftest"                                           ,
 "bspec"                                              ,
 "constrainedKriging"                                 ,
 "rCMA"                                               ,
 "TSfame"                                             ,
 "TSMySQL"                                            ,
 "TSodbc"                                             ,
 "TSPostgreSQL"                                       ,
 "TSSQLite"                                           ,
 "synbreedData"                                       ,
 "uniqtag"                                            ,
 "pairsD3"                                            ,
 "simpleNeural"                                       ,
 "timetree"                                           ,
 "mded"                                               ,
 "migui"                                              ,
 "LinRegInteractive"                                  ,
 "mapStats"                                           ,
 "probemod"                                           ,
 "treeperm"                                           ,
 "blockmodels"                                        ,
 "ensurer"                                            ,
 "managelocalrepo"                                    ,
 "bayescount"                                         ,
 "oro.dicom"                                          ,
 "prevalence"                                         ,
 "synRNASeqNet"                                       ,
 "R1magic"                                            ,
 "BivarP"                                             ,
 "datacheck"                                          ,
 "TKF"                                                ,
 "marl"                                               ,
 "dga"                                                ,
 "mi"                                                 ,
 "networksis"                                         ,
 "smoother"                                           ,
 "cernn"                                              ,
 "REST"                                               ,
 "rmaf"                                               ,
 "SPRT"                                               ,
 "bild"                                               ,
 "CRTSize"                                            ,
 "MissMech"                                           ,
 "fCertificates"                                      ,
 "HiLMM"                                              ,
 "NbClust"                                            ,
 "rstackdeque"                                        ,
 "smds"                                               ,
 "csrplus"                                            ,
 "resample"                                           ,
 "fdth"                                               ,
 "msSurv"                                             ,
 "BayesLCA"                                           ,
 "bentcableAR"                                        ,
 "chipPCR"                                            ,
 "expsmooth"                                          ,
 "groupRemMap"                                        ,
 "cqrReg"                                             ,
 "searchable"                                         ,
 "softImpute"                                         ,
 "SOUP"                                               ,
 "EcoSimR"                                            ,
 "RKEA"                                               ,
 "thsls"                                              ,
 "DiffCorr"                                           ,
 "HighDimOut"                                         ,
 "maptpx"                                             ,
 "PermAlgo"                                           ,
 "HarmonicRegression"                                 ,
 "KMDA"                                               ,
 "Laterality"                                         ,
 "orthogonalsplinebasis"                              ,
 "shinyBS"                                            ,
 "Ake"                                                ,
 "VdgRsm"                                             ,
 "AutoSEARCH"                                         ,
 "LPStimeSeries"                                      ,
 "SAMUR"                                              ,
 "StatMeasures"                                       ,
 "kmodR"                                              ,
 "ROCR"                                               ,
 "timeROC"                                            ,
 "tumblR"                                             ,
 "mapfit"                                             ,
 "rgexf"                                              ,
 "mht"                                                ,
 "hierDiversity"                                      ,
 "CARLIT"                                             ,
 "jtrans"                                             ,
 "saccades"                                           ,
 "DCluster"                                           ,
 "rsatscan"                                           ,
 "staTools"                                           ,
 "genalg"                                             ,
 "gvcm.cat"                                           ,
 "separationplot"                                     ,
 "betas"                                              ,
 "boottol"                                            ,
 "funFEM"                                             ,
 "GWAF"                                               ,
 "blatr"                                              ,
 "MCL"                                                ,
 "multigroup"                                         ,
 "NORRRM"                                             ,
 "binr"                                               ,
 "relevent"                                           ,
 "RVFam"                                              ,
 "informR"                                            ,
 "SID"                                                ,
 "alabama"                                            ,
 "recluster"                                          ,
 "CAM"                                                ,
 "DAMOCLES"                                           ,
 "gtop"                                               ,
 "abf2"                                               ,
 "monreg"                                             ,
 "BayesSummaryStatLM"                                 ,
 "BOG"                                                ,
 "coalescentMCMC"                                     ,
 "FieldSim"                                           ,
 "hornpa"                                             ,
 "LPTime"                                             ,
 "NAPPA"                                              ,
 "NSUM"                                               ,
 "RcmdrPlugin.EACSPIR"                                ,
 "simMSM"                                             ,
 "stheoreme"                                          ,
 "IsingSampler"                                       ,
 "lsr"                                                ,
 "PogromcyDanych"                                     ,
 "Conake"                                             ,
 "gendata"                                            ,
 "laketemps"                                          ,
 "timma"                                              ,
 "CoxRidge"                                           ,
 "GenForImp"                                          ,
 "dplRCon"                                            ,
 "gte"                                                ,
 "RcmdrPlugin.ROC"                                    ,
 "SurvCorr"                                           ,
 "aroma.apd"                                          ,
 "FGSG"                                               ,
 "ihs"                                                ,
 "multiPIM"                                           ,
 "SDD"                                                ,
 "autopls"                                            ,
 "babar"                                              ,
 "coloredICA"                                         ,
 "coxsei"                                             ,
 "FFD"                                                ,
 "likelihood"                                         ,
 "mwa"                                                ,
 "phenability"                                        ,
 "SvyNom"                                             ,
 "crrstep"                                            ,
 "games"                                              ,
 "SurvLong"                                           ,
 "wahc"                                               ,
 "wmlf"                                               ,
 "highTtest"                                          ,
 "R.huge"                                             ,
 "ssize.fdr"                                          ,
 "frm"                                                ,
 "lmeNBBayes"                                         ,
 "faisalconjoint"                                     ,
 "fastpseudo"                                         ,
 "msda"                                               ,
 "UBCRM"                                              ,
 "fgpt"                                               ,
 "Simile"                                             ,
 "clusterGeneration"                                  ,
 "lbfgsb3"                                            ,
 "MortalitySmooth"                                    ,
 "rjade"                                              ,
 "analyz"                                             ,
 "ATE"                                                ,
 "GaDiFPT"                                            ,
 "hpcwld"                                             ,
 "IBDhaploRtools"                                     ,
 "IBDLabels"                                          ,
 "bde"                                                ,
 "glba"                                               ,
 "LPS"                                                ,
 "PCIT"                                               ,
 "randomUniformForest"                                ,
 "nFCA"                                               ,
 "minimist"                                           ,
 "phytotools"                                         ,
 "demi"                                               ,
 "plotGoogleMaps"                                     ,
 "ROAuth"                                             ,
 "tbart"                                              ,
 "FastKNN"                                            ,
 "kselection"                                         ,
 "condMVNorm"                                         ,
 "rNMF"                                               ,
 "shinybootstrap2"                                    ,
 "tmg"                                                ,
 "treelet"                                            ,
 "cccp"                                               ,
 "ICAFF"                                              ,
 "ivpanel"                                            ,
 "cec2005benchmark"                                   ,
 "MetNorm"                                            ,
 "SVMMatch"                                           ,
 "klausuR"                                            ,
 "knncat"                                             ,
 "SimilarityMeasures"                                 ,
 "BayesMed"                                           ,
 "fdatest"                                            ,
 "svKomodo"                                           ,
 "BLCOP"                                              ,
 "poisDoubleSamp"                                     ,
 "RPPairwiseDesign"                                   ,
 "lmeNB"                                              ,
 "multicon"                                           ,
 "royston"                                            ,
 "SETPath"                                            ,
 "hbm"                                                ,
 "d3Network"                                          ,
 "infra"                                              ,
 "LMERConvenienceFunctions"                           ,
 "multiAssetOptions"                                  ,
 "kissmig"                                            ,
 "ppmlasso"                                           ,
 "mousetrack"                                         ,
 "portfolio"                                          ,
 "roughrf"                                            ,
 "compareC"                                           ,
 "flower"                                             ,
 "KFKSDS"                                             ,
 "metaMA"                                             ,
 "SLHD"                                               ,
 "SMNCensReg"                                         ,
 "geoCount"                                           ,
 "mvnpermute"                                         ,
 "tufterhandout"                                      ,
 "metaRNASeq"                                         ,
 "CensRegMod"                                         ,
 "pbivnorm"                                           ,
 "remMap"                                             ,
 "cec2013"                                            ,
 "ESEA"                                               ,
 "sig"                                                ,
 "ArfimaMLM"                                          ,
 "drmdel"                                             ,
 "ggm"                                                ,
 "imputeLCMD"                                         ,
 "HTMLUtils"                                          ,
 "anfis"                                              ,
 "OrdinalLogisticBiplot"                              ,
 "MultiMeta"                                          ,
 "ASPBay"                                             ,
 "mailR"                                              ,
 "mztwinreg"                                          ,
 "robustDA"                                           ,
 "SPODT"                                              ,
 "chromer"                                            ,
 "HierO"                                              ,
 "MBmca"                                              ,
 "readMLData"                                         ,
 "Sunder"                                             ,
 "traj"                                               ,
 "CNOGpro"                                            ,
 "DEEPR"                                              ,
 "penMSM"                                             ,
 "leiv"                                               ,
 "aoristic"                                           ,
 "neuRosim"                                           ,
 "randomizationInference"                             ,
 "textometry"                                         ,
 "interferenceCI"                                     ,
 "noia"                                               ,
 "sra"                                                ,
 "BEQI2"                                              ,
 "gdimap"                                             ,
 "tsc"                                                ,
 "moments"                                            ,
 "RadTran"                                            ,
 "SCEPtER"                                            ,
 "EntropyEstimation"                                  ,
 "jiebaRD"                                            ,
 "dropR"                                              ,
 "ForImp"                                             ,
 "TreePar"                                            ,
 "glrt"                                               ,
 "Storm"                                              ,
 "CNprep"                                             ,
 "YaleToolkit"                                        ,
 "CORE"                                               ,
 "ParDNAcopy"                                         ,
 "diffEq"                                             ,
 "dupiR"                                              ,
 "ecolMod"                                            ,
 "raters"                                             ,
 "TBEST"                                              ,
 "BNDataGenerator"                                    ,
 "lm.beta"                                            ,
 "extraTrees"                                         ,
 "LIM"                                                ,
 "NetIndices"                                         ,
 "RConics"                                            ,
 "additivityTests"                                    ,
 "historydata"                                        ,
 "manipulate"                                         ,
 "OutlierDM"                                          ,
 "OutrankingTools"                                    ,
 "MPDiR"                                              ,
 "PredictABEL"                                        ,
 "qmrparser"                                          ,
 "reports"                                            ,
 "NORTARA"                                            ,
 "orgR"                                               ,
 "SCEPtERbinary"                                      ,
 "tcltk2"                                             ,
 "sBF"                                                ,
 "DCGL"                                               ,
 "multiband"                                          ,
 "RoughSetKnowledgeReduction"                         ,
 "selfingTree"                                        ,
 "sprinter"                                           ,
 "RcppDL"                                             ,
 "rSFA"                                               ,
 "leaderCluster"                                      ,
 "Vdgraph"                                            ,
 "DSpat"                                              ,
 "c060"                                               ,
 "bnormnlr"                                           ,
 "DnE"                                                ,
 "gsalib"                                             ,
 "cvAUC"                                              ,
 "isopam"                                             ,
 "RDataCanvas"                                        ,
 "bdynsys"                                            ,
 "breakage"                                           ,
 "ccda"                                               ,
 "extrafont"                                          ,
 "rbounds"                                            ,
 "BayesianAnimalTracker"                              ,
 "dashboard"                                          ,
 "RColorBrewer"                                       ,
 "Rcgmin"                                             ,
 "okmesonet"                                          ,
 "BEDASSLE"                                           ,
 "extWeibQuant"                                       ,
 "HSSVD"                                              ,
 "NPS"                                                ,
 "sensitivityPStrat"                                  ,
 "rngSetSeed"                                         ,
 "sequences"                                          ,
 "gamboostMSM"                                        ,
 "MOrder"                                             ,
 "plgp"                                               ,
 "radar"                                              ,
 "turfR"                                              ,
 "VAR.etp"                                            ,
 "kriging"                                            ,
 "MindOnStats"                                        ,
 "ssd"                                                ,
 "rngwell19937"                                       ,
 "mco"                                                ,
 "rainfreq"                                           ,
 "DECIDE"                                             ,
 "miRtest"                                            ,
 "superbiclust"                                       ,
 "EstSimPDMP"                                         ,
 "kpodclustr"                                         ,
 "moduleColor"                                        ,
 "SunterSampling"                                     ,
 "acss"                                               ,
 "GPArotation"                                        ,
 "gsarima"                                            ,
 "Julia"                                              ,
 "maboost"                                            ,
 "musicNMR"                                           ,
 "setRNG"                                             ,
 "SMR"                                                ,
 "soma"                                               ,
 "choplump"                                           ,
 "sgr"                                                ,
 "someKfwer"                                          ,
 "magrittr"                                           ,
 "proteomics"                                         ,
 "BACprior"                                           ,
 "crmn"                                               ,
 "gset"                                               ,
 "ilc"                                                ,
 "RAPIDR"                                             ,
 "x12GUI"                                             ,
 "QuACN"                                              ,
 "SEAsic"                                             ,
 "sqlutils"                                           ,
 "spatialTailDep"                                     ,
 "randtests"                                          ,
 "entropy"                                            ,
 "toxtestD"                                           ,
 "catdata"                                            ,
 "lymphclon"                                          ,
 "episplineDensity"                                   ,
 "ionflows"                                           ,
 "ISBF"                                               ,
 "ModelGood"                                          ,
 "Rcpp11"                                             ,
 "Rothermel"                                          ,
 "riskSimul"                                          ,
 "LinCal"                                             ,
 "OptionPricing"                                      ,
 "BB"                                                 ,
 "BTYD"                                               ,
 "PLSbiplot1"                                         ,
 "hcp"                                                ,
 "MAT"                                                ,
 "MGL"                                                ,
 "strap"                                              ,
 "CombinePValue"                                      ,
 "ScoreGGUM"                                          ,
 "sdcTarget"                                          ,
 "bcpa"                                               ,
 "kaps"                                               ,
 "RMTstat"                                            ,
 "smac"                                               ,
 "tuple"                                              ,
 "isdals"                                             ,
 "ldr"                                                ,
 "CateSelection"                                      ,
 "lassoscore"                                         ,
 "qdm"                                                ,
 "ropensecretsapi"                                    ,
 "highD2pop"                                          ,
 "ivpack"                                             ,
 "FDGcopulas"                                         ,
 "JMdesign"                                           ,
 "icapca"                                             ,
 "MDSGUI"                                             ,
 "Rook"                                               ,
 "multispatialCCM"                                    ,
 "landpred"                                           ,
 "rareNMtests"                                        ,
 "stochprofML"                                        ,
 "gsg"                                                ,
 "pBrackets"                                          ,
 "RMKdiscrete"                                        ,
 "Metatron"                                           ,
 "NPCirc"                                             ,
 "RcmdrPlugin.steepness"                              ,
 "windex"                                             ,
 "AlgDesign"                                          ,
 "nsgp"                                               ,
 "vetools"                                            ,
 "dice"                                               ,
 "datamart"                                           ,
 "popRange"                                           ,
 "QualInt"                                            ,
 "minqa"                                              ,
 "mpcv"                                               ,
 "Rdsm"                                               ,
 "CINID"                                              ,
 "mnormpow"                                           ,
 "VecStatGraphs3D"                                    ,
 "GISTools"                                           ,
 "lllcrc"                                             ,
 "mvprpb"                                             ,
 "NB"                                                 ,
 "catIrt"                                             ,
 "SightabilityModel"                                  ,
 "vec2dtransf"                                        ,
 "TED"                                                ,
 "exsic"                                              ,
 "SQDA"                                               ,
 "steepness"                                          ,
 "atmcmc"                                             ,
 "GPFDA"                                              ,
 "httpRequest"                                        ,
 "rFDSN"                                              ,
 "neural"                                             ,
 "GenWin"                                             ,
 "Guerry"                                             ,
 "jSonarR"                                            ,
 "pvsR"                                               ,
 "documair"                                           ,
 "IBrokers"                                           ,
 "RcmdrPlugin.MA"                                     ,
 "sendmailR"                                          ,
 "trotter"                                            ,
 "Density.T.HoldOut"                                  ,
 "beanplot"                                           ,
 "edgeRun"                                            ,
 "iki.dataclim"                                       ,
 "opentraj"                                           ,
 "RidgeFusion"                                        ,
 "spatialnbda"                                        ,
 "bbo"                                                ,
 "modTempEff"                                         ,
 "qgtools"                                            ,
 "RadioSonde"                                         ,
 "compute.es"                                         ,
 "propOverlap"                                        ,
 "RcmdrPlugin.coin"                                   ,
 "phyloland"                                          ,
 "referenceIntervals"                                 ,
 "PepPrep"                                            ,
 "knnIndep"                                           ,
 "lefse"                                              ,
 "RankResponse"                                       ,
 "rChoiceDialogs"                                     ,
 "rsubgroup"                                          ,
 "eegkitdata"                                         ,
 "ForwardSearch"                                      ,
 "hwriter"                                            ,
 "saery"                                              ,
 "bpkde"                                              ,
 "globalOptTests"                                     ,
 "astro"                                              ,
 "lunar"                                              ,
 "minque"                                             ,
 "bionetdata"                                         ,
 "PerfMeas"                                           ,
 "GxM"                                                ,
 "BCA"                                                ,
 "cooptrees"                                          ,
 "helloJavaWorld"                                     ,
 "idr"                                                ,
 "insuranceData"                                      ,
 "MRCV"                                               ,
 "parcor"                                             ,
 "pqantimalarials"                                    ,
 "SDaA"                                               ,
 "sparseSEM"                                          ,
 "coreTDT"                                            ,
 "exactmeta"                                          ,
 "optrees"                                            ,
 "Rbitcoin"                                           ,
 "traitr"                                             ,
 "lbfgs"                                              ,
 "BIOM.utils"                                         ,
 "Eplot"                                              ,
 "raincpc"                                            ,
 "Rcapture"                                           ,
 "bfast"                                              ,
 "cgAUC"                                              ,
 "RcmdrPlugin.DoE"                                    ,
 "MultiSV"                                            ,
 "ptinpoly"                                           ,
 "BTSPAS"                                             ,
 "cents"                                              ,
 "KANT"                                               ,
 "commandr"                                           ,
 "granova"                                            ,
 "mkde"                                               ,
 "AMOEBA"                                             ,
 "labeling"                                           ,
 "mppa"                                               ,
 "pauwels2014"                                        ,
 "scaRabee"                                           ,
 "wpp2012"                                            ,
 "xlsxjars"                                           ,
 "SocialNetworks"                                     ,
 "matpow"                                             ,
 "extlasso"                                           ,
 "FD"                                                 ,
 "bigmemory.sri"                                      ,
 "mvrtn"                                              ,
 "snpar"                                              ,
 "CVTuningCov"                                        ,
 "gCat"                                               ,
 "phtt"                                               ,
 "pycno"                                              ,
 "rtype"                                              ,
 "AntWeb"                                             ,
 "RFLPtools"                                          ,
 "trimTrees"                                          ,
 "dicionariosIBGE"                                    ,
 "growthrate"                                         ,
 "LEAPFrOG"                                           ,
 "linkcomm"                                           ,
 "tvd"                                                ,
 "astrodatR"                                          ,
 "Xmisc"                                              ,
 "ibeemd"                                             ,
 "threeboost"                                         ,
 "hglasso"                                            ,
 "RWebLogo"                                           ,
 "tiger"                                              ,
 "UScancer"                                           ,
 "vrtest"                                             ,
 "astrolibR"                                          ,
 "ecipex"                                             ,
 "itertools2"                                         ,
 "osDesign"                                           ,
 "symmoments"                                         ,
 "interAdapt"                                         ,
 "sanitizers"                                         ,
 "MGRASTer"                                           ,
 "optiscale"                                          ,
 "subgroup"                                           ,
 "agsemisc"                                           ,
 "GeoDE"                                              ,
 "ExceedanceTools"                                    ,
 "rareGE"                                             ,
 "NeatMap"                                            ,
 "cosinor"                                            ,
 "cvxclustr"                                          ,
 "infotheo"                                           ,
 "stsm.class"                                         ,
 "nicheROVER"                                         ,
 "mtk"                                                ,
 "sensitivitymw"                                      ,
 "ssh.utils"                                          ,
 "ReorderCluster"                                     ,
 "rnaseqWrapper"                                      ,
 "fueleconomy"                                        ,
 "leapp"                                              ,
 "mewAvg"                                             ,
 "nasaweather"                                        ,
 "nontargetData"                                      ,
 "RMRAINGEN"                                          ,
 "GeoGenetix"                                         ,
 "ineq"                                               ,
 "gWidgetstcltk"                                      ,
 "ProgGUIinR"                                         ,
 "SBRect"                                             ,
 "functional"                                         ,
 "GCAI.bias"                                          ,
 "hillmakeR"                                          ,
 "translate"                                          ,
 "quad"                                               ,
 "ROSE"                                               ,
 "popKorn"                                            ,
 "CondReg"                                            ,
 "COSINE"                                             ,
 "metafolio"                                          ,
 "rysgran"                                            ,
 "crossReg"                                           ,
 "translateR"                                         ,
 "EvoRAG"                                             ,
 "RmixmodCombi"                                       ,
 "rkvo"                                               ,
 "ForIT"                                              ,
 "Thinknum"                                           ,
 "mcc"                                                ,
 "qualCI"                                             ,
 "FunctionalNetworks"                                 ,
 "MAVTgsa"                                            ,
 "sudoku"                                             ,
 "CORM"                                               ,
 "GenBinomApps"                                       ,
 "stressr"                                            ,
 "ezsim"                                              ,
 "pedigreemm"                                         ,
 "RelValAnalysis"                                     ,
 "sp23design"                                         ,
 "nlsMicrobio"                                        ,
 "FisHiCal"                                           ,
 "matlab"                                             ,
 "RWBP"                                               ,
 "rwt"                                                ,
 "rClinicalCodes"                                     ,
 "pbdNCDF4"                                           ,
 "TaoTeProgramming"                                   ,
 "bcpmeta"                                            ,
 "parallelMCMCcombine"                                ,
 "CommunityCorrelogram"                               ,
 "hazus"                                              ,
 "localsolver"                                        ,
 "tictoc"                                             ,
 "arnie"                                              ,
 "xhmmScripts"                                        ,
 "rbitcoinchartsapi"                                  ,
 "DeducerText"                                        ,
 "ElstonStewart"                                      ,
 "ESGtoolkit"                                         ,
 "PCDSpline"                                          ,
 "qVarSel"                                            ,
 "ustyc"                                              ,
 "tm.plugin.alceste"                                  ,
 "gsmoothr"                                           ,
 "HiCseg"                                             ,
 "ICsurv"                                             ,
 "ORCI"                                               ,
 "IsoCI"                                              ,
 "genasis"                                            ,
 "Interact"                                           ,
 "AR1seg"                                             ,
 "vbsr"                                               ,
 "ChargeTransport"                                    ,
 "rsdepth"                                            ,
 "sas7bdat"                                           ,
 "ebSNP"                                              ,
 "dissUtils"                                          ,
 "rHealthDataGov"                                     ,
 "pbo"                                                ,
 "interventionalDBN"                                  ,
 "kdetrees"                                           ,
 "SensitivityCaseControl"                             ,
 "ordBTL"                                             ,
 "bmmix"                                              ,
 "metagen"                                            ,
 "SRRS"                                               ,
 "rpart.utils"                                        ,
 "ucbthesis"                                          ,
 "BCE"                                                ,
 "spatial.gev.bma"                                    ,
 "hawkes"                                             ,
 "cstar"                                              ,
 "MC2toPath"                                          ,
 "boostr"                                             ,
 "dynia"                                              ,
 "erpR"                                               ,
 "hydroApps"                                          ,
 "rvHPDT"                                             ,
 "cutoffR"                                            ,
 "samplingEstimates"                                  ,
 "akmeans"                                            ,
 "msap"                                               ,
 "PPtree"                                             ,
 "rbundler"                                           ,
 "freestats"                                          ,
 "noncensus"                                          ,
 "bandit"                                             ,
 "FLR"                                                ,
 "KappaV"                                             ,
 "truncSP"                                            ,
 "sglr"                                               ,
 "irtProb"                                            ,
 "NominalLogisticBiplot"                              ,
 "MultiCNVDetect"                                     ,
 "nplplot"                                            ,
 "CMPControl"                                         ,
 "lar"                                                ,
 "SoilR"                                              ,
 "bbefkr"                                             ,
 "EasyMARK"                                           ,
 "mcheatmaps"                                         ,
 "plusser"                                            ,
 "DynClust"                                           ,
 "flux"                                               ,
 "RSvgDevice"                                         ,
 "clusterfly"                                         ,
 "EvCombR"                                            ,
 "MILC"                                               ,
 "classifly"                                          ,
 "meifly"                                             ,
 "micEconCES"                                         ,
 "SNPassoc"                                           ,
 "apmsWAPP"                                           ,
 "kcirt"                                              ,
 "micEconSNQP"                                        ,
 "GSAgm"                                              ,
 "RApiSerialize"                                      ,
 "RTextureMetrics"                                    ,
 "rWBclimate"                                         ,
 "ATmet"                                              ,
 "poisson.glm.mix"                                    ,
 "Johnson"                                            ,
 "NBPSeq"                                             ,
 "RGIFT"                                              ,
 "scar"                                               ,
 "scio"                                               ,
 "AMORE"                                              ,
 "LOGICOIL"                                           ,
 "RcppBDT"                                            ,
 "BootPR"                                             ,
 "earlywarnings"                                      ,
 "gRapfa"                                             ,
 "rvTDT"                                              ,
 "ora"                                                ,
 "cuttlefish.model"                                   ,
 "GHQp"                                               ,
 "disclap"                                            ,
 "acss.data"                                          ,
 "bbemkr"                                             ,
 "isotonic.pen"                                       ,
 "doRedis"                                            ,
 "Methplot"                                           ,
 "pcg"                                                ,
 "lpint"                                              ,
 "tosls"                                              ,
 "evdbayes"                                           ,
 "r2d2"                                               ,
 "partsm"                                             ,
 "aCRM"                                               ,
 "FacPad"                                             ,
 "invGauss"                                           ,
 "MMMS"                                               ,
 "polyaAeppli"                                        ,
 "iFad"                                               ,
 "Rphylip"                                            ,
 "clValid"                                            ,
 "CMF"                                                ,
 "Devore7"                                            ,
 "ivfixed"                                            ,
 "CCpop"                                              ,
 "lsdv"                                               ,
 "OutlierDC"                                          ,
 "Stack"                                              ,
 "deepnet"                                            ,
 "Paneldata"                                          ,
 "pspearman"                                          ,
 "fontcm"                                             ,
 "PKreport"                                           ,
 "IPMpack"                                            ,
 "uplift"                                             ,
 "sbioPN"                                             ,
 "pensim"                                             ,
 "ykmeans"                                            ,
 "PharmPow"                                           ,
 "binomlogit"                                         ,
 "itertools"                                          ,
 "LIStest"                                            ,
 "normtest"                                           ,
 "plotMCMC"                                           ,
 "SASmixed"                                           ,
 "hqmisc"                                             ,
 "XML2R"                                              ,
 "BurStFin"                                           ,
 "gpairs"                                             ,
 "regsubseq"                                          ,
 "geneSignatureFinder"                                ,
 "gptk"                                               ,
 "Jmisc"                                              ,
 "svyPVpack"                                          ,
 "covreg"                                             ,
 "eiwild"                                             ,
 "flowfield"                                          ,
 "mixRasch"                                           ,
 "bagRboostR"                                         ,
 "FDRreg"                                             ,
 "hbim"                                               ,
 "inarmix"                                            ,
 "bioPN"                                              ,
 "Table1Heatmap"                                      ,
 "ICGE"                                               ,
 "simFrame"                                           ,
 "SSsimple"                                           ,
 "widals"                                             ,
 "optimbase"                                          ,
 "partitionMetric"                                    ,
 "svDialogstcltk"                                     ,
 "svHttp"                                             ,
 "svSocket"                                           ,
 "svUnit"                                             ,
 "aqr"                                                ,
 "mcll"                                               ,
 "bPeaks"                                             ,
 "morgenstemning"                                     ,
 "GDELTtools"                                         ,
 "panelAR"                                            ,
 "rvgtest"                                            ,
 "snp.plotter"                                        ,
 "SimuChemPC"                                         ,
 "fractaldim"                                         ,
 "gridBase"                                           ,
 "mvc"                                                ,
 "RegClust"                                           ,
 "RYoudaoTranslate"                                   ,
 "blender"                                            ,
 "bursts"                                             ,
 "hcci"                                               ,
 "plumbr"                                             ,
 "arf3DS4"                                            ,
 "eulerian"                                           ,
 "lpmodeler"                                          ,
 "ncdf4.helpers"                                      ,
 "SPAr"                                               ,
 "zendeskR"                                           ,
 "lorec"                                              ,
 "OLScurve"                                           ,
 "RFreak"                                             ,
 "mGSZ"                                               ,
 "edeR"                                               ,
 "praktikum"                                          ,
 "RcmdrPlugin.pointG"                                 ,
 "turner"                                             ,
 "DTComPair"                                          ,
 "sdtoolkit"                                          ,
 "CRAC"                                               ,
 "prospectr"                                          ,
 "surface"                                            ,
 "rsae"                                               ,
 "bio.infer"                                          ,
 "mcr"                                                ,
 "QCGWAS"                                             ,
 "chromoR"                                            ,
 "HMMCont"                                            ,
 "npst"                                               ,
 "cpca"                                               ,
 "gpr"                                                ,
 "paf"                                                ,
 "MetABEL"                                            ,
 "vbdm"                                               ,
 "SBSA"                                               ,
 "pSI"                                                ,
 "VoxR"                                               ,
 "CVcalibration"                                      ,
 "ebal"                                               ,
 "MergeGUI"                                           ,
 "Synth"                                              ,
 "sqliter"                                            ,
 "PEMM"                                               ,
 "gconcord"                                           ,
 "LSC"                                                ,
 "interval"                                           ,
 "jpeg"                                               ,
 "saws"                                               ,
 "sigclust"                                           ,
 "psyphy"                                             ,
 "rateratio.test"                                     ,
 "HUM"                                                ,
 "blockmatrix"                                        ,
 "HBSTM"                                              ,
 "SCORER2"                                            ,
 "wpp2008"                                            ,
 "dfexplore"                                          ,
 "multinbmod"                                         ,
 "clustsig"                                           ,
 "iDynoR"                                             ,
 "linkim"                                             ,
 "RockFab"                                            ,
 "MF"                                                 ,
 "DSBayes"                                            ,
 "citbcmst"                                           ,
 "citccmst"                                           ,
 "jointPm"                                            ,
 "networkTomography"                                  ,
 "plRasch"                                            ,
 "poLCA"                                              ,
 "cwm"                                                ,
 "dagR"                                               ,
 "gpmap"                                              ,
 "rapportools"                                        ,
 "robcor"                                             ,
 "sfa"                                                ,
 "RClimMAWGEN"                                        ,
 "DESnowball"                                         ,
 "SWATmodel"                                          ,
 "binom"                                              ,
 "FAMT"                                               ,
 "PedCNV"                                             ,
 "robustgam"                                          ,
 "mmm"                                                ,
 "PLRModels"                                          ,
 "missForest"                                         ,
 "mmm2"                                               ,
 "TiddlyWikiR"                                        ,
 "dtt"                                                ,
 "seawaveQ"                                           ,
 "acm4r"                                              ,
 "rgabriel"                                           ,
 "extfunnel"                                          ,
 "ROC632"                                             ,
 "saves"                                              ,
 "SIN"                                                ,
 "cpk"                                                ,
 "mycobacrvR"                                         ,
 "evt0"                                               ,
 "RcmdrPlugin.sos"                                    ,
 "safeBinaryRegression"                               ,
 "TSTutorial"                                         ,
 "MigClim"                                            ,
 "SHIP"                                               ,
 "pa"                                                 ,
 "pmcgd"                                              ,
 "AssotesteR"                                         ,
 "BBRecapture"                                        ,
 "colortools"                                         ,
 "bvls"                                               ,
 "cellVolumeDist"                                     ,
 "HWEBayes"                                           ,
 "VHDClassification"                                  ,
 "ycinterextra"                                       ,
 "g.data"                                             ,
 "iRefR"                                              ,
 "reportRx"                                           ,
 "RobRSVD"                                            ,
 "fcd"                                                ,
 "MVB"                                                ,
 "svdvisual"                                          ,
 "DunnettTests"                                       ,
 "MiST"                                               ,
 "HIV.LifeTables"                                     ,
 "logconcens"                                         ,
 "Oncotree"                                           ,
 "skda"                                               ,
 "BACCO"                                              ,
 "dmt"                                                ,
 "mleur"                                              ,
 "sealasso"                                           ,
 "etable"                                             ,
 "mlogitBMA"                                          ,
 "rpartitions"                                        ,
 "SOAR"                                               ,
 "conics"                                             ,
 "fxregime"                                           ,
 "pathmox"                                            ,
 "sparseLTSEigen"                                     ,
 "DIME"                                               ,
 "vrmlgen"                                            ,
 "discrimARTs"                                        ,
 "hflights"                                           ,
 "DynamicDistribution"                                ,
 "MaXact"                                             ,
 "Sample.Size"                                        ,
 "snpStatsWriter"                                     ,
 "obs.agree"                                          ,
 "DEMEtics"                                           ,
 "Interpol.T"                                         ,
 "png"                                                ,
 "RCassandra"                                         ,
 "GrassmannOptim"                                     ,
 "superMDS"                                           ,
 "exptest"                                            ,
 "cycloids"                                           ,
 "aml"                                                ,
 "LCFdata"                                            ,
 "prognosticROC"                                      ,
 "LICORS"                                             ,
 "dils"                                               ,
 "frontiles"                                          ,
 "humanFormat"                                        ,
 "growthmodels"                                       ,
 "GMMBoost"                                           ,
 "msgpackR"                                           ,
 "osmar"                                              ,
 "PubBias"                                            ,
 "mlDNA"                                              ,
 "somebm"                                             ,
 "VLF"                                                ,
 "codadiags"                                          ,
 "eqs2lavaan"                                         ,
 "MetStaT"                                            ,
 "rbefdata"                                           ,
 "vscc"                                               ,
 "grade"                                              ,
 "DiscriMiner"                                        ,
 "PropScrRand"                                        ,
 "tester"                                             ,
 "HaploSim"                                           ,
 "mreg"                                               ,
 "colorfulVennPlot"                                   ,
 "robustfa"                                           ,
 "clv"                                                ,
 "GenKern"                                            ,
 "rGammaGamma"                                        ,
 "aqfig"                                              ,
 "dielectric"                                         ,
 "learningr"                                          ,
 "RGenetics"                                          ,
 "comparison"                                         ,
 "sspline"                                            ,
 "gskat"                                              ,
 "matie"                                              ,
 "mvShapiroTest"                                      ,
 "someMTP"                                            ,
 "pedigree"                                           ,
 "prinsimp"                                           ,
 "PAGI"                                               ,
 "ACD"                                                ,
 "lmf"                                                ,
 "MRsurv"                                             ,
 "cluster.datasets"                                   ,
 "DCL"                                                ,
 "dgof"                                               ,
 "kmconfband"                                         ,
 "SPARQL"                                             ,
 "BASIX"                                              ,
 "mcgibbsit"                                          ,
 "fbRanks"                                            ,
 "globalGSA"                                          ,
 "MFHD"                                               ,
 "HDtweedie"                                          ,
 "Rmisc"                                              ,
 "snapshot"                                           ,
 "sphereplot"                                         ,
 "mQTL"                                               ,
 "RTConnect"                                          ,
 "DAAGxtras"                                          ,
 "NormalGamma"                                        ,
 "cvq2"                                               ,
 "zoom"                                               ,
 "Correlplot"                                         ,
 "shinyRGL"                                           ,
 "cpa"                                                ,
 "mcGlobaloptim"                                      ,
 "GlobalDeviance"                                     ,
 "MRwarping"                                          ,
 "OptHedging"                                         ,
 "ScrabbleScore"                                      ,
 "bmem"                                               ,
 "caroline"                                           ,
 "censNID"                                            ,
 "migration.indices"                                  ,
 "TDD"                                                ,
 "kulife"                                             ,
 "PsumtSim"                                           ,
 "AUC"                                                ,
 "ParentOffspring"                                    ,
 "topsis"                                             ,
 "BAYSTAR"                                            ,
 "ssvd"                                               ,
 "SubCultCon"                                         ,
 "NPHMC"                                              ,
 "animalTrack"                                        ,
 "EffectsRelBaseline"                                 ,
 "hzar"                                               ,
 "LSMonteCarlo"                                       ,
 "SAMURAI"                                            ,
 "AtelieR"                                            ,
 "dataQualityR"                                       ,
 "svapls"                                             ,
 "prLogistic"                                         ,
 "GABi"                                               ,
 "R4dfp"                                              ,
 "SMCRM"                                              ,
 "speccalt"                                           ,
 "yhat"                                               ,
 "forensim"                                           ,
 "GPCSIV"                                             ,
 "EILA"                                               ,
 "RcmdrPlugin.qual"                                   ,
 "EcoTroph"                                           ,
 "logcondens.mode"                                    ,
 "pastis"                                             ,
 "calibrate"                                          ,
 "FREQ"                                               ,
 "DendSer"                                            ,
 "ttwa"                                               ,
 "ALSCPC"                                             ,
 "CommonTrend"                                        ,
 "G1DBN"                                              ,
 "tiff"                                               ,
 "RFGLS"                                              ,
 "automap"                                            ,
 "VaRES"                                              ,
 "dbEmpLikeGOF"                                       ,
 "corTools"                                           ,
 "RcmdrPlugin.sampling"                               ,
 "rtematres"                                          ,
 "tlemix"                                             ,
 "ssfit"                                              ,
 "kmlcov"                                             ,
 "RepeatedHighDim"                                    ,
 "SOLOMON"                                            ,
 "phenmod"                                            ,
 "SEchart"                                            ,
 "protiq"                                             ,
 "SynchWave"                                          ,
 "vwr"                                                ,
 "bitops"                                             ,
 "condGEE"                                            ,
 "blm"                                                ,
 "JOP"                                                ,
 "PCS"                                                ,
 "relaxnet"                                           ,
 "IntLik"                                             ,
 "EasyHTMLReport"                                     ,
 "ES"                                                 ,
 "RFinanceYJ"                                         ,
 "rLindo"                                             ,
 "mallet"                                             ,
 "DMwR"                                               ,
 "miRada"                                             ,
 "CompLognormal"                                      ,
 "lmeSplines"                                         ,
 "nlmeU"                                              ,
 "tipom"                                              ,
 "AMGET"                                              ,
 "ltsbase"                                            ,
 "timesboot"                                          ,
 "compactr"                                           ,
 "rbmn"                                               ,
 "gPCA"                                               ,
 "needy"                                              ,
 "SparseGrid"                                         ,
 "COBRA"                                              ,
 "Lambda4"                                            ,
 "pathdiagram"                                        ,
 "widenet"                                            ,
 "MPINet"                                             ,
 "pwt"                                                ,
 "eigenprcomp"                                        ,
 "AnDE"                                               ,
 "hint"                                               ,
 "neuroblastoma"                                      ,
 "somplot"                                            ,
 "wordmatch"                                          ,
 "mondate"                                            ,
 "suRtex"                                             ,
 "LogrankA"                                           ,
 "QSARdata"                                           ,
 "gpk"                                                ,
 "desire"                                             ,
 "optBiomarker"                                       ,
 "switchnpreg"                                        ,
 "portfolioSim"                                       ,
 "smoothie"                                           ,
 "ExactCIdiff"                                        ,
 "clusterGenomics"                                    ,
 "retimes"                                            ,
 "DTK"                                                ,
 "FField"                                             ,
 "loglognorm"                                         ,
 "r2dRue"                                             ,
 "b6e6rl"                                             ,
 "itree"                                              ,
 "tsModel"                                            ,
 "crrSC"                                              ,
 "ArrayBin"                                           ,
 "gwrr"                                               ,
 "MConjoint"                                          ,
 "SmithWilsonYieldCurve"                              ,
 "rsgcc"                                              ,
 "Bolstad2"                                           ,
 "twiddler"                                           ,
 "nsga2R"                                             ,
 "depthTools"                                         ,
 "govStatJPN"                                         ,
 "PDSCE"                                              ,
 "RcmdrPlugin.depthTools"                             ,
 "SoDA"                                               ,
 "tbdiag"                                             ,
 "cabootcrs"                                          ,
 "covLCA"                                             ,
 "sparktex"                                           ,
 "lshorth"                                            ,
 "pbs"                                                ,
 "IUPS"                                               ,
 "FGalgorithm"                                        ,
 "mopsocd"                                            ,
 "NPMVCP"                                             ,
 "RSADBE"                                             ,
 "ahaz"                                               ,
 "RobustRankAggreg"                                   ,
 "TrialSize"                                          ,
 "protoclass"                                         ,
 "OneTwoSamples"                                      ,
 "CoinMinD"                                           ,
 "DTMCPack"                                           ,
 "fitTetra"                                           ,
 "RVtests"                                            ,
 "compareODM"                                         ,
 "ggROC"                                              ,
 "McSpatial"                                          ,
 "profdpm"                                            ,
 "lbiassurv"                                          ,
 "InferenceSMR"                                       ,
 "SODC"                                               ,
 "cyphid"                                             ,
 "parallelize.dynamic"                                ,
 "trueskill"                                          ,
 "pom"                                                ,
 "pnmtrem"                                            ,
 "probsvm"                                            ,
 "CfEstimateQuantiles"                                ,
 "SMFI5"                                              ,
 "CDLasso"                                            ,
 "biglm"                                              ,
 "grnn"                                               ,
 "PurBayes"                                           ,
 "Nozzle.R1"                                          ,
 "RDSTK"                                              ,
 "csSAM"                                              ,
 "hgam"                                               ,
 "ergmharris"                                         ,
 "ArDec"                                              ,
 "CoxBoost"                                           ,
 "GAMBoost"                                           ,
 "NHEMOtree"                                          ,
 "timeline"                                           ,
 "wpp2010"                                            ,
 "lisrelToR"                                          ,
 "corrsieve"                                          ,
 "InteractiveIGraph"                                  ,
 "pnn"                                                ,
 "GibbsACOV"                                          ,
 "HydroMe"                                            ,
 "diffdepprop"                                        ,
 "iteRates"                                           ,
 "mixsep"                                             ,
 "RAP"                                                ,
 "MicroStrategyR"                                     ,
 "FuzzyToolkitUoN"                                    ,
 "dbEmpLikeNorm"                                      ,
 "hsmm"                                               ,
 "sendplot"                                           ,
 "dirmult"                                            ,
 "lars"                                               ,
 "siar"                                               ,
 "cdb"                                                ,
 "ibdreg"                                             ,
 "locfit"                                             ,
 "boussinesq"                                         ,
 "FrF2.catlg128"                                      ,
 "glmdm"                                              ,
 "spikeslab"                                          ,
 "catspec"                                            ,
 "CheckDigit"                                         ,
 "decon"                                              ,
 "GAIPE"                                              ,
 "spacejam"                                           ,
 "DeducerSpatial"                                     ,
 "orderbook"                                          ,
 "sideChannelAttack"                                  ,
 "clinUtiDNA"                                         ,
 "deseasonalize"                                      ,
 "df2json"                                            ,
 "survIDINRI"                                         ,
 "SportsAnalytics"                                    ,
 "persiandictionary"                                  ,
 "PlotRegionHighlighter"                              ,
 "BaySIC"                                             ,
 "ncbit"                                              ,
 "noweb"                                              ,
 "trioGxE"                                            ,
 "ENmisc"                                             ,
 "infutil"                                            ,
 "MLEcens"                                            ,
 "SPIn"                                               ,
 "gpclib"                                             ,
 "optAUC"                                             ,
 "ICE"                                                ,
 "sprsmdl"                                            ,
 "meteogRam"                                          ,
 "severity"                                           ,
 "HI"                                                 ,
 "HAP.ROR"                                            ,
 "hashFunction"                                       ,
 "hcc"                                                ,
 "orientlib"                                          ,
 "DATforDCEMRI"                                       ,
 "rCarto"                                             ,
 "BiplotGUI"                                          ,
 "tpe"                                                ,
 "crossdes"                                           ,
 "marqLevAlg"                                         ,
 "FitAR"                                              ,
 "BlockMessage"                                       ,
 "fpp"                                                ,
 "GriegSmith"                                         ,
 "hsicCCA"                                            ,
 "BayHap"                                             ,
 "perry"                                              ,
 "ber"                                                ,
 "cosso"                                              ,
 "patchPlot"                                          ,
 "BBMM"                                               ,
 "nls2"                                               ,
 "GESTr"                                              ,
 "LDOD"                                               ,
 "IM"                                                 ,
 "StandardizeText"                                    ,
 "survexp.fr"                                         ,
 "AtmRay"                                             ,
 "heatex"                                             ,
 "het.test"                                           ,
 "MOJOV"                                              ,
 "norm"                                               ,
 "HDMD"                                               ,
 "estout"                                             ,
 "iWeigReg"                                           ,
 "conf.design"                                        ,
 "cheb"                                               ,
 "epade"                                              ,
 "MEET"                                               ,
 "nonparaeff"                                         ,
 "Digiroo2"                                           ,
 "designGG"                                           ,
 "audiolyzR"                                          ,
 "GetR"                                               ,
 "HIest"                                              ,
 "multinomRob"                                        ,
 "Iboot"                                              ,
 "smdc"                                               ,
 "survC1"                                             ,
 "bayess"                                             ,
 "Compounding"                                        ,
 "modiscloud"                                         ,
 "beadarrayFilter"                                    ,
 "NestedCohort"                                       ,
 "nlsmsn"                                             ,
 "orddom"                                             ,
 "stoichcalc"                                         ,
 "exactLoglinTest"                                    ,
 "fractalrock"                                        ,
 "gaussquad"                                          ,
 "pacbpred"                                           ,
 "OOmisc"                                             ,
 "orthopolynom"                                       ,
 "pdist"                                              ,
 "TimeProjection"                                     ,
 "mlearning"                                          ,
 "randomGLM"                                          ,
 "YieldCurve"                                         ,
 "DivMelt"                                            ,
 "EMT"                                                ,
 "SUE"                                                ,
 "RcmdrPlugin.plotByGroup"                            ,
 "svSweave"                                           ,
 "emme2"                                              ,
 "shapefiles"                                         ,
 "UWHAM"                                              ,
 "FI"                                                 ,
 "misc3d"                                             ,
 "pln"                                                ,
 "dichromat"                                          ,
 "RcmdrPlugin.SLC"                                    ,
 "SLC"                                                ,
 "semPLS"                                             ,
 "stellaR"                                            ,
 "adaptsmoFMRI"                                       ,
 "currentSurvival"                                    ,
 "RcppXts"                                            ,
 "blockrand"                                          ,
 "CVThresh"                                           ,
 "edcc"                                               ,
 "obsSens"                                            ,
 "partitionMap"                                       ,
 "GWG"                                                ,
 "Bhat"                                               ,
 "rms.gof"                                            ,
 "spacodiR"                                           ,
 "ESG"                                                ,
 "subtype"                                            ,
 "survivalROC"                                        ,
 "gtx"                                                ,
 "PROTOLIDAR"                                         ,
 "RcmdrPlugin.mosaic"                                 ,
 "hier.part"                                          ,
 "displayHTS"                                         ,
 "GWASExactHW"                                        ,
 "proteomicdesign"                                    ,
 "partialOR"                                          ,
 "trapezoid"                                          ,
 "truncgof"                                           ,
 "MedOr"                                              ,
 "imputeMDR"                                          ,
 "ORMDR"                                              ,
 "ReCiPa"                                             ,
 "nose"                                               ,
 "dblcens"                                            ,
 "ConjointChecks"                                     ,
 "cosmoFns"                                           ,
 "MuFiCokriging"                                      ,
 "nutshell"                                           ,
 "nutshell.audioscrobbler"                            ,
 "nutshell.bbdb"                                      ,
 "OptInterim"                                         ,
 "IndependenceTests"                                  ,
 "ftnonpar"                                           ,
 "SampleSizeMeans"                                    ,
 "RDIDQ"                                              ,
 "rpartScore"                                         ,
 "simba"                                              ,
 "source.gist"                                        ,
 "KMsurv"                                             ,
 "muma"                                               ,
 "brainwaver"                                         ,
 "emdist"                                             ,
 "fracdiff"                                           ,
 "plmm"                                               ,
 "JASPAR"                                             ,
 "TFX"                                                ,
 "maptree"                                            ,
 "pairedCI"                                           ,
 "stepwise"                                           ,
 "ChoiceModelR"                                       ,
 "BAEssd"                                             ,
 "PAWL"                                               ,
 "bindata"                                            ,
 "linLIR"                                             ,
 "plsdepot"                                           ,
 "STAR"                                               ,
 "homtest"                                            ,
 "mlica2"                                             ,
 "tableplot"                                          ,
 "frmqa"                                              ,
 "smirnov"                                            ,
 "argosfilter"                                        ,
 "fpow"                                               ,
 "xgobi"                                              ,
 "NORMT3"                                             ,
 "cat"                                                ,
 "ScreenClean"                                        ,
 "speff2trial"                                        ,
 "ACCLMA"                                             ,
 "AdaptFit"                                           ,
 "allan"                                              ,
 "AllPossibleSpellings"                               ,
 "andrews"                                            ,
 "APSIMBatch"                                         ,
 "audit"                                              ,
 "batade"                                             ,
 "bayespref"                                          ,
 "BayHaz"                                             ,
 "betafam"                                            ,
 "bethel"                                             ,
 "BGSIMD"                                             ,
 "bimetallic"                                         ,
 "bivarRIpower"                                       ,
 "biwt"                                               ,
 "bootspecdens"                                       ,
 "CCA"                                                ,
 "CCP"                                                ,
 "CDFt"                                               ,
 "CMC"                                                ,
 "colorRamps"                                         ,
 "combinat"                                           ,
 "compoisson"                                         ,
 "concor"                                             ,
 "dbConnect"                                          ,
 "dinamic"                                            ,
 "discretization"                                     ,
 "divagis"                                            ,
 "dpa"                                                ,
 "DTDA"                                               ,
 "eigeninv"                                           ,
 "EngrExpt"                                           ,
 "eyetracking"                                        ,
 "FactMixtAnalysis"                                   ,
 "fgac"                                               ,
 "fit4NM"                                             ,
 "Flury"                                              ,
 "forensic"                                           ,
 "frt"                                                ,
 "FTICRMS"                                            ,
 "GAD"                                                ,
 "GANPAdata"                                          ,
 "GeneF"                                              ,
 "geomapdata"                                         ,
 "gibbs.met"                                          ,
 "globalboosttest"                                    ,
 "gPdtest"                                            ,
 "gwerAM"                                             ,
 "HadoopStreaming"                                    ,
 "HAPim"                                              ,
 "heatmap.plus"                                       ,
 "HPbayes"                                            ,
 "igraphtosonia"                                      ,
 "Imap"                                               ,
 "IMIS"                                               ,
 "ImpactIV"                                           ,
 "InfDim"                                             ,
 "insideRODE"                                         ,
 "Interpol"                                           ,
 "introgress"                                         ,
 "isopat"                                             ,
 "klin"                                               ,
 "kzft"                                               ,
 "lmec"                                               ,
 "LTR"                                                ,
 "MAclinical"                                         ,
 "mAr"                                                ,
 "MBCluster.Seq"                                      ,
 "Miney"                                              ,
 "minxent"                                            ,
 "MKLE"                                               ,
 "mpa"                                                ,
 "mvsf"                                               ,
 "NetCluster"                                         ,
 "NetData"                                            ,
 "NMFN"                                               ,
 "normwhn.test"                                       ,
 "OligoSpecificitySystem"                             ,
 "omd"                                                ,
 "oncomodel"                                          ,
 "ORDER2PARENT"                                       ,
 "ORIClust"                                           ,
 "paleoMAS"                                           ,
 "parviol"                                            ,
 "PASWR"                                              ,
 "pGLS"                                               ,
 "poistweedie"                                        ,
 "PopGenKit"                                          ,
 "pps"                                                ,
 "pragma"                                             ,
 "PredictiveRegression"                               ,
 "predmixcor"                                         ,
 "ProbForecastGOP"                                    ,
 "qtlDesign"                                          ,
 "rdetools"                                           ,
 "reams"                                              ,
 "Records"                                            ,
 "REQS"                                               ,
 "RHT"                                                ,
 "ROCwoGS"                                            ,
 "RODM"                                               ,
 "sddpack"                                            ,
 "seqRFLP"                                            ,
 "session"                                            ,
 "skewt"                                              ,
 "smco"                                               ,
 "SMVar"                                              ,
 "SNPmaxsel"                                          ,
 "spi"                                                ,
 "sporm"                                              ,
 "Stem"                                               ,
 "symbols"                                            ,
 "TERAplusB"                                          ,
 "tfer"                                               ,
 "titan"                                              ,
 "Tsphere"                                            ,
 "unbalhaar"                                          ,
 "UScensus2000cdp"                                    ,
 "UScensus2000tract"                                  ,
 "vegetarian"                                         ,
 "VIF"                                                ,
 "violinmplot"                                        ,
 "walkscoreAPI"                                       ,
 "wikibooks"                                          ,
 "WilcoxCV"                                           ,
 "write.snns"                                         ,
 "wSVM"                                               ,
 "nlmeODE"                                            ,
 "CARE1"                                              ,
 "dpglasso"                                           ,
 "pheno"                                              ,
 "detrendeR"                                          ,
 "landsat"                                            ,
 "MBI"                                                ,
 "TwoCop"                                             ,
 "bmk"                                                ,
 "linprog"                                            ,
 "PhaseType"                                          ,
 "Ridit"                                              ,
 "powerpkg"                                           ,
 "PoissonSeq"                                         ,
 "r2stl"                                              ,
 "Rivivc"                                             ,
 "FRCC"                                               ,
 "rworldxtra"                                         ,
 "biasbetareg"                                        ,
 "gbRd"                                               ,
 "loop"                                               ,
 "astroFns"                                           ,
 "ccChooser"                                          ,
 "risksetROC"                                         ,
 "smcure"                                             ,
 "emoa"                                               ,
 "binomialcftp"                                       ,
 "namespace"                                          ,
 "nparLD"                                             ,
 "abcdeFBA"                                           ,
 "matrixcalc"                                         ,
 "WCQ"                                                ,
 "spcov"                                              ,
 "BayesNI"                                            ,
 "FMStable"                                           ,
 "NISTnls"                                            ,
 "hbsae"                                              ,
 "ipdmeta"                                            ,
 "mlPhaser"                                           ,
 "survAUC"                                            ,
 "PIPS"                                               ,
 "pso"                                                ,
 "clusteval"                                          ,
 "NetComp"                                            ,
 "rbenchmark"                                         ,
 "regtest"                                            ,
 "SearchTrees"                                        ,
 "gaussDiff"                                          ,
 "muStat"                                             ,
 "Simpsons"                                           ,
 "smoothmest"                                         ,
 "Rlab"                                               ,
 "sitools"                                            ,
 "BigTSP"                                             ,
 "flashClust"                                         ,
 "cxxfunplus"                                         ,
 "MATTOOLS"                                           ,
 "RPCLR"                                              ,
 "PresenceAbsence"                                    ,
 "SAScii"                                             ,
 "capwire"                                            ,
 "aspace"                                             ,
 "rHpcc"                                              ,
 "kerdiest"                                           ,
 "SQN"                                                ,
 "PLIS"                                               ,
 "IC2"                                                ,
 "obliqueRF"                                          ,
 "bvenn"                                              ,
 "BVS"                                                ,
 "dlmap"                                              ,
 "FormalSeries"                                       ,
 "modelfree"                                          ,
 "p2distance"                                         ,
 "fugeR"                                              ,
 "notifyR"                                            ,
 "deamer"                                             ,
 "R2PPT"                                              ,
 "coexist"                                            ,
 "crn"                                                ,
 "Runiversal"                                         ,
 "rtape"                                              ,
 "HyPhy"                                              ,
 "rJython"                                            ,
 "BaylorEdPsych"                                      ,
 "gogarch"                                            ,
 "mbmdr"                                              ,
 "txtplot"                                            ,
 "c3net"                                              ,
 "FindAllRoots"                                       ,
 "FunCluster"                                         ,
 "geneListPie"                                        ,
 "GExMap"                                             ,
 "mcclust"                                            ,
 "ODB"                                                ,
 "parmigene"                                          ,
 "schoolmath"                                         ,
 "FRACTION"                                           ,
 "obliclus"                                           ,
 "knnGarden"                                          ,
 "UScensus2010"                                       ,
 "DeducerSurvival"                                    ,
 "HW.pval"                                            ,
 "hypothesestest"                                     ,
 "mlbench"                                            ,
 "mvtmeta"                                            ,
 "extraBinomial"                                      ,
 "SECP"                                               ,
 "NLRoot"                                             ,
 "WMDB"                                               ,
 "ncg"                                                ,
 "p3state.msm"                                        ,
 "myepisodes"                                         ,
 "CR"                                                 ,
 "rCUR"                                               ,
 "RAD"                                                ,
 "ezglm"                                              ,
 "psy"                                                ,
 "RVideoPoker"                                        ,
 "AMAP.Seq"                                           ,
 "crossmatch"                                         ,
 "bisectr"                                            ,
 "dummies"                                            ,
 "caribou"                                            ,
 "extrafontdb"                                        ,
 "rrlda"                                              ,
 "CHCN"                                               ,
 "R330"                                               ,
 "bios2mds"                                           ,
 "RIFS"                                               ,
 "MorseGen"                                           ,
 "Voss"                                               ,
 "SimpleTable"                                        ,
 "hiPOD"                                              ,
 "howmany"                                            ,
 "relaxo"                                             ,
 "OIdata"                                             ,
 "foodweb"                                            ,
 "plus"                                               ,
 "fgof"                                               ,
 "WeightedPortTest"                                   ,
 "irtrees"                                            ,
 "makeProject"                                        ,
 "QUIC"                                               ,
 "simSummary"                                         ,
 "M3"                                                 ,
 "ri"                                                 ,
 "cumplyr"                                            ,
 "cvTools"                                            ,
 "SortableHTMLTables"                                 ,
 "dafs"                                               ,
 "iCluster"                                           ,
 "CityPlot"                                           ,
 "clime"                                              ,
 "discreteMTP"                                        ,
 "tensor"                                             ,
 "lassoshooting"                                      ,
 "superdiag"                                          ,
 "labeledLoop"                                        ,
 "mseapca"                                            ,
 "LIHNPSD"                                            ,
 "mvnormtest"                                         ,
 "reweight"                                           ,
 "sROC"                                               ,
 "mmds"                                               ,
 "waffect"                                            ,
 "BayesDA"                                            ,
 "elec.strat"                                         ,
 "FinAsym"                                            ,
 "apsrtable"                                          ,
 "fossil"                                             ,
 "WWGbook"                                            ,
 "FBN"                                                ,
 "mlgt"                                               ,
 "MHadaptive"                                         ,
 "filehashSQLite"                                     ,
 "SPA3G"                                              ,
 "lle"                                                ,
 "mvtsplot"                                           ,
 "represent"                                          ,
 "QuantPsyc"                                          ,
 "AUCRF"                                              ,
 "mgpd"                                               ,
 "nnls"                                               ,
 "PSAgraphics"                                        ,
 "DeducerPlugInExample"                               ,
 "DeducerPlugInScaling"                               ,
 "bmd"                                                ,
 "R2wd"                                               ,
 "VarSwapPrice"                                       ,
 "PKgraph"                                            ,
 "zipcode"                                            ,
 "qLearn"                                             ,
 "eqtl"                                               ,
 "soobench"                                           ,
 "AmericanCallOpt"                                    ,
 "rBeta2009"                                          ,
 "benchden"                                           ,
 "superpc"                                            ,
 "RMallow"                                            ,
 "barcode"                                            ,
 "ops"                                                ,
 "gamlss.nl"                                          ,
 "idbg"                                               ,
 "pairheatmap"                                        ,
 "futile.paradigm"                                    ,
 "semdiag"                                            ,
 "geotools"                                           ,
 "tlmec"                                              ,
 "MIPHENO"                                            ,
 "SEER2R"                                             ,
 "apple"                                              ,
 "cytoDiv"                                            ,
 "ifa"                                                ,
 "DBGSA"                                              ,
 "PKPDmodels"                                         ,
 "lga"                                                ,
 "randaes"                                            ,
 "lisp"                                               ,
 "events"                                             ,
 "MImix"                                              ,
 "GOGANPA"                                            ,
 "rpsychi"                                            ,
 "cin"                                                ,
 "wasim"                                              ,
 "ADGofTest"                                          ,
 "biglars"                                            ,
 "ktspair"                                            ,
 "objectProperties"                                   ,
 "EMC"                                                ,
 "SMC"                                                ,
 "supclust"                                           ,
 "OrdMonReg"                                          ,
 "plugdensity"                                        ,
 "mpm"                                                ,
 "Survgini"                                           ,
 "ProfileLikelihood"                                  ,
 "SyNet"                                              ,
 "RArcInfo"                                           ,
 "NPsimex"                                            ,
 "mail"                                               ,
 "EL"                                                 ,
 "rationalfun"                                        ,
 "objectSignals"                                      ,
 "emma"                                               ,
 "AnnotLists"                                         ,
 "MultEq"                                             ,
 "ISOweek"                                            ,
 "viopoints"                                          ,
 "ieeeround"                                          ,
 "venneuler"                                          ,
 "mxkssd"                                             ,
 "REEMtree"                                           ,
 "mkssd"                                              ,
 "CLSOCP"                                             ,
 "minimax"                                            ,
 "pixmap"                                             ,
 "B2Z"                                                ,
 "limitplot"                                          ,
 "enrichvs"                                           ,
 "MetaPCA"                                            ,
 "reshapeGUI"                                         ,
 "GANPA"                                              ,
 "gtcorr"                                             ,
 "Kendall"                                            ,
 "pear"                                               ,
 "MAPLES"                                             ,
 "SPECIES"                                            ,
 "lmmlasso"                                           ,
 "brew"                                               ,
 "lcda"                                               ,
 "metacor"                                            ,
 "compeir"                                            ,
 "fpca"                                               ,
 "MUCflights"                                         ,
 "beadarrayMSV"                                       ,
 "Meth27QC"                                           ,
 "cmaes"                                              ,
 "outliers"                                           ,
 "multxpert"                                          ,
 "darts"                                              ,
 "factualR"                                           ,
 "reliaR"                                             ,
 "orQA"                                               ,
 "dcv"                                                ,
 "inference"                                          ,
 "fANCOVA"                                            ,
 "rocc"                                               ,
 "genepi"                                             ,
 "factorQR"                                           ,
 "laercio"                                            ,
 "psychometric"                                       ,
 "perm"                                               ,
 "mlmmm"                                              ,
 "RNCBIEUtilsLibs"                                    ,
 "ttutils"                                            ,
 "SEL"                                                ,
 "cgh"                                                ,
 "imprProbEst"                                        ,
 "ProDenICA"                                          ,
 "identity"                                           ,
 "space"                                              ,
 "nws"                                                ,
 "AIM"                                                ,
 "endogMNP"                                           ,
 "haarfisz"                                           ,
 "ic50"                                               ,
 "stam"                                               ,
 "scrapeR"                                            ,
 "dynamicGraph"                                       ,
 "fisheyeR"                                           ,
 "HMM"                                                ,
 "SampleSizeProportions"                              ,
 "datamap"                                            ,
 "LogitNet"                                           ,
 "rela"                                               ,
 "adaptTest"                                          ,
 "grouped"                                            ,
 "packClassic"                                        ,
 "HyperbolicDist"                                     ,
 "km.ci"                                              ,
 "crantastic"                                         ,
 "PearsonICA"                                         ,
 "svcm"                                               ,
 "HybridMC"                                           ,
 "bootStepAIC"                                        ,
 "SEMModComp"                                         ,
 "mcsm"                                               ,
 "sac"                                                ,
 "spe"                                                ,
 "CreditMetrics"                                      ,
 "Reliability"                                        ,
 "ETC"                                                ,
 "foba"                                               ,
 "kzs"                                                ,
 "expert"                                             ,
 "pack"                                               ,
 "RM2"                                                ,
 "SASPECT"                                            ,
 "LDtests"                                            ,
 "poilog"                                             ,
 "fuzzyFDR"                                           ,
 "allelic"                                            ,
 "BayesValidate"                                      ,
 "coxrobust"                                          ,
 "rk.ANOVA"                                           ,
 "rk.ClusterAnalysis"                                 ,
 "rk.CohenKappa"                                      ,
 "rk.FactorAnalysis"                                  ,
 "rk.gitInstall"                                      ,
 "rk.MPT"                                             ,
 "rk.MultidimensionalScaling"                         ,
 "rk.power"                                           ,
 "rk.subset"                                          ,
 "rkwarddev"                                          )
