#' @importFrom stats p.adjust
#' @importFrom stats rnorm
#' @export
#' @keywords internal
GARCHX_select.default <- function(eps, X, order = c(1, 1), delta = 2, alpha.level = 0.05, adjust.method = "fdr", optim.method = "NR"){
  # Declaring initial values for parameters
  # d = number of exogenous covariates
  d <- ncol(X)
  r <- 10

  # Allow users to have adjust.method = "None"

  # Error check: checks if X is a matrix
  if(!is.matrix(X)){
    stop("X must be a matrix with number of rows equal to the length of eps")
  } # End if

  # Error check: checks if number of rows of X is equal to length of eps.
  if(dim(X)[1] != (length(eps))){
    stop("Number of rows of matrix X must be the same as eps")
  } # End if

  # Error check: checks of order is a vector of length 2 for p and q values
  if(!is.vector(order) || length(order) != 2){
    stop("order must be a vector of length 2 with entries determining the order of the GARCH model")
  } # End if

  # Error check: order is c(0, 0)
  if(order[2] < 0){
    stop("q must be a non negative integer")
  } # End if

  if(order[1] <= 0){
    stop("p must be greater than 0")
  } # End if

  # Create object of class GARCHX
  obj <- create_GARCHXobj(eps = eps, X = X, order = order, d = d, delta = delta,
                    r = r, alpha.level = alpha.level, adjust.method = adjust.method)

  # Extract p and q from order
  p <- order[1]
  q <- order[2]

  # First estimation of coefficients
  # obj has the following values: omega, alpha, beta, pi, eps, X, order
  # optim.method is the optimization function user selected
  estimated_values <- estimate(obj, optim.method)

  # Extract estimated coefficients. Contains estimated values for omega, alpha, beta, and pi
  estimated_coefs <- estimated_values$coef

  # Adjust p values of exogenous covariates
  # estimated_values contains values for omega, alpha, and beta
  # We only need to extract the values for pi, hence: (p+q+2):(1+p+q+d)
  pval <- p.adjust(estimated_values$pval[(p+q+2):(1+p+q+d)], method = adjust.method)

  # Drop exogenous covariates with p-value higher than 0.05 and store index in object
  selected_pvals <- which(pval < 0.05)
  obj$selectedX_index <- selected_pvals

  # Check how many exogenous covariates were selected
  if(length(selected_pvals) == 0){
    # If zero were selected
    obj$selectedX <- NULL
    obj$d <- 0
    obj$pi <- NA
  } else {
    # If some at least one is selected
    # Drop unselected exogenous covariates
    obj$selectedX <- as.matrix(X[, selected_pvals], ncol = length(selected_pvals))
    # Adjust d for the dropped covariates
    obj$d <- length(selected_pvals)
    # Re-initialize pi with the new value of d
    obj$pi <- rep(0.05, obj$d)
  } # End if-else

  # Re-estimate coefficients with the new values of X, d, and pi
  estimated_values <- estimate(obj, optim.method)

  # Re-extract coefficients
  estimated_coefs <- estimated_values$coef
  obj$Likelihood <- estimated_values$minimum

  # Assign omega, alpha, beta, and pi coefficients to obj
  # Check if p, q, or d is 0
  obj$omega <- estimated_coefs[1]
  if(p != 0){
    # If p <= 1, assign estimated coefficients to the obj
    obj$alpha <- estimated_coefs[2:(p+1)]
  } # End if

  if(q != 0){
    # If q <= 1, assign estimated coefficients to the obj
    obj$beta <- estimated_coefs[(p+2):(p+q+1)]
  } # End if

  # Check the length of selected exogenous covariates
  if(length(selected_pvals) == 0){
    # If it is zero:
    # Assign NA to X, as we did not select any exogenous covariates
    obj$selectedX <- NA
    # Assign 0 to d
    obj$d <- 0
    # Assign NA to pi
    obj$pi <- NA
  } # End if

  d = obj$d
  
  if(obj$d != 0){
    # If d <= 1, assign estimated exogenous covariates to the obj
    obj$pi <- estimated_coefs[(p+q+2):(1+p+q+obj$d)]

    obj$p.values <- rep(NA, d)
    # Assign p-values of selected exogenous covariates
    obj$p.values <- p.adjust(estimated_values$pval[(p+q+2):(1+p+q+d)], method = adjust.method)
  } # End if

  # Return obj of GARCHX
  return(obj)
} # End GARCHX.select.default
