#' Full conditional for lambda
#'
#' @param lambda A scalar defining the parameter for the Truncate Poisson distribution
#'    that controls the number of change points (or its initial values)
#' @param kstar A scalar with the number maximum of change points in all clusters
#' @param N A scalar representing the number of data sequences
#' @param al The hyperparameter value for the shape parameter in the gamma prior for lambda
#' @param bl The hyperparameter value for the scale parameter in the gamma prior for lambda
#' @param K A vector containing the number of change points for each cluster (or its initial values)
#' @param cluster A vector containing the cluster assignments for the data sequences (or its initial values)
#'
#' @returns `full_cond` returns a numerical value corresponding to a sample from the full conditional for lambda
#' @note
#' This function is used within the Gibbs sampler, it is not expected to be used alone.
#'
#' @examples
#' # Using hypothetical values to exemplification purposes
#' clusters <- c(1,1,2,1,2)
#' full_cond(kstar = 2, lambda = 3, cluster = clusters, al = 2, bl = 1000, K = c(2, 2), N = 5)
#' @export
#'
full_cond <- function(kstar, lambda, cluster, al, bl, K, N){
  Nr <- as.vector(table(cluster))
  aln <- al + sum(Nr*K[sort(unique(cluster))])

  out <- (lambda^(aln - 1)*exp((-lambda/bl)))/(sum(sapply(0:kstar,FUN=function(l){lambda^l/factorial(l)}))^N)

}
