#' Search for the optimal value of GRNN smoothing parameter based on AUC 
#'
#' The function \code{grnn.search_auc} searches for the optimal value of GRNN smoothing parameter by cross-validation.
#' It is applicable to the classification.
#'
#' @param net    A GRNN object generated by grnn.fit() 
#' @param sigmas A numeric vector to search for the best smoothing parameter
#' @param nfolds A scalar for the number of n-fold, 4 by default
#' @param seed   The seed value for the n-fold cross-validation, 1 by default
#'
#' @return The list of all searching outcomes and the best outcome
#'
#' @examples
#' data(iris, package = "datasets")
#' Y <- ifelse(iris[, 5] == "setosa", 1, 0)
#' X <- scale(iris[, 1:4])
#' gnet <- grnn.fit(x = X, y = Y)
#' grnn.search_auc(net = gnet, sigmas = c(3, 5, 7), nfolds = 2)

grnn.search_auc <- function(net, sigmas, nfolds = 4, seed = 1) {
  if (class(net) != "General Regression Neural Net") stop("net needs to be a GRNN.", call. = F)
  if (is.vector(sigmas) != T) stop("sigmas needs to be a vector.", call. = F)

  fd <- folds(seq(nrow(net$x)), n = nfolds, seed = seed)
  cv <- function(s) {
    rs <- Reduce(rbind,
            lapply(fd,
              function(f) data.frame(ya = net$y[f],
                                     yp = grnn.predict(grnn.fit(net$x[-f, ], net$y[-f], sigma = s),
                                                       net$x[f, ]))))
    return(data.frame(sigma = s, auc = MLmetrics::AUC(y_pred = rs$yp, y_true = rs$ya)))
  }

  cls <- parallel::makeCluster(min(nfolds, parallel::detectCores() - 1), type = "PSOCK")
  obj <- c("fd", "net", "grnn.fit", "grnn.predone", "grnn.predict")
  parallel::clusterExport(cls, obj, envir = environment())
  rst <- Reduce(rbind, parallel::parLapply(cls, sigmas, cv))
  parallel::stopCluster(cls)
  return(list(test = rst, best = rst[rst$auc == max(rst$auc), ]))
}
