% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xmu_make_top_twin_models.R
\name{xmu_make_top_twin}
\alias{xmu_make_top_twin}
\title{Helper to make a basic top, MZ, and DZ model.}
\usage{
xmu_make_top_twin(
  mzData,
  dzData,
  selDVs,
  sep = NULL,
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"),
  allContinuousMethod = c("cumulants", "marginals"),
  nSib = 2,
  numObsMZ = NULL,
  numObsDZ = NULL,
  equateMeans = TRUE,
  weightVar = NULL,
  bVector = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{mzData}{Dataframe containing the MZ data}

\item{dzData}{Dataframe containing the DZ data}

\item{selDVs}{List of base (e.g. BMI) (i.e., NOT 'BMI_T1') variable names (OR, you don't set "sep", the full variable names)}

\item{sep}{(optional but desirable) string used to expand selDVs into selVars, i.e., "_T" to expand BMI into BMI_T1 and BMI_T2}

\item{type}{One of 'Auto','FIML','cov', 'cor', 'WLS','DWLS', or 'ULS'. Auto tries to react to the incoming mxData type (raw/cov).}

\item{allContinuousMethod}{"cumulants" or "marginals". Used in all-continuous WLS data to determine if a means model needed.}

\item{nSib}{Number of members per family (default = 2)}

\item{numObsMZ}{Number of MZ observations contributing (for summary data only)}

\item{numObsDZ}{Number of DZ observations contributing (for summary data only)}

\item{equateMeans}{Whether to equate T1 and T2 means (default = TRUE).}

\item{weightVar}{If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{verbose}{(default = FALSE)}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}}s for top, MZ and DZ.
}
}
\description{
\code{xmu_make_top_twin} makes basic \code{top}, \code{MZ}, and \code{DZ} models. It includes thresholds matrices in the twin models if needed.

This is used in  \code{\link[=umxCP]{umxCP()}}, and \code{\link[=umxACE]{umxACE()}} and \code{\link[=umxACEv]{umxACEv()}} and will be added to the other models: \code{\link[=umxGxE]{umxGxE()}}, \code{\link[=umxIP]{umxIP()}},
simplifying code maintenance.

\code{xmu_make_top_twin} takes \code{mzData} and \code{dzData}, a list of the \code{selDVs} to analyse (as well as \code{sep} and \code{nSib}), along with other
relevant information such as whether the user wants to \code{equateMeans}.
It can also handle a \code{weightVar}.

\code{varStarts} is computed as \code{sqrt(variance)/3} of the DVs and \code{meanStarts} as the variable means.
For raw data, a check is made for ordered variables. For Binary variables, means are fixed at 0 and
total variance (A+C+E) is fixed at 1. For ordinal variables, the first 2 thresholds are fixed.

\strong{Modeling}

\emph{top model}

For raw and WLS data, \code{top} contains a means matrix (if needed). For summary data, the top model contains only a name.

For ordinal data, \code{top} gains \code{top.threshMat} (from a call to \code{\link[=umxThresholdMatrix]{umxThresholdMatrix()}}).

\emph{MZ and DZ models}

\code{MZ} and \code{DZ} contain the data, and an expectation referencing \code{top.expCovMZ} and \code{top.expMean}, and, \code{vector = bVector}.
For continuous raw data, MZ and DZ contain \code{\link[OpenMx:mxExpectationNormal]{OpenMx::mxExpectationNormal()}} and \code{\link[OpenMx:mxFitFunctionML]{OpenMx::mxFitFunctionML()}}.
For WLS these the fit function is switched to \code{\link[OpenMx:mxFitFunctionWLS]{OpenMx::mxFitFunctionWLS()}} with appropriate \code{type} and \code{allContinuousMethod}.

For binary, a constraint and algebras are included to constrain \code{Vtot} (A+C+E) to 1.

If a \code{weightVar} is detected, this column is added to  mzWeightMatrix/mzWeightMatrix.

If \code{equateMeans} is \code{TRUE}, then the Twin-2 vars in the mean matrix are equated by label with Twin-1.

\strong{Matrices created}

Decent starts are guessed from the data.
Where needed, e.g. continuous raw data, top adds a means matrix "expMean".
For ordinal data, top adds a \code{\link[=umxThresholdMatrix]{umxThresholdMatrix()}}.

If binary variables are present, matrices and a constraint to hold \code{A+C+E == 1} are added to top.

If a weight variable is offered up, an \code{mzWeightMatrix} will be added.

\strong{Data handling}

In terms of data handling, \code{xmu_make_top_twin} was primarily designed to take
data.frames and process these into mxData.
It can also, however, handle cov and mxData input.

It can process data into all the types supported by \code{mxData}.

Raw data input with a target of \code{cov} or \code{cor} type requires the \code{numObsMZ} and \code{numObsDZ} to be set.

Type "WLS", "DWLS", or "ULS", data remain raw, but are handled as WLS in the \code{\link[OpenMx:mxFitFunctionWLS]{OpenMx::mxFitFunctionWLS()}}.

Unused columns are dropped.

If you pass in raw data, you can't request type cov/cor yet. Will work on this if desired.
}
\examples{
# ==============
# = Continuous =
# ==============
library(umx)
data(twinData)
selDVs = c("wt", "ht")
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)
names(bits) # "top" "MZ"  "DZ" "bVector" "mzWeightMatrix" "dzWeightMatrix"
class(bits$MZ$fitfunction)[[1]] == "MxFitFunctionML"
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, 
	selDVs= selDVs, sep= "", type = "WLS")
class(bits$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"
bits$MZ$fitfunction$type =="WLS"
bits$MZ$fitfunction$continuousType == "cumulants"

# Choose non-default type (DWLS)
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData,
	selDVs = selDVs, sep= "", type = "DWLS")
bits$MZ$fitfunction$type =="DWLS"
class(bits$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"

# Switch continuous method
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "",
	type = "WLS", allContinuousMethod = "marginals")
bits$MZ$fitfunction$continuousType == "marginals"
class(bits$MZ$fitfunction)[[1]] == "MxFitFunctionWLS"


# ============================================
# = Bivariate continuous and ordinal example =
# ============================================
data(twinData)
selDVs = c("wt", "obese")
# Cut BMI column to form ordinal obesity variables
ordDVs          = c("obese1", "obese2")
obesityLevels   = c('normal', 'overweight', 'obese')
cutPoints       = quantile(twinData[, "bmi1"], probs = c(.5, .2), na.rm = TRUE)
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
# Make the ordinal variables into mxFactors (ensure ordered is TRUE, and require levels)
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
mzData = twinData[twinData$zygosity \%in\%  "MZFF",] 
dzData = twinData[twinData$zygosity \%in\%  "DZFF",]
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, selDVs= selDVs, sep="", nSib= 2)
names(bits) # "top" "MZ"  "DZ" 

# ==============
# = One binary =
# ==============
data(twinData)
cutPoints       = quantile(twinData[, "bmi1"], probs = .2, na.rm = TRUE)
obesityLevels   = c('normal', 'obese')
twinData$obese1 = cut(twinData$bmi1, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
twinData$obese2 = cut(twinData$bmi2, breaks = c(-Inf, cutPoints, Inf), labels = obesityLevels) 
ordDVs = c("obese1", "obese2")
twinData[, ordDVs] = umxFactor(twinData[, ordDVs])
selDVs = c("wt", "obese")
mzData = twinData[twinData$zygosity \%in\% "MZFF",]
dzData = twinData[twinData$zygosity \%in\% "DZFF",]
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, selDVs= selDVs, sep= "", nSib= 2)

# ============
# = Cov data =
# ============
data(twinData)
mz = cov(twinData[twinData$zygosity \%in\%  "MZFF", tvars(c("wt", "ht"), sep="")], use = "complete")
dz = cov(twinData[twinData$zygosity \%in\%  "DZFF", tvars(c("wt", "ht"), sep="")], use = "complete")
bits = xmu_make_top_twin(mzData= mzData, dzData= dzData, selDVs= "wt", sep= "", nSib= 2)
class(bits$MZ$fitfunction)[[1]] =="MxFitFunctionML"
names(bits$MZ$data$observed) == c("wt1", "wt2") # height columns dropped

}
\seealso{
Other xmu internal not for end user: \code{\link{umxModel}()},
  \code{\link{umx_standardize}()}, \code{\link{umx}},
  \code{\link{xmuHasSquareBrackets}()},
  \code{\link{xmuLabel_MATRIX_Model}()},
  \code{\link{xmuLabel_Matrix}()},
  \code{\link{xmuLabel_RAM_Model}()},
  \code{\link{xmuMI}()},
  \code{\link{xmuMakeDeviationThresholdsMatrices}()},
  \code{\link{xmuMakeOneHeadedPathsFromPathList}()},
  \code{\link{xmuMakeTwoHeadedPathsFromPathList}()},
  \code{\link{xmuMaxLevels}()},
  \code{\link{xmuMinLevels}()},
  \code{\link{xmuPropagateLabels}()},
  \code{\link{xmu_CI_merge}()},
  \code{\link{xmu_CI_stash}()},
  \code{\link{xmu_DF_to_mxData_TypeCov}()},
  \code{\link{xmu_PadAndPruneForDefVars}()},
  \code{\link{xmu_assemble_twin_supermodel}()},
  \code{\link{xmu_check_levels_identical}()},
  \code{\link{xmu_check_needs_means}()},
  \code{\link{xmu_check_variance}()},
  \code{\link{xmu_clean_label}()},
  \code{\link{xmu_describe_data_WLS}()},
  \code{\link{xmu_dot_make_paths}()},
  \code{\link{xmu_dot_make_residuals}()},
  \code{\link{xmu_dot_maker}()},
  \code{\link{xmu_dot_move_ranks}()},
  \code{\link{xmu_dot_rank_str}()},
  \code{\link{xmu_get_CI}()},
  \code{\link{xmu_lavaan_process_group}()},
  \code{\link{xmu_make_bin_cont_pair_data}()},
  \code{\link{xmu_make_mxData}()},
  \code{\link{xmu_name_from_lavaan_str}()},
  \code{\link{xmu_safe_run_summary}()},
  \code{\link{xmu_set_sep_from_suffix}()},
  \code{\link{xmu_show_fit_or_comparison}()},
  \code{\link{xmu_simplex_corner}()},
  \code{\link{xmu_standardize_ACEcov}()},
  \code{\link{xmu_standardize_ACEv}()},
  \code{\link{xmu_standardize_ACE}()},
  \code{\link{xmu_standardize_CP}()},
  \code{\link{xmu_standardize_IP}()},
  \code{\link{xmu_standardize_RAM}()},
  \code{\link{xmu_standardize_SexLim}()},
  \code{\link{xmu_standardize_Simplex}()},
  \code{\link{xmu_start_value_list}()},
  \code{\link{xmu_starts}()},
  \code{\link{xmu_twin_check}()}
}
\concept{xmu internal not for end user}
