% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ubiquity.R
\name{system_set_tt_cond}
\alias{system_set_tt_cond}
\title{Define Titration Triggers and Actions}
\usage{
system_set_tt_cond(cfg, name, cond, action, value = "-1")
}
\arguments{
\item{cfg}{ubiquity system object}

\item{name}{string containing the name for the titration rule to which this condition applies}

\item{cond}{string that evaluates a boolean value that is \code{TRUE} when the action should be triggered}

\item{action}{stringing that evaluates to what should be done when the condition is met (e.g. changing the dose, state change, etc)}

\item{value}{code to be stored in the titration history to track when this condition has been triggered}
}
\value{
Ubiquity system object with the titration condition defined
}
\description{
Once a rule has been defined using
\code{\link{system_new_tt_rule}}, it can then be used by specifying checks at
each of the titration time points that, when true, will perform some actions.
}
\details{
The general syntax for setting a new condition is:

\preformatted{
cfg = system_new_tt_cond(cfg,
                        name   = "rname",
                        cond   = "BOOLEAN EXPRESSION",
                        action = "EXPRESSION",
                        value  = "VALUE")
}

The \code{name}
input will associate this condition with a previously defined rule. For each
time defined when the rule was created, the condition (\code{cond}) will be
evaluated. If that condition evaluates as \code{TRUE} then the \code{action} will be
evaluated. Lastly, when a condition action is evaluated, the \code{value} is stored
in the titration history.

Multiple conditions can be associated with a rule. The internal titration
history will track each one where a condition has been evaluated as true, but
the simulation output will only show the \bold{last} condition to be evaluated as
true.

The \code{cond} field is a string that, when evaluated, will produce a boolean value
(\code{TRUE} or \code{FALSE}). If you simply want to force an action at each of the times
for a given rule you can use: \code{cond = "TRUE"}. Alternatively you can provide
mathematical expressions or even complicated user defined functions.

The \code{action} field is evaluated when \code{cond} is true. To modify how a simulation
is going to be performed, you will want to modify the \code{SIMINT_cfgtt}
variable using the different system commands. Certain common tasks have
prototype functions created to make it easier for the user:
\itemize{
\item \code{SI_TT_BOLUS} - Set bolus dosing
\item \code{SI_TT_RATE} - Set infusion inputs
\item \code{SI_TT_STATE} - Reset system states
}

\bold{Note:} Protype functions are strings but sometimes it is necessary to
specify strings within this string. For the main string use double quotes (")
and for the internal strings use single quotes (')

\bold{\code{SI_TT_BOLUS}}

The simplest way to apply a bolus when the condition is true is to use the following:

\preformatted{
action = "SI_TT_BOLUS[state=’At’, 
                     values=c(10, 10, 10), 
                     times=c(0, 1, 2)]"
}

The \code{values} and \code{times} are vectors of numbers of equal length. The dosing and
time units are those specified in the \code{system.txt} file for the \code{<B:?>} delimiter. The
times are relative to the titration time. So \code{0} above means at the titration time.

It’s possible to specify an interval and a number of times to repeat the last dose
using the following:

\preformatted{
action = "SI_TT_BOLUS[state    = ’At’, 
                     values   = c(5, 5, 10), 
                     times    = c(0, 2, 4), 
                     repdose  = ’last’, 
                     number   = 7, 
                     interval = 4]"
}

This will give a dose of \code{5} at the titration point and \code{2} time units later. The dose of \code{10}
at time \code{4} will be repeated \code{7} times every \code{4} time units. So a total of 8 (\code{7 + 1}) doses
at \code{10} will be administered. Remember the time units were those defined in \code{<B:?>}.
The input \code{repdose} can be either \code{’last’} or \code{’none’}.

\bold{Note:} The main string is in double quotes \code{" "} but the strings in the protype
argument (e.g. \code{’last’}) are in single quotes \code{’ ’}.

\bold{\code{SI_TT_RATE}} 

If you created an infusion named \code{Dinf} using \code{<R:?>} and the infusion units
are min (times) and mg/min (rates). To have a 60 minute infusion of 20
mg/min then we would do the following:

\preformatted{
action = "SI_TT_RATE[rate=’Dinf’, times=c(0, 60), levels=c(20.0, 0)]"
}

If we wanted to do this every day for 9 more days (a total of 10 days) we can repeat
the sequence:

\preformatted{
action = "SI_TT_RATE[rate     = ’Dinf’, 
                    times    = c(0, 60), 
                    levels   = c(20, 0), 
                    repdose  = ’sequence’, 
                    number   = 9, 
                    interval = 24*60]"
}

The input \code{repdose} can be either \code{’sequence’} or \code{’none’}.

\bold{Note:} The time units and dosing rate are those specified using \code{<R:?>}.

\bold{\code{SI_TT_STATE}} 

To provide fine control over states at titration points the state reset
prototype is provided. For example, if you are modeling an assay where
there is a wash step and you want to drop a concentration to zero. If you
have a state named \code{Cc} defined in your \code{system.txt} and you want to set
it to \code{0.0} in a condition the following action would work.

\preformatted{
action = "SI_TT_STATE[Cc][0.0]"
}

The value here is a number but you can use any mathematical
combination of variables available in the titration environment. Also you
can create your own user function and place the function call within the
brackets above.

\bold{Titration Environment}
The \code{cond}, \code{action}, and \code{value} statements can use any variables available in
the titration environment. If you want to perform complicated actions, you can
simply create a user defined functions and pass it the variables from the
titration environment that you need. These include named variables from the
model as well as internal variables used to control the titration.

\bold{States and Parameters}

The state values (at the current titration time), system parameters (\code{<P>}),
static secondary parameters (\code{<As>}) and the initial value of covariates are
available as the names specified in the \code{system.txt} file. Since system resets
(\code{SI_TT_STATE}) are processed first, any changes made to states are the values
that are active for other actions.

\bold{Internal Simulation Variables}

Internal variables are used to control titration activities. These variables can also be used in the conditions and actions.
\itemize{
  \item \code{SIMINT_p} - list of system parameters
  \item \code{SIMINT_cfg} - system configuration sent into the titration routine
  \item \code{SIMINT_cfgtt}-systemconfigurationatthecurrenttitrationeventtime
  \item \code{SIMINT_ttimes} - vector of titration times (in simulation units)
  \item \code{SIMINT_tt_ts} - list of time scales for the current titration
  \item \code{SIMINT_history} - data frame tracking the history of conditions that evaluated true with the following structure:
  \item \itemize{
        \item \code{tname} - name of titration rule
        \item \code{value} - value indicating condition that was satisfied
        \item \code{simtime} - simulation time when that rule/value were triggered
        \item \code{timescale} -  time at the rule timescale when that rule/value were triggered
}
}

\bold{Individual Simulations}

To run an individual titration simulation use the follwoing:

\preformatted{
som = run_simulation_titrate(parameters, cfg)
}

 This provides the same output as \code{\link{run_simulation_ubiquity}} with
 two extra fields. The first, \code{som$titration}, contains three columns for each
 titration rule. The columns will have a length equal and corresponding to the
 simulation times. If the rule name is rname, then the column headers will have
 the following names and meanings:
\itemize{
  \item \code{tt.rname.value} - Value of the rule for the active condition or -1 if not triggered
  \item \code{tt.rname.simtime} - Simulation time where the last condition became active
  \item \code{tt.rname.timescale} - Simulation time in the time scale the rule was specified in
}

 The second field is \code{som$titration_history} which contains a summary list of all of the titration events that were triggered.
\itemize{
   \item \code{tname} - Titration rule name
   \item \code{value} - Value of the rule for the active condition or -1 if not triggered   
   \item \code{simtime} - Simulation time where the last condition became active
   \item \code{timescale} - Simulation time in the time scale the rule was specified in
}

To convert this structured list into a data frame the \code{\link{som_to_df}} command can be used:

\preformatted{
sdf = som_to_df(cfg, som)
}

To run stochastic titration simulations, the same function is used:

\preformatted{
som = simulate_subjects(parameters, cfg)
}

This will add a data a list element called \code{som$titration} with three
fields for each titration rule:

\itemize{
  \item \code{tt.rname.value} - Value of the rule for the active condition or -1 if not triggered
  \item \code{tt.rname.simtime} - Simulation time where the last condition became active
  \item \code{tt.rname.timescale} - Simulation time in the time scale the rule was specified in
} 

Each of these fields is a matrix with an entry for each simulation time
(column) and each subject (row). This data structure can also be converted to
a data frame using \code{som_to_df}.
}
\seealso{
\code{\link{system_new_tt_rule}}, \code{\link{run_simulation_titrate}},  \code{\link{som_to_df}}, \code{\link{simulate_subjects}}
}
