# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Simulates species tree using constant rate birth-death process
#'
#' @description Forward simulates to a number of tips. This function does so using
#'     the general algorithm of Hartmann et al. 2010.
#' @param sbr species birth rate (i.e. speciation rate)
#' @param sdr species death rate (i.e. extinction rate)
#' @param numbsim number of species trees to simulate
#' @param n_tips number of tips to simulate to
#' @param gsa_stop_mult number of tips to simulate the GSA tip to
#' @return List of objects of the tree class (as implemented in APE)
#' @references
#' K. Hartmann, D. Wong, T. Stadler. Sampling trees from evolutionary models.
#'     Syst. Biol., 59(4): 465-476, 2010.
#'
#' T. Stadler. Simulating trees on a fixed number of extant species.
#'     Syst. Biol., 60: 676-684, 2011.
#' @examples
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' numb_extant_tips <- 4
#' # simulate trees under the GSA so first simulates a tree with
#' # numb_extant_tips * 100 tips counting each time we have a tree with 10 tips
#' # then randomly picks one of those trees
#'
#' tree_list <- sim_sptree_bdp(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 n_tips = numb_extant_tips)
sim_sptree_bdp <- function(sbr, sdr, numbsim, n_tips, gsa_stop_mult = 10L) {
    .Call(`_treeducken_sim_sptree_bdp`, sbr, sdr, numbsim, n_tips, gsa_stop_mult)
}

#' Simulates species tree using constant rate birth-death process to a time
#'
#' @description Forward simulates a tree until a provided time is reached.
#' @param sbr species birth rate (i.e. speciation rate)
#' @param sdr species death rate (i.e. extinction rate)
#' @param numbsim number of species trees to simulate
#' @param t time to simulate to
#' @return List of objects of the tree class (as implemented in APE)
#' @references
#' K. Hartmann, D. Wong, T. Stadler. Sampling trees from evolutionary models.
#'     Syst. Biol., 59(4): 465-476, 2010.
#'
#' T. Stadler. Simulating trees on a fixed number of extant species.
#'     Syst. Biol., 60: 676-684, 2011.
#' @examples
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' time <- 1
#'
#' tree_list <- sim_sptree_bdp_time(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 t = time)
sim_sptree_bdp_time <- function(sbr, sdr, numbsim, t) {
    .Call(`_treeducken_sim_sptree_bdp_time`, sbr, sdr, numbsim, t)
}

#' Simulates locus tree using constant rate birth-death-transfer process
#'
#' @description Given a species tree simulates a locus or gene family tree along
#'     the species tree.
#' @param species_tree species tree to simulate along
#' @param gbr gene birth rate
#' @param gdr gene death rate
#' @param lgtr gene transfer rate
#' @param num_loci number of locus trees to simulate
#' @param transfer_type The type of transfer input. Acceptable options: "cladewise" or "random"
#' @return List of objects of the tree class (as implemented in APE)
#' @details Given a species tree will perform a birth-death process coupled with transfer.
#' The simulation runs along the species tree speciating and going extinct in addition to locus tree birth and deaths.
#' Thus with parameters set to 0.0 a tree identical to the species tree is returned (it is relabel however).
#'
#' Transfers are implemented as a birth-death process.
#' One descendant lineage retains its species identity the other gains a new identity.
#' At present, two types of transfers are implemented: "random" an "cladewise".
#' The random transfer mode transfers one randomly chooses a contemporaneous lineage.
#' Cladewise transfers choose lineages based on relatedness with more closely related lineages being more likely.
#' @references
#' Rasmussen MD, Kellis M. Unified modeling of gene duplication, loss, and
#'     coalescence using a locus tree. Genome Res. 2012;22(4):755–765.
#'     doi:10.1101/gr.123901.111
#' @examples
#' # first simulate a species tree
#' mu <- 0.5 # death rate
#' lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' numb_extant_tips <- 4
#' # simulate trees under the GSA so first simulates a tree with
#' # numb_extant_tips * 100 tips counting each time we have a tree with 10 tips
#' # then randomly picks one of those trees
#'
#' sp_tree <- sim_sptree_bdp(sbr = lambda,
#'                 sdr = mu,
#'                 numbsim = numb_replicates,
#'                 n_tips = numb_extant_tips)
#'
#' gene_br <- 1.0
#' gene_dr <- 0.2
#' transfer_rate <- 0.2
#' sim_locustree_bdp(species_tree = sp_tree[[1]],
#'                   gbr = gene_br,
#'                   gdr = gene_dr,
#'                   lgtr = transfer_rate,
#'                   num_loci = 10)
sim_locustree_bdp <- function(species_tree, gbr, gdr, lgtr, num_loci, transfer_type = "random") {
    .Call(`_treeducken_sim_locustree_bdp`, species_tree, gbr, gdr, lgtr, num_loci, transfer_type)
}

#' Simulates a cophylogenetic system using a paired birth-death process
#'
#' @details Simulates a cophylogenetic system using birth-death processes. The
#'     host tree is simulated following a constant rate birth-death process
#'     with an additional parameter - the cospeciation rate. This rate works as
#'     the speciation rate with the additional effect that if cospeciation
#'     occurs the symbiont tree also speciates. The symbiont tree is related to
#'     the host tree via an association matrix that describes which lineages
#'     are associated with which. The symbiont tree has an independent
#'     birth-death process with the addition of a host shift speciation rate
#'     that allows for the addition of more associated hosts upon symbiont
#'     speciation.
#' @param hbr host tree birth rate
#' @param hdr host tree death rate
#' @param sbr symbiont tree birth rate
#' @param sdr symbiont tree death rate
#' @param host_exp_rate host shift speciation rate
#' @param cosp_rate cospeciation rate
#' @param time_to_sim time units to simulate until
#' @param numbsim number of replicates
#' @return A list containing the `host_tree`, the `symbiont_tree`, the
#'     association matrix at present, and the history of events that have
#'     occurred.
#' @examples
#'
#' host_mu <- 0.5 # death rate
#' host_lambda <- 2.0 # birth rate
#' numb_replicates <- 10
#' time <- 1.0
#' symb_mu <- 0.2
#' symb_lambda <- 0.4
#' host_shift_rate <- 0.0
#' cosp_rate <- 2.0
#'
#' cophylo_pair <- sim_cophylo_bdp(hbr = host_lambda,
#'                            hdr = host_mu,
#'                            cosp_rate = cosp_rate,
#'                            host_exp_rate = host_shift_rate,
#'                            sdr = symb_mu,
#'                            sbr = symb_lambda,
#'                            numbsim = numb_replicates,
#'                            time_to_sim = time)
#'
sim_cophylo_bdp <- function(hbr, hdr, sbr, sdr, host_exp_rate, cosp_rate, time_to_sim, numbsim) {
    .Call(`_treeducken_sim_cophylo_bdp`, hbr, hdr, sbr, sdr, host_exp_rate, cosp_rate, time_to_sim, numbsim)
}

#' Simulate multispecies coalescent on a species tree
#'
#' @description Simulates the multispecies coalescent on a species tree.
#' @param species_tree input species tree of class "phylo"
#' @param ne Effective population size
#' @param generation_time The number of time units per generation
#' @param num_sampled_individuals number of individuals sampled within each lineage
#' @param num_genes number of genes to simulate within each locus
#' @param mutation_rate The rate of mutation per generation
#' @param rescale Rescale the tree into coalescent units (otherwise assumes it is in those units)
#' @details
#' This a multispecies coalescent simulator with two usage options.
#' The function can rescale the given tree into coalescent units given the `mutation_rate`, `ne`, and the `generation_time`.
#' These result in a tree with coalescent times in units of expected number of mutations per site.
#' The generation_time parameter default is 1 time unit per generation if the units of the tree are in millions of years
#' The mutation_rate parameter is by default set to 1 mutations per site per generation (which is nonsensical).
#' Rescale is set to true by default.
#'
#' If rescale is set to false the tree is assumed to be in coalescent units and `ne` is used as the population
#' genetic parameter theta.
#' @return A list of coalescent trees
#' @seealso sim_locustree_bdp, sim_sptree_bdp, sim_sptree_bdp_time
#'
#' @examples
#' # first simulate a species tree
#' mu <- 0.5
#' lambda <- 1.0
#' nt <- 6
#' tr <- sim_sptree_bdp(sbr = lambda, sdr = mu, numbsim = 1, n_tips = nt)
#' # for a locus tree with 100 genes sampled per locus tree
#' gentrees <- sim_multispecies_coal(tr[[1]],
#'                                     ne = 10000,
#'                                     mutation_rate = 1e-9,
#'                                     generation_time = 1e-6,
#'                                     num_sampled_individuals = 1,
#'                                     num_genes = 100)
#'
#' @references
#' Bruce Rannala and Ziheng Yang (2003) Bayes Estimation of Species Divergence Times and Ancestral Population Sizes Using DNA Sequences From Multiple Loci Genetics August 1, 2003 vol. 164 no. 4 1645-1656
#' Mallo D, de Oliveira Martins L, Posada D (2015) SimPhy: Phylogenomic Simulation of Gene, Locus and Species Trees. Syst. Biol. doi: http://dx.doi.org/10.1093/sysbio/syv082
sim_multispecies_coal <- function(species_tree, ne, num_sampled_individuals, num_genes, rescale = TRUE, mutation_rate = 1L, generation_time = 1L) {
    .Call(`_treeducken_sim_multispecies_coal`, species_tree, ne, num_sampled_individuals, num_genes, rescale, mutation_rate, generation_time)
}

