% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collapse_index.R
\name{collapse_index}
\alias{collapse_index}
\title{Collapse an index vector so that all observations in an interval share the
same date}
\usage{
collapse_index(index, period = "yearly", start_date = NULL, side = "end",
  ...)
}
\arguments{
\item{index}{An index vector.}

\item{period}{A character specification used for time-based grouping. The
general format to use is \code{"frequency period"} where frequency is a number
like 1 or 2, and period is an interval like weekly or yearly. There must be
a space between the two.

Note that you can pass the specification in a flexible way:
\itemize{
\item 1 Year: \code{'1 year'} / \code{'1 Y'} / \code{'1 yearly'} / \code{'yearly'}
}

This shorthand is available for year, quarter, month, day, hour, minute,
and second periodicities. Note that "m" is the 1 letter specification used
for "month" and "M" is used for "minute". Otherwise case does not matter.}

\item{start_date}{Optional argument used to specify the start date for the
first group. The default is to start at the closest period boundary
below the minimum date in the supplied index.}

\item{side}{Whether to return the date at the beginning or the end of
the new period. By default, the "end" of the period.
Use "start" to change to the start of the period.}

\item{...}{Not currently used.}
}
\description{
When \code{collapse_index()} is used, the index vector is altered
so that all dates that fall in a specified interval share a common date.
The most common use case for this is to then group on the collapsed index.
}
\details{
Because this is often used for end of period summaries, the default is to
use \code{side = "end"}. Note that this is the opposite of \code{\link[=as_period]{as_period()}} where
the default is \code{side = "start"}.
}
\examples{

# Basic functionality -------------------------------------------------------

# Facebook stock prices
data(FB)
FB <- as_tbl_time(FB, date)

# Collapse to weekly dates
dplyr::mutate(FB, date = collapse_index(date, "weekly"))

# A common workflow is to group on the new date column
# to perform a time based summary
FB \%>\%
  dplyr::mutate(date = collapse_index(date, "yearly")) \%>\%
  dplyr::group_by(date) \%>\%
  dplyr::summarise_if(is.numeric, mean)

# You can also assign the result to a separate column and use that
# to nest on, allowing for 'period nests' that keep the
# original dates in the nested tibbles.
FB \%>\%
  dplyr::mutate(nest_date = collapse_index(date, "2 year")) \%>\%
  dplyr::group_by(nest_date) \%>\%
  tidyr::nest()

# Grouped functionality -----------------------------------------------------

data(FANG)
FANG <- FANG \%>\%
  as_tbl_time(date) \%>\%
  dplyr::group_by(symbol)

# Collapse each group to monthly,
# calculate monthly standard deviation for each column
FANG \%>\%
  dplyr::mutate(date = collapse_index(date, "monthly")) \%>\%
  dplyr::group_by(date, add = TRUE) \%>\%
  dplyr::summarise_all(sd)


}
