% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/repair-names.R
\name{name-repair}
\alias{name-repair}
\title{Repair the names of a vector}
\description{
tibble deals with a few levels of name repair:
\itemize{
\item \code{minimal} names exist. The \code{names} attribute is not \code{NULL}. The name of
an unnamed element is \code{""} and never \code{NA}. Tibbles created by the tibble
package have names that are, at least, \code{minimal}.
\item \code{unique} names are \code{minimal}, have no duplicates, and are never empty
(literally, no \code{""}s).
\itemize{
\item All columns can be accessed by name via \code{df[["name"]]}.
}
\item \code{universal} names are \code{unique} and syntactic (see Details for more).
\itemize{
\item Names work everywhere, without quoting: \code{df$name} and
\code{lm(name1 ~ name2, data = df)} and \code{dplyr::select(df, name)} all work.
}
}

\code{universal} implies \code{unique}, \code{unique} implies \code{minimal}. These levels are
nested.

The \code{.name_repair} argument of \code{\link[=tibble]{tibble()}} and \code{\link[=as_tibble]{as_tibble()}} refers to these
levels. Alternatively, the user can pass their own name repair function. It
should anticipate \code{minimal} names as input and should, likewise, return names
that are at least \code{minimal}.

The existing functions \code{\link[=tidy_names]{tidy_names()}}, \code{\link[=set_tidy_names]{set_tidy_names()}},
and \code{\link[=repair_names]{repair_names()}} are soft-deprecated.
}
\section{\code{minimal} names}{


\code{minimal} names exist. The \code{names} attribute is not \code{NULL}. The name of an
unnamed element is \code{""} and never \code{NA}.

Examples:\preformatted{Original names of a vector with length 3: NULL
                           minimal names: "" "" ""

                          Original names: "x" NA
                           minimal names: "x" ""
}

Request \code{.name_repair = "minimal"} to suppress almost all name munging. This
is useful when the first row of a data source -- allegedly variable names --
actually contains \emph{data} and the resulting tibble is destined for reshaping
with, e.g., \code{tidyr::gather()}.
}

\section{\code{unique} names}{


\code{unique} names are \code{minimal}, have no duplicates, and are never empty
(literally, no \code{""}s). If a data frame has \code{unique} names, you can index it
by name, e.g., \code{df[["name"]]} works.

There are many ways to make names \code{unique}. We append a suffix of the form
\code{..j} to any name that is \code{""} or a duplicate, where \code{j} is the position.

Example:\preformatted{Original names:    ""    "x"    "" "y"    "x"
  unique names: "..1" "x..2" "..3" "y" "x..5"
}

Pre-existing suffixes of the form \code{..j} are always stripped, prior to making
names \code{unique}, i.e. reconstructing the suffixes. If this interacts poorly
with your names, you should take control of name repair.
}

\section{\code{universal} names}{


\code{universal} names are \code{unique} and syntactic, meaning they:
\itemize{
\item Are never empty (inherited from \code{unique}).
\item Have no duplicates (inherited from \code{unique}).
\item Consist of letters, numbers, and the dot \code{.} or underscore \code{_}
characters.
\item Start with a letter or start with the dot \code{.} not followed by a number.
\item Are not a \link{reserved} word, e.g., \code{if} or \code{function} or \code{TRUE}.
\item Are not \code{...}. Do not have the form \code{..i}, where \code{i} is a number.
}

If a data frame has \code{universal} names, variable names can be used "as is" in
code. They work well with nonstandard evaluation, e.g., \code{df$name} works.

Tibble has a different method of making names syntactic than
\code{\link[base:make.names]{base::make.names()}}. In general, tibble prepends one or more dots \code{.} until
the name is syntactic.

Examples:\preformatted{ Original names:     ""    "x"    NA     "x"
universal names: "...1" "x..2" "...3" "x..4"

  Original names: "(y)"  "_z"  ".2fa"  "FALSE"  "..."  "..3"
 universal names: ".y." "._z" "..2fa" ".FALSE" "...." "...6"
}
}

\examples{
\dontrun{
## by default, duplicate names are not allowed
tibble(x = 1, x = 2)
}
## you can authorize duplicate names
tibble(x = 1, x = 2, .name_repair = "minimal")
## or request that the names be made unique
tibble(x = 1, x = 2, .name_repair = "unique")

## by default, non-syntactic names are allowed
df <- tibble(`a 1` = 1, `a 2` = 2)
## because you can still index by name
df[["a 1"]]
df$`a 1`

## syntactic names are easier to work with, though, and you can request them
df <- tibble(`a 1` = 1, `a 2` = 2, .name_repair = "universal")
df$a.1

## you can specify your own name repair function
tibble(x = 1, x = 2, .name_repair = make.unique)

fix_names <- function(x) gsub("\%", " percent", x)
tibble(`25\%` = 1, `75\%` = 2, .name_repair = fix_names)

fix_names <- function(x) gsub("\\\\s+", "_", x)
tibble(`year 1` = 1, `year 2` = 2, .name_repair = fix_names)

## purrr-style anonymous functions and constants
## are also supported
tibble(x = 1, x = 2, .name_repair = ~ make.names(., unique = TRUE))

tibble(x = 1, x = 2, .name_repair = ~ c("a", "b"))

## the names attibute will be non-NULL, with "" as the default element
df <- as_tibble(list(1:3, letters[1:3]), .name_repair = "minimal")
names(df)
}
\seealso{
\code{\link[rlang:names2]{rlang::names2()}} returns the names of an object, after making them
\code{minimal}.

The \href{https://principles.tidyverse.org/names-attribute.html}{Names attribute}
section in the "tidyverse package development principles".
}
