% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tokenize.R
\name{step_tokenize}
\alias{step_tokenize}
\alias{tidy.step_tokenize}
\title{Tokenization of character variables}
\usage{
step_tokenize(
  recipe,
  ...,
  role = NA,
  trained = FALSE,
  columns = NULL,
  training_options = list(),
  options = list(),
  token = "words",
  engine = "tokenizers",
  custom_token = NULL,
  skip = FALSE,
  id = rand_id("tokenize")
)

\method{tidy}{step_tokenize}(x, ...)
}
\arguments{
\item{recipe}{A \link{recipe} object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variables are affected by the step. See \code{\link[recipes:selections]{recipes::selections()}}
for more details.}

\item{role}{Not used by this step since no new variables are
created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{columns}{A character string of variable names that will
be populated (eventually) by the \code{terms} argument. This is \code{NULL}
until the step is trained by \code{\link[recipes:prep]{recipes::prep.recipe()}}.}

\item{training_options}{A list of options passed to the tokenizer when it is
being trained. Only applicable for engine == "tokenizers.bpe".}

\item{options}{A list of options passed to the tokenizer.}

\item{token}{Unit for tokenizing. See details for options. Defaults to
"words".}

\item{engine}{Package that will be used for tokenization. See details for
options. Defaults to "tokenizers".}

\item{custom_token}{User supplied tokenizer. Use of this argument
will overwrite the token and engine arguments. Must take a character vector
as input and output a list of character vectors.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[recipes:bake]{recipes::bake.recipe()}}? While all operations are baked
when \code{\link[recipes:prep]{recipes::prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = FALSE}.}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_tokenize} object.}
}
\value{
An updated version of \code{recipe} with the new step added
to the sequence of existing steps (if any).
}
\description{
\code{\link[=step_tokenize]{step_tokenize()}} creates a \emph{specification} of a recipe step that
will convert a character predictor into a \link{tokenlist}.
}
\details{
Tokenization is the act of splitting a character string into smaller parts
to be further analyzed. This step uses the \code{tokenizers} package which
includes heuristics to split the text into paragraphs tokens, word tokens
among others. \code{textrecipes} keeps the tokens in a \link{tokenlist} and other
steps will do their tasks on those \link{tokenlist}s before transforming them
back to numeric.

Working will \code{textrecipes} will almost always start by calling
\code{step_tokenize} followed by modifying and filtering steps. This is not always
the case as you sometimes want to do apply pre-tokenization steps, this can
be done with \code{\link[recipes:step_mutate]{recipes::step_mutate()}}.
}
\section{Engines}{
The choice of \code{engine} determines the possible choices of \code{token}.

The following is some small example data used in the following examples\if{html}{\out{<div class="sourceCode r">}}\preformatted{text_tibble <- tibble(
  text = c("This is words", "They are nice!")
)
}\if{html}{\out{</div>}}
\subsection{tokenizers}{

The tokenizers package is the default \code{engine} and it comes with the
following unit of \code{token}. All of these options correspond to a function in
the tokenizers package.
\itemize{
\item "words" (default)
\item "characters"
\item "character_shingles"
\item "ngrams"
\item "skip_ngrams"
\item "sentences"
\item "lines"
\item "paragraphs"
\item "regex"
\item "tweets"
\item "ptb" (Penn Treebank)
\item "skip_ngrams"
\item "word_stems"
}

The default tokenizer is \code{"word"} which splits the text into a series of
words. By using \code{step_tokenize()} without setting any arguments you get word
tokens\if{html}{\out{<div class="sourceCode r">}}\preformatted{recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(text) \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
## [1] "this"  "is"    "words"
## 
## [[2]]
## [1] "they" "are"  "nice"
}

This tokenizer has arguments that change how the tokenization occurs and can
accessed using the \code{options} argument by passing a named list. Here we are
telling \link[tokenizers:basic-tokenizers]{tokenizers::tokenize_words} that we don't want to turn the words to
lowercase\if{html}{\out{<div class="sourceCode r">}}\preformatted{recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(text, 
                options = list(lowercase = FALSE)) \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
## [1] "This"  "is"    "words"
## 
## [[2]]
## [1] "They" "are"  "nice"
}

We can also stop removing punctuation.\if{html}{\out{<div class="sourceCode r">}}\preformatted{recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(text, 
                options = list(strip_punct = FALSE,
                               lowercase = FALSE)) \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
## [1] "This"  "is"    "words"
## 
## [[2]]
## [1] "They" "are"  "nice" "!"
}

The tokenizer can be changed by setting a different \code{token}. Here we change
it to return character tokens.\if{html}{\out{<div class="sourceCode r">}}\preformatted{recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(text, token = "characters") \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
##  [1] "t" "h" "i" "s" "i" "s" "w" "o" "r" "d" "s"
## 
## [[2]]
##  [1] "t" "h" "e" "y" "a" "r" "e" "n" "i" "c" "e"
}

It is worth noting that not all these token methods are appropriate but are
included for completeness.
}

\subsection{spacyr}{
\itemize{
\item "words"
}
}

\subsection{tokenizers.bpe}{

The tokeenizers.bpe engine performs Byte Pair Encoding Text Tokenization.
\itemize{
\item "words"
}

This tokenizer is trained on the training set and will thus need to be passed
training arguments. These are passed to the \code{training_options} argument and
the most important one is \code{vocab_size}. The determines the number of unique
tokens the tokenizer will produce. It is generally set to a much higher
value, typically in the thousands, but is set to 22 here for demonstration
purposes.\if{html}{\out{<div class="sourceCode r">}}\preformatted{recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(
    text, 
    engine = "tokenizers.bpe", 
    training_options = list(vocab_size = 22)
  ) \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
##  [1] "_Th" "is"  "_"   "is"  "_"   "w"   "o"   "r"  
##  [9] "d"   "s"  
## 
## [[2]]
##  [1] "_Th" "e"   "y"   "_"   "a"   "r"   "e"   "_"  
##  [9] "n"   "i"   "c"   "e"   "!"
}
}

\subsection{udpipe}{
\itemize{
\item "words"
}
}

\subsection{custom_token}{

Sometimes you need to perform tokenization that is not covered by the
supported engines. In that case you can use the \code{custom_token} argument to
pass a function in that performs the tokenization you want.

Below is an example of a very simple space tokenization. This is a very fast
way of tokenizing.\if{html}{\out{<div class="sourceCode r">}}\preformatted{space_tokenizer <- function(x) \{
  strsplit(x, " +")
\}

recipe(~ text, data = text_tibble) \%>\%
  step_tokenize(
    text, 
    custom_token = space_tokenizer
  ) \%>\%
  show_tokens(text)
}\if{html}{\out{</div>}}\preformatted{## [[1]]
## [1] "This"  "is"    "words"
## 
## [[2]]
## [1] "They"  "are"   "nice!"
}
}
}

\examples{
library(recipes)
library(modeldata)
data(okc_text)

okc_rec <- recipe(~., data = okc_text) \%>\%
  step_tokenize(essay0)

okc_obj <- okc_rec \%>\%
  prep()

bake(okc_obj, new_data = NULL, essay0) \%>\%
  slice(1:2)

bake(okc_obj, new_data = NULL) \%>\%
  slice(2) \%>\%
  pull(essay0)

tidy(okc_rec, number = 1)
tidy(okc_obj, number = 1)

okc_obj_chars <- recipe(~., data = okc_text) \%>\%
  step_tokenize(essay0, token = "characters") \%>\%
  prep()

bake(okc_obj, new_data = NULL) \%>\%
  slice(2) \%>\%
  pull(essay0)

}
\seealso{
\code{\link[=step_untokenize]{step_untokenize()}} to untokenize.
}
\concept{character to tokenlist steps}
