% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/date.R, R/extract.R,
%   R/refine.R
\docType{methods}
\name{date}
\alias{date}
\alias{get_dates}
\alias{get_dates-method}
\alias{set_dates<-}
\alias{set_dates-method}
\alias{date_mcd}
\alias{date_mcd-method}
\alias{date_event}
\alias{date_event-method}
\alias{refine_dates}
\alias{refine_dates-method}
\alias{date_mcd,CountMatrix-method}
\alias{date_event,CountMatrix-method}
\alias{get_dates,AbundanceMatrix-method}
\alias{set_dates<-,AbundanceMatrix-method}
\alias{set_dates,AbundanceMatrix-method}
\alias{refine_dates,DateModel-method}
\title{Date Archaeological Assemblages}
\usage{
get_dates(object)

set_dates(object) <- value

date_mcd(object, ...)

date_event(object, ...)

refine_dates(object, ...)

\S4method{date_mcd}{CountMatrix}(object, dates, errors = NULL,
  level = 0.95, n = 1000, ...)

\S4method{date_event}{CountMatrix}(object, level = 0.95, cutoff = 90,
  ...)

\S4method{get_dates}{AbundanceMatrix}(object)

\S4method{set_dates}{AbundanceMatrix}(object) <- value

\S4method{refine_dates}{DateModel}(object, method = c("jackknife",
  "bootstrap"), n = 1000, ...)
}
\arguments{
\item{object}{A \eqn{m \times p}{m x p} matrix of count data (typically of
class \linkS4class{CountMatrix}).}

\item{value}{A possible value for the element(s) of \code{object} (see
below).}

\item{...}{Further arguments to be passed to internal methods.}

\item{dates}{A length-\eqn{p} numeric vector giving the mid-date of each type
(year AD).}

\item{errors}{A length-\eqn{p} numeric vector giving the absolute error of
each date.}

\item{level}{A length-one \code{\link{numeric}} vector giving the
confidence level.}

\item{n}{A non-negative \code{\link{integer}} giving the number of bootstrap
replications (see below).}

\item{cutoff}{An \code{\link{integer}} giving the cumulative percentage of
variance used to select CA factorial components for linear model fitting
(see details). All compounds with a cumulative percentage of variance of
less than the \code{cutoff} value will be retained.}

\item{method}{A \code{\link{character}} string specifying the resampling
method to be used. This must be one of "\code{jackknife}",
"\code{bootstrap}" (see details). Any unambiguous substring can be given.}
}
\value{
\code{date_mcd} returns a \code{\link{data.frame}} with the following
 columns:
 \describe{
  \item{id}{An identifier to link each row to an assemblage.}
  \item{date}{The Mean Ceramic Date.}
  \item{error}{The error on the MCD.}
  \item{lower}{The lower boundary of the confidence interval.}
  \item{upper}{The upper boundary of the confidence interval.}
 }

 \code{date_event} returns an object of class \linkS4class{DateModel}.

 \code{refine_dates} returns a \code{\link{data.frame}}.
}
\description{
\code{date_mcd} estimates the Mean Ceramic Date of an assemblage.
}
\details{
\code{date_event} estimates the event and accumulation dates of an
assemblage.

\code{refine_dates} checks the stability of a date model with resampling
methods.

\code{set_dates} and \code{get_dates} allow to get or set the dates of an
object.
}
\note{
Bellanger \emph{et al.} did not publish the data supporting their
 demonstration: no replication of their results is possible and this
 implementation must be considered \strong{experimental}.
 \code{date_event} may be subject to major changes in a future release.

 Refining methods can lead to much longer execution times and larger output
 objects. To monitor the execution of these re-sampling procedures, a
 progress bar will automatically be displayed if the
 \code{\link[pbapply]{pbapply}} package is installed on your machine.
}
\section{Set dates}{

 An attempt is made to interpret the argument \code{value} in a suitable way.
 \emph{Note} that errors are assumed to be given at \code{1} sigma.

 If \code{value} is a:
 \describe{
  \item{\code{character} vector}{it is assumed to contain Roman numerals.}
  \item{\code{numeric} or \code{integer} \code{vector}}{these values are
  assumed to represent dates without known errors.}
  \item{\code{list}}{containing components "\code{value}" and "\code{error}",
  these are used to define dates and corresponding errors.}
  \item{\code{matrix} or \code{data.frame} with two or more columns}{the
  first is assumed to contain the dates and the second the error values.
  \emph{Note} that if \code{value} has columns named "\code{value}" and
  "\code{error}", these columns will be used.}
 }
}

\section{Mean Ceramic Date}{

 The Mean Ceramic Date (MCD) is a point estimate of the occupation of an
 archaeological site (South 1977). The MCD is estimated as the weighted mean
 of the date midpoints of the ceramic types (based on absolute dates or the
 known production interval) found in a given assemblage. The weights are the
 relative frequencies of the respective types in the assemblage.

 A bootstrapping procedure is used to estimate the confidence interval of a
 given MCD. For each assemblage, a large number of new bootstrap replicates
 is created, with the same sample size, by resampling the original
 assemblage with replacement. MCDs are calculated for each replicates and
 upper and lower boundaries of the confidence interval associated with each
 MCD are then returned. Confidence interval are not estimated for assemblages
 with only a single type (\code{NA}s are returned).
}

\section{Event and Accumulation Dates}{

 This is an implementation of the chronological modeling method proposed by
 Bellanger and Husi (2012, 2013).

 Event and accumulation dates are density estimates of the occupation and
 duration of an archaeological site (Bellanger and Husi 2012, 2013).
 The event date is an estimation of the \emph{terminus post-quem} of an
 archaeological assemblage. The accumulation date represents the
 "chronological profile" of the assemblage. According to Bellanger and Husi
 (2012), accumulation date can be interpreted "at best [...] as a formation
 process reflecting the duration or succession of events on the scale of
 archaeological time, and at worst, as imprecise dating due to contamination
 of the context by residual or intrusive material." In other words,
 accumulation dates estimate occurrence of archaeological events and rhythms
 of the long term.

 This method relies on strong archaeological and statistical assumptions.
 Use it only if you know what you are doing (see references below and the
 vignette: \code{utils::vignette("dating", package = "tabula")}).
}

\section{Date Model Checking}{

 \code{refine_date} can be used to check the stability of the resulting
 \linkS4class{DateModel} with resampling methods.

 If \code{jackknife} is used, one type/fabric is removed at a
 time and all statistics are recalculated. In this way, one can assess
 whether certain type/fabric has a substantial influence on the date
 estimate.
 A six columns \code{\link{data.frame}} is returned, giving the results of
 the resampling procedure (jackknifing fabrics) for each assemblage (in rows)
 with the following columns:
 \describe{
  \item{id}{An identifier to link each row to an assemblage.}
  \item{date}{The jackknife event date estimate.}
  \item{lower}{The lower boundary of the associated prediction interval.}
  \item{upper}{The upper boundary of the associated prediction interval.}
  \item{error}{The standard error of predicted means.}
  \item{bias}{The jackknife estimate of bias.}
 }

 If \code{bootstrap} is used, a large number of new
 bootstrap assemblages is created, with the same sample size, by resampling
 each of the original assemblage with replacement. Then, examination of the
 bootstrap statistics makes it possible to pinpoint assemblages that require
 further investigation.
 A six columns \code{\link{data.frame}} is returned, giving the bootstrap
 distribution statistics for each replicated assemblage (in rows)
 with the following columns:
 \describe{
  \item{id}{An identifier to link each row to an assemblage.}
  \item{min}{Minimum value.}
  \item{Q05}{Sample quantile to 0.05 probability.}
  \item{mean}{Mean value (event date).}
  \item{Q95}{Sample quantile to 0.95 probability.}
  \item{max}{Maximum value.}
 }
}

\examples{
## Mean Ceramic Date
## Coerce the zuni dataset to an abundance (count) matrix
zuni_counts <- as_count(zuni)

## Set the start and end dates for each ceramic type
zuni_dates <- list(
  LINO = c(600, 875), KIAT = c(850, 950), RED = c(900, 1050),
  GALL = c(1025, 1125), ESC = c(1050, 1150), PUBW = c(1050, 1150),
  RES = c(1000, 1200), TULA = c(1175, 1300), PINE = c(1275, 1350),
  PUBR = c(1000, 1200), WING = c(1100, 1200), WIPO = c(1125, 1225),
  SJ = c(1200, 1300), LSJ = c(1250, 1300), SPR = c(1250, 1300),
  PINER = c(1275, 1325), HESH = c(1275, 1450), KWAK = c(1275, 1450)
)

## Calculate date midpoints and errors
zuni_mid <- vapply(X = zuni_dates, FUN = mean, FUN.VALUE = numeric(1))
zuni_error <- vapply(X = zuni_dates, FUN = diff, FUN.VALUE = numeric(1)) / 2

## Calculate MCD
## (we use a bootstrapping procedure to estimate the confidence interval)
zuni_mcd <- date_mcd(zuni_counts, dates = zuni_mid, errors = zuni_error)
head(zuni_mcd)

## Plot dates
keep_sites <- c("CS11", "CS12", "CS144", "CS195", "CS40", "LZ0219", "LZ0280",
                "LZ0367", "LZ0508", "LZ0560", "LZ1076", "LZ1087")
set_dates(zuni_counts) <- list(value = zuni_mcd$date, error = zuni_mcd$error)
plot_date(zuni_counts, select = keep_sites, sort = "asc") +
  ggplot2::theme_bw()

## Event and accumulation dates (Bellanger et al.)
## See the vignette:
\donttest{
utils::vignette("dating", package = "tabula")
}
}
\references{
Bellanger, L. & Husi, P. (2013). Mesurer et modéliser le temps inscrit dans
 la matière à partir d'une source matérielle : la céramique médiévale.
 In \emph{Mesure et Histoire Médiévale}. Histoire ancienne et médiévale.
 Paris: Publication de la Sorbonne, p. 119-134.

 Bellanger, L. & Husi, P. (2012). Statistical Tool for Dating and
 Interpreting Archaeological Contexts Using Pottery. \emph{Journal of
 Archaeological Science}, 39(4), 777-790.
 DOI: \href{https://doi.org/10.1016/j.jas.2011.06.031}{10.1016/j.jas.2011.06.031}.

 Bellanger, L., Tomassone, R. & Husi, P. (2008). A Statistical Approach for
 Dating Archaeological Contexts. \emph{Journal of Data Science}, 6, 135-154.

 Bellanger, L., Husi, P. & Tomassone, R. (2006). Une approche statistique
 pour la datation de contextes archéologiques. \emph{Revue de Statistique
 Appliquée}, 54(2), 65-81.

 Bellanger, L., Husi, P. & Tomassone, R. (2006). Statistical Aspects of
 Pottery Quantification for the Dating of Some Archaeological Contexts.
 \emph{Archaeometry}, 48(1), 169-183.
 DOI: \href{https://doi.org/10.1111/j.1475-4754.2006.00249.x}{10.1111/j.1475-4754.2006.00249.x}.

 Poblome, J. & Groenen, P. J. F. (2003). Constrained Correspondence Analysis
 for Seriation of Sagalassos Tablewares. In Doerr, M. & Apostolis, S. (eds.),
 \emph{The Digital Heritage of Archaeology}. Athens: Hellenic Ministry of
 Culture.

 South, S. A. (1977). \emph{Method and Theory in Historical Archaeology}.
 New York: Academic Press.
}
\seealso{
\link{refine}
}
\author{
N. Frerebeau
}
\concept{dating}
