% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stdmod.R, R/stdmod_bootci.R, R/stdmod-package.R
\docType{package}
\name{stdmod}
\alias{stdmod}
\alias{stdmod_boot}
\title{Standardized Moderation Effect Given an 'lm' Output}
\usage{
stdmod(
  lm_out,
  x = NULL,
  w = NULL,
  y = NULL,
  x_rescale = TRUE,
  w_rescale = TRUE,
  y_rescale = TRUE
)

stdmod_boot(
  lm_out,
  ...,
  nboot = 100,
  conf = 0.95,
  boot_args = NULL,
  full_output = FALSE
)
}
\arguments{
\item{lm_out}{The output from \code{\link[=lm]{lm()}}.}

\item{x}{The focal variable, that is, the variable with its effect
being moderated. If supplied, its standard deviation will be
used
for rescaling. Also called the independent variable in some
models. Default is \code{NULL}.}

\item{w}{The moderator. If supplied, its standard deviation will be
used
for rescaling. Default is \code{NULL}.}

\item{y}{The outcome variable (dependent variable) . If supplied,
its standard
deviation will be used for rescaling. Default is NULL.}

\item{x_rescale}{If \code{TRUE}, will rescale x by its standard deviation.
Default is \code{TRUE}.}

\item{w_rescale}{If \code{TRUE}, will rescale w by its standard deviation.
Default is \code{TRUE}.}

\item{y_rescale}{If \code{TRUE}, will rescale y by its standard deviation.
Default is \code{TRUE}.}

\item{...}{Parameters to be passed to \code{\link[=stdmod]{stdmod()}}.}

\item{nboot}{The number of bootstrap samples. Default is 100.}

\item{conf}{The level of confidence for the confidence interval.
Default is .95.}

\item{boot_args}{A named list of arguments to be passed to \code{\link[boot:boot]{boot::boot()}}.
Default is \code{NULL}.}

\item{full_output}{Whether the full output from \code{\link[boot:boot]{boot::boot()}} is returned.
Default is \code{FALSE}.}
}
\value{
\code{\link[=stdmod]{stdmod()}} returns a scalar: The standardized moderation effect.

\code{\link[=stdmod_boot]{stdmod_boot()}} returns a list with two elements. The element \code{ci} is
a numeric vector of the bootstrap confidence interval. The element \code{boot_out},
if not \code{NA}, is the output of \code{\link[boot:boot]{boot::boot()}}, which is used to do the
bootstrapping.
}
\description{
Compute the standardized moderation effect in a moderated
regression model.
}
\details{
Two more general functions, \code{\link[=std_selected]{std_selected()}} and
\code{\link[=std_selected_boot]{std_selected_boot()}}, have been developed and can do what these functions
do and more. Users are recommended to use them instead of \code{\link[=stdmod]{stdmod()}} and
\code{\link[=stdmod_boot]{stdmod_boot()}}. These two functions will not be updated in the near
future.

Nevertheless, if computing the standardized moderation effect and
forming its nonparametric
bootstrap interval are all required, then these functions can still
be used.

\code{\link[=stdmod]{stdmod()}} computes the standardized moderation effect given an
\code{\link[=lm]{lm()}} output using the formula from Cheung, Cheung, Lau, Hui, and Vong
(2022). Users specify
the moderator, the focal variable (the variable with its effect on
the outcome variable moderated), the outcome variable (dependent variable)
, and the corresponding
standardized moderation
effect. Users can also select which variable(s) will be standardized.

\code{\link[=stdmod_boot]{stdmod_boot()}} is a wrapper of \code{\link[=stdmod]{stdmod()}}. It computes the nonparametric
bootstrap confidence interval of the standardized moderation effect, as
suggested by Cheung, Cheung, Lau, Hui, and Vong (2022), given
the output of \code{\link[=lm]{lm()}}

Percentile interval from \code{\link[boot:boot.ci]{boot::boot.ci()}} is returned by this function.
If other types of
confidence intervals are desired, set \code{full_output = TRUE} and use
\code{\link[boot:boot.ci]{boot::boot.ci()}} on the element \code{boot_out} in the output of this
function.
}
\section{Functions}{
\itemize{
\item \code{stdmod}: The base function for computing standardized
moderation effect

\item \code{stdmod_boot}: A wrapper of \code{\link[=stdmod]{stdmod()}} that computes the nonparametric
bootstrap confidence interval of the standardized
moderation effect.
}}

\examples{

# Load a test data of 500 cases

dat <- test_x_1_w_1_v_2_n_500

# Do regression as usual:
lm_raw <- lm(dv ~ iv*mod + v1 + v2, dat)
summary(lm_raw)

# The standard deviations of iv, dv, and mod:
sds <- apply(dat, 2, sd)
sds

# Compute the standardized moderation effect:
stdmod_xyw <- stdmod(lm_raw, x = iv, y = dv, w = mod)
stdmod_xyw
# By default, all three variables will be standardized.

# Check against self-computed standardized moderation effect:
coef(lm_raw)["iv:mod"] * sds["iv"] * sds["mod"] / sds["dv"]

# Standardize only the iv, i.e., do not standardized dv and the moderator:
stdmod_x <- stdmod(lm_raw, x = iv, y = dv, w = mod,
                   x_rescale = TRUE,  y_rescale = FALSE, w_rescale = FALSE)
stdmod_x
# Check against self-computed moderation effect with only iv standardized:
coef(lm_raw)["iv:mod"] * sds["iv"]


dat <- test_x_1_w_1_v_2_n_500
# Do regression as usual:
lm_raw <- lm(dv ~ iv*mod + v1 + v2, dat)

# Compute the standardized moderation effect.
# Form its confidence interval by nonparametric bootstrapping.
set.seed(85740917)
stdmod_xyw_boot <- stdmod_boot(lm_raw, x = iv, w = mod, y = dv, nboot = 100)
# In real analysis, nboot should be at least 2000.

# Print the ci
stdmod_xyw_boot$ci

# Repeat the analysis but keep the results from boot:
set.seed(85740917)
stdmod_xyw_boot <- stdmod_boot(lm_raw, x = iv, w = mod, y = dv,
                                 nboot = 200, full_output = TRUE)
# In real analysis, nboot should be at least 2000.

# Print the 95\% percentile confidence interval
stdmod_xyw_boot$ci


}
\references{
Cheung, S. F., Cheung, S.-H., Lau, E. Y. Y., Hui, C. H., & Vong, W. N.
(2022) Improving an old way to measure moderation effect in standardized
units. Advance online publication. \emph{Health Psychology}.
\doi{10.1037/hea0001188}
}
\author{
Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}
}
