#' @title Expression for one sample *t*-test and its non-parametric and
#'   robust equivalents
#' @name expr_t_onesample
#' @author \href{https://github.com/IndrajeetPatil}{Indrajeet Patil}
#'
#' @param x A numeric variable.
#' @param test.value A number specifying the value of the null hypothesis
#'   (Default: `0`).
#' @param type Type of statistic expected (`"parametric"` or `"nonparametric"`
#'   or `"robust"` or `"bayes"`).Corresponding abbreviations are also accepted:
#'   `"p"` (for parametric), `"np"` (nonparametric), `"r"` (robust), or
#'   `"bf"`resp.
#' @param robust.estimator If `type = "robust"`, a robust estimator to be used
#'   (`"onestep"` (Default), `"mom"`, or `"median"`). For more, see
#'   `?WRS2::onesampb`.
#' @param ... Additional arguments (currently ignored).
#' @inheritParams t1way_ci
#' @inheritParams expr_t_parametric
#' @inheritParams bf_corr_test
#' @inheritParams expr_anova_parametric
#'
#' @return Expression containing results from a one-sample test. The exact test
#'   and the effect size details contained will be dependent on the `type`
#'   argument.
#'
#' @references For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @importFrom dplyr select bind_rows summarize mutate mutate_at mutate_if
#' @importFrom dplyr group_by n arrange
#' @importFrom WRS2 onesampb
#' @importFrom crayon green blue yellow red
#' @importFrom rcompanion wilcoxonOneSampleR
#'
#' @examples
#' \donttest{
#' # for reproducibility
#' set.seed(123)
#'
#' # ----------------------- parametric ---------------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "parametric"
#' )
#'
#' # ----------------------- non-parametric -----------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "nonparametric"
#' )
#'
#' # ----------------------- robust --------------------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "robust"
#' )
#'
#' # ----------------------- Bayes Factor -----------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "bayes",
#'   bf.prior = 0.8
#' )
#' }
#' @export

expr_t_onesample <- function(data,
                             x,
                             type = "parametric",
                             test.value = 0,
                             bf.prior = 0.707,
                             robust.estimator = "onestep",
                             effsize.type = "g",
                             effsize.noncentral = TRUE,
                             conf.level = 0.95,
                             conf.type = "norm",
                             nboot = 100,
                             k = 2,
                             stat.title = NULL,
                             messages = TRUE,
                             ...) {

  # check the dots
  x <- rlang::ensym(x)

  # ====================== dataframe ========================================

  # preparing a dataframe out of provided inputs
  data %<>%
    dplyr::select(.data = ., {{ x }}) %>%
    tidyr::drop_na(data = .) %>%
    tibble::as_tibble(x = .)

  # sample size
  sample_size <- nrow(data)

  # standardize the type of statistics
  stats.type <- stats_type_switch(type)

  # ========================= parametric ====================================

  if (stats.type == "parametric") {
    # deciding which effect size to use (Hedge's g or Cohen's d)
    if (effsize.type %in% c("unbiased", "g")) {
      hedges.correction <- TRUE
      effsize.text <- quote(widehat(italic("g")))
    } else {
      hedges.correction <- FALSE
      effsize.text <- quote(widehat(italic("d")))
    }

    # creating model object
    mod_object <-
      stats::t.test(
        x = data %>% dplyr::pull({{ x }}),
        mu = test.value,
        conf.level = conf.level,
        alternative = "two.sided",
        na.action = na.omit
      )

    # tidy dataframe
    stats_df <- broomExtra::tidy(mod_object)

    # creating effect size info
    effsize_df <-
      effsize_t_parametric(
        formula = rlang::new_formula(NULL, {{ x }}),
        data = data,
        tobject = mod_object,
        mu = test.value,
        hedges.correction = hedges.correction,
        conf.level = conf.level
      )

    # preparing subtitle parameters
    statistic.text <- quote(italic("t"))
    no.parameters <- 1L
  }

  # ========================== non-parametric ==============================

  if (stats.type == "nonparametric") {
    # setting up the Mann-Whitney U-test and getting its summary
    stats_df <-
      broomExtra::tidy(
        x = stats::wilcox.test(
          x = data %>% dplyr::pull({{ x }}),
          alternative = "two.sided",
          na.action = na.omit,
          mu = test.value,
          exact = FALSE,
          correct = TRUE
        )
      ) %>%
      dplyr::mutate(.data = ., statistic = log(statistic))

    # effect size dataframe
    effsize_df <-
      rcompanion::wilcoxonOneSampleR(
        x = data %>% dplyr::pull({{ x }}),
        mu = test.value,
        ci = TRUE,
        conf = conf.level,
        type = conf.type,
        R = nboot,
        histogram = FALSE,
        digits = k,
        reportIncomplete = TRUE
      ) %>%
      rcompanion_cleaner(.)

    # preparing subtitle parameters
    statistic.text <- quote("log"["e"](italic("V")))
    no.parameters <- 0L
    effsize.text <- quote(widehat(italic("r")))

    # message about effect size measure
    if (isTRUE(messages)) effsize_ci_message(nboot, conf.level)
  }

  # preparing subtitle
  if (stats.type %in% c("parametric", "nonparametric")) {
    subtitle <-
      expr_template(
        stat.title = stat.title,
        no.parameters = no.parameters,
        stats.df = stats_df,
        effsize.df = effsize_df,
        statistic.text = statistic.text,
        effsize.text = effsize.text,
        n = sample_size,
        n.text = quote(italic("n")["obs"]),
        conf.level = conf.level,
        k = k
      )
  }

  # ======================= robust =========================================

  if (stats.type == "robust") {
    # running one-sample percentile bootstrap
    stats_df <-
      WRS2::onesampb(
        x = data %>% dplyr::pull({{ x }}),
        est = robust.estimator,
        nboot = nboot,
        nv = test.value,
        alpha = 1 - conf.level
      )

    # displaying message about bootstrap
    if (isTRUE(messages)) effsize_ci_message(nboot, conf.level)

    # preparing the subtitle
    subtitle <-
      substitute(
        expr = paste(
          italic("M")["robust"],
          " = ",
          estimate,
          ", CI"[conf.level],
          " [",
          LL,
          ", ",
          UL,
          "], ",
          italic("p"),
          " = ",
          p.value,
          ", ",
          italic("n")["obs"],
          " = ",
          n
        ),
        env = list(
          estimate = specify_decimal_p(x = stats_df$estimate[[1]], k = k),
          conf.level = paste(conf.level * 100, "%", sep = ""),
          LL = specify_decimal_p(x = stats_df$ci[[1]], k = k),
          UL = specify_decimal_p(x = stats_df$ci[[2]], k = k),
          p.value = specify_decimal_p(x = stats_df$p.value[[1]], k = k, p.value = TRUE),
          n = sample_size
        )
      )
  }

  # ======================== bayes ============================================

  # running Bayesian one-sample t-test
  if (stats.type == "bayes") {
    subtitle <-
      bf_one_sample_ttest(
        data = data,
        x = {{ x }},
        test.value = test.value,
        bf.prior = bf.prior,
        caption = NULL,
        output = "h1",
        k = k
      )
  }

  # return the subtitle
  return(subtitle)
}
