% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learnGraphTopology.R
\name{learn_k_component_graph}
\alias{learn_k_component_graph}
\title{Learn the Laplacian matrix of a k-component graph

Learns a k-component graph on the basis of an observed data matrix.
Check out https://mirca.github.io/spectralGraphTopology for code examples.}
\usage{
learn_k_component_graph(S, is_data_matrix = FALSE, k = 1,
  w0 = "naive", lb = 0, ub = 10000, alpha = 0, beta = 10000,
  beta_max = 1e+06, fix_beta = TRUE, rho = 0.01, m = 7,
  maxiter = 10000, abstol = 1e-06, reltol = 1e-04, eigtol = 1e-09,
  record_objective = FALSE, record_weights = FALSE, verbose = TRUE)
}
\arguments{
\item{S}{either a pxp sample covariance/correlation matrix, or a pxn data
matrix, where p is the number of nodes and n is the number of
features (or data points per node)}

\item{is_data_matrix}{whether the matrix S should be treated as data matrix
or sample covariance matrix}

\item{k}{the number of components of the graph}

\item{w0}{initial estimate for the weight vector the graph or a string
selecting an appropriate method. Available methods are: "qp": finds w0 that minimizes
||ginv(S) - L(w0)||_F, w0 >= 0; "naive": takes w0 as the negative of the
off-diagonal elements of the pseudo inverse, setting to 0 any elements s.t.
w0 < 0}

\item{lb}{lower bound for the eigenvalues of the Laplacian matrix}

\item{ub}{upper bound for the eigenvalues of the Laplacian matrix}

\item{alpha}{L1 regularization hyperparameter}

\item{beta}{regularization hyperparameter for the term ||L(w) - U Lambda U'||^2_F}

\item{beta_max}{maximum allowed value for beta}

\item{fix_beta}{whether or not to fix the value of beta. In case this parameter
is set to false, then beta will increase (decrease) depending whether the number of
zero eigenvalues is lesser (greater) than k}

\item{rho}{how much to increase (decrease) beta in case fix_beta = FALSE}

\item{m}{in case is_data_matrix = TRUE, then we build an affinity matrix based
on Nie et. al. 2017, where m is the maximum number of possible connections
for a given node}

\item{maxiter}{the maximum number of iterations}

\item{abstol}{absolute tolerance on the weight vector w}

\item{reltol}{relative tolerance on the weight vector w}

\item{eigtol}{value below which eigenvalues are considered to be zero}

\item{record_objective}{whether to record the objective function values at
each iteration}

\item{record_weights}{whether to record the edge values at each iteration}

\item{verbose}{whether to output a progress bar showing the evolution of the
iterations}
}
\value{
A list containing possibly the following elements:
\item{\code{Laplacian}}{the estimated Laplacian Matrix}
\item{\code{Adjacency}}{the estimated Adjacency Matrix}
\item{\code{w}}{the estimated weight vector}
\item{\code{lambda}}{optimization variable accounting for the eigenvalues of the Laplacian matrix}
\item{\code{U}}{eigenvectors of the estimated Laplacian matrix}
\item{\code{elapsed_time}}{elapsed time recorded at every iteration}
\item{\code{beta_seq}}{sequence of values taken by beta in case fix_beta = FALSE}
\item{\code{convergence}}{boolean flag to indicate whether or not the optimization converged}
\item{\code{obj_fun}}{values of the objective function at every iteration in case record_objective = TRUE}
\item{\code{loglike}}{values of the negative loglikelihood at every iteration in case record_objective = TRUE}
\item{\code{w_seq}}{sequence of weight vectors at every iteration in case record_weights = TRUE}
}
\description{
Learn the Laplacian matrix of a k-component graph

Learns a k-component graph on the basis of an observed data matrix.
Check out https://mirca.github.io/spectralGraphTopology for code examples.
}
\examples{
# design true Laplacian
Laplacian <- rbind(c(1, -1, 0, 0),
                   c(-1, 1, 0, 0),
                   c(0, 0, 1, -1),
                   c(0, 0, -1, 1))
n <- ncol(Laplacian)
# sample data from multivariate Gaussian
Y <- MASS::mvrnorm(n * 500, rep(0, n), MASS::ginv(Laplacian))
# estimate graph on the basis of sampled data
graph <- learn_k_component_graph(cov(Y), k = 2, beta = 10)
graph$Laplacian
}
\references{
S. Kumar, J. Ying, J. V. de Miranda Cardoso, D. P. Palomar. A unified
            framework for structured graph learning via spectral constraints (2019).
            https://arxiv.org/pdf/1904.09792.pdf
}
\author{
Ze Vinicius and Daniel Palomar
}
