% 2021-10-27 A. Papritz
% cd ~/R.user.home/soilhypfit/soilhypfit_01-1/pkg/man
% R CMD Rdconv -t html -o bla.html soilhypfit_fit_wrc_hcc.Rd ; open bla.html; R CMD Rd2pdf --force soilhypfit_fit_wrc_hcc.Rd;
% hunspell -d en_GB-ise fit_wrc_hcc.Rd

\encoding{utf8}

\name{fit_wrc_hcc}
\alias{fit_wrc_hcc}
\alias{default_fit_param}

\concept{soil water}
\concept{soil water characteristic}
\concept{soil water retention curve}
\concept{hydraulic conductivity function}
\concept{saturated hydraulic conductivity}

% \newcommand{\mbvec}{\eqn{\mbox{\boldmath$#1$\unboldmath}^\mathrm{#2} #4}{#1#3#4}}

\title{
  Parametric Modelling of Soil Hydraulic Properties
}

% description

\description{

  The function \code{fit_wrc_hcc} estimates parameters of models for the
  soil water retention curve and/or soil hydraulic conductivity function
  from respective measurements by nonlinear regression methods, optionally
  subject to physical constraints on the estimated parameters.
  \code{fit_wrc_hcc} uses optimisation algorithms of the NLopt library
  (Johnson, see \code{\link[nloptr]{nloptr-package}}) and the Shuffled
  Complex Evolution (SCE) algorithm (Duan et al., 1994) implemented in the
  function \code{\link[SoilHyP]{SCEoptim}}.

}

% usage

\usage{

fit_wrc_hcc(
    wrc_formula, hcc_formula, data,
    subset = NULL, wrc_subset = subset, hcc_subset = subset,
    weights = NULL, wrc_weights = weights, hcc_weights = weights,
    na.action, param = NULL, lower_param = NULL, upper_param = NULL,
    fit_param = default_fit_param(),
    e0 = 2.5e-3, ratio_lc_lt_bound = c(lower = 0.5, upper = 2),
    control = control_fit_wrc_hcc(), verbose = 0)

default_fit_param(
    alpha = TRUE, n = TRUE, tau = FALSE,
    thetar = TRUE, thetas = TRUE, k0 = TRUE)

}

% arguments

\arguments{
  \item{wrc_formula}{an optional two-sided formula such as \code{wc ~ head}
  or \code{wc ~ head | id}, specifying the variables for the water content
  (\code{wc}), the capillary pressure \code{head} and, optionally, for
  sample \code{id}s when model parameters are estimated for several soil
  samples at the same time, see \code{\link[stats]{formula}} and
  \emph{Details}.}

  \item{hcc_formula}{an optional two-sided formula such as \code{hc ~ head}
  or \code{hc ~ head | id}, specifying the variables for the hydraulic
  conductivity (\code{hc}), the capillary pressure \code{head} and,
  optionally, for sample \code{id}s when model parameters are estimated for
  several soil samples at the same time.  See \code{\link[stats]{formula}}
  and \emph{Details}.}

  \item{data}{a mandatory data frame containing the variables specified in
  the formula, the subset and weights arguments.}

  \item{subset}{an optional expression generating a vector to choose a
  subset of water content and hydraulic conductivity data.  The expression
  is evaluated in the environment generated by
  \code{model.frame(wrc_formula, data)} and\cr \code{model.frame(hcc_formula,
  data)}, respectively.}

  \item{wrc_subset}{an optional expression generating a vector to choose a
  subset of water content data.  The expression is evaluated in the
  environment generated by\cr  \code{model.frame(wrc_formula, data)}.  Defaults
  to \code{subset}.}

  \item{hcc_subset}{an optional expression generating a vector to choose a
  subset of hydraulic conductivity data.
  The expression is evaluated in the environment generated by
  \code{model.frame(hcc_formula, data)}.  Defaults to \code{subset}.}

  \item{weights}{an optional expression generating a numeric vector of case
  weights \eqn{w^\prime_{\theta,i}} and \eqn{w^\prime_{K,i}} (default: 1,
  see \code{\link{soilhypfitIntro}}) for water content and hydraulic
  conductivity data.  The expression is evaluated in the environment
  generated by \code{model.frame(wrc_formula, data)} and
  \code{model.frame(hcc_formula, data)}, respectively.}

  \item{wrc_weights}{an optional expression generating a numeric vector of
  case weights \eqn{w^\prime_{\theta,i}} (see
  \code{\link{soilhypfitIntro}}) for water content data.  The expression is
  evaluated in the environment generated by \code{model.frame(wrc_formula,
  data)}.  Defaults to \code{weights}.}

  \item{hcc_weights}{an optional expression generating a numeric vector of
  case weights \eqn{w^\prime_{K,i}} (see \code{\link{soilhypfitIntro}}) for
  hydraulic conductivity data.  The expression is evaluated in the
  environment generated by \code{model.frame(hcc_formula, data)}.  Defaults
  to \code{weights}.}

  \item{na.action}{a function which indicates what should happen when the
  data contain \code{NA}s.  The default is set by the \code{na.action}
  argument of \code{\link{options}}, and is \code{\link{na.fail}} if that
  is unset.  The \dQuote{factory-fresh} default is \code{\link{na.omit}}.
  Another possible value is \code{NULL}, no action.  Value
  \code{\link{na.exclude}} can be useful.}

  \item{param}{an optional named numeric vector (or a numeric matrix or a
  dataframe with specified column names, see \emph{Details}) with initial
  values for the model parameters.  Currently, \code{param} may have
  elements (or columns) named \code{"alpha"}, \code{"n"}, \code{"tau"},
  \code{"thetar"}, \code{"thetas"}, \code{"k0"}, see \code{\link{wc_model}}
  and \code{\link{hc_model}} and \emph{Details}.  For consistency with
  other quantities, the unit of \eqn{\alpha} should be \bold{1/meter}
  [\eqn{\mathrm{m}^{-1}}{m^-1}] and the unit of \eqn{K_0} \bold{meter/day}
  [\eqn{\mathrm{m}\,\mathrm{d}^{-1}}{m d^-1}].}

  \item{lower_param}{an optional named numeric vector (or a numeric matrix
  or a dataframe with specified column names, see \emph{Details}) with
  lower bounds for the parameters of the models.  Currently,
  \code{lower_param} may have elements (or columns) named \code{"alpha"},
  \code{"n"}, \code{"tau"}, \code{"thetar"}, \code{"thetas"}, \code{"k0"},
  see \code{\link{wc_model}}, \code{\link{hc_model}} and
  \code{\link{param_boundf}}.  For consistency with other quantities, the
  unit of \eqn{\alpha} should be \bold{1/meter}
  [\eqn{\mathrm{m}^{-1}}{m^-1}] and the unit of \eqn{K_0} \bold{meter/day}
  [\eqn{\mathrm{m}\,\mathrm{d}^{-1}}{m d^-1}].  If lower bounds are
  specified for \eqn{\theta_r} but not for \eqn{\theta_s} then the lower
  bounds specified for \eqn{\theta_r} will also be used for
  \eqn{\theta_s}.}

  \item{upper_param}{an optional named numeric vector (or a numeric matrix
  or a dataframe with specified column names, see \emph{Details}) with
  upper bounds for the
  parameters of the models.  Currently, \code{upper_param} may have
  elements (or columns) named \code{"alpha"}, \code{"n"}, \code{"tau"},
  \code{"thetar"}, \code{"thetas"}, \code{"k0"}, see
  \code{\link{wc_model}}, \code{\link{hc_model}} and
  \code{\link{param_boundf}}.  For consistency with other quantities, the
  unit of \eqn{\alpha} should be \bold{1/meter}
  [\eqn{\mathrm{m}^{-1}}{m^-1}] and the unit of \eqn{K_0} \bold{meter/day}
  [\eqn{\mathrm{m}\,\mathrm{d}^{-1}}{m d^-1}].  If upper bounds are
  specified for \eqn{\theta_s} but not for \eqn{\theta_r} then the upper
  bounds specified for \eqn{\theta_s} will also be used for \eqn{\theta_r}.}

  \item{fit_param}{a named logical vector (or a logical matrix or a
  dataframe with specified column names, see \emph{Details}) containing
  flags that control whether model parameters are estimated (\code{TRUE})
  or kept fixed at the initial values (\code{FALSE}) as specified in
  \code{param}.  This vector can be generated easily by the function
  \code{default_fit_param}.  Currently, \code{fit_param} may have elements
  (or columns) named \code{"alpha"}, \code{"n"}, \code{"tau"},\cr
  \code{"thetar"}, \code{"thetas"}, \code{"k0"}, see \emph{Details},
  \code{\link{wc_model}} and \code{\link{hc_model}}.}

  \item{e0}{a numeric scalar (or vector, see \emph{Details}) with the stage-I rate of
  evaporation \eqn{E_0} from a soil (default \eqn{2.5\cdot 10^{-3} \
  \mathrm{m}\,\mathrm{d}^{-1}}{0.0025 m d^-1}) required to evaluate the
  \emph{characteristic evaporative length}, see
  \code{\link{evaporative-length}} and \code{\link{soilhypfitIntro}}.  For
  consistency with other quantities, the unit of \eqn{E_0} should
  \bold{meter/day} [\eqn{\mathrm{m}\,\mathrm{d}^{-1}}{m d^-1}]. Note
  that \code{e0} is ignored when an unconstrained nonlinear
  optimisation algorithm is chosen (argument \code{settings} of
  \code{\link{control_fit_wrc_hcc}} equal to \code{"uglobal"},
  \code{"sce"} or \code{"ulocal"}).}

  \item{ratio_lc_lt_bound}{a named numeric vector of length 2 (or a matrix
  with two columns, see \emph{Details}) defining the default \code{lower}
  and \code{upper} bounds of the ratio
  \eqn{L_\mathrm{c}/L_\mathrm{t}}{L_c/L_t} (\cite{Lehmann et al., 2008,
  2020}) for constrained estimation of the \emph{nonlinear} parameters
  \mbvec{\nu}{T}{^T}{=(\alpha, n, \tau)} (default values 0.5 (\code{lower})
  and 2 (\code{upper})), see \code{\link{evaporative-length}} and
  \code{\link{soilhypfitIntro}}.  Note that \code{ratio_lc_lt_bound} is
  ignored when an unconstrained nonlinear optimisation algorithm is chosen
  (argument \code{settings} of \code{\link{control_fit_wrc_hcc}} equal to
  \code{"uglobal"}, \code{"sce"} or \code{"ulocal"}).}

  \item{control}{a list with options to control \code{fit_wrc_hcc} or a
  function such as\cr \code{\link{control_fit_wrc_hcc}} that generates such
  a list.  The main argument \code{settings} of \code{control_fit_wrc_hcc}
  selects a nonlinear optimisation approach, see\cr
  \code{\link{soilhypfitIntro}} and \code{\link{control_fit_wrc_hcc}}
  for full details.}

  \item{verbose}{positive integer controlling logging of diagnostic
  messages to the console and plotting of data and model curves
  during fitting.

  \describe{

    \item{\code{verbose < 0}}{suppresses all output,}

    \item{\code{verbose >= 0}}{suppresses all output except
    warning messages,}

    \item{\code{verbose >= 1}}{displays at the end the measurements along
    with the fitted model curves,}

    \item{\code{verbose >= 2}}{prints for each iteration the parameters
    values, the value of the objective function and the ratio of
    \eqn{L_\mathrm{c}/L_\mathrm{t}}{L_c/L_t} (see
    \code{\link{evaporative-length}}),}

    \item{\code{verbose >= 3}}{plots for each iteration the measurements
    along with the fitted model curves.}

  }

  Logging of further diagnostics during fitting is controlled in addition
  by the arguments \code{print_level}, \code{check_derivatives},
  \code{check_derivatives_print} when \code{\link[nloptr]{nloptr}} is used
  (see \code{\link[nloptr]{nloptr.print.options}} and
  \code{\link{control_nloptr}}) and by the argument \code{trace} of
  \code{\link[SoilHyP]{SCEoptim}} (see \code{\link{control_sce}}).}

  \item{alpha}{a logical scalar controlling whether the inverse air entry
  pressure \eqn{\alpha} should be estimated (\code{TRUE}, default) or kept
  fixed at the initial value (\code{FALSE}) specified by \code{param}, see
  \code{\link{wc_model}} and \code{\link{hc_model}}.}

  \item{n}{a logical scalar controlling whether the shape parameter \eqn{n}
  should be estimated (\code{TRUE}, default) or kept fixed at the initial
  value (\code{FALSE}) specified by \code{param}, see
  \code{\link{wc_model}} and \code{\link{hc_model}}.}

  \item{tau}{a logical scalar controlling whether the tortuosity parameter
  \eqn{\tau} should be estimated (\code{TRUE}) or kept fixed at the initial
  value (\code{FALSE}, default) specified by \code{param}, see
  \code{\link{hc_model}}.}

  \item{thetar}{a logical scalar controlling whether the residual water
  content \eqn{\theta_r} should be estimated (\code{TRUE}, default) or kept
  fixed at the initial value (\code{FALSE}) specified by \code{param}, see
  \code{\link{wc_model}}.}

  \item{thetas}{a logical scalar controlling whether the saturated water
  content \eqn{\theta_s} should be estimated (\code{TRUE}, default) or kept
  fixed at the initial value (\code{FALSE}) specified by \code{param}, see
  \code{\link{wc_model}}.}

  \item{k0}{a logical scalar controlling whether the saturated hydraulic
  conductivity \eqn{K_0} should be estimated (\code{TRUE}, default) or kept
  fixed at the initial value (\code{FALSE}) specified by \code{param}, see
  \code{\link{hc_model}}.}

}

% details

\details{

  \subsection{Estimating model parameters of water retention curves and/or
  hydraulic conductivity functions}{

    The function \code{fit_wrc_hcc} estimates model parameters of water
    retention curves and/or hydraulic conductivity functions by
    \emph{maximum posterior density} (mpd, default) \emph{maximum likelihood}
    (ml) or \emph{nonlinear least squares} (wls), see argument
    \code{method} of \code{\link{control_fit_wrc_hcc}}.  Measurements must
    be available for at least one of the two material functions.  If one
    type of data is missing then the respective \code{formula},
    \code{subset} and \code{weights} arguments should be omitted.

    If both types of data are available then measurements are weighed when
    computing the residual sum of squares for wls, but unit weights are
    used by default for mpd and ml estimation, see
    \code{\link{soilhypfitIntro}}.  The wls weights are the product of the
    \emph{case weights} \eqn{w^\prime_{\theta,i}} and \eqn{w^\prime_{K,i}}
    given by \code{weights}, \code{wrc_weights} and \code{hcc_weights},
    respectively, and the \emph{variable weights} as specified by the
    argument \code{variable_weight} of \code{\link{control_fit_wrc_hcc}}.
    Note that the \code{variable_weights} are not used ``as is'', but they
    are multiplied by the inverse variances of the water content or the
    (log-transformed) conductivity data per sample to obtain the variable
    weights \eqn{w_\theta}, \eqn{w_K} mentioned in
    \code{\link{soilhypfitIntro}}.}

  \subsection{Estimating model parameters for a single or multiple soil
  samples}{

    When parameters are estimated for a single soil sample only, then model
    formulae are of the form \code{wc ~ head} and/or \code{hc ~head}, and
    there is no need to specify a sample \code{id}.  In this case
    \code{param}, \code{lower_param}, \code{upper_param}, \code{fit_param}
    and \code{ratio_lc_lt_bound} are vectors and \code{e0} is a scalar.

    \code{fit_wrc_hcc} allows to fit models to datasets containing data for
    multiple soil samples.  The model formula must then be of the form
    \code{wc ~ head|id} and/or \code{hc ~ head|id} where the factor
    \code{id} identifies the soil samples.  If \code{param},
    \code{lower_param}, \code{upper_param}, \code{fit_param} and
    \code{ratio_lc_lt_bound} are vectors and \code{e0} is a scalar then
    this information is used for processing all the soil samples.  If
    specific information per sample should be used then \code{param},
    \code{lower_param}, \code{upper_param}, \code{fit_param} and
    \code{ratio_lc_lt_bound} must be matrices (or data frames) with as many
    rows as there are soil samples, and \code{e0} is a vector with as many
    elements as there are soil samples.

    By default, \code{fit_wrc_hcc} processes data of multiple soil samples
    in parallel, see \code{\link{control_pcmp}} for options to control
    parallel computing.  Note that diagnostic output (except warning
    messages) is suppressed in parallel computations.  If computations fail
    for a particular soil sample, the id of the sample with the failed fit
    can be extracted by the utility function
    \code{\link{select_failed_fits}} and the respective error messages by
    \code{\link{extract_error_messages}}.

  }

  \subsection{Controlling \code{fit_wrc_hcc}}{

    The argument \code{control} is used to pass a list of options to
    \code{fit_wrc_hcc} that steer the function, see
    \code{\link{soilhypfit-package}} and \code{\link{control_fit_wrc_hcc}}
    for full details.


  }

}

% value

\value{

  \code{fit_wrc_hcc} returns an object of \code{\link[base]{class}}
  \code{fit_wrc_hcc}, which is a list with the following components:

  \item{fit}{a list with as many components as there are soils samples.
  Each component is in turn a list that contains the estimated parameters
  and accompanying information:

    \itemize{

      \item{\code{converged}}: an integer code issued by
      \code{\link[SoilHyP]{SCEoptim}} or \code{\link{nloptr}} on (non-)
      convergence, see \code{\link{convergence_message}} and
      \href{https://nlopt.readthedocs.io/en/latest/NLopt_Reference/#return-values}{NLopt
      return values}.

      \item{\code{message}}: a character string issued by
      \code{\link[SoilHyP]{SCEoptim}} or \code{\link{nloptr}} on
      (non-)convergence.

      \item{\code{evaluation}}: the number of evaluations of the objective
      function and of the gradient.

      \item{\code{objective}}: the value of the objective function
      evaluated at the solution.  The contributions of the water retention
      curve and the hydraulic conductivity function to \code{objective} are
      reported as attributes \code{obj_wc} and \code{obj_hc}.  The
      attributes \code{ssq_wc} and \code{ssq_hc} are the respective
      residual sums of squares \eqn{Q_\theta} and \eqn{Q_K}, see
      \code{\link{soilhypfitIntro}}.
      
      \item{\code{gradient}}: the gradient of the objective function at the
      solution with respect to the (possibly transformed) nonlinear parameters.

      \item{\code{lp}}: the estimated values for the linear parameters
      \mbvec{\mu}{T}{^T}{=(\theta_r, \theta_s, K_0)}, see
      \code{\link{wc_model}} and \code{\link{hc_model}}.

      \item{\code{nlp}}: the estimated values for the nonlinear parameters
      \mbvec{\nu}{T}{^T}{=(\alpha, n, \tau)}, see \code{\link{wc_model}} and
      \code{\link{hc_model}}.

      \item{\code{inequality_constraint}}: a list with the values of the
      inequality constraints evaluated at the solutions.  Currently, values
      are reported for the expressions.

      \deqn{-(\frac{L_\mathrm{c}}{L_\mathrm{t}} - l)}{-(L_c/L_t - l)}
      \deqn{\frac{L_\mathrm{c}}{L_\mathrm{t}} - u}{L_c/L_t - u }

      in the only component \code{lc}, where \eqn{l} and \eqn{u} are the
      lower and upper bounds of the ratio, see argument
      \code{ratio_lc_lt_bound} and \code{\link{evaporative-length}}.  The
      values of \eqn{L_\mathrm{c}}{L_c}, \eqn{L_\mathrm{t}}{L_t}, the
      imposed bounds on their ratio and \code{e0} are
      returned as attributes of \code{lc}.

      \item{\code{hessian}}: optionally the Hessian matrix of the objective
      function with respect to the possibly transformed nonlinear
      parameters at the solution.

      \item{\code{variable_weight}}: a named vector with the variable
      weights \eqn{w_{\theta}} and \eqn{w_{K}}, see \emph{Details}.

      \item{\code{weight}}: a list with one or two components named
      \code{weights_wc} and or \code{weights_hc} with the case weights
      \eqn{w_{\theta,i}} and \eqn{w_{K,i}} used in the objective function,
      see \emph{Details}.

      \item{\code{fitted}}: a list with one or two components named
      \code{wrc} and/or \code{hcc} with the fitted values for the water
      retention and the (possibly log-transformed) hydraulic conductivity
      data.

      \item{\code{residuals}}: a list with one or two components named
      \code{wrc} and/or \code{hcc} with residuals for the water retention
      and the (possibly log-transformed) hydraulic conductivity data.

      \item{\code{model}}: a list with one or two components named \code{wrc}
      and/or \code{hcc} with the \code{\link[stats]{model.frame}}s for the
      water retention and hydraulic conductivity data.

      \item{\code{initial_objects}}: a list with the values for
      \code{param}, \code{fit_param}, \code{variable_weight},
      \code{param_bound}, \code{e0} and \code{ratio_lc_lt_bound} as taken
      from the (processed) arguments of the call of \code{fit_wrc_hcc}.

    }

  }

  \item{control}{a list with the options used to control
  \code{fit_wrc_hcc}, see \code{\link{control_fit_wrc_hcc}}.}

  \item{call}{the matched call.}

  \item{na.action}{information returned by \code{\link[stats]{model.frame}}
  on the special handling of \code{NA}s.}


}

% references

\references{
  Duan, Q., Sorooshian, S., and Gupta, V. K. (1994) Optimal use of the
  SCE-UA global optimisation method for calibrating watershed models,
  \emph{Journal of Hydrology} \bold{158}, 265--284,
  \doi{10.1016/0022-1694(94)90057-4}.

  Johnson, S.G. The NLopt nonlinear-optimisation package.
  \url{https://github.com/stevengj/nlopt}.

  Lehmann, P., Assouline, S., Or, D. (2008) Characteristic lengths
  affecting evaporative drying of porous media.  \emph{Physical Review E},
  \bold{77}, 056309, \doi{10.1103/PhysRevE.77.056309}.

  Lehmann, P., Bickel, S., Wei, Z., Or, D. (2020) Physical Constraints for
  Improved Soil Hydraulic Parameter Estimation by Pedotransfer Functions.
  \emph{Water Resources Research} \bold{56}, e2019WR025963,
  \doi{10.1029/2019WR025963}.

}

% author

\author{
   Andreas Papritz \email{andreas.papritz@env.ethz.ch}.
}

% see also

\seealso{
  \code{\link{soilhypfitIntro}} for a description of the models and a brief
  summary of the parameter estimation approach;

%   \code{\link{fit_wrc_hcc}} for (constrained) estimation of parameters of
%   models for soil water retention and hydraulic conductivity data;
%
  \code{\link{control_fit_wrc_hcc}} for options to control
  \code{fit_wrc_hcc};

  \code{\link{soilhypfitmethods}} for common S3 methods for class
  \code{fit_wrc_hcc};

  \code{\link{vcov}} for computing (co-)variances of the estimated
  nonlinear parameters;

  \code{\link{wc_model}} and \code{\link{hc_model}} for currently
  implemented models for soil water retention curves and hydraulic
  conductivity functions;

  \code{\link{evaporative-length}} for physically constraining parameter
  estimates of soil hydraulic material functions.
}

% examples

\examples{
\donttest{
# use of \donttest{} because execution time exceeds 5 seconds

data(sim_wrc_hcc)

# define number of cores for parallel computations
if(interactive()) ncpu <- parallel::detectCores() - 1L else ncpu <- 1L


# estimate parameters for single sample ...

#  ... from wrc and hcc data
plot(rfit_wrc_hcc <- fit_wrc_hcc(
  wrc_formula = wc ~ head, hcc_formula = hc ~ head,
  data = sim_wrc_hcc, subset = id == 1))
coef(rfit_wrc_hcc, gof = TRUE)

#  ... from wrc data
plot(rfit_wrc <- fit_wrc_hcc(
  wrc_formula = wc ~ head,
  data = sim_wrc_hcc, subset = id == 1))
coef(rfit_wrc, gof = TRUE)

#  ... from  hcc data
plot(rfit_hcc <- fit_wrc_hcc(
  hcc_formula = hc ~ head,
  data = sim_wrc_hcc, subset = id == 1))
coef(rfit_hcc, gof = TRUE)

# ... from wrc and hcc data
#     keeping some parameters fixed
plot(rfit_wrc_hcc_fixed <- fit_wrc_hcc(
  wrc_formula = wc ~ head, hcc_formula = hc ~ head,
  data = sim_wrc_hcc, subset = id == 1,
  param = c(alpha = 2.1, thetar = 0.1),
  fit_param = default_fit_param(alpha = FALSE, thetar = FALSE)),
  y = rfit_wrc_hcc)
coef(rfit_wrc_hcc, gof = TRUE)
coef(rfit_wrc_hcc_fixed, gof = TRUE)


# estimate parameters for 3 samples simultaneously by ...

# ... unconstrained, global optimisation algorithm NLOPT_GN_MLSL (default)
rfit_uglob <- fit_wrc_hcc(
  wrc_formula = wc ~ head | id, hcc_formula = hc ~ head | id,
  data = sim_wrc_hcc,
  control = control_fit_wrc_hcc(pcmp = control_pcmp(ncores = ncpu)))
summary(rfit_uglob)
op <- par(mfrow = c(3, 2))
plot(rfit_uglob)
on.exit(par(op))

# ... unconstrained, global optimisation algorithm SCEoptim
rfit_sce <- update(
  rfit_uglob,
  control = control_fit_wrc_hcc(
    settings = "sce", pcmp = control_pcmp(ncores = ncpu)))
coef(rfit_sce, gof = TRUE, lc = TRUE)
convergence_message(2, sce = TRUE)
op <- par(mfrow = c(3, 2))
plot(rfit_sce, y = rfit_uglob)
on.exit(par(op))

# ... unconstrained, local optimisation algorithm NLOPT_LN_NELDERMEAD,
#     logging iteration results to console
rfit_uloc <- update(
  rfit_uglob,
  param = as.matrix(coef(rfit_uglob, what = "nonlinear")),
  control = control_fit_wrc_hcc(
    settings = "ulocal", pcmp = control_pcmp(ncores = 1L)),
  verbose = 2)
coef(rfit_uloc, gof = TRUE, lc = TRUE)

# ... constrained, global optimisation algorithm NLOPT_GN_ISRES
rfit_cglob <- update(
  rfit_uglob,
  ratio_lc_lt_bound = c(lower = 0.8, upper = 1.2),
  control = control_fit_wrc_hcc(
    settings = "cglobal", nloptr = control_nloptr(ranseed = 1),
    pcmp = control_pcmp(ncores = ncpu)))
coef(rfit_cglob, gof = TRUE, lc = TRUE)

# ... constrained, local optimisation algorithm NLOPT_LD_MMA
#     starting from unconstrained, locally fitted initial values
rfit_cloc_1 <- update(
  rfit_uglob,
  param = coef(rfit_uloc, what = "nonlinear"),
  ratio_lc_lt_bound = c(lower = 0.8, upper = 1.2),
  control = control_fit_wrc_hcc(
    settings = "clocal", pcmp = control_pcmp(ncores = ncpu)))
coef(rfit_cloc_1, gof = TRUE, lc = TRUE)
op <- par(mfrow = c(3, 2))
plot(x = rfit_uloc, y = rfit_cloc_1)
on.exit(par(op))

# ... constrained, local optimisation algorithm NLOPT_LD_MMA
#     starting from constrained, globally fitted initial values,
#     using sample-specific evaporation rates and limits for ratio lc/lt
rfit_cloc_2 <- update(
  rfit_uglob,
  param = as.matrix(coef(rfit_cglob, what = "nonlinear")),
  e0 = c(0.002, 0.0025, 0.003),
  ratio_lc_lt_bound = rbind(
    c(lower = 0.7, upper = 2),
    c(lower = 0.8, upper = 1.4),
    c(lower = 0.8, upper = 2)
  ),
  control = control_fit_wrc_hcc(
    settings = "clocal", pcmp = control_pcmp(ncores = ncpu)))
coef(rfit_cloc_2, gof = TRUE, lc = TRUE, e0 = TRUE)

# ... global optimisation algorithm NLOPT_GN_MLSL
#     with sample-specific box-constraints
rfit_uglob_bc <- update(
  rfit_uglob,
  lower_param = rbind(
    c(alpha = 2.4, n = 1.11, thetar = 0.2, thetas = 0.6),
    c(alpha = 1.2, n = 1.12, thetar = 0.2, thetas = 0.6),
    c(alpha = 1.2, n = 1.13, thetar = 0.2, thetas = 0.6)
  ),
  upper_param = rbind(
    c(alpha = 20.1, n = 2.51, thetar = 0.6, thetas = 0.6),
    c(alpha = 20.2, n = 1.5,  thetar = 0.6, thetas = 0.6),
    c(alpha = 1.3, n = 2.53,  thetar = 0.6, thetas = 0.6)
  )
)
coef(rfit_uglob, gof = TRUE)
coef(rfit_uglob_bc, gof = TRUE)

}}
