% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_pars.R
\name{sk_pars_update}
\alias{sk_pars_update}
\title{Convert covariance parameter list to/from vectorized form}
\usage{
sk_pars_update(
  pars,
  p = NULL,
  iso = FALSE,
  eps_scaled = FALSE,
  na_omit = FALSE
)
}
\arguments{
\item{pars}{list of kernel parameters (in form understood by \code{sk_pars_make})}

\item{p}{numeric vector (optional) kernel parameters to update in \code{pars}}

\item{iso}{logical, indicating to treat y and x kernel parameters as equal (see details)}

\item{eps_scaled}{logical, indicates to drop partial sill from input/output}

\item{na_omit}{logical, toggles handling of \code{NA}s (see details)}
}
\value{
numeric vector of parameters, or, if \code{p} is supplied, the updated \code{pars} list
}
\description{
Converts parameter list \code{pars} to vector \code{p} and back again. A convenience function for
numerical optimization where objective functions accept numeric vectors only.
}
\details{
When \code{p} is not supplied, the function un-lists the numeric values of \code{pars}, returning
them as a vector. When \code{na_omit=TRUE}, only the non-NA values are returned; and when
\code{iso=TRUE} the x kernel parameters are also omitted from the output vector.

When \code{p} is supplied, the function copies its values to the corresponding entries in
\code{pars}, returning the result as a list. In this case, when \code{na_omit=TRUE}, only the
NA entries of \code{pars} are filled; and when \code{iso=TRUE}, parameters for the y kernel
are recycled to fill entries for the x kernel.

The order returned (and expected in \code{p}, if supplied) is:

'eps', 'psill', 'y.rho', ('y.kap'), 'x.rho', ('x.kap')

where 'y.kap' and 'x.kap' are omitted in kernels without shape parameters (see
\code{sk_corr}). Only the order matters here, as names are ignored in \code{p}.

With \code{na_omit=FALSE}, \code{p} should have length 3-6, the same as the vector returned by
\code{sk_pars_update(pars)}; NAs in \code{p} are copied over in this case, effectively inverting
the vectorization.

With \code{na_omit=TRUE}, values in \code{p} are copied only to the NA entries of \code{pars}; ie the
length of \code{p} should equal the number of NA entries in \code{pars} (less any redundant x
kernel parameters when \code{iso=TRUE}). Note that this does NOT invert the vectorization
\code{p=sk_pars_update(pars, na_omit=TRUE)}, which returns non-NA entries of \code{pars}.
}
\examples{
# initialize a parameter list and pass to sk_pars_update
k = c('gau', 'mat')
pars_empty = sk_pars_make(k)
sk_pars_update(pars_empty)

# pars can be a character vector passed directly to sk_pars_update
sk_pars_update(k)

# single kernel definitions are duplicated
sk_pars_update('mat')

# example parameter vector to illustrate ordering
p_update = seq(5)

# (inverse) pass a modified vector back to the function to update pars
pars_filled = sk_pars_update(pars_empty, p_update)
print(pars_filled)

# p_update is unchanged by round trip
sk_pars_update(pars_filled)

# NAs in pars can be filtered out with na_omit=TRUE
pars_filled$eps = NA
pars_filled$x$kp[1] = NA
sk_pars_update(pars_filled)
sk_pars_update(pars_filled, na_omit=TRUE)

# when updating parameters, NAs in pars identify parameters to receive the new values
p_update = abs(stats::rnorm(2))
sk_pars_update(pars_filled, p_update, na_omit=TRUE)

# when na_omit=FALSE, all parameters in pars are updated
p_update = stats::rnorm(5)
sk_pars_update(pars_filled, p_update)

# iso=TRUE is for when x kernel parameters are assigned values from the y kernel
pars_empty = sk_pars_make('mat')
p_update = seq(length(sk_pars_update(pars_empty, iso=TRUE)))
pars_filled = sk_pars_update(pars_empty, p_update, iso=TRUE)
print(pars_filled)

# notice NA shape parameter in x ignored while NA in eps is copied
pars_filled$eps = NA
pars_filled$x$kp[2] = NA
sk_pars_update(pars_filled, iso=TRUE)

# update calls with iso=TRUE should omit the x kernel parameters (p is shorter)
sk_pars_update(pars_filled, seq(4), iso=TRUE)

# if eps_scaled=TRUE, psill is ignored (useful for when eps is scaled by psill)
sk_pars_update(pars_filled)
sk_pars_update(pars_filled, eps_scaled=TRUE)

# notice value of psill not updated
sk_pars_update(pars_filled, -seq(5), eps_scaled=TRUE)

# compare/combine with other modes
pars_filled$y$kp[2] = NA
sk_pars_update(pars_filled)
sk_pars_update(pars_filled, -seq(2), iso=TRUE, na_omit=TRUE)
sk_pars_update(pars_filled, -seq(4), iso=TRUE)
sk_pars_update(pars_filled, -seq(3), iso=TRUE, eps_scaled=TRUE)
sk_pars_update(pars_filled, -seq(3), na_omit=TRUE)

}
\seealso{
Other parameter managers: 
\code{\link{sk_bds}()},
\code{\link{sk_fit}()},
\code{\link{sk_kp}()},
\code{\link{sk_pars_make}()},
\code{\link{sk_pars}()},
\code{\link{sk_to_string}()}
}
\concept{parameter managers}
\keyword{internal}
