#' Difference of estimated K functions
#' 
#' \code{kdest} determines the difference in estimated K
#' functions for a set of cases and controls.  Non-rejection
#' envelopes can also be produced.
#' 
#' This function relies internally on the 
#' \code{\link[spatstat.core]{Kest}} and 
#' \code{\link[spatstat.core]{eval.fv}} functions from the 
#' \code{spatstat} package.  The arguments are essentially
#' the same as the \code{\link[spatstat.core]{Kest}} function, 
#' and the user is referred there for more details about
#' the various arguments.
#' 
#' @param x A \code{\link[spatstat.geom]{ppp}} object 
#'   package with marks for the case and control groups.
#' @param case The position of the name of the "case" group 
#'   in \code{levels(x$marks)}.  The default is 2. 
#'   \code{x$marks} is assumed to be a factor.  Automatic 
#'   conversion is attempted if it is not.
#' @param nsim An non-negative integer.  Default is 0.  The
#'   difference in estimated K functions will be calculated
#'   for \code{nsim} data sets generated under the random
#'   labeling hypothesis.  These will be used to construct
#'   the non-rejection envelopes.
#' @param level The level used for the non-rejection envelopes. 
#'   Ignored if \code{nsim} is 0.
#' @param domain Optional. Calculations will be restricted
#'   to this subset of the window. See Details of
#'   \code{\link[spatstat.core]{Kest}}.
#' @inheritParams spatstat.core::Kest
#'   
#' @return Returns a \code{kdenv} object.  See documentation
#'   for \code{spatstat::Kest}.
#' @author Joshua French
#' @export
#' @seealso \code{\link[spatstat.core]{Kest}},
#'   \code{\link[spatstat.core]{eval.fv}}
#' @references Waller, L.A. and Gotway, C.A. (2005). 
#'   Applied Spatial Statistics for Public Health Data. 
#'   Hoboken, NJ: Wiley.
#' @examples 
#' data(grave)
#' kd1 = kdest(grave)
#' plot(kd1, iso ~ r, ylab = "difference", legend = FALSE, main = "")
#' kd2 = kdest(grave, nsim = 9, level = 0.8)
#' plot(kd2)
kdest = function(x, case = 2, nsim = 0, level = 0.95, r = NULL, 
                 rmax = NULL, breaks = NULL, 
                 correction = c("border", "isotropic", "Ripley", "translate"), 
                 nlarge = 3000, domain = NULL, 
                 var.approx = FALSE, ratio = FALSE) {
  x = arg_check_ppp_marks(x)
  cases = arg_check_case(case, x)
  arg_check_nsim(nsim)
  arg_check_level(level)

  if (nsim == 0) {
    out = kd(x, case = case, 
             r = r, rmax = rmax, breaks = breaks, 
             correction = correction, 
             nlarge = nlarge, domain = domain, 
             var.approx = var.approx, ratio = ratio)
    out = list(out = out)
  } else {
    #min/max envelope
    out = spatstat.core::envelope(x, kd, case = case, nsim = nsim, 
                             savefuns = TRUE, 
                             simulate = expression(spatstat.core::rlabel(x, permute = TRUE)), 
                             r = r, rmax = rmax, 
                             breaks = breaks, 
                             correction = correction, 
                             nlarge = nlarge, 
                             domain = domain, 
                             var.approx = var.approx, 
                             ratio = ratio)
    simfuns <- as.data.frame(attr(out, "simfuns"))
    simfuns[,1] <- out$obs
    l = apply(simfuns, 1, stats::quantile, prob  = (1 - level)/2)
    u = apply(simfuns, 1, stats::quantile, prob = 1 - (1 - level)/2)
    out = list(out = out, qlo = l, qhi = u)
  }
  out$r = out$out$r
  out$case_label = levels(x$marks)[case]
  out$control_label = levels(x$marks)[-case]
  out$nsim = nsim
  out$level = level
  class(out) = "kdenv"
  return(out)
}