#' Gather Draft Pick Trades
#'
#' Given a draft ID, gather draft pick trades for that draft.
#'
#' @return Returns a data frame containing information about the draft pick trade(s).
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, December 2021
#' @keywords draft pick trades
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#' @export
#' @examples
#' \dontrun{get_draft_pick_trades(688281872463106048)}
#'
#' @param draft_id Draft ID generated by Sleeper (numeric or character)
#'
get_draft_pick_trades <- function(draft_id) {
  # Send request to API given draft ID specified
  x <- jsonlite::fromJSON(httr::content(httr::GET(paste0("https://api.sleeper.app/v1/draft/", draft_id, "/traded_picks")), as = "text"))
  # Check if class of data returned from request is list
  if(is.list(x)) {
    # If class is list, it is an empty list and no data was found - inform user and do not return anything
    message("No data found - were there any trades in the draft?")
  } else if(is.null(x)) {
    # If class is NULL, invalid draft ID was entered - inform user and do not return anything
    message("No data found - was the draft ID entered correctly?")
    } else {
    # If class is not list or NULL, a data frame is returned and data was found - return data
    return(x)
  }
}
