% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/to_factor.R
\name{to_factor}
\alias{to_factor}
\title{Convert variable into factor and keep value labels}
\usage{
to_factor(x, ..., add.non.labelled = FALSE, ref.lvl = NULL)
}
\arguments{
\item{x}{A vector or data frame.}

\item{...}{Optional, unquoted names of variables that should be selected for
further processing. Required, if \code{x} is a data frame (and no
vector) and only selected variables from \code{x} should be processed.
You may also use functions like \code{:} or dplyr's \code{\link[dplyr]{select_helpers}}.
See 'Examples' or \href{../doc/design_philosophy.html}{package-vignette}.}

\item{add.non.labelled}{Logical, if \code{TRUE}, non-labelled values also
get value labels.}

\item{ref.lvl}{Numeric, specifies the reference level for the new factor. Use
this parameter if a different factor level than the lowest value
should be used as reference level. If \code{NULL}, lowest value
will become the reference level. See \code{\link{ref_lvl}} for
details.}
}
\value{
A factor, including variable and value labels. If \code{x}
          is a data frame, the complete data frame \code{x} will be returned,
          where variables specified in \code{...} are coerced
          to factors (including variable and value labels);
          if \code{...} is not specified, applies to all variables in the
          data frame.
}
\description{
This function converts a variable into a factor, but preserves
               variable and value label attributes. See 'Examples'.
}
\details{
\code{to_factor} converts numeric values into a factor with numeric
           levels. \code{\link{to_label}}, however, converts a vector into
           a factor and uses value labels as factor levels.
}
\note{
This function is intended for use with vectors that have value and variable
       label attributes. Unlike \code{\link{as.factor}}, \code{to_factor} converts
       a variable into a factor and preserves the value and variable label attributes.
       \cr \cr
       Adding label attributes is automatically done by importing data sets
       with one of the \code{read_*}-functions, like \code{\link{read_spss}}.
       Else, value and variable labels can be manually added to vectors
       with \code{\link[sjlabelled]{set_labels}} and \code{\link[sjlabelled]{set_label}}.
}
\examples{
library(sjlabelled)
data(efc)
# normal factor conversion, loses value attributes
x <- as.factor(efc$e42dep)
frq(x)

# factor conversion, which keeps value attributes
x <- to_factor(efc$e42dep)
frq(x)

# create parially labelled vector
x <- set_labels(efc$e42dep,
                labels = c(`1` = "independent", `4` = "severe dependency",
                           `9` = "missing value"))

# only copy existing value labels
to_factor(x)
get_labels(to_factor(x), include.values = "p")

# also add labels to non-labelled values
to_factor(x, add.non.labelled = TRUE)
get_labels(to_factor(x, add.non.labelled = TRUE), include.values = "p")


# Convert to factor, using different reference level
x <- to_factor(efc$e42dep)
str(x)
table(x)

x <- to_factor(efc$e42dep, ref.lvl = 3)
str(x)
table(x)


# easily coerce specific variables in a data frame to factor
# and keep other variables, with their class preserved
to_factor(efc, e42dep, e16sex, c172code)

# use select-helpers from dplyr-package
library(dplyr)
to_factor(efc, contains("cop"), c161sex:c175empl)


}
\seealso{
\code{\link{to_value}} to convert a factor into a numeric vector and
           \code{\link{to_label}} to convert a vector into a factor with labelled
           factor levels.
}
