% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/icauchy.R
\name{icauchy}
\alias{icauchy}
\title{Visualization of Random Variate Generation for the Cauchy Distribution}
\usage{
icauchy(
  u = runif(1),
  location = 0,
  scale = 1,
  minPlotQuantile = 0.05,
  maxPlotQuantile = 0.95,
  plot = TRUE,
  showCDF = TRUE,
  showPDF = TRUE,
  showECDF = TRUE,
  show = NULL,
  maxInvPlotted = 50,
  plotDelay = 0,
  sampleColor = "red3",
  populationColor = "grey",
  showTitle = TRUE,
  respectLayout = FALSE,
  ...
)
}
\arguments{
\item{u}{vector of uniform(0,1) random numbers, or NULL to show population
figures only}

\item{location}{Location parameter (default 0)}

\item{scale}{Scale parameter (default 1)}

\item{minPlotQuantile}{minimum quantile to plot}

\item{maxPlotQuantile}{maximum quantile to plot}

\item{plot}{logical; if \code{TRUE} (default), one or more plots will appear
(see parameters below); otherwise no plots appear}

\item{showCDF}{logical; if \code{TRUE} (default), cdf plot appears, otherwise cdf 
plot is suppressed}

\item{showPDF}{logical; if \code{TRUE} (default), PDF plot 
appears, otherwise PDF plot is suppressed}

\item{showECDF}{logical; if \code{TRUE} (default), ecdf plot appears,
otherwise ecdf plot is suppressed}

\item{show}{octal number (0-7) indicating plots to display;  4: CDF, 2: PDF, 
1: ECDF; sum for desired combination}

\item{maxInvPlotted}{number of inversions to plot across CDF before switching to 
plotting quantiles only}

\item{plotDelay}{delay in seconds between CDF plots}

\item{sampleColor}{Color used to display random sample from distribution}

\item{populationColor}{Color used to display population}

\item{showTitle}{logical; if \code{TRUE} (default), displays a title in the 
first of any displayed plots}

\item{respectLayout}{logical; if \code{TRUE} (default), respects existing 
settings for device layout}

\item{...}{Possible additional arguments. Currently, additional arguments not considered.}
}
\value{
A vector of Cauchy random variates
}
\description{
Generates random variates from the Cauchy distribution by inversion.
 Optionally graphs the population cumulative distribution function and
 associated random variates, the population probability density
 function and a histogram of the random variates, and the empirical
 cumulative distribution function versus the population cumulative
 distribution function.
}
\details{
Generates random variates from the Cauchy distribution, and optionally,
 illustrates
 \itemize{
   \item the use of the inverse-CDF technique,
   \item the effect of random sampling variability in relation to the PDF and CDF.
 }
 When all of the graphics are requested,
 \itemize{
   \item the first graph illustrates the use of the inverse-CDF technique by
       graphing the population CDF and the transformation of the random numbers
       to random variates,
   \item the second graph illustrates the effect of random sampling variability
       by graphing the population PDF and the histogram associated with the
       random variates, and
   \item the third graph illustrates effect of random sampling variability by
       graphing the population CDF and the empirical CDF associated with the
       random variates.
 }
 All aspects of the random variate generation algorithm are output in red by
 default, which can be changed by specifying \code{sampleColor}.
 All aspects of the population distribution are output in gray by default,
 which can be changed by specifying \code{populationColor}.

The Cauchy distribution has density
    \deqn{f(x) = \frac{1}{\pi s} \ \left(1 + \left( \frac{x - l}{s} \right)^2
              \right)^{-1}}{%
          f(x) = f(x) = 1 / (π s (1 + ((x-l)/s)^2))}
  for all \eqn{x}.

  The mean is \eqn{a/(a+b)} and the variance is \eqn{ab/((a+b)^2 (a+b+1))}.

The algorithm for generating random variates from the Cauchy distribution is
  synchronized (one random variate for each random number) and monotone in u.
  This means that the variates generated here might be useful in some variance
  reduction techniques used in Monte Carlo and discrete-event simulation.

  Values from the u vector are plotted in the cdf plot along the vertical axis
  as colored dots.  A horizontal, dashed, colored line extends from the dot to
  the population cdf.  At the intersection, a vertical, dashed colored line
  extends downward to the horizontal axis, where a second colored dot, denoting
  the associated Cauchy random variate is plotted.

  This is not a particularly fast variate generation algorithm because it uses
  the base R \code{qcauchy} function to invert the values contained in \code{u}.

  All of the elements of the \code{u} vector must be between 0 and 1.
  Alternatively, \code{u} can be \code{NULL} in which case plot(s) of the
  theoretical PDF and cdf are displayed according to plotting parameter
  values (defaulting to display of both the PDF and cdf).

  The \code{show} parameter can be used as a shortcut way to denote plots to
  display.  The argument to \code{show} can be either:
  \itemize{
    \item a binary vector of length three, where the entries from left to right
        correspond to \code{showCDF}, \code{showPDF}, and \code{showECDF},
        respectively.  For each entry, a 1 indicates the plot should be
        displayed, and a 0 indicates the plot should be suppressed.
    \item an integer in [0,7] interpreted similar to the Unix chmod command.  That
        is, the integer's binary representation can be transformed into a
        length-three vector discussed above (e.g., 6 corresponds to c(1,1,0)).
        See examples.
  }
  Any valid value for \code{show} takes precedence over existing individual
  values for \code{showCDF}, \code{showPDF}, and \code{showECDF}.

  If \code{respectLayout} is \code{TRUE}, the function respects existing
  settings for device layout.  Note, however, that if the number of plots
  requested (either via \code{show} or via \code{showCDF}, \code{showPMF}, and
  \code{showECDF}) exceeds the number of plots available in the current layout
  (as determined by \code{prod(par("mfrow"))}), the function will display all
  requested plots but will also display a warning message indicating that the
  current layout does not permit simultaneous viewing of all requested plots.
  The most recent plot with this attribute can be further annotated after the call.

  If \code{respectLayout} is \code{FALSE}, any existing user settings for device
  layout are ignored.  That is, the function uses \code{par} to explicitly set
  \code{mfrow} sufficient to show all requested plots stacked vertically to
  align their horizontal axes, and then resets row, column, and margin settings
  to their prior state on exit.

  The \code{minPlotQuantile} and \code{maxPlotQuantile} arguments are present in
  order to compress the plots horizontally.   The random variates generated are
  not impacted by these two arguments.  Vertical, dotted, black lines are
  plotted at the associated quantiles on the plots.

  \code{plotDelay} can be used to slow down or halt the variate generation for
  classroom explanation.

  In the plot associated with the PDF, the maximum plotting height is
  associated with 125\% of the maximum height of PDF. Any histogram cell
  that extends above this limit will have three dots appearing above it.
}
\examples{
 icauchy(0.5, location = 3, scale = 1)

 set.seed(8675309)
 icauchy(runif(10), 0, 3, showPDF = TRUE)

 set.seed(8675309)
 icauchy(runif(10), 0, 3, showECDF = TRUE)

 set.seed(8675309)
 icauchy(runif(10), 0, 3, showPDF = TRUE, showECDF = TRUE, sampleColor = "blue3")

 set.seed(8675309)
 icauchy(runif(10), 0, 3, showPDF = TRUE, showCDF = FALSE)

 \dontrun{
 icauchy(runif(100), 0, 3, showPDF = TRUE, minPlotQuantile = 0.02, maxPlotQuantile = 0.98)
 }

 # plot the PDF and CDF without any variates
 icauchy(NULL, 0, 3, showPDF = TRUE, showCDF = TRUE)

 # plot CDF with inversion and PDF using show
 \dontrun{
 icauchy(runif(10), 0, 3, show = c(1,1,0))
 }
 icauchy(runif(10), 0, 3, show = 6)

 # plot CDF with inversion and ECDF using show, using vunif
 \dontrun{
 icauchy(vunif(10), 0, 3, show = c(1,0,1))
 }
 icauchy(vunif(10), 0, 3, show = 5)

 # plot CDF with inversion, PDF, and ECDF using show
 \dontrun{
 icauchy(vunif(10), 0, 3, show = c(1,1,1))
 }
 icauchy(vunif(10), 0, 3, show = 7)

 # plot three different CDF+PDF+ECDF horizontal displays,
 # with title only on the first display
 par(mfrow = c(3,3))  # 3 rows, 3 cols, filling rows before columns
 set.seed(8675309)
 icauchy(runif(20), 0, 3, show = 7, respectLayout = TRUE)
 icauchy(runif(20), 0, 3, show = 7, respectLayout = TRUE, showTitle = FALSE)
 icauchy(runif(20), 0, 3, show = 7, respectLayout = TRUE, showTitle = FALSE)

 # overlay visual exploration of ks.test results
 set.seed(54321)
 vals <- icauchy (runif(10), 0, 3, showECDF = TRUE)
 D <- as.numeric(ks.test(vals, "pcauchy", 0, 3)$statistic)
 for (x in seq(0.75, 1.25, by = 0.05)) {
  y <- pcauchy(x, 0, 3)
  segments(x, y, x, y + D, col = "darkgreen", lwd = 2, xpd = NA)
 }

 \dontrun{
 # display animation of all components
 icauchy(runif(10), 0, 3, show = 7, plotDelay = 0.1)

 # display animation of CDF and PDF components only
 icauchy(runif(10), 0, 3, show = 5, plotDelay = 0.1)

 # pause at each stage of inversion
 icauchy(runif(10), 0, 3, show = 7, plotDelay = -1)
 }

}
\seealso{
\code{\link[=rcauchy]{stats::rcauchy}}

\code{\link[=runif]{stats::runif}}, 
         \code{\link[=vunif]{simEd::vunif}}
}
\author{
Barry Lawson (\email{blawson@richmond.edu}), \cr
   Larry Leemis (\email{leemis@math.wm.edu}), \cr
   Vadim Kudlay (\email{vadim.kudlay@richmond.edu})
}
\concept{random variate generation}
\keyword{distribution}
\keyword{dynamic}
\keyword{hplot}
