% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/explanatory_performance.R
\name{ExplanatoryPerformance}
\alias{ExplanatoryPerformance}
\title{Prediction performance in regression}
\usage{
ExplanatoryPerformance(
  xdata,
  ydata,
  new_xdata = NULL,
  new_ydata = NULL,
  stability = NULL,
  family = NULL,
  implementation = NULL,
  prediction = NULL,
  resampling = "subsampling",
  K = 1,
  tau = 0.8,
  seed = 1,
  n_thr = NULL,
  time = 1000,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{xdata}{matrix of predictors with observations as rows and variables as
columns.}

\item{ydata}{optional vector or matrix of outcome(s). If \code{family} is set
to \code{"binomial"} or \code{"multinomial"}, \code{ydata} can be a vector
with character/numeric values or a factor.}

\item{new_xdata}{optional test set (predictor data).}

\item{new_ydata}{optional test set (outcome data).}

\item{stability}{output of \code{\link{VariableSelection}}. If
\code{stability=NULL} (the default), a model including all variables in
\code{xdata} as predictors is fitted. Argument \code{family} must be
provided in this case.}

\item{family}{type of regression model. Possible values include
\code{"gaussian"} (linear regression), \code{"binomial"} (logistic
regression), and \code{"cox"} (survival analysis). If provided, this
argument must be consistent with input \code{stability}.}

\item{implementation}{optional function to refit the model. If
\code{implementation=NULL} and \code{stability} is the output of
\code{\link{VariableSelection}}, \code{\link[stats]{lm}} (linear
regression), \code{\link[survival]{coxph}} (Cox regression),
\code{\link[stats]{glm}} (logistic regression), or
\code{\link[nnet]{multinom}} (multinomial regression) is used.}

\item{prediction}{optional function to compute predicted values from the
model refitted with \code{implementation}.}

\item{resampling}{resampling approach to create the training set. The default
is \code{"subsampling"} for sampling without replacement of a proportion
\code{tau} of the observations. Alternatively, this argument can be a
function to use for resampling. This function must use arguments named
\code{data} and \code{tau} and return the IDs of observations to be
included in the resampled dataset.}

\item{K}{number of training-test splits. Only used if \code{new_xdata} and
\code{new_ydata} are not provided.}

\item{tau}{proportion of observations used in the training set. Only used if
\code{new_xdata} and \code{new_ydata} are not provided.}

\item{seed}{value of the seed to ensure reproducibility of the results. Only
used if \code{new_xdata} and \code{new_ydata} are not provided.}

\item{n_thr}{number of thresholds to use to construct the ROC curve. If
\code{n_thr=NULL}, all predicted probability values are iteratively used as
thresholds. For faster computations on large data, less thresholds can be
used. Only applicable to logistic regression.}

\item{time}{numeric indicating the time for which the survival probabilities
are computed. Only applicable to Cox regression.}

\item{verbose}{logical indicating if a loading bar and messages should be
printed.}

\item{...}{additional parameters passed to the function provided in
\code{resampling}.}
}
\value{
A list with: \item{TPR}{True Positive Rate (for logistic regression
  only).} \item{FPR}{False Positive Rate (for logistic regression only).}
  \item{AUC}{Area Under the Curve (for logistic regression only).}
  \item{concordance}{Concordance index (for Cox regression only).}
  \item{Beta}{matrix of estimated beta coefficients across the \code{K}
  iterations. Coefficients are extracted using the \code{\link[stats]{coef}}
  function.}
}
\description{
Calculates model performance for linear (measured by Q-squared), logistic
(AUC) or Cox (C-statistic) regression. This is done by (i) refitting the
model on a training set including a proportion \code{tau} of the
observations, and (ii) evaluating the performance on the remaining
observations (test set). For more reliable results, the procedure can be
repeated \code{K} times (default \code{K=1}).
}
\details{
For a fair evaluation of the prediction performance, the data is
  split into a training set (including a proportion \code{tau} of the
  observations) and test set (remaining observations). The regression model
  is fitted on the training set and applied on the test set. Performance
  metrics are computed in the test set by comparing predicted and observed
  outcomes.

  For logistic regression, a Receiver Operating Characteristic (ROC) analysis
  is performed: the True and False Positive Rates (TPR and FPR), and Area
  Under the Curve (AUC) are computed for different thresholds in predicted
  probabilities.

  For Cox regression, the Concordance Index (as implemented in
  \code{\link[survival]{concordance}}) looking at survival probabilities up
  to a specific \code{time} is computed.

  For linear regression, the squared correlation between predicted and
  observed outcome in the test set (Q-squared) is reported.
}
\examples{
\donttest{
# Data simulation
set.seed(1)
simul <- SimulateRegression(
  n = 1000, pk = 10,
  family = "binomial", ev_xy = 0.8
)

# Data split: selection, training and test set
ids <- Split(
  data = simul$ydata,
  family = "binomial",
  tau = c(0.4, 0.3, 0.3)
)
xselect <- simul$xdata[ids[[1]], ]
yselect <- simul$ydata[ids[[1]], ]
xtrain <- simul$xdata[ids[[2]], ]
ytrain <- simul$ydata[ids[[2]], ]
xtest <- simul$xdata[ids[[3]], ]
ytest <- simul$ydata[ids[[3]], ]

# Stability selection
stab <- VariableSelection(
  xdata = xselect,
  ydata = yselect,
  family = "binomial"
)

# Performances in test set of model refitted in training set
roc <- ExplanatoryPerformance(
  xdata = xtrain, ydata = ytrain,
  new_xdata = xtest, new_ydata = ytest,
  stability = stab
)
plot(roc)
roc$AUC

# Alternative with multiple training/test splits
roc <- ExplanatoryPerformance(
  xdata = rbind(xtrain, xtest),
  ydata = c(ytrain, ytest),
  stability = stab, K = 100
)
plot(roc)
boxplot(roc$AUC)

# Partial Least Squares Discriminant Analysis
if (requireNamespace("sgPLS", quietly = TRUE)) {
  stab <- VariableSelection(
    xdata = xselect,
    ydata = yselect,
    implementation = SparsePLS,
    family = "binomial"
  )

  # Defining wrapping functions for predictions from PLS-DA
  PLSDA <- function(xdata, ydata, family = "binomial") {
    model <- mixOmics::plsda(X = xdata, Y = as.factor(ydata), ncomp = 1)
    return(model)
  }
  PredictPLSDA <- function(xdata, model) {
    xdata <- xdata[, rownames(model$loadings$X), drop = FALSE]
    predicted <- predict(object = model, newdata = xdata)$predict[, 2, 1]
    return(predicted)
  }

  # Performances with custom models
  roc <- ExplanatoryPerformance(
    xdata = rbind(xtrain, xtest),
    ydata = c(ytrain, ytest),
    stability = stab, K = 100,
    implementation = PLSDA, prediction = PredictPLSDA
  )
  plot(roc)
}
}
}
\seealso{
\code{\link{VariableSelection}}, \code{\link{Refit}}

Other prediction performance functions: 
\code{\link{Incremental}()}
}
\concept{prediction performance functions}
