% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/explanatory_performance.R
\name{PlotIncremental}
\alias{PlotIncremental}
\title{Visualisation of incremental performance}
\usage{
PlotIncremental(
  perf,
  quantiles = c(0.05, 0.95),
  ylab = "Performance",
  pch = 18,
  col = "black",
  col.axis = NULL,
  cex = 1,
  cex.lab = 1.5,
  xcex.axis = 1,
  ycex.axis = 1,
  xlas = 2,
  ylas = 1,
  sfrac = 0.005,
  ylim = NULL,
  bty = "o",
  xgrid = FALSE,
  ygrid = FALSE,
  output_data = FALSE
)
}
\arguments{
\item{perf}{output of \code{\link{Incremental}}.}

\item{quantiles}{quantiles defining the lower and upper bounds.}

\item{ylab}{label of the y-axis.}

\item{pch}{type of point, as in \code{\link[graphics]{points}}.}

\item{col}{vector of point colours.}

\item{col.axis}{optional vector of label colours. If \code{col.axis=NULL},
the colours provided in argument \code{col} are used.}

\item{cex}{size of point.}

\item{cex.lab}{font size for labels.}

\item{xcex.axis}{size of labels along the x-axis.}

\item{ycex.axis}{size of labels along the y-axis.}

\item{xlas}{orientation of labels on the x-axis, as \code{las} in
\code{\link[graphics]{par}}.}

\item{ylas}{orientation of labels on the y-axis, as \code{las} in
\code{\link[graphics]{par}}.}

\item{sfrac}{size of the end bars, as in \code{\link[plotrix]{plotCI}}.}

\item{ylim}{displayed range along the y-axis. Only used if \code{stability}
is the output of \code{\link{BiSelection}}.}

\item{bty}{character string indicating if the box around the plot should be
drawn. Possible values include: \code{"o"} (default, the box is drawn), or
\code{"n"} (no box).}

\item{xgrid}{logical indicating if a vertical grid should be drawn. Only used
if \code{stability} is the output of \code{\link{BiSelection}}.}

\item{ygrid}{logical indicating if a horizontal grid should be drawn. Only
used if \code{stability} is the output of \code{\link{BiSelection}}.}

\item{output_data}{logical indicating if the median and quantiles should be
returned in a matrix.}
}
\value{
A plot.
}
\description{
Represents prediction performances upon sequential inclusion of the
predictors in a logistic or Cox regression model as produced by
\code{\link{Incremental}}. The median and \code{quantiles} of the performance
metric are reported.
}
\examples{
\donttest{
## Logistic regression

# Data simulation
set.seed(1)
simul <- SimulateRegression(n = 1000, pk = 50, family = "binomial")

# Balanced split: 50\% variable selection set and 50\% for evaluation of performances
ids_train <- Resample(
  data = simul$ydata,
  tau = 0.5, family = "binomial"
)
xtrain <- simul$xdata[ids_train, ]
ytrain <- simul$ydata[ids_train, ]
xtest <- simul$xdata[-ids_train, ]
ytest <- simul$ydata[-ids_train, ]

# Stability selection
stab <- VariableSelection(xdata = xtrain, ydata = ytrain, family = "binomial")

# Evaluating marginal contribution of the predictors
perf <- Incremental(xdata = xtest, ydata = ytest, stability = stab, K = 10)

# Basic visualisation
PlotIncremental(perf)

# Adding grids
PlotIncremental(perf, xgrid = TRUE, ygrid = TRUE)

# Changing colours
PlotIncremental(perf,
  bty = "n",
  col = colorRampPalette(c("blue", "red"))(length(perf$names))
)


## Cox regression

# Data simulation
set.seed(1)
simul <- SimulateRegression(n = 1000, pk = 50, family = "binomial")
ydata <- cbind(
  time = runif(nrow(simul$ydata), min = 100, max = 2000),
  case = simul$ydata[, 1]
) # including dummy time to event

# Balanced split: 50\% variable selection set and 50\% for evaluation of performances
ids_train <- Resample(
  data = simul$ydata,
  tau = 0.5, family = "binomial"
)
xtrain <- simul$xdata[ids_train, ]
ytrain <- ydata[ids_train, ]
xtest <- simul$xdata[-ids_train, ]
ytest <- ydata[-ids_train, ]

# Stability selection
stab <- VariableSelection(xdata = xtrain, ydata = ytrain, family = "cox")

# Marginal contribution
perf <- Incremental(xdata = xtest, ydata = ytest, stability = stab, K = 10)
PlotIncremental(perf)

# Faster computations on a single data split
perf <- Incremental(xdata = xtest, ydata = ytest, stability = stab, ij_method = TRUE)
PlotIncremental(perf)


## Linear regression

# Data simulation
set.seed(1)
simul <- SimulateRegression(n = 1000, pk = 50, family = "gaussian")

# Balanced split: 50\% variable selection set and 50\% for evaluation of performances
ids_train <- Resample(
  data = simul$ydata,
  tau = 0.5, family = "gaussian"
)
xtrain <- simul$xdata[ids_train, ]
ytrain <- simul$ydata[ids_train, ]
xtest <- simul$xdata[-ids_train, ]
ytest <- simul$ydata[-ids_train, ]

# Stability selection
stab <- VariableSelection(xdata = xtrain, ydata = ytrain, family = "gaussian")

# Evaluating marginal contribution of the predictors
perf <- Incremental(xdata = xtest, ydata = ytest, stability = stab, K = 10)
PlotIncremental(perf)

# Evaluating marginal contribution of the predictors beyond stably selected
perf <- Incremental(
  xdata = xtest, ydata = ytest, stability = stab,
  K = 10, n_predictors = 20
)
N <- sum(SelectedVariables(stab))
PlotIncremental(perf, col = c(rep("red", N), rep("grey", 20 - N)))
}

}
\seealso{
\code{\link{VariableSelection}}, \code{\link{Refit}}

Other prediction performance functions: 
\code{\link{ExplanatoryPerformance}()},
\code{\link{Incremental}()},
\code{\link{PlotROC}()},
\code{\link{ROC}()}
}
\concept{prediction performance functions}
