% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count_kmers.R
\name{count_kmers}
\alias{count_kmers}
\title{Count k-mers of one, particular type for a given collection of sequences}
\usage{
count_kmers(
  sequences,
  k = length(kmer_gaps) + 1,
  kmer_alphabet = getOption("seqR_kmer_alphabet_default"),
  positional = getOption("seqR_positional_default"),
  kmer_gaps = c(),
  with_kmer_counts = getOption("seqR_with_kmer_counts_default"),
  with_kmer_names = getOption("seqR_with_kmer_names_default"),
  batch_size = getOption("seqR_batch_size_default"),
  hash_dim = getOption("seqR_hash_dim_default"),
  verbose = getOption("seqR_verbose_default")
)
}
\arguments{
\item{sequences}{input sequences of one of two supported types,
either \code{string vector} or \code{list} of \code{string vectors}}

\item{k}{an \code{integer} representing the length of a k-mer}

\item{kmer_alphabet}{a \code{string vector} representing the elements that should be used
during the construction of k-mers. By default, all elements that are present in sequences
are taking into account}

\item{positional}{a single \code{logical} value that determines whether positional k-mer
variant should be considered}

\item{kmer_gaps}{an \code{integer vector} representing the lengths of gaps between consecutive
k-mer elements. The length of the vector should be equal to \code{k - 1}}

\item{with_kmer_counts}{a single \code{logical} value that determines whether the result
should contain k-mer frequencies}

\item{with_kmer_names}{a single \code{logical} value that determines whether the result
should contain human-readable k-mer names}

\item{batch_size}{a single \code{integer} value that represents the number of sequences
that are being processed in a single step}

\item{hash_dim}{a single \code{integer} value (\verb{1 <= hash_dim <= 500}) representing the length of a hash vector
that is internally used in the algorithm}

\item{verbose}{a single \code{logical} value that denotes whether a user wants to get
extra information on the current state of computations}
}
\value{
a \code{\link[Matrix]{Matrix}} value that represents a result k-mer matrix.
The result is a sparse matrix in order to reduce memory consumption.
The i-th row of the matrix represents k-mers found in the i-th input sequence.
Each column represents a distinct k-mer.
The names of columns conform to human-readable schema for k-mers,
if parameter \code{with_kmer_names = TRUE}
}
\description{
This is an in-memory, probabilistic, highly-optimized,
and multi-threaded implementation of k-mer counting algorithm.

The function supports
\enumerate{
\item several types of k-mers (contiguous, gapped, and positional variants)
\item all biological sequences (in particular, nucleic acids and proteins)
\item two common in-memory representations of sequences, i.e., string vectors and list of string vectors
}

Moreover, several extra features are provided
(for more information see \code{details}'):
\enumerate{
\item configurable k-mer alphabet
(i.e., which elements of a sequence should be considered during the k-mer counting procedure)
\item verbose mode
\item configurable batch size (i.e., how many sequences are processed in a single step)
\item configurable dimension of the hash value of a k-mer
\item possibility to compute k-mers with or without their frequencies
\item possibility to compute a result k-mer matrix with or without human-readable k-mer (column) names
}
}
\details{
The comprehensive description of supported features is available
in \code{vignette("features-overview", package = "seqR")}.
}
\examples{

batch_size <- 1

# Counting 1-mers af two DNA sequences
count_kmers(
   c("ACAT", "ACC"),
   batch_size=batch_size)

# Counting 2-mers of two DNA sequences
count_kmers(
    c("ACAT", "ACC"),
    k=2,
    batch_size=batch_size)

# Counting positional 2-mers of two DNA sequences
count_kmers(
    c("ACAT", "ACC"),
    k=2,
    positional=TRUE,
    batch_size=batch_size)

# Counting positional 2-mers of two DNA sequences (second representation)
count_kmers(
     list(c("A", "C", "A", "T"), c("A", "C", "C")),
     k=2,
     positional=TRUE,
     batch_size=batch_size)

# Counting 3-mers of two DNA sequences, considering only A and C elements
count_kmers(
    c("ACAT", "ACC"),
    k=2,
    kmer_alphabet=c("A", "C"),
    batch_size=batch_size)

# Counting gapped 3-mers with lengths of gaps 1 and 2
count_kmers(
    c("ACATACTAT", "ACCCCCC"),
    kmer_gaps=c(1,2),
    batch_size=batch_size)

}
\seealso{
Function that counts many k-mer variants in the single invocation: \link[seqR]{count_multimers}

Function that merges several k-mer matrices (rbind): \link[seqR]{rbind_columnwise}
}
