% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runMI-Wald.R
\name{lavTestWald.mi}
\alias{lavTestWald.mi}
\title{Wald Test for Multiple Imputations}
\usage{
lavTestWald.mi(object, constraints = NULL, test = c("D1", "D2"),
  asymptotic = FALSE, scale.W = FALSE, verbose = FALSE,
  warn = TRUE)
}
\arguments{
\item{object}{An object of class \code{\linkS4class{lavaan.mi}}.}

\item{constraints}{A \code{character} string (typically between single
quotes) containing one or more equality constraints.
See examples for more details}

\item{test}{\code{character} indicating which pooling method to use.
\code{"D1"} or \code{"Rubin"} (default) indicates Rubin's (1987) rules
will be applied to the point estimates and the asymptotic covariance
matrix of model parameters, and those pooled values will be used to
calculate the Wald test in the usual manner. \code{"D2"}, \code{"LMRR"},
or \code{"Li.et.al"} indicate that the complete-data Wald test statistic
should be calculated using each imputed data set, which will then be
pooled across imputations, as described in Li, Meng, Raghunathan, & Rubin
(1991) and Enders (2010, chapter 8).}

\item{asymptotic}{\code{logical}. If \code{FALSE} (default), the pooled test
will be returned as an \emph{F}-distributed statistic with numerator
(\code{df1}) and denominator (\code{df2}) degrees of freedom.
If \code{TRUE}, the pooled \emph{F} statistic will be multiplied by its
\code{df1} on the assumption that its \code{df2} is sufficiently large
enough that the statistic will be asymptotically \eqn{\chi^2} distributed
with \code{df1}.}

\item{scale.W}{\code{logical}. If \code{FALSE} (default), the pooled
asymptotic covariance matrix of model parameters is calculated as the
weighted sum of the within-imputation and between-imputation components.
Otherwise, the pooled asymptotic covariance matrix of model parameters is
calculated by scaling the within-imputation component by the
average relative increase in variance (ARIV; see Enders, 2010, p. 235).
Not recommended, and ignored (irrelevant) if \code{test = "D2"}.}

\item{verbose}{\code{logical}. If \code{TRUE}, print the restriction
matrix and the estimated restricted values.}

\item{warn}{\code{logical}. If \code{TRUE}, print warnings if they occur.}
}
\value{
A vector containing the Wald test statistic (either an \code{F} or
  \eqn{\chi^2} statistic, depending on the \code{asymptotic} argument),
  the degrees of freedom (numerator and denominator, if
  \code{asymptotic = FALSE}), and a \emph{p} value.
}
\description{
Wald test for testing a linear hypothesis about the parameters of lavaan
models fitted to multiple imputed data sets. Statistics for constraining
one or more free parameters in a model can be calculated from the pooled
point estimates and asymptotic covariance matrix of model parameters
using Rubin's (1987) rules, or by pooling the Wald  test statistics
across imputed data sets (Li, Meng, Raghunathan, & Rubin, 1991).
}
\details{
The constraints are specified using the \code{"=="} operator.
Both the left-hand side and the right-hand side of the equality can contain
a linear combination of model parameters, or a constant (like zero).
The model parameters must be specified by their user-specified labels from
the \code{link[lavaan]{model.syntax}}. Names of defined parameters
(using the ":=" operator) can be included too.
}
\examples{
 \dontrun{
## impose missing data for example
HSMiss <- HolzingerSwineford1939[ , c(paste("x", 1:9, sep = ""),
                                      "ageyr","agemo","school")]
set.seed(12345)
HSMiss$x5 <- ifelse(HSMiss$x5 <= quantile(HSMiss$x5, .3), NA, HSMiss$x5)
age <- HSMiss$ageyr + HSMiss$agemo/12
HSMiss$x9 <- ifelse(age <= quantile(age, .3), NA, HSMiss$x9)

## impute missing data
library(Amelia)
set.seed(12345)
HS.amelia <- amelia(HSMiss, m = 20, noms = "school", p2s = FALSE)
imps <- HS.amelia$imputations

## specify CFA model from lavaan's ?cfa help page
HS.model <- '
  visual  =~ x1 + b1*x2 + x3
  textual =~ x4 + b2*x5 + x6
  speed   =~ x7 + b3*x8 + x9
'

fit <- cfa.mi(HS.model, data = imps)

## Testing whether a single parameter equals zero yields the 'chi-square'
## version of the Wald z statistic from the summary() output, or the
## 'F' version of the t statistic from the summary() output, depending
## whether asymptotic = TRUE or FALSE
lavTestWald.mi(fit, constraints = "b1 == 0")      # default D1 statistic
lavTestWald.mi(fit, constraints = "b1 == 0", test = "D2") # D2 statistic

## The real advantage is simultaneously testing several equality
## constraints, or testing more complex constraints:
con <- '
   2*b1 == b3
   b2 - b3 == 0
'
lavTestWald.mi(fit, constraints = con) # default F statistic
lavTestWald.mi(fit, constraints = con, asymptotic = TRUE) # chi-squared

}

}
\references{
Enders, C. K. (2010). \emph{Applied missing data analysis}.
  New York, NY: Guilford.

  Li, K.-H., Meng, X.-L., Raghunathan, T. E., & Rubin, D. B. (1991).
  Significance levels from repeated \emph{p}-values with multiply-imputed
  data. \emph{Statistica Sinica, 1}(1), 65--92. Retrieved from
  \url{https://www.jstor.org/stable/24303994}

  Rubin, D. B. (1987). \emph{Multiple imputation for nonresponse in surveys}.
  New York, NY: Wiley.
}
\seealso{
\code{\link[lavaan]{lavTestWald}}
}
\author{
Terrence D. Jorgensen (University of Amsterdam;
  \email{TJorgensen314@gmail.com})

  Adapted from \pkg{lavaan} source code, written by
  Yves Rosseel (Ghent University; \email{Yves.Rosseel@UGent.be})
}
