#' Scatter plot HTML widget
#'
#' Interactive scatter plots based on htmlwidgets and d3.js
#'
#' @param x numerical vector of x values
#' @param y numerical vector of y values
#' @param lab optional character vector of text labels
#' @param point_size points size. Ignored if size_var is not NULL.
#' @param labels_size text labels size
#' @param point_opacity points opacity, as an integer (same opacity for all points) or a vector of integers
#' @param fixed force a 1:1 aspect ratio
#' @param col_var optional vector for points color mapping
#' @param colors vector of custom points colors. Colors must be
#'          defined as an hexadecimal string (eg "#FF0000").  If
#'          \code{colors} is a named list or vector, then the colors will
#'          be associated with their name within \code{col_var}.
#' @param ellipses draw confidence ellipses for points or the different color mapping groups
#' @param ellipses_level confidence level for ellipses (0.95 by default)
#' @param symbol_var optional vector for points symbol mapping
#' @param size_var optional vector for points size mapping
#' @param size_range numeric vector of length 2, giving the minimum and maximum point sizes when mapping with size_var
#' @param col_lab color legend title
#' @param symbol_lab symbols legend title
#' @param size_lab size legend title
#' @param key_var optional vector of rows ids. This is passed as a key to d3, and is only added in shiny apps where displayed rows are filtered interactively.
#' @param type_var optional vector of points type : "point" for adot (default), "arrow" for an arrow starting from the origin.
#' @param unit_circle set tot TRUE to draw a unit circle
#' @param tooltips logical value to display tooltips when hovering points
#' @param tooltip_text optional character vector of tooltips text
#' @param xlab x axis label
#' @param ylab y axis label
#' @param xlim numeric vector of length 2, manual x axis limits
#' @param ylim numeric vector of length 2, manual y axis limits
#' @param lasso logical value to add {https://github.com/skokenes/D3-Lasso-Plugin}{d3-lasso-plugin} feature
#' @param lasso_callback the body of a JavaScript callback function with the argument \code{sel} to be applied to a lasso plugin selection
#' @param html_id manually specify an HTML id for the svg root node. A random one is generated by default.
#' @param dom_id_reset_zoom HTML DOM id of the element to bind the "reset zoom" control to.
#' @param dom_id_svg_export HTML DOM id of the element to bind the "svg export" control to.
#' @param dom_id_lasso_toggle HTML DOM id of the element to bind the "toggle lasso" control to.
#' @param transitions if TRUE, data updates are displayed with smooth transitions, if FALSE the whole chart is redrawn. Only used within shiny apps.
#' @param legend_width legend area width, in pixels. Set to 0 to disable legend completely.
#' @param width figure width, computed when displayed
#' @param height figure height, computed when displayed
#'
#' @description Generates an interactive scatter plot based on d3.js.
#' Interactive features include zooming, panning, text labels moving, tooltips,
#' fading effects in legend. Additional handlers are provided to change label
#' size, point opacity or export the figure as an SVG file via HTML form controls.
#'
#' @author Julien Barnier <julien.barnier@@ens-lyon.fr>
#'
#' @source
#' D3.js was created by Michael Bostock. See \url{http://d3js.org/}
#'
#' @examples
#' scatterD3(x = mtcars$wt, y = mtcars$mpg, lab = rownames(mtcars),
#'           col_var = mtcars$cyl, symbol_var = mtcars$am,
#'           xlab = "Weight", ylab = "Mpg", col_lab = "Cylinders",
#'           symbol_lab = "Manual transmission", html_id = NULL)
#'
#' @importFrom ellipse ellipse
#' @importFrom stats cov
#' @importFrom htmlwidgets JS
#' @export
#'
scatterD3 <- function(x, y, lab = NULL,
                      point_size = 64, labels_size = 10,
                      point_opacity = 1,
                      fixed = FALSE, col_var = NULL,
                      colors = NULL,
                      ellipses = FALSE,
                      ellipses_level = 0.95,
                      symbol_var = NULL,
                      size_var = NULL,
                      size_range = c(10,300),
                      col_lab = NULL, symbol_lab = NULL,
                      size_lab = NULL,
                      key_var = NULL,
                      type_var = NULL,
                      unit_circle = FALSE,
                      tooltips = TRUE,
                      tooltip_text = NULL,
                      xlab = NULL, ylab = NULL,
                      html_id = NULL,
                      width = NULL, height = NULL,
                      legend_width = 150,
                      xlim = NULL, ylim = NULL,
                      dom_id_reset_zoom = "scatterD3-reset-zoom",
                      dom_id_svg_export = "scatterD3-svg-export",
                      dom_id_lasso_toggle = "scatterD3-lasso-toggle",
                      transitions = FALSE,
                      lasso = FALSE,
                      lasso_callback = NULL) {

  ## Variable names as default labels
  if (is.null(xlab)) xlab <- deparse(substitute(x))
  if (is.null(ylab)) ylab <- deparse(substitute(y))
  if (is.null(col_lab)) col_lab <- deparse(substitute(col_var))
  if (is.null(symbol_lab)) symbol_lab <- deparse(substitute(symbol_var))
  if (is.null(size_lab)) size_lab <- deparse(substitute(size_var))
  if (is.null(html_id)) html_id <- paste0("scatterD3-", paste0(sample(LETTERS,8,replace = TRUE),collapse = ""))

  # colors can be named
  #  we'll need to convert named vector to a named list
  #  for the JSON conversion
  if (!is.null(colors) && !is.null(names(colors))) {
    colors <- as.list(colors)
    if (!setequal(names(colors), unique(col_var))) warning("Set of colors and col_var values do not match")
  }

  ## data element
  data <- data.frame(x = x, y = y)
  if (!is.null(lab)) data <- cbind(data, lab = lab)
  if (!is.null(point_opacity)) data <- cbind(data, point_opacity = point_opacity)
  if (!is.null(col_var)) {
    col_var <- as.character(col_var)
    col_var[is.na(col_var)] <- "NA"
    data <- cbind(data, col_var = col_var)
  }
  if (!is.null(symbol_var)) {
    symbol_var <- as.character(symbol_var)
    symbol_var[is.na(symbol_var)] <- "NA"
    data <- cbind(data, symbol_var = symbol_var)
  }
  if (!is.null(size_var)) data <- cbind(data, size_var = size_var)
  if (!is.null(type_var)) data <- cbind(data, type_var = type_var)
  if (!is.null(key_var)) data <- cbind(data, key_var = key_var)
  else data <- cbind(data, key_var = seq_along(x))
  if (!is.null(tooltip_text)) data <- cbind(data, tooltip_text = tooltip_text)

  ## Compute confidence ellipses point positions with ellipse::ellipse.default()
  compute_ellipse <- function(x, y, level = ellipses_level, npoints = 50) {
    cx <- mean(x)
    cy <- mean(y)
    data.frame(ellipse::ellipse(stats::cov(cbind(x,y)), centre = c(cx, cy), level = level, npoints = npoints))
  }

  ## Compute ellipses points data
  ellipses_data <- list()
  if (ellipses) {
    ## Only one ellipse
    if (is.null(col_var)) {
      ell <- compute_ellipse(x, y)
      ellipses_data <- append(ellipses_data, list(list(level = "_scatterD3_all", data = ell)))
    } else {
      ## One ellipse per col_var level
      for (l in unique(col_var)) {
        sel <- col_var == l & !is.na(col_var)
        if (sum(sel) > 2) {
          tmpx <- x[sel]
          tmpy <- y[sel]
          ell <- compute_ellipse(tmpx, tmpy)
          ellipses_data <- append(ellipses_data, list(list(level = l, data = ell)))
        }
      }
    }
  }

  ## List of hashes for each data variable, to track which data elements changed
  ## to apply updates and transitions in shiny app.
  hashes <- list()
  if (transitions) {
    for (var in c("x", "y", "lab", "key_var", "col_var", "symbol_var", "size_var", "ellipses_data", "point_opacity")) {
      hashes[[var]] <- digest::digest(get(var), algo = "sha256")
    }
  }

  # create a list that contains the settings
  settings <- list(
    labels_size = labels_size,
    point_size = point_size,
    xlab = xlab,
    ylab = ylab,
    has_labels = !is.null(lab),
    col_var = col_var,
    col_lab = col_lab,
    colors = colors,
    ellipses = ellipses,
    ellipses_data = ellipses_data,
    symbol_var = symbol_var,
    symbol_lab = symbol_lab,
    size_var = size_var,
    size_range = size_range,
    size_lab = size_lab,
    key_var = key_var,
    type_var = type_var,
    unit_circle = unit_circle,
    has_color_var = !is.null(col_var),
    has_symbol_var = !is.null(symbol_var),
    has_size_var = !is.null(size_var),
    has_legend = !is.null(col_var) || !is.null(symbol_var) || !is.null(size_var),
    has_tooltips = tooltips,
    tooltip_text = tooltip_text,
    has_custom_tooltips = !is.null(tooltip_text),
    fixed = fixed,
    legend_width = legend_width,
    html_id = html_id,
    xlim = xlim,
    ylim = ylim,
    lasso = lasso,
    lasso_callback = htmlwidgets::JS(lasso_callback),
    dom_id_reset_zoom = dom_id_reset_zoom,
    dom_id_svg_export = dom_id_svg_export,
    dom_id_lasso_toggle = dom_id_lasso_toggle,
    transitions = transitions,
    hashes = hashes
  )

  # pass the data and settings using 'x'
  x <- list(
    data = data,
    settings = settings
  )

  # create widget
  htmlwidgets::createWidget(
      name = 'scatterD3',
      x,
      width = width,
      height = height,
      package = 'scatterD3',
      sizingPolicy = htmlwidgets::sizingPolicy(
          browser.fill = TRUE,
          viewer.fill = TRUE
      )
  )
}

#' @rdname scatterD3-shiny
#' @export
scatterD3Output <- function(outputId, width = '100%', height = '600px'){
  htmlwidgets::shinyWidgetOutput(outputId, 'scatterD3', width, height, package = 'scatterD3')
}

#' @rdname scatterD3-shiny
#' @export
renderScatterD3 <- function(expr, env = parent.frame(), quoted = FALSE) {
  if (!quoted) { expr <- substitute(expr) } # force quoted
  htmlwidgets::shinyRenderWidget(expr, scatterD3Output, env, quoted = TRUE)
}




