% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CDORemap.R
\name{CDORemap}
\alias{CDORemap}
\title{Interpolates arrays with longitude and latitude dimensions using CDO}
\usage{
CDORemap(data_array = NULL, lons, lats, grid, method, avoid_writes = TRUE,
  crop = TRUE, force_remap = FALSE, write_dir = tempdir())
}
\arguments{
\item{data_array}{Multidimensional numeric array to be interpolated. If 
provided, it must have at least a longitude and a latitude dimensions, 
identified by the array dimension names. The names for these dimensions 
must be one of the recognized by s2dverification (can be checked with 
\code{s2dverification:::.KnownLonNames()} and 
\code{s2dverification:::.KnownLatNames()}).}

\item{lons}{Numeric vector or array of longitudes of the centers of the grid 
cells. Its size must match the size of the longitude/latitude dimensions 
of the input array.}

\item{lats}{Numeric vector or array of latitudes of the centers of the grid 
cells. Its size must match the size of the longitude/latitude dimensions 
of the input array.}

\item{grid}{Character string specifying either a name of a target grid 
(recognized by CDO; e.g.: 'r256x128', 't106grid') or a path to another 
NetCDF file which to read the target grid from (a single grid must be 
defined in such file).}

\item{method}{Character string specifying an interpolation method 
(recognized by CDO; e.g.: 'con', 'bil', 'bic', 'dis'). The following 
long names are also supported: 'conservative', 'bilinear', 'bicubic' and 
'distance-weighted'.}

\item{avoid_writes}{The step of permutation is needed when the input array 
has more than 3 dimensions and none of the longitude or latitude dimensions
 in the right-most position (CDO would not accept it without permuting 
previously). This step, executed by default when needed, can be avoided 
for the price of writing more intermediate files (whis usually is 
unconvenient) by setting the parameter \code{avoid_writes = TRUE}.}

\item{crop}{Whether to crop the data after interpolation with 
'cdo sellonlatbox' (TRUE) or to extend interpolated data to the whole 
world as CDO does by default (FALSE). If \code{crop = TRUE} then the 
longitude and latitude borders which to crop at are taken as the limits of 
the cells at the borders ('lons' and 'lats' are perceived as cell centers), 
i.e. the resulting array will contain data that covers the same area as 
the input array. This is equivalent to specifying \code{crop = 'preserve'}, 
i.e. preserving area. If \code{crop = 'tight'} then the borders which to 
crop at are taken as the minimum and maximum cell centers in 'lons' and 
'lats', i.e. the area covered by the resulting array may be smaller if 
interpolating from a coarse grid to a fine grid. The parameter 'crop' also 
accepts a numeric vector of custom borders which to crop at: 
c(western border, eastern border, southern border, northern border).}

\item{force_remap}{Whether to force remapping, even if the input data array 
is already on the target grid.}

\item{write_dir}{Path to the directory where to create the intermediate 
files for CDO to work. By default, the R session temporary directory is 
used (\code{tempdir()}).}
}
\value{
A list with the following components:
 \item{'data_array'}{The interpolated data array (if an input array 
 is provided at all, NULL otherwise).}
 \item{'lons'}{The longitudes of the data on the destination grid.}
 \item{'lats'}{The latitudes of the data on the destination grid.}
}
\description{
This function takes as inputs a multidimensional array (optional), a vector 
or matrix of longitudes, a vector or matrix of latitudes, a destination grid 
specification, and the name of a method to be used to interpolate (one of 
those available in the 'remap' utility in CDO). The interpolated array is 
returned (if provided) together with the new longitudes and latitudes.\cr\cr 
\code{CDORemap()} permutes by default the dimensions of the input array (if 
needed), splits it in chunks (CDO can work with data arrays of up to 4 
dimensions), generates a file with the data of each chunk, interpolates it 
with CDO, reads it back into R and merges it into a result array. If no 
input array is provided, the longitude and latitude vectors will be 
transformed only. If the array is already on the desired destination grid, 
no transformation is performed (this behvaiour works only for lonlat and 
gaussian grids). \cr\cr
Any metadata attached to the input data array, longitudes or latitudes will 
be preserved or accordingly modified.
}
\examples{
 \dontrun{
# Interpolating only vectors of longitudes and latitudes
lon <- seq(0, 360 - 360/50, length.out = 50)
lat <- seq(-90, 90, length.out = 25)
tas2 <- CDORemap(NULL, lon, lat, 't170grid', 'bil', TRUE)

# Minimal array interpolation
tas <- array(1:50, dim = c(25, 50))
names(dim(tas)) <- c('lat', 'lon')
lon <- seq(0, 360 - 360/50, length.out = 50)
lat <- seq(-90, 90, length.out = 25)
tas2 <- CDORemap(tas, lon, lat, 't170grid', 'bil', TRUE)

# Metadata can be attached to the inputs. It will be preserved and 
# accordignly modified.
tas <- array(1:50, dim = c(25, 50))
names(dim(tas)) <- c('lat', 'lon')
lon <- seq(0, 360 - 360/50, length.out = 50)
metadata <- list(lon = list(units = 'degrees_east'))
attr(lon, 'variables') <- metadata
lat <- seq(-90, 90, length.out = 25)
metadata <- list(lat = list(units = 'degrees_north'))
attr(lat, 'variables') <- metadata
metadata <- list(tas = list(dim = list(lat = list(len = 25,
                                                 vals = lat),
                                      lon = list(len = 50,
                                                 vals = lon)
                                     )))
attr(tas, 'variables') <- metadata
tas2 <- CDORemap(tas, lon, lat, 't170grid', 'bil', TRUE)

# Arrays of any number of dimensions in any order can be provided.
num_lats <- 25
num_lons <- 50
tas <- array(1:(10*num_lats*10*num_lons*10), 
            dim = c(10, num_lats, 10, num_lons, 10))
names(dim(tas)) <- c('a', 'lat', 'b', 'lon', 'c')
lon <- seq(0, 360 - 360/num_lons, length.out = num_lons)
metadata <- list(lon = list(units = 'degrees_east'))
attr(lon, 'variables') <- metadata
lat <- seq(-90, 90, length.out = num_lats)
metadata <- list(lat = list(units = 'degrees_north'))
attr(lat, 'variables') <- metadata
metadata <- list(tas = list(dim = list(a = list(),
                                      lat = list(len = num_lats,
                                                 vals = lat),
                                      b = list(),
                                      lon = list(len = num_lons,
                                                 vals = lon),
                                      c = list()
                                     )))
attr(tas, 'variables') <- metadata
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil', TRUE)
# The step of permutation can be avoided but more intermediate file writes
# will be performed.
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil', FALSE)

# If the provided array has the longitude or latitude dimension in the 
# right-most position, the same number of file writes will be performed,
# even if avoid_wrties = FALSE.
num_lats <- 25
num_lons <- 50
tas <- array(1:(10*num_lats*10*num_lons*10), 
            dim = c(10, num_lats, 10, num_lons))
names(dim(tas)) <- c('a', 'lat', 'b', 'lon')
lon <- seq(0, 360 - 360/num_lons, length.out = num_lons)
metadata <- list(lon = list(units = 'degrees_east'))
attr(lon, 'variables') <- metadata
lat <- seq(-90, 90, length.out = num_lats)
metadata <- list(lat = list(units = 'degrees_north'))
attr(lat, 'variables') <- metadata
metadata <- list(tas = list(dim = list(a = list(),
                                      lat = list(len = num_lats,
                                                 vals = lat),
                                      b = list(),
                                      lon = list(len = num_lons,
                                                 vals = lon)
                                     )))
attr(tas, 'variables') <- metadata
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil', TRUE)
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil', FALSE)

# An example of an interpolation from and onto a rectangular regular grid
num_lats <- 25
num_lons <- 50
tas <- array(1:(1*num_lats*num_lons), dim = c(num_lats, num_lons))
names(dim(tas)) <- c('y', 'x')
lon <- array(seq(0, 360 - 360/num_lons, length.out = num_lons), 
            dim = c(num_lons, num_lats))
metadata <- list(lon = list(units = 'degrees_east'))
names(dim(lon)) <- c('x', 'y')
attr(lon, 'variables') <- metadata
lat <- t(array(seq(-90, 90, length.out = num_lats), 
              dim = c(num_lats, num_lons)))
metadata <- list(lat = list(units = 'degrees_north'))
names(dim(lat)) <- c('x', 'y')
attr(lat, 'variables') <- metadata
tas2 <- CDORemap(tas, lon, lat, 'r100x50', 'bil')

# An example of an interpolation from an irregular grid onto a gaussian grid
num_lats <- 25
num_lons <- 50
tas <- array(1:(10*num_lats*10*num_lons*10), 
            dim = c(10, num_lats, 10, num_lons))
names(dim(tas)) <- c('a', 'j', 'b', 'i')
lon <- array(seq(0, 360 - 360/num_lons, length.out = num_lons), 
            dim = c(num_lons, num_lats))
metadata <- list(lon = list(units = 'degrees_east'))
names(dim(lon)) <- c('i', 'j')
attr(lon, 'variables') <- metadata
lat <- t(array(seq(-90, 90, length.out = num_lats), 
        dim = c(num_lats, num_lons)))
metadata <- list(lat = list(units = 'degrees_north'))
names(dim(lat)) <- c('i', 'j')
attr(lat, 'variables') <- metadata
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil')

# Again, the dimensions can be in any order
num_lats <- 25
num_lons <- 50
tas <- array(1:(10*num_lats*10*num_lons), 
            dim = c(10, num_lats, 10, num_lons))
names(dim(tas)) <- c('a', 'j', 'b', 'i')
lon <- array(seq(0, 360 - 360/num_lons, length.out = num_lons), 
            dim = c(num_lons, num_lats))
names(dim(lon)) <- c('i', 'j')
lat <- t(array(seq(-90, 90, length.out = num_lats), 
              dim = c(num_lats, num_lons)))
names(dim(lat)) <- c('i', 'j')
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil')
tas2 <- CDORemap(tas, lon, lat, 't17grid', 'bil', FALSE)
# It is ossible to specify an external NetCDF file as target grid reference
tas2 <- CDORemap(tas, lon, lat, 'external_file.nc', 'bil')
}
}
\author{
History:\cr
 0.0 - 2017-01 (N. Manubens, \email{nicolau.manubens@bsc.es}) - Original code.
}
\keyword{datagen}

