context("forest")
library(ggplot2)

test_that("forest checks arguments properly", {
  data("MIsim", package = "rsimsum")
  s1 <- simsum(data = MIsim, estvarname = "b", true = 0.5, se = "se", methodvar = "method")
  expect_error(forest(obj = MIsim))
  expect_error(forest(obj = s1))
  expect_error(forest(obj = s1, sstat = "BIAS"))
  expect_error(forest(obj = s1, sstat = "nsim"))
  expect_error(forest(obj = s1, sstat = "bias", level = 101))
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model", by = "fv_dist")
  expect_error(forest(obj = sm))
  expect_error(forest(obj = sm, sstat = "BIAS"))
  expect_warning(expect_error(forest(obj = sm, sstat = "nsim")))
  expect_error(forest(obj = sm, sstat = "bias", level = 101))
})

test_that("forest returns a ggplot object", {
  data("MIsim", package = "rsimsum")
  s <- simsum(data = MIsim, estvarname = "b", true = 0.5, se = "se", methodvar = "method")
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model")
  expect_s3_class(forest(s, sstat = "bias"), class = c("gg", "ggplot"))
  expect_s3_class(forest(sm, sstat = "bias"), class = c("gg", "ggplot"))
  expect_s3_class(forest(sm, sstat = "bias", par = "trt"), class = c("gg", "ggplot"))
})

test_that("forest does not plot CI when simsum in called with mcse = FALSE", {
  data("MIsim", package = "rsimsum")
  s <- simsum(data = MIsim, estvarname = "b", true = 0.5, se = "se", methodvar = "method", mcse = FALSE)
  forest(s, sstat = "bias")
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model", mcse = FALSE)
  forest(sm, sstat = "bias")
  forest(sm, sstat = "bias", par = "trt")
})

test_that("forest works with 'by' factors", {
  data("relhaz", package = "rsimsum")
  s <- simsum(data = relhaz, estvarname = "theta", true = -0.5, se = "se", methodvar = "model", by = c("n", "baseline"))
  expect_warning(forest(s, sstat = "bias"))
  expect_error(forest(s, sstat = "bias", by = "random-name"))
  expect_s3_class(forest(s, sstat = "bias", by = c("n", "baseline")), class = c("gg", "ggplot"))
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model", by = "fv_dist")
  expect_warning(forest(sm, sstat = "bias", par = "trt"))
  expect_error(forest(sm, sstat = "bias", par = "trt", by = "random-name"))
  expect_s3_class(forest(sm, sstat = "bias", par = "trt", by = "fv_dist"), class = c("gg", "ggplot"))
  expect_warning(forest(sm, sstat = "bias"))
  expect_error(forest(sm, sstat = "bias", by = "random-name"))
  expect_s3_class(forest(sm, sstat = "bias", by = "fv_dist"), class = c("gg", "ggplot"))
})

test_that("forest works when changing graphical parameters", {
  data("MIsim", package = "rsimsum")
  s <- simsum(data = MIsim, estvarname = "b", true = 0.5, se = "se", methodvar = "method")
  forest(s, sstat = "bias", gpars = list(target.shape = 1))
  forest(s, sstat = "bias", gpars = list(target.colour = 2))
  forest(s, sstat = "bias", gpars = list(width = 0.5))
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model")
  forest(sm, sstat = "bias", par = "trt", gpars = list(target.shape = 1))
  forest(sm, sstat = "bias", par = "trt", gpars = list(target.colour = 2))
  forest(sm, sstat = "bias", par = "trt", gpars = list(width = 0.5))
  forest(sm, sstat = "bias", gpars = list(target.shape = 1))
  forest(sm, sstat = "bias", gpars = list(target.colour = 2))
  forest(sm, sstat = "bias", gpars = list(width = 0.5))
})

test_that("forest with all 'sstat' options, with and without 'target'", {
  data("MIsim", package = "rsimsum")
  s <- simsum(data = MIsim, estvarname = "b", true = 0.5, se = "se", methodvar = "method")
  forest(s, sstat = "nsim", target = 1000)
  forest(s, sstat = "thetamean")
  forest(s, sstat = "thetamean", target = 0.50)
  forest(s, sstat = "thetamedian")
  forest(s, sstat = "thetamedian", target = 0.50)
  forest(s, sstat = "se2mean")
  forest(s, sstat = "se2mean", target = 0)
  forest(s, sstat = "se2median")
  forest(s, sstat = "se2median", target = 0)
  forest(s, sstat = "bias")
  forest(s, sstat = "bias", target = 0)
  forest(s, sstat = "empse")
  forest(s, sstat = "empse", target = 0)
  forest(s, sstat = "mse")
  forest(s, sstat = "mse", target = 0)
  forest(s, sstat = "relprec")
  forest(s, sstat = "relprec", target = 1)
  forest(s, sstat = "modelse")
  forest(s, sstat = "modelse", target = 0)
  forest(s, sstat = "relerror")
  forest(s, sstat = "relerror", target = 0)
  forest(s, sstat = "cover")
  forest(s, sstat = "cover", target = 0.95)
  forest(s, sstat = "bccover")
  forest(s, sstat = "bccover", target = 0.95)
  forest(s, sstat = "power")
  forest(s, sstat = "power", target = 0.95)
  data("frailty", package = "rsimsum")
  sm <- multisimsum(data = frailty, par = "par", true = c(trt = -0.50, fv = 0.75), estvarname = "b", se = "se", methodvar = "model", by = "fv_dist")
  forest(sm, par = "trt", by = "fv_dist", sstat = "nsim", target = 1000)
  forest(sm, par = "trt", by = "fv_dist", sstat = "thetamean")
  forest(sm, par = "trt", by = "fv_dist", sstat = "thetamean", target = -0.50)
  forest(sm, par = "trt", by = "fv_dist", sstat = "thetamedian")
  forest(sm, par = "trt", by = "fv_dist", sstat = "thetamedian", target = -0.50)
  forest(sm, par = "trt", by = "fv_dist", sstat = "se2mean")
  forest(sm, par = "trt", by = "fv_dist", sstat = "se2mean", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "se2median")
  forest(sm, par = "trt", by = "fv_dist", sstat = "se2median", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "bias")
  forest(sm, par = "trt", by = "fv_dist", sstat = "bias", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "empse")
  forest(sm, par = "trt", by = "fv_dist", sstat = "empse", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "mse")
  forest(sm, par = "trt", by = "fv_dist", sstat = "mse", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "relprec")
  forest(sm, par = "trt", by = "fv_dist", sstat = "relprec", target = 1)
  forest(sm, par = "trt", by = "fv_dist", sstat = "modelse")
  forest(sm, par = "trt", by = "fv_dist", sstat = "modelse", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "relerror")
  forest(sm, par = "trt", by = "fv_dist", sstat = "relerror", target = 0)
  forest(sm, par = "trt", by = "fv_dist", sstat = "cover")
  forest(sm, par = "trt", by = "fv_dist", sstat = "cover", target = 0.95)
  forest(sm, par = "trt", by = "fv_dist", sstat = "bccover")
  forest(sm, par = "trt", by = "fv_dist", sstat = "bccover", target = 0.95)
  forest(sm, par = "trt", by = "fv_dist", sstat = "power")
  forest(sm, par = "trt", by = "fv_dist", sstat = "power", target = 0.95)
  forest(sm, by = "fv_dist", sstat = "nsim", target = 1000)
  forest(sm, by = "fv_dist", sstat = "thetamean")
  forest(sm, by = "fv_dist", sstat = "thetamean", target = -0.50)
  forest(sm, by = "fv_dist", sstat = "thetamedian")
  forest(sm, by = "fv_dist", sstat = "thetamedian", target = -0.50)
  forest(sm, by = "fv_dist", sstat = "se2mean")
  forest(sm, by = "fv_dist", sstat = "se2mean", target = 0)
  forest(sm, by = "fv_dist", sstat = "se2median")
  forest(sm, by = "fv_dist", sstat = "se2median", target = 0)
  forest(sm, by = "fv_dist", sstat = "bias")
  forest(sm, by = "fv_dist", sstat = "bias", target = 0)
  forest(sm, by = "fv_dist", sstat = "empse")
  forest(sm, by = "fv_dist", sstat = "empse", target = 0)
  forest(sm, by = "fv_dist", sstat = "mse")
  forest(sm, by = "fv_dist", sstat = "mse", target = 0)
  forest(sm, by = "fv_dist", sstat = "relprec")
  forest(sm, by = "fv_dist", sstat = "relprec", target = 1)
  forest(sm, by = "fv_dist", sstat = "modelse")
  forest(sm, by = "fv_dist", sstat = "modelse", target = 0)
  forest(sm, by = "fv_dist", sstat = "relerror")
  forest(sm, by = "fv_dist", sstat = "relerror", target = 0)
  forest(sm, by = "fv_dist", sstat = "cover")
  forest(sm, by = "fv_dist", sstat = "cover", target = 0.95)
  forest(sm, by = "fv_dist", sstat = "bccover")
  forest(sm, by = "fv_dist", sstat = "bccover", target = 0.95)
  forest(sm, by = "fv_dist", sstat = "power")
  forest(sm, by = "fv_dist", sstat = "power", target = 0.95)
})
