#' Test homogeneous Poisson assumption of disease incidence.
#'
#' This function compares the actual variance of the yearly incidence rates with
#' rates simulated from a Poisson process with overall rate equal to the overall
#' mean rate.
#'
#' @param inc Vector of absolute incidence values for each included year
#'   of the registry as generated by \code{\link{raw_incidence}}.
#' @param N_sim Number of simulations to perform.
#' @return Vector of p-values for over- and under-dispersion based on the
#'   position of the observed sequence variance in the distribution.
#' @examples
#' data(prevsim)
#'
#' inc <- raw_incidence(prevsim$entrydate)
#'
#' test_incidence_fit(inc)
#' @export
#' @family incidence functions
test_incidence_fit <- function(inc, N_sim = 1e5) {
    var_sim <- vapply(seq(N_sim), function(i) var(rpois(length(inc), mean(inc))), numeric(1))
    c(sum(var_sim > var(inc))/N_sim, sum(var_sim <= var(inc))/N_sim)
}

#' Test simulated prevalence fit.
#'
#' Calculates a Chi squared test between predicted yearly contributions to
#' prevalence, and the observed values obtained from the registry, indicating
#' whether the simulated prevalence values are accurate.
#'
#' @param object A \code{prevalence} object.
#' @return P-value from a chi-squared test of difference between prevalence
#'   prediction and counted prevalence at the index date.
#' @examples
#' data(prevsim)
#'
#' \dontrun{
#'
#' obj <- prevalence(Surv(time, status) ~ age(age) + sex(sex) + entry(entrydate) + event(eventdate),
#'                   data=prevsim, num_years_to_estimate = c(5, 10), population_size=1e6,
#'                   start = "2005-09-01",
#'                   num_reg_years = 8, cure = 5)
#'
#' test_prevalence_fit(obj)
#' }
#'
#' @export
#' @family prevalence functions
test_prevalence_fit <- function(object) {
    predicted <- rev(object$simulated$mean_yearly_contributions[1:object$nregyears])
    chi <- sum(((object$counted - predicted)^2) / predicted)
    1 - pchisq(chi, object$nregyears - 1)
}

#' Plot the age distribution of incident cases.
#'
#' This function plots the age distribution of incident cases from the registry.
#'
#' @param agedata Vector of ages at diagnosis for each patient in the registry.
#' @param df Degrees of freedom for the smooth.
#' @return Plot of the raw data and smoothed age distribution function.
#' @examples
#' data(prevsim)
#'
#' incidence_age_distribution(prevsim$age)
#'
#' incidence_age_distribution(prevsim$age, df=5)
#'
#' @export
#' @family incidence functions
incidence_age_distribution <- function(agedata, df=10) {

    ages <- vapply(seq(100), function(i) sum(floor(agedata) + 1 == i), numeric(1))
    p <- ggplot2::ggplot(data.frame(count=ages, age=seq(100)),
                         ggplot2::aes_string(x='age', y='count')) +
            ggplot2::geom_point() +
            ggplot2::geom_smooth(method='lm', formula=y ~ poly(x, df, raw=TRUE)) +
            ggplot2::xlim(0, 100) +
            ggplot2::labs(x='Age (years)', y='Number incident cases') +
            ggplot2::theme_bw()
    p
}

#' View the suitability of a linear effect of age on hazard.
#'
#' Inspects whether a non-linear transform of age is more appropriate for
#' survival modeling than the linear method which is currently used in the
#' calculations in \code{\link{prevalence}}.
#'
#' @param form Formula where the LHS is represented by a standard \code{Surv}
#'   object, and the RHS provides the name of the variable containing age in the
#'   supplied dataset.
#' @param data A data frame with the corresponding column names provided in
#'   \code{form}.
#' @param df The desired degrees of freedom for the cubic spline, must be >= 3.
#' @param plot_fit Whether to plot the relationship between age and relative
#'   hazard.
#' @param num_points The number of values of age to calculate the hazard ratio at.
#' @return An \code{rms::cph} object containing the Cox model fit between a
#'   cubic spline transform of age and the survival outcomes.
#' @examples
#' data(prevsim)
#'
#' functional_form_age(Surv(time, status) ~ age, prevsim)
#'
#' functional_form_age(Surv(time, status) ~ age, prevsim, df=3)
#'
#' functional_form_age(Surv(time, status) ~ age, prevsim, df=3, plot_fit=FALSE)
#'
#' @export
#' @importFrom rms cph
#' @importFrom rms rcs
#' @importFrom ggplot2 ggplot
#' @importFrom ggplot2 geom_ribbon
#' @importFrom ggplot2 geom_line
#' @importFrom ggplot2 aes_string
functional_form_age <- function(form, data, df=4, plot_fit=TRUE, num_points=200) {

    trms <- attr(terms(form), 'variables')
    if (length(trms) != 3)
        stop("Error: Please provide just Surv and age terms in the formula.")

    resp <- trms[[2]]
    age_var <- trms[[3]]
    survobj <- eval(resp, data)
    mydf <- data.frame(time=survobj[, 1], status=survobj[, 2], age=eval(age_var, data))

    myform <- survival::Surv(time, status) ~ rms::rcs(age, df)
    mod_rms <- rms::cph(myform, mydf, x=TRUE, y=TRUE, surv=TRUE, time.inc=1)

    if (plot_fit) {
        min_age <- quantile(mydf[, 'age'], 0.01)
        max_age <- quantile(mydf[, 'age'], 0.99)
        preds <- .hazard_varying_age(mod_rms, min_age, max_age, num_points)

        p <- ggplot2::ggplot(preds, ggplot2::aes_string(x='age', y='yhat')) +
                ggplot2::geom_ribbon(ggplot2::aes_string(ymin='lower', ymax='upper'),
                                 colour='#d2d2d2', alpha=0.30) +
                ggplot2::geom_line(colour='#0080ff', size=1.2) +
                ggplot2::theme_bw() +
                ggplot2::labs(x='Age', y='Log relative hazard')
        print(p)
    }

    mod_rms
}

.hazard_varying_age <- function(model, min_age, max_age, num_points) {
    # Calculates log hazard linear predictor at multiple values of age
    preds <- lapply(seq(min_age, max_age, length.out=num_points),
                    function(a) {
                         p <- predict(model, newdata=a, se.fit=T)
                         lp <- p$linear.predictors
                         c(a, lp, lp - 2*p$se.fit, lp + 2*p$se.fit)
                         }
                     )
    preds_df <- as.data.frame(do.call('rbind', preds))
    colnames(preds_df) <- c('age', 'yhat', 'lower', 'upper')
    preds_df
}