# This file was generated by Rcpp::compileAttributes
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

connect_ <- function(keywords, values) {
    .Call('rpg_connect_', PACKAGE = 'rpg', keywords, values)
}

#' Database connection utilities
#' 
#' Conection reporting and defaults
#' 
#' @param opts a libpq connection string
#' 
#' @details \code{ping} will ignore any keywords not directly
#' related to the database host (e.g., username, dbname) as it
#' does not connect; it only detect the server port is responding.
#' 
#' @return \code{ping} returns one of the following:
#' \tabular{ll}{
#' \code{PQPING_OK}  \tab Server reachable \cr
#' \code{PQPING_REJECT } \tab Server reachable but not accepting
#' connections \cr
#' \code{PQPING_NO_RESPONSE} \tab Server unreachable \cr
#' \code{PQPING_NO_ATTEMPT} \tab Connection string is nonsense \cr}
#' 
#' @author Timothy H. Keitt
#' 
#' @examples
#' \dontrun{
#' ping("connect_timeout = 3, host = www.keittlab.org")
#' connect()
#' get_conn_defaults()
#' set_conn_defaults(dbname = "test")
#' get_conn_defaults()
#' reset_conn_defaults()
#' get_conn_defaults()
#' get_conn_defaults(all = TRUE)
#' get_conn_info()
#' get_conn_error()
#' disconnect()}
#' 
#' @export ping
#' @rdname connection-utils
ping <- function(opts = "") {
    .Call('rpg_ping', PACKAGE = 'rpg', opts)
}

#' @details \code{disconnect} will free any query results as well
#' as clean up the connection data. It is called in the pakcage
#' \code{\link{.Last.lib}} function when exiting \code{R}.
#' 
#' @export disconnect
#' @rdname connection
disconnect <- function() {
    invisible(.Call('rpg_disconnect', PACKAGE = 'rpg'))
}

clean_up_all <- function() {
    invisible(.Call('rpg_clean_up_all', PACKAGE = 'rpg'))
}

#' @return get_conn_error: an error string
#' @export get_conn_error
#' @rdname connection-utils
get_conn_error <- function() {
    .Call('rpg_get_conn_error', PACKAGE = 'rpg')
}

get_conn_info_ <- function() {
    .Call('rpg_get_conn_info_', PACKAGE = 'rpg')
}

#' @return \code{result_dim} returns the number of tuples and fields
#' @rdname query
#' @export
result_dim <- function() {
    .Call('rpg_result_dim', PACKAGE = 'rpg')
}

get_tuple_info <- function() {
    .Call('rpg_get_tuple_info', PACKAGE = 'rpg')
}

#' PostgreSQL query
#' 
#' Issue a query to the current database connection
#' 
#' @param sql a query string
#' @param pars a character vector of substitution values
#' 
#' @return \code{query} returns:
#' \tabular{ll}{
#' PGRES_EMPTY_QUERY \tab The string sent to the server was empty \cr
#' PGRES_COMMAND_OK \tab Successful completion of a command returning no data \cr
#' PGRES_TUPLES_OK \tab Successful completion of a command returning data (such as a SELECT or SHOW) \cr
#' PGRES_COPY_OUT \tab Copy Out (from server) data transfer started \cr
#' PGRES_COPY_IN \tab Copy In (to server) data transfer started \cr
#' PGRES_BAD_RESPONSE \tab The server's response was not understood. \cr
#' PGRES_NONFATAL_ERROR \tab A nonfatal error (a notice or warning) occurred \cr
#' PGRES_FATAL_ERROR \tab A fatal error occurred \cr
#' PGRES_COPY_BOTH \tab Copy In/Out (to and from server) data transfer started. This is currently used only for streaming replication \cr}
#' 
#' @author Timothy H. Keitt
#' 
#' @seealso \code{\link{psql}}
#' 
#' @examples
#' \dontrun{
#' system("createdb rpgtesting")
#' connect("rpgtesting")
#' begin()
#' execute("DROP SCHEMA IF EXISTS rpgtesting CASCADE")
#' execute("CREATE SCHEMA rpgtesting")
#' execute("SET search_path TO rpgtesting")
#' execute("DROP TABLE IF EXISTS test")
#' execute("CREATE TABLE test (id integer, field text)")
#' query("INSERT INTO test VALUES ($1, $2)", c(1, "test"))
#' fetch("SELECT * FROM test")
#' result_dim()
#' fetch("SELECT * FROM testing")
#' query_error()
#' rollback()
#' disconnect()
#' system("dropdb rpgtesting")}
#' 
#' @rdname query
#' @export query
query <- function(sql = "", pars = NULL) {
    .Call('rpg_query', PACKAGE = 'rpg', sql, pars)
}

#' @return \code{query_error} returns an error string
#' @export query_error
#' @rdname query
query_error <- function() {
    .Call('rpg_query_error', PACKAGE = 'rpg')
}

fetch_matrix <- function() {
    .Call('rpg_fetch_matrix', PACKAGE = 'rpg')
}

fetch_dataframe <- function() {
    .Call('rpg_fetch_dataframe', PACKAGE = 'rpg')
}

#' PostgeSQL connection tracing
#' 
#' Functions to manage connection tracing
#' 
#' @param filename where to send the tracing
#' @param append if true, append to existing file
#' @param remove if true, unlink the tracing file
#' 
#' @details
#' PostgeSQL tracing lets you observe all information
#' passing between \code{rpg} and the database server.
#' 
#' \code{trace_conn} begins tracing and \code{untrace_conn}
#' stops tracing.
#' 
#' @author Timothy H. Keitt
#' 
#' @examples
#' \dontrun{
#' system("createdb rpgtesting")
#' connect("rpgtesting")
#' trace_conn()
#' list_tables()
#' dump_conn_trace(n = 40)
#' untrace_conn(remove = TRUE)
#' disconnect()
#' system("dropdb rpgtesting")}
#' @rdname tracing
#' @export
trace_conn <- function(filename = "", append = FALSE) {
    invisible(.Call('rpg_trace_conn', PACKAGE = 'rpg', filename, append))
}

#' @rdname tracing
#' @export
untrace_conn <- function(remove = FALSE) {
    invisible(.Call('rpg_untrace_conn', PACKAGE = 'rpg', remove))
}

#' @return \code{trace_filename}: the name of the file containing
#' trace information.
#' 
#' @rdname tracing
#' @export
trace_filename <- function() {
    .Call('rpg_trace_filename', PACKAGE = 'rpg')
}

#' @param all if false return only defaults with settings
#' @details \code{get_conn_defaults} returns a data frame containing
#' all of the possible connection string keywords, the names of environment
#' variables used to override the defaults, the compiled in default value
#' and the current value of the keyword.
#' @return \code{get_conn_defaults}: a data frame with defaults listed
#' @rdname connection-utils
#' @export
get_conn_defaults <- function(all = FALSE) {
    .Call('rpg_get_conn_defaults', PACKAGE = 'rpg', all)
}

#' Miscelaneous functions
#' 
#' Various utility functions
#' 
#' @author Timothy H. Keitt
#' 
#' @examples
#' \dontrun{
#' # try connecting to default database
#' system("createdb rpgtesting")
#' connect("rpgtesting")
#' begin()
#' 
#' libpq_version()
#' encrypt_password("test", "tester")
#' get_encoding()
#' set_encoding("UTF8")
#' set_error_verbosity("terse")
#' set_error_verbosity("verbose")
#' set_error_verbosity("default")
#'
#' # cleanup
#' rollback()
#' disconnect()
#' system("dropdb rpgtesting")}
#' 
#' @rdname misc
#' @export
libpq_version <- function() {
    .Call('rpg_libpq_version', PACKAGE = 'rpg')
}

#' @param passwd the password
#' @param user the user name
#' @rdname misc
#' @export
encrypt_password <- function(passwd, user) {
    .Call('rpg_encrypt_password', PACKAGE = 'rpg', passwd, user)
}

#' @rdname misc
#' @export
get_encoding <- function() {
    .Call('rpg_get_encoding', PACKAGE = 'rpg')
}

#' @param encoding the character encoding
#' @rdname misc
#' @export
set_encoding <- function(encoding) {
    .Call('rpg_set_encoding', PACKAGE = 'rpg', encoding)
}

#' @param verbosity one of "terse", "default", "verbose"
#' @rdname misc
#' @export
set_error_verbosity <- function(verbosity) {
    invisible(.Call('rpg_set_error_verbosity', PACKAGE = 'rpg', verbosity))
}

check_transaction <- function() {
    .Call('rpg_check_transaction', PACKAGE = 'rpg')
}

#' Prepared queries
#' 
#' Prepare and execute queries
#' 
#' @param sql a valid query string
#' @param name an optional statement name
#' 
#' @details These functions allow the preparation of an incomplete
#' SQL statement and then application of that statement to a arrays
#' of input parameters.
#' 
#' If the number of parameters supplied to \code{execute_prepared} is a
#' multiple of the number of open parameters in query prepared
#' using \code{prepare}, then the prepared query will be executed
#' repeatedly for each successive set of parameters. This repeated
#' execution loop is evaluted in C++ and so is quite fast. The
#' supplied parameter values will be coerced to a matrix of the
#' appropriate dimensions. If the number of supplied parameter
#' values is not an even multiple of the number of parameters
#' specified in the prepared statement, an error condition is raised.
#' The passed parameters will be coerced to character strings.
#' 
#' If you supply a \code{name}, then you must use the same name
#' when calling \code{execute_prepared}. This allows multiple prepared queries.
#' If you do not supply a name, each call to \code{prepare} will
#' overwrite the previous prepared statement. The lifetime of
#' a prepared statement is the lifetime of the current connection
#' or transaction.
#' 
#' @note One can use pure SQL to achieve the same result and
#' \code{execute_prepared} will work on any prepared statement regardless
#' of whether it was defined using \code{prepare} or directly with
#' \code{\link{query}}.
#' 
#' It is generally a good idea to wrap \code{prepare}--\code{execute_prepared}
#' in a transaction. If not in a transaction, you cannot rollback any updates
#' and it will be much slower as PostgreSQL initiates a transaction-per-query
#' by default.
#' 
#' @return A status string.
#' 
#' In the case of \code{execute_prepared} the status of the last command
#' executed. In case of error, the loop will terminate prematurely and
#' the status string will contain the error message.
#' 
#' @author Timothy H. Keitt
#' 
#' @examples
#' \dontrun{
#' # try connecting to default database
#' system("createdb rpgtesting")
#' connect("rpgtesting")
#' begin()
#' 
#' # write data frame contents
#' data(mtcars)
#' write_table(mtcars)
#' 
#' # delete the rows
#' query("truncate mtcars")
#' read_table(mtcars)
#' 
#' # use prepare-execute to write rows
#' pars = paste0("$", 1:11, collapse = ", ")
#' sql = paste0("INSERT INTO mtcars VALUES (", pars, ")", collapse = " ")
#' prepare(sql, "test_statement")
#' execute_prepared(mtcars, "test_statement")
#' read_table(mtcars, limit = 5)
#' 
#' # cleanup
#' rollback()
#' disconnect()
#' system("dropdb rpgtesting")}
#' 
#' @rdname prepare
#' @export
prepare <- function(sql, name = "") {
    .Call('rpg_prepare', PACKAGE = 'rpg', sql, name)
}

execute_prepared_ <- function(pars, name = "") {
    .Call('rpg_execute_prepared_', PACKAGE = 'rpg', pars, name)
}

num_prepared_params <- function(name = "") {
    .Call('rpg_num_prepared_params', PACKAGE = 'rpg', name)
}

#' Multiple PostgreSQL connections
#' 
#' Manage multiple connections on a stack
#' 
#' @details
#' These functions allow you to store multiple connections on a stack. They are
#' only used for their side-effects. \code{\link{rpg}} stores an active connection
#' pointer internally. This pointer can be moved onto the stack and manipulated.
#' Once on the stack, the pointer is no longer active. You must use
#' \code{swap_conn} or \code{pop_conn} to reactive a pushed connection, or call
#' \code{\link{connect}} to create a new active connection.
#' 
#' \code{push_conn} pushes the current connection onto the connection stack
#' leaving the active connection null.
#' 
#' @examples
#' \dontrun{
#' # make some databases
#' dbs = paste0("rpgdb", 1:3)
#' lapply(paste("createdb", dbs), system)
#'
#' # connect
#' connect(dbname = dbs[1]); push_conn()
#' connect(dbname = dbs[2]); push_conn()
#' connect(dbname = dbs[3]); push_conn()
#' 
#' show_conn_stack()
#' rotate_stack()
#' show_conn_stack()
#' rotate_stack(2)
#' show_conn_stack()
#' pop_conn()
#' show_conn_stack()
#' get_conn_info("dbname")
#' swap_conn()
#' show_conn_stack()
#' get_conn_info("dbname")
#' pop_conn()
#' show_conn_stack()
#' pop_conn()
#' show_conn_stack()
#' disconnect()
#' connect()
#' lapply(paste("dropdb", dbs), system)}
#' 
#' @rdname stack
#' @export
push_conn <- function() {
    invisible(.Call('rpg_push_conn', PACKAGE = 'rpg'))
}

#' @details
#' \code{pop_conn} pops a connection off the stack and makes it active. Whatever
#' connection was active when \code{pop_conn} is called will be disconnected and
#' cleared. Use \code{swap_conn} to preserve the active connection.
#' 
#' @rdname stack
#' @export
pop_conn <- function() {
    invisible(.Call('rpg_pop_conn', PACKAGE = 'rpg'))
}

#' @details
#' \code{swap_conn} swaps the active connection with the connection on the top
#' of the stack. If the stack is empty, the connection is swapped with a null
#' connection.
#' 
#' @rdname stack
#' @export
swap_conn <- function() {
    invisible(.Call('rpg_swap_conn', PACKAGE = 'rpg'))
}

#' @param n number of shifts
#' @details
#' \code{rotate_stack} moves the bottom of the stack to the top.
#' 
#' @rdname stack
#' @export
rotate_stack <- function(n = 1L) {
    invisible(.Call('rpg_rotate_stack', PACKAGE = 'rpg', n))
}

#' @details
#' \code{show_conn_stack} returns a data frame with information about the
#' connections on the stack.
#' 
#' @rdname stack
#' @export
show_conn_stack <- function() {
    .Call('rpg_show_conn_stack', PACKAGE = 'rpg')
}

#' Asynchronous query processing
#' 
#' Manage an asynchronous query
#' 
#' @param sql a query string
#' @param pars a vector of parameters
#' 
#' @details
#' These functions expose the asynchronous query interface from
#' \code{libpq}. The function \code{async_query} issues a query. Its
#' call is identical to \code{\link{query}} except that it will return
#' immediately. When the issued command is ready, the function
#' \code{async_status} will return a query status object exactly
#' as \code{\link{query}}. Otherwise it will return \code{"BUSY"} to
#' indicate the server has not finished or \code{"DONE"} to indicate
#' there is nothing more to fetch.
#' 
#' If \code{async_status} does not return \code{"DONE"}, then
#' you should call \code{finish_async} to free pending results. Note
#' that a call to \code{finish_async} may block until the server is finished
#' processing the command. It calls \code{cancel} internally but there is
#' no guarantee the command will abort.
#' 
#' @return
#' \code{async_query}: true if query was successfully sent (an invalid query
#' will still return true)
#' 
#' \code{async_status}: a results status object, possibly indicating an
#' invalid query
#' 
#' \code{is_busy}: a boolean
#' 
#' @note In practice, you will be much better off using \code{\link{cursor}}
#' as that will usually return very quickly even for large queries, and has
#' the advantage of retrieving the results in chunks. You can call \code{cancel}
#' while a cursor is active. The cursor will return \code{PGRES_FATAL_ERROR} if
#' the \code{cancel} is effective. Alternately, issuing any query that sets the
#' result status will have the same effect as \code{finish_async}.
#' 
#' @author Timothy H. Keitt
#'  
#' @examples
#' \dontrun{
#' # create a database
#' system("createdb rpgtesting")
#' connect("rpgtesting")
#' begin()
#' 
#' # write data frame contents
#' data(mtcars)
#' write_table(mtcars)
#' 
#' # async processing on smallish result
#' # this wont be interesting if your machine is very fast
#' async_query("SELECT a.* FROM mtcars a, mtcars b")
#' repeat
#' {
#'   status = async_status()
#'   if ( status != "BUSY" ) break
#'   cat("busy...\n")
#'   Sys.sleep(1)
#' }
#' print(status)
#' head(fetch())
#' finish_async()
#' Sys.sleep(1)
#' 
#' # async processing on larger result
#' async_query("SELECT a.* FROM mtcars a, mtcars b, mtcars c")
#' count = 0
#' repeat
#' {
#'   status = async_status()
#'   if ( status == "BUSY" )
#'   {
#'     if ( count > 2 )
#'     {
#'       cat("calling cancel...\n")
#'       cancel()
#'     }
#'   }
#'   else break
#'   cat("busy... \n")
#'   Sys.sleep(1)
#'   count = count + 1
#' }
#' print(status)
#' finish_async()
#' 
#' # you can run multiple queries with async_query
#' rollback(); begin()
#' write_table(mtcars)
#' sql1 = "SELECT mpg FROM mtcars LIMIT 3"
#' sql2 = "SELECT cyl FROM mtcars LIMIT 4"
#' async_query(paste(sql1, sql2, sep = "; "))
#' while ( async_status() == "BUSY" ) NULL
#' fetch()
#' while ( is_busy() ) NULL
#' async_status()
#' fetch()
#' finish_async()
#' 
#' # issue an async query and come back later
#' async_query(sql1)
#' push_conn()
#' connect("rpgtesting")
#' 
#' # fails because of transaction isolation
#' fetch(sql2)
#' pop_conn()
#' async_status()
#' 
#' # results from sql1
#' fetch()
#' 
#' # this is automatic if you issue new queries
#' finish_async()
#' 
#' # cleanup
#' rollback()
#' disconnect()
#' system("dropdb rpgtesting")} 
#' 
#' @export
#' @rdname async
async_query <- function(sql = "", pars = NULL) {
    .Call('rpg_async_query', PACKAGE = 'rpg', sql, pars)
}

#' @details Any pending results will be lost if you call \code{\link{query}},
#' \code{\link{execute}} or \code{\link{fetch}} with a \code{sql} string prior
#' to \code{async_query} returning \code{DONE}. If you need to issue queries
#' while waiting on an async call, then use \code{\link{push_conn}} to save
#' the query state, \code{\link{connect}} to make a new connetion, and then
#' \code{\link{pop_conn}} followed by \code{async_status}.
#' @export
#' @rdname async
async_status <- function() {
    .Call('rpg_async_status', PACKAGE = 'rpg')
}

#' @details \code{is_busy} is a slightly faster shortcut to check whether the
#' server has completed the query. You must still call \code{async_status} to
#' fetch the results.
#' @export
#' @rdname async
is_busy <- function() {
    .Call('rpg_is_busy', PACKAGE = 'rpg')
}

#' @param stop_on_error call \code{\link{stop}} if cancel request cannot be issued
#' @export
#' @rdname async
cancel <- function(stop_on_error = TRUE) {
    invisible(.Call('rpg_cancel', PACKAGE = 'rpg', stop_on_error))
}

#' @export
#' @rdname async
finish_async <- function() {
    invisible(.Call('rpg_finish_async', PACKAGE = 'rpg'))
}

exec_param_serialize <- function(sql, obj) {
    .Call('rpg_exec_param_serialize', PACKAGE = 'rpg', sql, obj)
}

fetch_stowed <- function(sql, par) {
    .Call('rpg_fetch_stowed', PACKAGE = 'rpg', sql, par)
}

