\name{sparseLTSGrid}
\alias{print.optSparseLTSGrid}
\alias{print.sparseLTSGrid}
\alias{sparseLTSGrid}
\alias{sparseLTSGrid.default}
\alias{sparseLTSGrid.formula}
\title{Sparse least trimmed squares regression}
\usage{
  sparseLTSGrid(x, ...)

  \method{sparseLTSGrid}{formula} (formula, data, ...)

  \method{sparseLTSGrid}{default} (x, y, lambda,
    mode = c("lambda", "fraction"), ...,
    crit = c("BIC", "PE"), splits = foldControl(),
    cost = rtmspe, costArgs = list(),
    selectBest = c("hastie", "min"), seFactor = 1,
    ncores = 1, cl = NULL, seed = NULL, model = TRUE)
}
\arguments{
  \item{formula}{a formula describing the model.}

  \item{data}{an optional data frame, list or environment
  (or object coercible to a data frame by
  \code{\link{as.data.frame}}) containing the variables in
  the model.  If not found in data, the variables are taken
  from \code{environment(formula)}, typically the
  environment from which \code{sparseLTS} is called.}

  \item{x}{a numeric matrix containing the predictor
  variables.}

  \item{y}{a numeric vector containing the response
  variable.}

  \item{lambda}{a numeric vector of non-negative numeric
  values to be used as penalty parameter.}

  \item{mode}{a character string specifying the type of
  penalty parameter.  If \code{"lambda"}, \code{lambda}
  gives the grid of values for the penalty parameter
  directly.  If \code{"fraction"}, the smallest value of
  the penalty parameter that sets all coefficients to 0 is
  first estimated based on bivariate winsorization, then
  \code{lambda} gives the fractions of that estimate to be
  used (hence all values of \code{lambda} should be in the
  interval [0,1] in that case).}

  \item{\dots}{additional arguments to be passed down,
  eventually to \code{\link{sparseLTS}}.}

  \item{crit}{a character string specifying the optimality
  criterion to be used for selecting the final model.
  Possible values are \code{"BIC"} for the Bayes
  information criterion and \code{"PE"} for
  resampling-based prediction error estimation.}

  \item{splits}{an object giving data splits to be used for
  prediction error estimation (see
  \code{\link[perry]{perryTuning}}).}

  \item{cost}{a cost function measuring prediction loss
  (see \code{\link[perry]{perryTuning}} for some
  requirements).  The default is to use the root trimmed
  mean squared prediction error (see
  \code{\link[perry]{cost}}).}

  \item{costArgs}{a list of additional arguments to be
  passed to the prediction loss function \code{cost}.}

  \item{selectBest,seFactor}{arguments specifying a
  criterion for selecting the best model (see
  \code{\link[perry]{perryTuning}}).  The default is to use
  a one-standard-error rule.}

  \item{ncores}{a positive integer giving the number of
  processor cores to be used for parallel computing (the
  default is 1 for no parallelization).  If this is set to
  \code{NA}, all available processor cores are used.  For
  prediction error estimation, parallel computing is
  implemented on the \R level using package \pkg{parallel}.
  Otherwise parallel computing is implemented on the C++
  level via OpenMP (\url{http://openmp.org/}).}

  \item{cl}{a \pkg{parallel} cluster for parallel computing
  as generated by \code{\link[parallel]{makeCluster}}.
  This is preferred over \code{ncores} for prediction error
  estimation, in which case \code{ncores} is only used on
  the C++ level for computing the optimal model.}

  \item{seed}{optional initial seed for the random number
  generator (see \code{\link{.Random.seed}}).  On parallel
  \R worker processes for prediction error estimation,
  random number streams are used and the seed is set via
  \code{\link{clusterSetRNGStream}}.}

  \item{model}{a logical indicating whether the data
  \code{x} and \code{y} should be added to the return
  object.  If \code{intercept} is \code{TRUE}, a column of
  ones is added to \code{x} to account for the intercept.}
}
\value{
  An object of class \code{"sparseLTSGrid"} (if
  \code{crit="BIC"}) or \code{"optSparseLTSGrid"} (if
  \code{crit="PE"}) with the following components:

  \item{best}{an integer matrix in which each column
  contains the best subset of \eqn{h} observations found
  and used for computing the corresponding raw estimates
  (\code{"sparseLTSGrid"}); or an integer vector containing
  the best subset for the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{objective}{a numeric vector giving the values of
  the sparse LTS objective function, i.e., the
  \eqn{L_{1}}{L1} penalized sum of the \eqn{h} smallest
  squared residuals from the raw fits
  (\code{"sparseLTSGrid"}); or a numeric giving the value
  of the objective function from the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{coefficients}{a numeric matrix in which each column
  contains the coefficient estimates of the corresponding
  reweighted fit (\code{"sparseLTSGrid"}); or a numeric
  vector containing the coefficients of the optimal
  reweighted fit (\code{"optSparseLTSGrid"}).}

  \item{fitted.values}{a numeric matrix in which each
  column contains the fitted values of the response of the
  corresponding reweighted fit (\code{"sparseLTSGrid"}); or
  a numeric vector containing the fitted values of the
  optimal reweighted fit (\code{"optSparseLTSGrid"}).}

  \item{residuals}{a numeric matrix in which each column
  contains the residuals of the corresponding reweighted
  fit (\code{"sparseLTSGrid"}); or a numeric vector
  containing the residuals of the optimal reweighted fit
  (\code{"optSparseLTSGrid"}).}

  \item{center}{a numeric vector giving the robust center
  estimates of the residuals from the reweighted fits
  (\code{"sparseLTSGrid"}); or a numeric giving the robust
  center estimate of the optimal reweighted fit
  (\code{"optSparseLTSGrid"}).}

  \item{scale}{a numeric vector giving the robust scale
  estimates of the residuals from the reweighted fits
  (\code{"sparseLTSGrid"}); or a numeric giving the robust
  scale estimate of the optimal reweighted fit
  (\code{"optSparseLTSGrid"}).}

  \item{lambda}{a numeric vector giving the values of the
  penalty parameter.}

  \item{intercept}{a logical indicating whether the model
  includes a constant term.}

  \item{alpha}{a numeric value giving the percentage of the
  residuals for which the \eqn{L_{1}}{L1} penalized sum of
  squares was minimized.}

  \item{quan}{the number \eqn{h} of observations used to
  compute the raw estimates.}

  \item{cnp2}{a numeric vector giving the consistency
  factors applied to the scale estimates of the residuals
  from the reweighted fits (\code{"sparseLTSGrid"}); or a
  numeric giving the consistency factor from the optimal
  reweighted fit (\code{"optSparseLTSGrid"}).}

  \item{wt}{an integer matrix in which each column contains
  binary weights that indicate outliers from the
  corresponding reweighted fit, i.e., the weights are
  \eqn{1} for observations with reasonably small reweighted
  residuals and \eqn{0} for observations with large
  reweighted residuals (\code{"sparseLTSGrid"}); or an
  integer vector containing the outlier weights of the
  optimal reweighted fit (\code{"optSparseLTSGrid"}).}

  \item{df}{an integer vector giving the degrees of freedom
  of the obtained reweighted model fits, i.e., the number
  of nonzero coefficient estimates
  (\code{"sparseLTSGrid"}); or an integer giving the
  degrees of freedom of the optimal reweighted fit
  (\code{"optSparseLTSGrid"}).}

  \item{raw.coefficients}{a numeric matrix in which each
  column contains the coefficient estimates of the
  corresponding raw fit (\code{"sparseLTSGrid"}); or a
  numeric vector containing the coefficients of the optimal
  raw fit (\code{"optSparseLTSGrid"}).}

  \item{raw.fitted.values}{a numeric matrix in which each
  column contains the fitted values of the response of the
  corresponding raw fit (\code{"sparseLTSGrid"}); or a
  numeric vector containing the fitted values of the
  optimal raw fit (\code{"optSparseLTSGrid"}).}

  \item{raw.residuals}{a numeric matrix in which each
  column contains the residuals of the corresponding raw
  fit (\code{"sparseLTSGrid"}); or a numeric vector
  containing the residuals of the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{raw.center}{a numeric vector giving the robust
  center estimates of the residuals from the raw fits
  (\code{"sparseLTSGrid"}); or a numeric giving the robust
  center estimate of the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{raw.scale}{a numeric vector giving the robust scale
  estimates of the residuals from the raw fits
  (\code{"sparseLTSGrid"}); or a numeric giving the robust
  scale estimate of the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{raw.cnp2}{a numeric vector giving the consistency
  factors applied to the scale estimates of the residuals
  from the raw fits (\code{"sparseLTSGrid"}); or a numeric
  giving the consistency factor from the optimal raw fit
  (\code{"optSparseLTSGrid"}).}

  \item{raw.wt}{an integer matrix in which each column
  contains binary weights that indicate outliers of the
  corresponding raw fit, i.e., the weights used for the
  reweighted fits (\code{"sparseLTSGrid"}); or an integer
  vector containing the outlier weights of the optimal raw
  fit (\code{"optSparseLTSGrid"}).}

  \item{crit}{a character string specifying the optimality
  criterion used for selecting the optimal model.}

  \item{critValues}{a numeric vector containing the values
  of the optimality criterion from the reweighted fits
  (\code{"sparseLTSGrid"}); or an object of class
  \code{"perrySparseLTSGrid"} (inheriting from
  \code{"\link[perry]{perryTuning}"}) that contains the
  estimated prediction errors of the reweighted and raw
  fits (\code{"optSparseLTSGrid"}).}

  \item{sOpt}{an integer giving the optimal reweighted fit
  (only \code{"sparseLTSGrid"}).}

  \item{raw.critValues}{a numeric vector containing the
  values of the optimality criterion from the raw fits
  (only \code{"sparseLTSGrid"}).}

  \item{raw.sOpt}{an integer giving the optimal raw fit
  (only \code{"sparseLTSGrid"}).}

  \item{x}{the predictor matrix (if \code{model} is
  \code{TRUE}).}

  \item{y}{the response variable (if \code{model} is
  \code{TRUE}).}

  \item{call}{the matched function call.}
}
\description{
  Compute least trimmed squares regression with an
  \eqn{L_{1}}{L1} penalty on the regression coefficients,
  which allows for sparse model estimates, over a grid of
  values for the penalty parameter.
}
\note{
  Package \pkg{robustHD} has a built-in back end for sparse
  least trimmed squares using the C++ library Armadillo.
  Another back end is available through package
  \pkg{sparseLTSEigen}, which uses the C++ library Eigen.
  The latter is faster, currently does not work on 32-bit
  \R for Windows.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points

## fit sparse LTS models over a grid of values for lambda
frac <- seq(0.25, 0.05, by = -0.05)
sparseLTSGrid(x, y, lambda = frac, mode = "fraction")
}
\author{
  Andreas Alfons
}
\seealso{
  \code{\link{sparseLTS}},
  \code{\link[=coef.sparseLTS]{coef.sparseLTSGrid}},
  \code{\link[=fitted.sparseLTS]{fitted.sparseLTSGrid}},
  \code{\link[=plot.seqModel]{plot.sparseLTSGrid}},
  \code{\link{diagnosticPlot}},
  \code{\link[=predict.sparseLTS]{predict.sparseLTSGrid}},
  \code{\link[=residuals.sparseLTS]{residuals.sparseLTSGrid}},
  \code{\link[=wt.sparseLTS]{wt.sparseLTSGrid}},
}
\keyword{multivariate}
\keyword{robust}

