% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source.R
\name{rust_source}
\alias{rust_source}
\alias{rust_function}
\title{Compile Rust code and call from R}
\usage{
rust_source(
  file,
  code = NULL,
  module_name = "rextendr",
  dependencies = NULL,
  patch.crates_io = getOption("rextendr.patch.crates_io"),
  profile = c("dev", "release", "perf"),
  toolchain = getOption("rextendr.toolchain"),
  extendr_deps = NULL,
  features = NULL,
  env = parent.frame(),
  use_extendr_api = TRUE,
  generate_module_macro = TRUE,
  cache_build = TRUE,
  quiet = FALSE,
  use_rtools = TRUE,
  use_dev_extendr = FALSE
)

rust_function(
  code,
  extendr_fn_options = NULL,
  env = parent.frame(),
  quiet = FALSE,
  use_dev_extendr = FALSE,
  ...
)
}
\arguments{
\item{file}{Input rust file to source.}

\item{code}{Input rust code, to be used instead of \code{file}.}

\item{module_name}{Name of the module defined in the Rust source via
\verb{extendr_module!}. Default is \code{"rextendr"}. If \code{generate_module_macro} is \code{FALSE}
or if \code{file} is specified, should \emph{match exactly} the name of the module defined in the source.}

\item{dependencies}{Character vector of dependencies lines to be added to the
\code{Cargo.toml} file.}

\item{patch.crates_io}{Character vector of patch statements for crates.io to
be added to the \code{Cargo.toml} file.}

\item{profile}{Rust profile. Can be either \code{"dev"}, \code{"release"} or \code{"perf"}.
The default, \code{"dev"}, compiles faster but produces slower code.}

\item{toolchain}{Rust toolchain. The default, \code{NULL}, compiles with the
system default toolchain. Accepts valid Rust toolchain qualifiers,
such as \code{"nightly"}, or (on Windows) \code{"stable-msvc"}.}

\item{extendr_deps}{Versions of \verb{extendr-*} crates. Defaults to \code{rextendr.extendr_deps} option
(\code{list(`extendr-api` = "*")}) if \code{use_dev_extendr} is not \code{TRUE},
otherwise, uses \code{rextendr.extendr_dev_deps} option
(\code{list(`extendr-api` = list(git = "https://github.com/extendr/extendr")}).}

\item{features}{A vector of \code{extendr-api} features that should be enabled.
Supported values are \code{"ndarray"}, \code{"num-complex"}, \code{"serde"}, and \code{"graphics"}.
Unknown features will produce a warning if \code{quiet} is not \code{TRUE}.}

\item{env}{The R environment in which the wrapping functions will be defined.}

\item{use_extendr_api}{Logical indicating whether
\verb{use extendr_api::prelude::*;} should be added at the top of the Rust source
provided via \code{code}. Default is \code{TRUE}. Ignored for Rust source provided
via \code{file}.}

\item{generate_module_macro}{Logical indicating whether the Rust module
macro should be automatically generated from the code. Default is \code{TRUE}.
Ignored for Rust source provided via \code{file}. The macro generation is done
with \code{\link[=make_module_macro]{make_module_macro()}} and it may fail in complex cases. If something
doesn't work, try calling \code{\link[=make_module_macro]{make_module_macro()}} on your code to see whether
the generated macro code has issues.}

\item{cache_build}{Logical indicating whether builds should be cached between
calls to \code{\link[=rust_source]{rust_source()}}.}

\item{quiet}{Logical indicating whether compile output should be generated or not.}

\item{use_rtools}{Logical indicating whether to append the path to Rtools
to the \code{PATH} variable on Windows using the \code{RTOOLS40_HOME} environment
variable (if it is set). The appended path depends on the process
architecture. Does nothing on other platforms.}

\item{use_dev_extendr}{Logical indicating whether to use development version of
\code{extendr}. Has no effect if \code{extendr_deps} are set.}

\item{extendr_fn_options}{A list of extendr function options that are inserted into
\verb{#[extendr(...)]} attribute}

\item{...}{Other parameters handed off to \code{\link[=rust_source]{rust_source()}}.}
}
\value{
The result from \code{\link[=dyn.load]{dyn.load()}}, which is an object of class \code{DLLInfo}.
See \code{\link[=getLoadedDLLs]{getLoadedDLLs()}} for more details.
}
\description{
\code{\link[=rust_source]{rust_source()}} compiles and loads a single Rust file for use in R. \code{\link[=rust_function]{rust_function()}}
compiles and loads a single Rust function for use in R.
}
\examples{
\dontrun{
# creating a single rust function
rust_function("fn add(a:f64, b:f64) -> f64 { a + b }")
add(2.5, 4.7)

# creating multiple rust functions at once
code <- r"(
#[extendr]
fn hello() -> &'static str {
    "Hello, world!"
}

#[extendr]
fn test( a: &str, b: i64) {
    rprintln!("Data sent to Rust: {}, {}", a, b);
}
)"

rust_source(code = code)
hello()
test("a string", 42)


# use case with an external dependency: a function that converts
# markdown text to html, using the `pulldown_cmark` crate.
code <- r"(
  use pulldown_cmark::{Parser, Options, html};

  #[extendr]
  fn md_to_html(input: &str) -> String {
    let mut options = Options::empty();
    options.insert(Options::ENABLE_TABLES);
    let parser = Parser::new_ext(input, options);
    let mut output = String::new();
    html::push_html(&mut output, parser);
    output
  }
)"
rust_source(
  code = code,
  dependencies = list(`pulldown-cmark` = "0.8")
)

md_text <- "# The story of the fox
The quick brown fox **jumps over** the lazy dog.
The quick *brown fox* jumps over the lazy dog."

md_to_html(md_text)
}
}
