#' @name exportNextRecordName
#' 
#' @title Generate Next Record Name from a REDCap Database
#' @description To be used by projects with record auto-numbering enabled, 
#' this method exports the next potential record ID for a project. 
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#'   
#' @details 
#' It generates the next record name by determining the current maximum numerical 
#' record ID and then incrementing it by one.
#' 
#' @return 
#' Returns the maximum integer record ID + 1.
#' 
#' @section REDCap API Documentation:
#' 
#' NOTE: This method does not create a new record, but merely determines 
#' what the next record name would be.
#' 
#' @section REDCap Version:
#' 8.1.8+ 
#' 
#' @author Xuefei Jia
#' 
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#'
#' @export

exportNextRecordName <- function(rcon, 
                                 ...){
  UseMethod("exportNextRecordName")
}

#' @rdname exportNextRecordName
#' @export


exportNextRecordName.redcapApiConnection <- function(rcon, 
                                                     ...,
                                                     error_handling = getOption("redcap_error_handling"), 
                                                     config         = list(), 
                                                     api_param      = list()){
   ##################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Make the Body List
  
  body <- list(token = rcon$token, 
               content = 'generateNextRecordName')
  
  body <- body[length(body) > 0]
  
   ##################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) redcap_error(response, error_handling)
  
  as.numeric(rawToChar(response$content))
}
