% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rotate_3d.R
\name{rotate_3d}
\alias{rotate_3d}
\title{Rotate the values around an origin in 3 dimensions}
\usage{
rotate_3d(
  data,
  x_col,
  y_col,
  z_col,
  x_deg = 0,
  y_deg = 0,
  z_deg = 0,
  suffix = "_rotated",
  origin = NULL,
  origin_fn = NULL,
  keep_original = TRUE,
  degrees_col_name = ".degrees",
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{x_col, y_col, z_col}{Name of x/y/z column in \code{`data`}. All must be specified.}

\item{x_deg, y_deg, z_deg}{Degrees to rotate values around the x/y/z-axis counterclockwise. In \code{[-360, 360]}.
Can be \code{vector}s with multiple degrees.

\code{`x_deg`} is \emph{roll}. \code{`y_deg`} is \emph{pitch}. \code{`z_deg`} is \emph{yaw}.}

\item{suffix}{Suffix to add to the names of the generated columns.

Use an empty string (i.e. \code{""}) to overwrite the original columns.}

\item{origin}{Coordinates of the origin to rotate around.
\code{Vector} with 3 elements (i.e. origin_x, origin_y, origin_z).
Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{keep_original}{Whether to keep the original columns. (Logical)

Some columns may have been overwritten, in which case only the newest versions are returned.}

\item{degrees_col_name}{Name of new column with the degrees. If \code{NULL}, no column is added.

Also adds a string version with the same name + \code{"_str"}, making it easier to group by the degrees
when plotting multiple rotations.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with six new columns containing
the rotated x-,y- and z-values and the degrees and origin coordinates.
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

The values are rotated counterclockwise around a specified origin.

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and rotating around e.g. the centroid
of each group.
}
\details{
Applies the following rotation matrix:\tabular{llll}{
   [ \eqn{cos \alpha cos \beta} \tab , \eqn{cos \alpha sin \beta sin \gamma - sin \alpha cos \gamma} \tab , \eqn{cos \alpha sin \beta cos \gamma + sin \alpha sin \gamma} \tab ] \cr
   [ \eqn{sin \alpha cos \beta} \tab , \eqn{sin \alpha sin \beta sin \gamma + cos \alpha cos \gamma} \tab , \eqn{sin \alpha sin \beta cos \gamma - cos \alpha sin \gamma} \tab ] \cr
   [ \eqn{-sin \beta} \tab , \eqn{cos \beta sin \gamma } \tab , \eqn{cos \beta cos \gamma} \tab ] \cr
}


Where \eqn{\alpha =} \code{`z_deg`} in radians, \eqn{\beta =} \code{`y_deg`} in radians, \eqn{\gamma =} \code{`x_deg`} in radians.

As specified at \href{https://en.wikipedia.org/wiki/Rotation_matrix}{Wikipedia/Rotation_matrix}.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
library(ggplot2)

# Set seed
set.seed(3)

# Create a data frame
df <- data.frame(
  "x" = 1:12,
  "y" = c(1:4, 9:12, 15:18),
  "z" = runif(12),
  "g" = rep(1:3, each = 4)
)

# Rotate values 45 degrees around x-axis at (0, 0, 0)
rotate_3d(df, x_col = "x", y_col = "y", z_col = "z", x_deg = 45, origin = c(0, 0, 0))

# Rotate all axes around the centroid
df_rotated <- df \%>\%
  rotate_3d(
    x_col = "x",
    y_col = "y",
    z_col = "z",
    x_deg = c(0, 72, 144, 216, 288),
    y_deg = c(0, 72, 144, 216, 288),
    z_deg = c(0, 72, 144, 216, 288),
    origin_fn = centroid
  )
df_rotated

# Plot rotations
ggplot(df_rotated, aes(x = x_rotated, y = y_rotated, color = .degrees_str, alpha = z_rotated)) +
  geom_vline(xintercept = mean(df$x), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_hline(yintercept = mean(df$y), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_line(alpha = .4) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "degrees", alpha = "z (opacity)")

\dontrun{
# Plot 3d with plotly
plotly::plot_ly(
  x = df_rotated$x_rotated,
  y = df_rotated$y_rotated,
  z = df_rotated$z_rotated,
  type = "scatter3d",
  mode = "markers",
  color = df_rotated$.degrees_str
)
}
\donttest{

# Rotate randomly around all axes
df_rotated <- df \%>\%
  rotate_3d(
    x_col = "x",
    y_col = "y",
    z_col = "z",
    x_deg = round(runif(10, min = 0, max = 360)),
    y_deg = round(runif(10, min = 0, max = 360)),
    z_deg = round(runif(10, min = 0, max = 360)),
    origin_fn = centroid
  )
df_rotated

# Plot rotations
ggplot(df_rotated, aes(x = x_rotated, y = y_rotated, color = .degrees_str, alpha = z_rotated)) +
  geom_vline(xintercept = mean(df$x), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_hline(yintercept = mean(df$y), size = 0.2, alpha = .4, linetype = "dashed") +
  geom_line(alpha = .4) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "degrees", alpha = "z (opacity)")

}
\dontrun{
# Plot 3d with plotly
plotly::plot_ly(
  x = df_rotated$x_rotated,
  y = df_rotated$y_rotated,
  z = df_rotated$z_rotated,
  type = "scatter3d",
  mode = "markers",
  color = df_rotated$.degrees_str
)
}
\donttest{

# Rotate around group centroids
df_grouped <- df \%>\%
  dplyr::group_by(g) \%>\%
  rotate_3d(
    x_col = "x",
    y_col = "y",
    z_col = "z",
    x_deg = c(0, 72, 144, 216, 288),
    y_deg = c(0, 72, 144, 216, 288),
    z_deg = c(0, 72, 144, 216, 288),
    origin_fn = centroid
  )

# Plot A and A rotated around group centroids
ggplot(df_grouped, aes(x = x_rotated, y = y_rotated, color = .degrees_str, alpha = z_rotated)) +
  geom_point() +
  theme_minimal() +
  labs(x = "x", y = "y", color = "degrees", alpha = "z (opacity)")

}
\dontrun{
# Plot 3d with plotly
plotly::plot_ly(
  x = df_grouped$x_rotated,
  y = df_grouped$y_rotated,
  z = df_grouped$z_rotated,
  type = "scatter3d",
  mode = "markers",
  color = df_grouped$.degrees_str
)
}
}
\seealso{
Other mutate functions: 
\code{\link{apply_transformation_matrix}()},
\code{\link{cluster_groups}()},
\code{\link{dim_values}()},
\code{\link{expand_distances_each}()},
\code{\link{expand_distances}()},
\code{\link{flip_values}()},
\code{\link{roll_values}()},
\code{\link{rotate_2d}()},
\code{\link{shear_2d}()},
\code{\link{shear_3d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}

Other rotation functions: 
\code{\link{rotate_2d}()},
\code{\link{swirl_2d}()},
\code{\link{swirl_3d}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{mutate functions}
\concept{rotation functions}
