\name{.hpc}
\alias{.hpc}

\title{
High Precision Calibration
}
\description{
Only basic support (not 100\% identical results) for Bruker Daltonics' HPC.
HPC stands for \bold{H}igh \bold{P}recision \bold{C}alibration.\cr
This is an internal function and should normally not used by the user.
}
\usage{
.hpc(mass, minMass, maxMass, hpcCoefficients)
}
\arguments{
    \item{mass}{
        vector of doubles. Mass calculated traditionally.
}
    \item{minMass}{
        double. Lower Threshold for HPC. HPC is only defined for a
        range of mass.
}
    \item{maxMass}{
        double. Upper Threshold for HPC. HPC is only defined for a
        range of mass.
}
    \item{hpcCoefficients}{
        vector of doubles. Coefficients needed by the HPC algorithm.
}
}
\details{
Bruker Daltonics' don't explain how HPC works. All formula a results of
\dQuote{trial and error}. That is why mass calculated by .hpc differs little from
original HPC mass. (In example file 214 of 24860 mass are incorrect;
deviations: min: 6.103515625e-05, max: 0.02935791015625.) \cr
In the manual of mass spectrometry instruments of Bruker Daltonics' flex-series
you can find an article about HPC principles: \cr
Johan Gobom, Martin Mueller, Volker Egelhofer, Dorothea Theiss,
Hans Lehrach, and Eckhard Nordhoff \cr
\dQuote{A Calibration Method That Simplifies and Improves Accurate Determination of
Peptide Molecular mass by MALDI-TOF MS} \cr
Anal Chem. 2002 Aug 1; 74(15): 3915-23 \cr
\url{http://www.ncbi.nlm.nih.gov/pubmed/12175185}
}
\value{
A vector of HPC corrected mass (double). 
}
\references{
Johan Gobom, Martin Mueller, Volker Egelhofer, Dorothea Theiss,
Hans Lehrach, and Eckhard Nordhoff \cr
\dQuote{A Calibration Method That Simplifies and Improves Accurate Determination
of Peptide Molecular mass by MALDI-TOF MS} \cr
Anal Chem. 2002 Aug 1; 74(15): 3915-23 \cr
\url{http://www.ncbi.nlm.nih.gov/pubmed/12175185}
}
\author{
Sebastian Gibb
}
\note{
Please note that .hpc is not correct! You have been warned.
}

\seealso{
    \code{\link[readBrukerFlexData]{readBrukerFlexDir}},
    \code{\link[readBrukerFlexData]{readBrukerFlexFile}},
    \code{\link[readBrukerFlexData]{.double2singlePrecision}}
}
\examples{
## load library
library("readBrukerFlexData");

## get examples directory
exampleDirectory <- system.file("Examples", package="readBrukerFlexData");

## read example spectra
spec <- readBrukerFlexFile(file.path(exampleDirectory, 
    "hpc/fid/0_A20/1/1SRef/fid"));

## plot spectrum 
plot(spec$metaData$backup$mass, spec$spectrum$intensity, type="l", col="red", 
     xlim=c(1296, 1300));
lines(spec$spectrum$mass, spec$spectrum$intensity, type="l", col="green",
      xlim=c(1296, 1300));
legend(x="topright", legend=c("no hpc", "hpc"), col=c("red", "green"), lwd=1);

## show difference between .hpc and original HPC
## load mzXML generated by Bruker Daltonics' compassXport 1.3.5
## you could do it like this:
#library("readMzXmlData");
#cpSpecHpcMzXml <- readMzXmlFile(file.path(exampleDirectory, 
#   "hpc/mzXML/hpc.mzXML"));

## or easily use:
data(cpSpecHpcMzXml);

## reduce R double precision to single precision because our compassXport 1.3.5
## supports only mzXML with precision=32 (only for compatibility reasons)
spec$metaData$backup$mass32 <-
    readBrukerFlexData:::.double2singlePrecision(spec$metaData$backup$mass);
spec$spectrum$mass32 <- readBrukerFlexData:::.double2singlePrecision(spec$spectrum$mass);

## calculate deviance
d <- spec$metaData$backup$mass32-cpSpecHpcMzXml$spectrum$mass;
dHPC <- spec$spectrum$mass32-cpSpecHpcMzXml$spectrum$mass;

## a little summary
cat("without .hpc:\n",
    "not matching: ", length(cpSpecHpcMzXml$spectrum$mass[d!=0]), " of ",
    length(cpSpecHpcMzXml$spectrum$mass), "; range: ", range(abs(d[d!=0])),
    "\nwith .hpc:\n",
    "not matching: ", length(cpSpecHpcMzXml$spectrum$mass[dHPC!=0]), " of ",
    length(cpSpecHpcMzXml$spectrum$mass), "; range: ", range(abs(d[dHPC!=0])),
    "\n");

##
## doing things manually
##
hpcMass <- .hpc(mass=spec$metaData$backup$mass,
                minMass=spec$metaData$hpc$limits["minMass"],
                maxMass=spec$metaData$hpc$limits["maxMass"],
                hpcCoefficients=spec$metaData$hpc$coefficients);
}
\keyword{IO}

