% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rcure.R
\name{rcure}
\alias{rcure}
\title{Robust cure model}
\usage{
rcure(formula, cureform, offset.s = NULL, data, na.action = na.omit,
  model = c("aft", "ph"), link = "logit", Var = TRUE, emmax = 50,
  eps = 1e-07, nboot = 100, family = binomial(link = "logit"),
  method = "glm.fit", prior.mean = 0, prior.scale = NULL, prior.df = 1,
  prior.mean.for.intercept = 0, prior.scale.for.intercept = NULL,
  prior.df.for.intercept = 1, min.prior.scale = 1e-12, scaled = TRUE,
  n.iter = 100, Warning = TRUE, eva_model = NULL, cutpoint = c(0.1,
  0.25, 0.5, 0.75, 0.9))
}
\arguments{
\item{formula}{a formula object for the survival part in cure model. left must be a survival object as returned by the Surv function}

\item{cureform}{specifies the variables in the uncure probability part in cure model}

\item{offset.s}{variable(s) with coefficient 1 in PH model or AFT model}

\item{data}{a a data.frame}

\item{na.action}{a missing-data filter function. By default na.action = na.omit}

\item{model}{specifies survival part in cure model, "ph" or "aft"}

\item{link}{specifies the link in incidence part. The "logit", "probit" or complementary loglog ("cloglog") links are available. By default link = "logit".}

\item{Var}{If it is TRUE, the program returns Std.Error by bootstrap method. If set to False, the program only returns estimators of coefficients. By default, Var = TRUE}

\item{emmax}{specifies the maximum iteration number. If the convergence criterion is not met, the EM iteration will be stopped after emmax iterations and the estimates will be based on the last maximum likelihood iteration. The default emmax = 100.}

\item{eps}{sets the convergence criterion. The default is eps = 1e-7. The iterations are considered to be converged when the maximum relative change in the parameters and likelihood estimates between iterations is less than the value specified.}

\item{nboot}{specifies the number of bootstrap sampling. The default nboot = 100}

\item{family}{a description of the error distribution and link function to be used in the model. Default is binomial(link="logit")}

\item{method}{the method to be used in fitting the glmbayes model. The default method "glm.fit" uses iteratively reweighted least squares (IWLS). The only current alternative is "model.frame" which returns the model frame and does no fitting}

\item{prior.mean}{prior mean for the coefficients: default is 0. Can be a vector of length equal to the number of predictors (not counting the intercept, if any). If it is a scalar, it is expanded to the length of this vector.}

\item{prior.scale}{prior scale for the coefficients: default is NULL; if is NULL, for a logit model, prior.scale is 2.5; for a probit model, prior scale is 2.5*1.6. Can be a vector of length equal to the number of predictors (not counting the intercept, if any). If it is a scalar, it is expanded to the length of this vector}

\item{prior.df}{prior degrees of freedom for the coefficients. For t distribution: default is 1 (Cauchy). Set to Inf to get normal prior distributions. Can be a vector of length equal to the number of predictors (not counting the intercept, if any). If it is a scalar, it is expanded to the length of this vector}

\item{prior.mean.for.intercept}{prior mean for the intercept: default is 0.}

\item{prior.scale.for.intercept}{prior scale for the intercept: default is NULL; for a logit model, prior scale for intercept is 10; for probit model, prior scale for intercept is rescaled as 10*1.6.}

\item{prior.df.for.intercept}{prior degrees of freedom for the intercept: default is 1.}

\item{min.prior.scale}{Minimum prior scale for the coefficients: default is 1e-12.}

\item{scaled}{scaled=TRUE, the scales for the prior distributions of the coefficients are determined as follows: For a predictor with only one value, we just use prior.scale. For a predictor with two values, we use prior.scale/range(x). For a predictor with more than two values, we use prior.scale/(2*sd(x)). If the response is Gaussian, prior.scale is also multiplied by 2 * sd(y). Default is TRUE}

\item{n.iter}{integer giving the maximal number of bayesglm IWLS iterations, default is 100.}

\item{Warning}{default is TRUE, which will show the error messages of not convergence and separation}

\item{eva_model}{for Cox PH model, the default is "PH". For AFT model, it can be "PO".}

\item{cutpoint}{the cut points for ROC to calculate AUC}

\item{...}{further arguments passed to or from other methods}
}
\description{
Fits robust cure model by incorporating a weakly informative prior distribution for uncure probability part in cure models
}
\examples{
library(survival)
library(smcure)
library(arm)
data(e1684)

# fit PH robust cure model
pd <- rcure(Surv(FAILTIME,FAILCENS)~TRT+SEX+AGE,cureform=~TRT+SEX+AGE,
data=e1684,model="ph",Var =FALSE,
method = "glm.fit", prior.mean = 0, prior.scale = NULL, prior.df = 1,
prior.mean.for.intercept = 0, prior.scale.for.intercept = NULL,
prior.df.for.intercept = 1, min.prior.scale = 1e-12,
scaled = FALSE, n.iter = 100, Warning=F)
printrcure(pd,Var = FALSE, ROC=FALSE)
# plot predicted survival curves for male with median centered age by treatment groups
predm=predictrcure(pd,newX=cbind(c(1,0),c(0,0),c(0.579,0.579)),
newZ=cbind(c(1,0),c(0,0),c(0.579,0.579)),model="ph")
plotpredictrcure(predm,model="ph")

# just a test: this should be identical to classical cure model
pd2 <- rcure(Surv(FAILTIME,FAILCENS)~TRT+SEX+AGE,cureform=~TRT+SEX+AGE,
data=e1684,model="ph",Var = FALSE,
method = "glm.fit", prior.mean = 0, prior.scale = Inf, prior.df = Inf,
prior.mean.for.intercept = 0, prior.scale.for.intercept = Inf,
prior.df.for.intercept = Inf, Warning=F)
printrcure(pd2,Var = FALSE, ROC=FALSE)
pd3 <- smcure(Surv(FAILTIME,FAILCENS)~TRT+SEX+AGE,cureform=~TRT+SEX+AGE,
data=e1684,model="ph",Var = FALSE)

data(bmt)
# fit AFT robust cure model
bmtfit <- rcure(formula = Surv(Time, Status) ~ TRT, cureform = ~TRT,
data = bmt, model = "aft", Var = FALSE,
method = "glm.fit", prior.mean = 0, prior.scale = NULL, prior.df = 1,
prior.mean.for.intercept = 0, prior.scale.for.intercept = NULL,
prior.df.for.intercept = 1, min.prior.scale = 1e-12,
scaled = TRUE, n.iter = 100, Warning=F, eva_mode="PO")
printrcure(bmtfit,Var = FALSE, ROC=FALSE)
## plot predicted Survival curves by treatment groups
predbmt=predictrcure(bmtfit,newX=c(0,1),newZ=c(0,1),model="aft")
plotpredictrcure(predbmt,model="aft")
}
\author{
Xiaoxia Han
}
\references{
Cai, C., Zou, Y., Peng, Y., & Zhang, J. (2012). smcure: An R-Package for estimating semiparametric mixture cure models. Computer methods and programs in biomedicine, 108(3), 1255-1260.

Gelman, A., Jakulin, A., Pittau, M. G., & Su, Y. S. (2008). A weakly informative default prior distribution for logistic and other regression models. The Annals of Applied Statistics, 1360-1383.
}

