% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_MDP.R
\name{solve_MDP}
\alias{solve_MDP}
\alias{q_values_MDP}
\alias{random_MDP_policy}
\alias{approx_MDP_policy_evaluation}
\title{Solve an MDP Problem}
\usage{
solve_MDP(
  model,
  horizon = NULL,
  discount = NULL,
  terminal_values = NULL,
  method = "value",
  eps = 0.01,
  max_iterations = 1000,
  k_backups = 10,
  verbose = FALSE
)

q_values_MDP(model, U = NULL)

random_MDP_policy(model, prob = NULL)

approx_MDP_policy_evaluation(pi, model, U = NULL, k_backups = 10)
}
\arguments{
\item{model}{a POMDP problem specification created with \code{\link[=POMDP]{POMDP()}}.
Alternatively, a POMDP file or the URL for a POMDP file can be specified.}

\item{horizon}{an integer with the number of epochs for problems with a
finite planning horizon. If set to \code{Inf}, the algorithm continues
running iterations till it converges to the infinite horizon solution. If
\code{NULL}, then the horizon specified in \code{model} will be used.  For
time-dependent POMDPs a vector of horizons can be specified (see Details
section).}

\item{discount}{discount factor in range \eqn{[0, 1]}. If \code{NULL}, then the
discount factor specified in \code{model} will be used.}

\item{terminal_values}{a vector with terminal utilities for each state. If
\code{NULL}, then a vector of all 0s is used.}

\item{method}{string; one of the following solution methods: \code{'value'},
\code{'policy'}.}

\item{eps}{maximum error allowed in the utility of any state
(i.e., the maximum policy loss).}

\item{max_iterations}{maximum number of iterations allowed to converge. If the
maximum is reached then the non-converged solution is returned with a
warning.}

\item{k_backups}{number of look ahead steps used for approximate policy evaluation
used by method \code{'policy'}.}

\item{verbose}{logical, if set to \code{TRUE}, the function provides the
output of the pomdp solver in the R console.}

\item{U}{a vector with state utilities (expected sum of discounted rewards from that point on).}

\item{prob}{probability vector for actions.}

\item{pi}{a policy as a data.frame with columns state and action.}
}
\value{
\code{solve_MDP()} returns an object of class POMDP which is a list with the
model specifications (\code{model}), the solution (\code{solution}).
The solution is a list with the elements:
\itemize{
\item \code{policy} a list representing the policy graph. The list only has one element for converged solutions.
\item \code{converged} did the algorithm converge (\code{NA}) for finite-horizon problems.
\item \code{delta} final delta (infinite-horizon only)
\item \code{iterations} number of iterations to convergence (infinite-horizon only)
}

\code{q_values_MDP()} returns a state by action matrix specifying the Q-function,
i.e., the utility value of executing each action in each state.

\code{random_MDP_policy()} returns a data.frame with columns state and action to define a policy.

\code{approx_MDP_policy_evaluation()} is used by the modified policy
iteration algorithm and returns an approximate utility vector U estimated by evaluating policy \code{pi}.
}
\description{
A simple implementation of value iteration and modified policy iteration.
}
\examples{
data(Maze)
Maze

# use value iteration
maze_solved <- solve_MDP(Maze, method = "value")
policy(maze_solved)

# value function (utility function U)
plot_value_function(maze_solved)

# Q-function (states times action)
q_values_MDP(maze_solved)

# use modified policy iteration
maze_solved <- solve_MDP(Maze, method = "policy")
policy(maze_solved)

# finite horizon
maze_solved <- solve_MDP(Maze, method = "value", horizon = 3)
policy(maze_solved)

# create a random policy where action n is very likely and approximate 
#  the value function. We change the discount factor to .9 for this.
Maze_discounted <- Maze
Maze_discounted$discount <- .9
pi <- random_MDP_policy(Maze_discounted, prob = c(n = .7, e = .1, s = .1, w = 0.1))
pi

# compare the utility function for the random policy with the function for the optimal 
#  policy found by the solver.
maze_solved <- solve_MDP(Maze)

approx_MDP_policy_evaluation(pi, Maze, k_backup = 100)
approx_MDP_policy_evaluation(policy(maze_solved)[[1]], Maze, k_backup = 100)

# Note that the solver already calculates the utility function and returns it with the policy
policy(maze_solved)
}
\seealso{
Other solver: 
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()}
}
\author{
Michael Hahsler
}
\concept{MDP}
\concept{solver}
