#' new LazyFrame from csv
#' @description will scan the csv when collect(), not now.
#' read_csv is a shortcut for scan_csv$collect().
#' @name scan_csv
#' @details  Read a file from path into a polars lazy frame. Not yet supporting eol_char and with_column_names
#' @keywords LazyFrame_new
#'
#' @param path string, Path to a file or URL
#' @param sep Single char to use as delimiter in the file.
#' @param has_header bool, indicate if the first row of dataset is a header or not.
#'  If set to False, column names will be autogenerated in the following format:
#'  column_x, with x being an enumeration over every column in the dataset starting at 1.
#' @param ignore_errors bool, try to keep reading lines if some lines yield errors. First try
#' infer_schema_length=0 to read all columns as pl.Utf8 to check which values might cause
#' an issue.
#' @param skip_rows integer, Start reading after skip_rows lines. The header will be parsed
#' at this offset.
#' @param n_rows int (NULL is disable),Stop reading from CSV file after reading n_rows.
#' @param cache bool, cache the result after reading.
#' @param overwrite_dtype (NULL is disable) named list of dtypes OR dtype-names, where name points
#' to a column. Can overwrite dtypes during inference.
#' Supported types so far are:
#'  name | alias | polars side dtype
#' "Boolean" | "logical" => DataType::Boolean,
#' "Float32" | "double" => DataType::Float32,
#' "Float64" | "float64" => DataType::Float64,
#' "Int32" | "integer" => DataType::Int32,
#' "Int64" | "integer64" => DataType::Int64,
#' "Utf8" | "character" => DataType::Utf8,
#'
#' @param low_memory bool, reduce memory usage in expense of performance
#' @param comment_char (NULL is disable) Single byte character that indicates the start of a comment line,
#' for instance #.
#' @param quote_char (NULL is disable) Single byte character used for csv quoting, default = ". Set to None
#' to turn off special handling and escaping of quotes.
#' @param null_values  (NULL is disable) Values to interpret as null values. You can provide a
#' String : All values equal to this string will be null.
#' Unnamed char vector: A null value per column.
#' Named char vector.  A mapping from (names)column to a null value string(values).
#'
#' @param infer_schema_length Maximum number of rows to read to infer the column
#'   types. If set to 0, all columns will be read as UTF-8. If `NULL`, a full
#'   table scan will be done (slow).
#' @param skip_rows_after_header bool Skip this number of rows when the header is parsed.
#' @param encoding either "utf8" or "utf8-lossy". Lossy means that invalid utf8 values are replaced with "?" characters.
#' @param row_count_name String(NULL is disable), name of a added row count column
#' @param row_count_offset integer, Offset to start the row_count column (only used if the name is set).
#' @param parse_dates bool Try to automatically parse dates. If this does not succeed, the column remains of data type pl.Utf8.
#' @param reuse_downloaded Boolean. If `TRUE`(default) and a URL was provided,
#' cache the downloaded files in session for an easy reuse.
#' @return scan_csv returns a LazyFrame. read_csv returns a DataFrame.
#' @examples
#' my_file = tempfile()
#' write.csv(iris, my_file)
#' lazy_frame = pl$scan_csv(path = my_file)
#' lazy_frame$collect()
#' unlink(my_file)
pl$scan_csv = function(
    path,
    sep = ",",
    has_header = TRUE,
    ignore_errors = FALSE,
    skip_rows = 0,
    n_rows = NULL,
    cache = FALSE,
    overwrite_dtype = NULL,
    low_memory = FALSE,
    comment_char = NULL,
    quote_char = '"',
    null_values = NULL,
    infer_schema_length = 100,
    skip_rows_after_header = 0,
    encoding = "utf8",
    row_count_name = NULL,
    row_count_offset = 0,
    parse_dates = FALSE,
    reuse_downloaded = TRUE) {
  # capture all args and modify some to match lower level function
  args = as.list(environment())
  args[["reuse_downloaded"]] = NULL

  path = check_is_link(path, reuse_downloaded = reuse_downloaded)
  args[["path"]] = path.expand(path)

  # overwrite_dtype: convert named list of DataType's to DataTypeVector obj
  if (!is.null(args$overwrite_dtype)) {
    owdtype = args$overwrite_dtype

    if (!is.list(owdtype) || !is_named(owdtype)) {
      stopf("could not interpret overwrite_dtype, must be a named list of DataTypes")
    }
    datatype_vector = DataTypeVector$new() # mutable
    mapply(
      name = names(owdtype),
      type = unname(owdtype),
      FUN = function(name, type) {
        # convert possible string to datatype
        if (is_string(type)) {
          type = DataType$new(type)
        }
        if (!inherits(type, "RPolarsDataType")) {
          stopf("arg overwrite_dtype must be a named list of dtypes or dtype names")
        }
        datatype_vector$push(name, type)
      }
    )
    args$overwrite_dtype = datatype_vector
  }


  # null_values: convert string or un/named  char vec into RNullValues obj
  if (!is.null(args$null_values)) {
    nullvals = args$null_values
    ## TODO support also unnamed list, like will be interpreted as positional dtypes args by polars.
    RNullValues = (function() {
      # one string is used as one NULL marker for all columns
      if (is_string(nullvals)) {
        return(RNullValues$new_all_columns(nullvals))
      }

      # many unnamed strings(char vec) is used one mark for each column
      if (is.character(nullvals) && !is_named(nullvals)) {
        return(RNullValues = RNullValues$new_columns(nullvals))
      }

      # named char vec is used as column(name) marker(value) pairs
      if (is.character(nullvals) && is_named(nullvals)) {
        return(RNullValues$new_named(null_values))
      }

      stopf("null_values arg must be a string OR unamed char vec OR named char vec")
    })()

    args$null_values = RNullValues
  }

  ## call low level function with args
  check_no_missing_args(rlazy_csv_reader, args)
  unwrap(do.call(rlazy_csv_reader, args))
}

#' Shortcut for pl$scan_csv()$collect()
#' @rdname scan_csv
#' @return DataFrame
#' @name read_csv
pl$read_csv = function(
    path,
    sep = ",",
    has_header = TRUE,
    ignore_errors = FALSE,
    skip_rows = 0,
    n_rows = NULL,
    cache = FALSE,
    overwrite_dtype = NULL,
    low_memory = FALSE,
    comment_char = NULL,
    quote_char = '"',
    null_values = NULL,
    infer_schema_length = 100,
    skip_rows_after_header = 0,
    encoding = "utf8",
    row_count_name = NULL,
    row_count_offset = 0,
    parse_dates = FALSE,
    reuse_downloaded = TRUE) {
  mc = match.call()
  mc[[1]] = quote(pl$scan_csv)
  eval.parent(mc)$collect()
}


check_is_link = function(path, reuse_downloaded) {
  if (!file.exists(path)) {
    con = NULL

    # check if possible to open url connection
    assumed_schemas = c("", "https://", "http://", "ftp://")
    for (i_schema in assumed_schemas) {
      if (!is.null(con)) break
      actual_url = paste0(i_schema, path)
      suppressWarnings(
        tryCatch(
          {
            con = url(actual_url, open = "rt")
          },
          error = function(e) {}
        )
      )
    }

    # try download file if valid url
    if (!is.null(con)) {
      close(con)
      tmp_file = paste0(tempdir(), "/", make.names(actual_url))
      if (isFALSE(reuse_downloaded) || isFALSE(file.exists(tmp_file))) {
        download.file(url = actual_url, destfile = tmp_file)
        message(paste("tmp file placed in \n", tmp_file))
      }

      path = tmp_file # redirect path to tmp downloaded file
    } else {
      # do nothing let path fail on rust side
      path = NULL
    }
  }

  path
}
