#
#                _         _    _      _                _    
#               (_)       | |  | |    | |              | |   
#  _ __    ___   _  _ __  | |_ | |__  | |  __ _  _ __  | | __
# | '_ \  / _ \ | || '_ \ | __|| '_ \ | | / _` || '_ \ | |/ /
# | |_) || (_) || || | | || |_ | |_) || || (_| || | | ||   < 
# | .__/  \___/ |_||_| |_| \__||_.__/ |_| \__,_||_| |_||_|\_\
# | |                                                        
# |_|                                                        
# 
# This file is part of the 'rich-iannone/pointblank' package.
# 
# (c) Richard Iannone <riannone@me.com>
# 
# For full copyright and license information, please look at
# https://rich-iannone.github.io/pointblank/LICENSE.html
#


#' Read a **pointblank** YAML file to create an *informant* object
#'
#' With `yaml_read_informant()` we can read a **pointblank** YAML file that
#' describes table information (typically generated by the [yaml_write()]
#' function. What's returned is a new *informant* object with the information
#' intact. The *informant* object can be given more information through use of
#' the `info_*()` functions.
#' 
#' @param path A path to a **pointblank** YAML file that contains fields related
#'   to an *informant*.
#' 
#' @examples 
#' # Create a pointblank `informant`
#' # object with `create_informant()`
#' # and the `small_table` dataset
#' informant <- create_informant(small_table)
#' 
#' # An `informant` object can be written
#' # to a YAML file with the `yaml_write()`
#' # function
#' # yaml_write(
#' #   informant = informant,
#' #   filename = "informant-small_table.yml"
#' # )
#' 
#' # The `informant-small_table.yml` file
#' # looks like this when written
#' 
#' #> info_label: '[2020-09-06|13:37:38]'
#' #> table:
#' #>   name: small_table
#' #> _columns: 8
#' #> _rows: 13
#' #> _type: tbl_df
#' #> columns:
#' #>   date_time:
#' #>     _type: POSIXct, POSIXt
#' #>   date:
#' #>     _type: Date
#' #>   a:
#' #>     _type: integer
#' #>   b:
#' #>     _type: character
#' #>   c:
#' #>     _type: numeric
#' #>   d:
#' #>     _type: numeric
#' #>   e:
#' #>     _type: logical
#' #>   f:
#' #>     _type: character
#' 
#' # We can add keys and values to
#' # add more pertinent information; with
#' # some direct editing of the file we get:
#' 
#' #> info_label: '[2020-09-06|13:37:38]'
#' #> table:
#' #>   name: small_table
#' #>   _columns: 8
#' #>   _rows: 13
#' #>   _type: tbl_df
#' #> columns:
#' #>   date_time:
#' #>     _type: POSIXct, POSIXt
#' #>     info: Date-time values.
#' #>   date:
#' #>     _type: Date
#' #>     info: Date values (the date part of `date_time`).
#' #>   a:
#' #>     _type: integer
#' #>     info: Small integer values (no missing values).
#' #>   b:
#' #>     _type: character
#' #>     info: Strings with a common pattern.
#' #>   c:
#' #>     _type: numeric
#' #>     info: Small numeric values (contains missing values).
#' #>   d:
#' #>     _type: numeric
#' #>     info: Large numeric values (much greater than `c`).
#' #>   e:
#' #>     _type: logical
#' #>     info: TRUE and FALSE values.
#' #>   f:
#' #>     _type: character
#' #>     info: Strings of the set `"low"`, `"mid"`, and `"high"`.
#' 
#' # We could also have done the same
#' # with the `informant` object by use of
#' # the `info_columns()` function
#' 
#' # The 'informant-small_table.yml' file
#' # is available in the package through
#' # `system.file()`
#' yml_file <- 
#'   system.file(
#'     "informant-small_table.yml",
#'     package = "pointblank"
#'   )
#' 
#' # We can read this YAML file back
#' # as an `informant` object by using
#' # `yaml_read_informant()`
#' informant <- 
#'   yaml_read_informant(path = yml_file)
#' 
#' class(informant)
#' 
#' @family pointblank YAML
#' @section Function ID:
#' 9-3
#' 
#' @export
yaml_read_informant <- function(path) {

  informant_list <- 
    expr_from_informant_yaml(path = path, incorporate = FALSE)

  informant <- 
    informant_list$expr_str %>%
    rlang::parse_expr() %>%
    rlang::eval_tidy()
  
  informant$metadata <- informant_list$metadata
  informant
}

#' Get an *informant* from **pointblank** YAML and `incorporate()`
#'
#' The `yaml_informant_incorporate()` function operates much like the
#' [yaml_read_informant()] function (reading a **pointblank** YAML file and
#' generating an *informant* with all information in place). The key difference
#' is that this function takes things a step further and incorporates aspects
#' from the the target table (defined by table-reading, `read_fn`, function that
#' is required in the YAML file). The additional auto-invocation of
#' [incorporate()] uses the default options of that function. As with
#' [yaml_read_informant()] the informant is returned except, this time, it has
#' been updated with the latest information from the target table.
#'
#' @param path A path to a YAML file that specifies a information for an
#'   *informant*.
#'
#' @examples
#' # Let's go through the process of
#' # developing an informant with information
#' # about the `small_table` dataset and then
#' # move all that to a pointblank YAML
#' # file; this will later be read in as a
#' # new informant and the target data will
#' # be incorporated into the info text
#' # (in one step) with
#' # `yaml_informant_incorporate()`
#' 
#' # Now create a pointblank `informant`
#' # object; the data will be referenced
#' # in a `read_fn` (a requirement for
#' # writing to YAML)
#' informant <- 
#'   create_informant(
#'     read_fn = ~small_table,
#'     label = "A simple example with the `small_table`."
#'   )
#' 
#' # Then, as with any `informant` object, we
#' # can add information by using as many
#' # `info_*()` functions as we want
#' informant <-
#'   informant %>%
#'   info_columns(
#'    columns = vars(a),
#'    info = "In the range of 1 to 10. (SIMPLE)"
#'   ) %>%
#'   info_columns(
#'     columns = starts_with("date"),
#'     info = "Time-based values (e.g., `Sys.time()`)."
#'   ) %>%
#'   info_columns(
#'     columns = "date",
#'     info = "The date part of `date_time`. (CALC)"
#'   ) %>%
#'   info_section(
#'     section_name = "rows",
#'     row_count = "There are {row_count} rows available."
#'   ) %>%
#'   info_snippet(
#'     snippet_name = "row_count",
#'     fn = ~ . %>% nrow()
#'   ) %>%
#'   incorporate()
#'
#' # The informant can be written to a pointblank
#' # YAML file with `yaml_write()`
#' # yaml_write(
#' #   informant = informant,
#' #   filename = "informant-small_table.yml"
#' # )
#' 
#' # The 'informant-small_table.yml' file
#' # is available in the package through
#' # `system.file()`
#' yml_file <- 
#'   system.file(
#'     "informant-small_table.yml",
#'     package = "pointblank"
#'   )
#' 
#' # We can incorporate the data (which
#' # is accessible through the `read_fn`)
#' # into the info text through direct
#' # use of the YAML file with
#' # `yaml_informant_incorporate()`
#' informant <- 
#'   yaml_informant_incorporate(path = yml_file)
#' 
#' class(informant)
#'
#' # If it's desired to only create a new
#' # informant with the information in place
#' # (stopping short of processing), then the
#' # `yaml_read_informant()` function will
#' # be useful
#' informant <- 
#'   yaml_read_informant(path = yml_file)
#' 
#' class(informant)
#'
#' @family pointblank YAML
#' @section Function ID:
#' 9-7
#'
#' @export
yaml_informant_incorporate <- function(path) {
  
  informant_list <- 
    expr_from_informant_yaml(path = path)
  
  informant <- 
    informant_list$expr_str %>%
    rlang::parse_expr() %>%
    rlang::eval_tidy()
  
  informant$metadata <- informant_list$metadata
  
  informant <- informant %>% incorporate()
  informant
}

expr_from_informant_yaml <- function(path,
                                     incorporate = FALSE) {
  
  # Read the YAML file with `yaml::read_yaml()`
  y <- yaml::read_yaml(file = path)
  
  # Perform checks on elements of `y`
  check_info_yaml_table(y)
  check_info_yaml_columns(y)
  check_info_yaml_others(y)

  if ("read_fn" %in% names(y)) {
    read_fn <- paste0("  read_fn = ", y$read_fn)
  } else {
    read_fn <- NULL
  }
  
  if (!is.null(y$table$name)) {
    tbl_name <- paste0("  tbl_name = \"", y$table$name, "\"")
  } else {
    tbl_name <- NULL
  }
  
  if (!is.null(y$info_label)) {
    label <- paste0("  label = \"", y$info_label, "\"")
  } else {
    label <- NULL
  }
  
  if (!is.null(y$lang) && y$lang != "en") {
    lang <- paste0("  lang = \"", y$lang, "\"")
  } else {
    lang <- NULL
  }
  
  if (!is.null(y$locale) && y$locale != "en") {
    locale <- paste0("  locale = \"", y$locale, "\"")
  } else {
    locale <- NULL
  }
  
  # Generate `info_snippet()` expressions
  info_snippets <- make_info_snippets(y$meta_snippets)
  
  # Generate the expression string
  expr_str <-
    paste0(
      "create_informant(\n",
      paste(
        c(
          read_fn, tbl_name, label, 
          lang, locale
        ),
        collapse = ",\n"
      ),
      "\n) ",
      info_snippets
    )
  
  # Add the `incorporate()` statement if needed (this is
  # for the `yaml_informant_incorporate()` function)
  if (incorporate) {
    expr_str <- paste0(expr_str, "%>%\nincorporate()")
  }
  
  y$read_fn <- NULL
  y$lang <- NULL
  y$locale <- NULL
  y$meta_snippets <- NULL

  list(
    expr_str = expr_str,
    metadata = y
  )
}

check_info_yaml_table <- function(y) {
  
  # If `table` is present, perform a few validations on that component
  if ("table" %in% names(y)) {
    
    # Validate that 2nd-level elements have unique names
    if (any(duplicated(names(y[["table"]])))) {
      stop("Duplicate column names provided in `table`.", call. = FALSE)
    }
    
    # Get component names of `table`
    table_names <- names(y[["table"]])
    
    # Validate that there are only character vectors inside `table`
    checks <- 
      lapply(
        table_names,
        FUN = function(x) {
          x_names <- names(y[["table"]][x])
          
          for (z in x_names) {
            if (is.list(y[["table"]][[z]])) {
              stop("All subcomponents inside of `table` should be a ",
                   "character vector.",
                   call. = FALSE)
            }
          }
        }
      )
  }
}

check_info_yaml_columns <- function(y) {
  
  # If `columns` is present, perform a few validations on that component
  if ("columns" %in% names(y)) {
    
    # Validate that 2nd-level elements have unique names
    if (any(duplicated(names(y[["columns"]])))) {
      stop("Duplicate column names provided in `columns`.", call. = FALSE)
    }
    
    # Get listed column names
    column_names <- names(y[["columns"]])
    
    # Validate that there is no more than only a single level below
    # the column names
    checks <- 
      lapply(
        column_names,
        FUN = function(x) {
          x_names <- names(y[["columns"]][x])
          
          for (z in x_names) {
            
            if (is.list(y[["columns"]][[z]])) {
              
              components_are_char <-
                unname(unlist(lapply(y[["columns"]][[z]], is.character)))
              
              if (!all(components_are_char)) {
                stop("All components inside of `columns/", z,
                     "` should either be text or text under a single heading.",
                     call. = FALSE)
              }
            }
          }
        }
      )
  }
}

check_info_yaml_others <- function(y) {
  
  # If any other items are present, perform a few validations on those
  exclusions <- c("table", "columns", "actions", "steps")
  other_names <- base::setdiff(names(y), exclusions)
  
  if (length(other_names) > 0) {
    
    # Validate that there is no more than only a single level below
    # the column names
    checks <- 
      lapply(
        other_names,
        FUN = function(x) {
          
          if (is.list(y[[x]])) {
            
            if (any(unname(unlist(lapply(y[[x]], Negate(is.character)))))) {
              
              idx <- which(unname(unlist(lapply(y[[x]], Negate(is.character)))))
              
              stop("All components inside `", x, "/", names(y[[x]][idx]),
                   "` should be a character vector.",
                   call. = FALSE)
            }
            
          } else if (!is.list(y[[x]])) {
            if (!is.character(y[[x]])) {
              stop("The component inside `", x,
                   "` should be a character vector.",
                   call. = FALSE)
            }
          }
        }
      )
  }
}

make_info_snippets <- function(snippets) {
  
  if (length(snippets) == 0) return("")
  
  str_exprs <- 
    vapply(
      seq_along(snippets),
      FUN.VALUE = character(1),
      USE.NAMES = FALSE,
      FUN = function(x) {
        
        snippet_name <- names(snippets[x])
        snippet_fun <- snippets[[x]]
        
        paste0(
          "%>% info_snippet(",
          "snippet_name = \"", snippet_name, "\", ",
          "fn = ", snippet_fun, ")"
        )
      }
    )
  
  paste(str_exprs, collapse = " ")
}
