parser <- function( file, encoding = "unknown", text ){
	if( !missing( text ) ){
		file <- tempfile( );
		cat( text , file = file, sep = "\n" )
	}
	p <- .External( "do_parser", file = file, encoding = encoding )
	data <- as.data.frame( t(attr(p,"data")) )
	colnames( data ) <- c( "line1", "col1", "byte1", 
		 	"line2", "col2", "byte2", "token", "id", "parent" )
	m <- match( data$token, grammar_symbols$token )
	data$token.desc <- as.character(grammar_symbols$desc)[ m ]
	data$terminal <- grammar_symbols$terminal[m]
	data$text     <- rep( "", nrow(data) )
	toks <- getTokens( data= subset(data, terminal), 
		encoding = encoding, file = file, 
		sort = FALSE )
	data$text[ match( toks$id, data$id) ] <- toks$text
	attr( p, "data" ) <- data
	attr( p, "file" ) <- file
	attr( p, "encoding") <- encoding
	oldClass( p ) <- "parser"
	p
}

getChilds <- function( x, i = 0, 
	parent = sapply( x[i], function(.) attr(.,"id" ) ) ){
	
	if(missing(parent) && ( missing(i) || is.null(i) || i==0 ) ){
		return( attr(x, "data")[,'id'] )
	}
	all.childs <- c()
	data    <- attr( x, "data" )
	parents <- abs( data[, "parent"] )
	id      <- data[, "id" ]
	childs  <- function( index ){
		kids <- id[ parents %in% index ]
		if( length(kids) ){
			all.childs <<- c( all.childs, kids )
			childs( kids )
		}
	}
	childs( parent )
	sort( all.childs )
}


#' Gets the terminal tokens
getTokens <- function( x, 
	data = subset( attr( x, "data" ), terminal ), 
	encoding = attr( x, "encoding"), 
	file = attr( x, "file" ), 
	sort = TRUE ){
	
	if( sort ){
		data <- data[ do.call( order, data[, c("line1", "col1") ] ), ]
	} else{
		if( is.unsorted(data[["line1"]] ) ){
			stop( "data is not in increasing order of line1" )
		}	
	}
	data.frame( id = data[, "id"], 
		text = .External( "do_getTokens", 
			file = file, 
			encoding = encoding, 
			line1 = data[, "line1" ], 
			col1  = data[, "byte1"  ], 
			line2 = data[, "line2" ], 
			col2  = data[, "byte2"  ] ), 
		stringsAsFactors = FALSE
		)
	
}


#' parses the grammar.output file that is generated by bison
grammar.symbols <- function(  ){
	# the output file from bison
	gram.output.file <- system.file( "grammar", "gram.output", package = "parser" )
	rl <- readLines( gram.output.file ) 
	
	.extract <- function( start.rx, end.rx, type = "terminal" ){
		start <- grep( start.rx, rl )[1] + 1L
		end <- grep( end.rx, rl)[1] - 1L
		
		rl <- rl[ start:end ]
		rx <- "(^.*) \\((\\d+)\\).*"
		rl <- grep( rx, rl, perl = T, value = T )
		desc   <- gsub( rx, "\\1", rl, perl = TRUE )
		token  <- as.integer( gsub( rx, "\\2", rl, perl = TRUE ) )
		data.frame( desc = desc, token = token, 
			terminal = rep.int( type , length(token) ), 
			stringsAsFactors = FALSE )
	}
	rbind( 
		.extract( 
			"^Terminals, with rules where they appear",
			"^Nonterminals, with rules where they appear", 
			TRUE ), 
		.extract( 
			"^Nonterminals, with rules where they appear",
			"^state 0", 
			FALSE )
		)
}

#' counts the number of bytes and columns in each line of the file
#'
#' @param file file to analyze
#' @param encoding encoding to assume for the file
count.chars <- function( file, encoding = "unknown" ){
	out <- .External( "do_countchars", file = file, encoding = encoding )
	dimnames(out) <- list( 1:nrow(out), c("char", "byte") )
	out
}

#' counts the number of lines of a file
#' 
#' @param file file from which to count lines
nlines <- function( file ){
	.External( "do_nlines", file = file )
}

